// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

extern "C" {
#include <ctype.h>
#include <stdlib.h>
}
#include <qlist.h>
#include "Options.h"
#include "Database.h"

//------------------------------------------------------------------------------
// Search the Database for the CompactDisc with a CompactDiscID matching id.
// If an entry is found in the database, create a new CompactDisc from the
// information in the Database.  If no matching entry is found, create a new
// default CompactDisc.  Return the resulting CompactDisc.
//------------------------------------------------------------------------------
CompactDisc* Database::search( const CompactDiscID& id ) const
{
    CompactDisc* cd = 0;
    
    // Open the database.
    FILE* fptr = fopen( options->dbasePath(), "r" );

    // Scan each entry for a match.
    if ( fptr ) {
        while ( !feof( fptr ) ) {
            cd = nextEntry( fptr );
            if (cd) {
                if ( cd->isEqual( id ) ) {
                    fclose( fptr );
                    return cd;
                }
                delete cd;
            }
        }
        fclose(fptr);
    }

    // Create a default CompactDisc.
    QString title = "No Title";
    QString artist = "No Artist";
    QString* tracks = new QString[id.numTracks()];
    for (int i = 0; i < id.numTracks(); i++) {
        tracks[i].sprintf("Track %02d", i+1);
    }
    cd = new CompactDisc(id, title, artist, tracks, 0, 0);
    delete [] tracks;

    return cd;
}

//------------------------------------------------------------------------------
// If the CompactDisc cd is already in the Database, replace the existing entry.
// If cd is not in the Database, create a new entry in the Database for it.
//------------------------------------------------------------------------------
void Database::update(CompactDisc& cd)
{
    // Read in the current Database.
    QList<CompactDisc>  db;
    db.setAutoDelete(1);
    CompactDisc* disc;
    FILE* fptr = fopen(options->dbasePath(), "r");

    if ( fptr ) {
        while (!feof(fptr)) {
            disc = nextEntry(fptr);
            if (disc) {
                if (!disc->isEqual(cd)) {
                    db.append(disc);
                } else {
                    delete disc;
                }
            }
        }
        fclose(fptr);
    }

    // Write out the database.
    fptr = fopen(options->dbasePath(), "w");
    if ( fptr ) {
        while ( ( disc = db.getFirst() ) ) {
            writeEntry(fptr, *disc);
            db.remove(disc);
        }
        
        // Write out the new entry.
        writeEntry(fptr, cd);
        fclose(fptr);
    } else {
        debug( "Database::update() -- error, could not open '%s' for writing", options->dbasePath() );
    }
}

QString Database::getToken( FILE* fp ) const
{
	QString tok;
	char ch = 0;
	
	while ( !feof( fp ) && isspace( ch = getc( fp ) ) );
	tok += ch;
	while ( !feof( fp ) && !isspace( ch = getc( fp ) ) ) {
		tok += ch;
	}
	return tok;
}

QString Database::getToEOL( FILE* fp ) const
{
	QString tok;
	char ch = 0;
	
	while ( !feof( fp ) && isspace( ch = getc( fp ) ) );
	tok += ch;
	while ( !feof( fp ) && ( ( ch = getc( fp ) ) != '\n' ) ) {
		tok += ch;
	}
	return tok;
}

//------------------------------------------------------------------------------
// Read in the next entry from the Database, and return the resulting
// CompactDisc.
//------------------------------------------------------------------------------
CompactDisc* Database::nextEntry(FILE* fptr) const
{
    CompactDisc*    cd;
    CompactDiscID*  cid;
    int             numTracks;
    int*            trackStart;
    QString         title;
    QString         artist;
    QString*        tracks;
    int             progLength;
    int*            progTracks;
    QString         token;
    int             i;

    // Read number of tracks.
	if ( getToken( fptr ) != "tracks" ) return 0;
	numTracks = atoi( getToken( fptr ) );
	if ( numTracks <= 0 ) return 0;

    // Read track start times.
    trackStart = new int[numTracks+1];
    for (i = 0; i < numTracks; i++) {
		trackStart[i] = atoi( getToken( fptr ) );
    }

    // Read leadout track start time.
	trackStart[numTracks] = atoi( getToken( fptr ) ) * XcdFrames;
    if ( trackStart[numTracks] <= 0 ) {
		delete [] trackStart;
		return 0;
	}

    cid = new CompactDiscID(numTracks, 1, trackStart);
    delete [] trackStart;
	
	// Read disc program.
	token = getToken( fptr );
	if ( token == "program" ) {
		progLength = atoi( getToken( fptr ) );
		progTracks = new int[progLength];
		for ( i = 0; i < progLength; i++ ) {
			progTracks[i] = atoi( getToken( fptr ) );
		}
		token = getToken( fptr );
	} else {
		progLength = 0;
		progTracks = 0;
	}

    // Read disc title.
	if ( token != "cdname" ) {
		delete cid;
		return 0;
	}
	title = getToEOL( fptr );
            
    // Read disc artist.
	if ( getToken( fptr ) != "artist" ) {
		delete cid;
		return 0;
	}
	artist = getToEOL( fptr );

    // Read track names.
    tracks = new QString[numTracks];
    for (i = 0; i < numTracks; i++) {
		if ( getToken( fptr ) != "track" ) {
			delete cid;
			delete [] tracks;
			return 0;
		}
		tracks[i] = getToEOL( fptr );
    }

    // Create the CompactDisc.
    cd = new CompactDisc( *cid, title, artist, tracks, progLength, progTracks );

    delete [] tracks;
	delete [] progTracks;
    delete cid;
    return cd;
}

//------------------------------------------------------------------------------
// Write out an entry for the given CompactDisc.
//------------------------------------------------------------------------------
void Database::writeEntry(FILE* fptr, CompactDisc& cd)
{
    int i;
    int numTracks = cd.id().numTracks();

    fprintf( fptr, "tracks %d", numTracks );
    for ( i = 0; i < numTracks; i++ ) {
        fprintf( fptr, " %d", cd.id().trackStart(i) );
    }
    fprintf( fptr, " %d\n", cd.id().trackStart( numTracks ) / XcdFrames );
	
	fprintf( fptr, "program %d", cd.program()->ordered().count() );
	for ( int* ip = cd.program()->ordered().first(); ip; ip = cd.program()->ordered().next() ) {
		fprintf( fptr, " %d", *ip );
	}
	fprintf( fptr, "\n" );

    fprintf( fptr, "cdname %s\n", cd.title() );
    fprintf( fptr, "artist %s\n", cd.artist() );
    for ( i = 0; i < numTracks; i++ ) {
        fprintf( fptr, "track %s\n", cd.track( i ) );
    }
}
