// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <qkeycode.h>
#include "InfoWindow.h"

//------------------------------------------------------------------------------
// Emit special signals when the Up and Down arrow keys are pressed.  For all
// other key press events, use our parents event handler.
//------------------------------------------------------------------------------
void MyLineEdit::keyPressEvent(QKeyEvent* e)
{
    switch (e->key()) {
        case Key_Up:
            emit upPressed();
            break;

        case Key_Down:
            emit downPressed();
            break;

        default:
            QLineEdit::keyPressEvent(e);
            break;
    }
}

//------------------------------------------------------------------------------
// Create the PrefsWindow.
//------------------------------------------------------------------------------
InfoWindow::InfoWindow( QWidget* parent, const char* name ) :
        QWidget( parent, name )
{
    setCaption( "XCd: CD Info" );

    _discTitleLabel = new QLabel( "Disc Title", this );
	_discTitleLabel->setAlignment(AlignRight | AlignVCenter);

    _discTitle = new QLineEdit( this );
    connect( _discTitle, SIGNAL(returnPressed()), this, SLOT(procDiscTitle()) );

    _discArtistLabel = new QLabel( "Disc Artist", this );
	_discArtistLabel->setAlignment(AlignRight | AlignVCenter);

    _discArtist = new QLineEdit( this );
    connect( _discArtist, SIGNAL(returnPressed()), this, SLOT(procDiscArtist()) );

    _trackTitleLabel = new QLabel( "Track Title", this );
	_trackTitleLabel->setAlignment(AlignRight | AlignVCenter);

    _trackTitle = new MyLineEdit( this );
    connect( _trackTitle, SIGNAL(returnPressed()), this, SLOT(procTrackTitle()) );
    connect( _trackTitle, SIGNAL(upPressed()), this, SLOT(procUpPressed()) );
    connect( _trackTitle, SIGNAL(downPressed()), this, SLOT(procDownPressed()) );

	_trackListLabel = new QLabel( "All Tracks", this );
	_trackListLabel->setAlignment( AlignCenter );
	
    _trackList = new MyListBox( this );
    connect( _trackList, SIGNAL(highlighted(int)), this, SLOT(highlighted(int)));

	_append = new QPushButton( "Append", this );
	connect( _append, SIGNAL( clicked() ), this, SLOT( append() ) );
	
	_prepend = new QPushButton( "Prepend", this );
	connect( _prepend, SIGNAL( clicked() ), this, SLOT( prepend() ) );
	
	_remove = new QPushButton( "Remove", this );
	connect( _remove, SIGNAL( clicked() ), this, SLOT( remove() ) );
	
	_progListLabel = new QLabel( "Programmed Tracks", this );
	_progListLabel->setAlignment( AlignCenter );
	
	_progList = new QListBox( this );

    // Separator
    _separator = new QFrame( this );
    _separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    
    // Buttons
    _save = new QPushButton( "Save", this );
	connect(_save, SIGNAL(clicked()), this, SLOT(save()));

    _dismiss = new QPushButton( "Dismiss", this );
    connect( _dismiss, SIGNAL(clicked()), this, SLOT(dismiss()) );

	resize( 450, 400 );
}

void InfoWindow::resizeEvent( QResizeEvent* )
{
	const int pad             = 4;
	const int lineHeight      = 24;
	const int labelWidth      = 70;
	const int buttonWidth     = 75;
	const int buttonHeight    = 24;
	const int listWidth       = ( width() - 4 * pad - buttonWidth ) / 2;
	const int listHeight      = height() - 9 * pad - 4 * lineHeight - buttonHeight;
	const int editWidth       = 2 * listWidth + pad + buttonWidth - labelWidth;
	const int separatorWidth  = 2 * listWidth + 2 * pad + buttonWidth;
	
	_discTitleLabel->setGeometry( pad, pad, labelWidth, lineHeight );
	_discTitle->setGeometry( 2 * pad + labelWidth, pad, editWidth, lineHeight);
	_discArtistLabel->setGeometry(pad, 2 * pad + lineHeight, labelWidth, lineHeight);
	_discArtist->setGeometry( 2 * pad +labelWidth, 2 * pad + lineHeight, editWidth, lineHeight);
	_trackTitleLabel->setGeometry(pad, 3 * pad + 2 * lineHeight, labelWidth, lineHeight);
	_trackTitle->setGeometry( 2 * pad + labelWidth, 3 * pad + 2 * lineHeight, editWidth, lineHeight);
	_trackListLabel->setGeometry( pad, 4 * pad + 3 * lineHeight, listWidth, lineHeight );
	_trackList->setGeometry( pad, 5 * pad + 4 * lineHeight, listWidth, listHeight);
	_append->setGeometry( 2 * pad + listWidth, 5 * pad + 4 * lineHeight, buttonWidth, buttonHeight );
	_prepend->setGeometry( 2 * pad + listWidth, 6 * pad + 4 * lineHeight + buttonHeight, buttonWidth, buttonHeight );
	_remove->setGeometry( 2 * pad + listWidth, 7 * pad + 4 * lineHeight + 2 * buttonHeight, buttonWidth, buttonHeight );
	_progListLabel->setGeometry( 3 * pad + listWidth + buttonWidth, 4 * pad + 3 * lineHeight, listWidth, lineHeight );
	_progList->setGeometry( 3 * pad + listWidth + buttonWidth, 5 * pad + 4 * lineHeight, listWidth, listHeight );
	_separator->setGeometry( pad, 6 * pad + 4 * lineHeight + listHeight, separatorWidth, pad );
	_save->setGeometry( pad, _separator->y() + pad + pad, buttonWidth, buttonHeight );
	_dismiss->setGeometry( 2 * pad + buttonWidth, _separator->y() + pad + pad, buttonWidth, buttonHeight);
}

//------------------------------------------------------------------------------
// Hide the prefs window.
//------------------------------------------------------------------------------
void InfoWindow::dismiss()
{
    hide();
}

//------------------------------------------------------------------------------
// Save the current user preferences, and popdown the dialog.
//------------------------------------------------------------------------------
void InfoWindow::save()
{
	int length = _progList->count();
	if ( length <= 0 ) {
		// %%% popup error.
		debug("No tracks in program!!!");
		return;
	}
	
	int* tracks = new int[length];
	
	for ( int i = 0; i < length; i++ ) {
		tracks[i] = nameToTrack( _progList->text( i ) );
	}

    emit updateProgram( length, tracks );

	delete [] tracks;

	emit saveDiscInfo();
	hide();
}

//------------------------------------------------------------------------------
// Configure the InfoWindow for no disc in the CDROM drive.
//------------------------------------------------------------------------------
void InfoWindow::noDisc()
{
    _discTitle->setText( "" );
    _discArtist->setText( "" );
    _trackTitle->setText( "" );
    _trackList->clear();
	_progList->clear();
}

//------------------------------------------------------------------------------
// Configure the InfoWindow for a new disc in the CDROM drive.
//------------------------------------------------------------------------------
void InfoWindow::newDisc(CompactDisc* cd)
{
    _discTitle->setText( cd->title() );
    _discArtist->setText( cd->artist() );
	
	_trackList->clear();
    for ( int i = 0; i < cd->id().numTracks(); i++ ) {
        _trackList->insertItem( cd->track(i) );
    }

	_progList->clear();
	for ( int *ip = cd->program()->ordered().first(); ip; ip = cd->program()->ordered().next() ) {
		_progList->insertItem( cd->track( *ip - 1 ) );
	}
	if ( _trackList->count() ) _trackList->setCurrentItem( 0 );
	if ( _progList->count() ) _progList->setCurrentItem( 0 );

    // Connect the InfoWindow to the CompactDisc.
    connect( this, SIGNAL(updateDiscTitle(const char*))         , cd, SLOT(setTitle(const char*)));
    connect( this, SIGNAL(updateDiscArtist(const char*))        , cd, SLOT(setArtist(const char*)));
    connect( this, SIGNAL(updateTrackTitle(int, const char*))   , cd, SLOT(setTrack(int, const char*)));
    connect( this, SIGNAL(saveDiscInfo())                       , cd, SLOT(saveDiscInfo()));
}

//------------------------------------------------------------------------------
// Process the highlighting of an item in the track list.
//------------------------------------------------------------------------------
void InfoWindow::highlighted(int index)
{
    _trackTitle->setText( _trackList->text( index ) );
}

//------------------------------------------------------------------------------
// Process the pressing of the return key for the disc title.
//------------------------------------------------------------------------------
void InfoWindow::procDiscTitle()
{
    emit updateDiscTitle( _discTitle->text() );
    _discArtist->setFocus();
}

//------------------------------------------------------------------------------
// Process the pressing of the return key for the disc artist.
//------------------------------------------------------------------------------
void InfoWindow::procDiscArtist()
{
    emit updateDiscArtist( _discArtist->text() );
    _trackTitle->setFocus();
}

//------------------------------------------------------------------------------
// Process the pressing of the return key for the track title.
//------------------------------------------------------------------------------
void InfoWindow::procTrackTitle()
{
    int track = _trackList->currentItem();
    if (track >= 0) {
        QString oldName = _trackList->text( track );

        _trackList->changeItem( _trackTitle->text(), track );
        emit updateTrackTitle( track, _trackTitle->text() );

        int count = _progList->count();

        // Change all occurences in the program list.
	for ( int i = 0; i < count; i++ ) {
            if ( !strcmp( oldName, _progList->text( i ) ) ) {
                _progList->changeItem( _trackTitle->text(), i );
            }
	}
    }

    track++;
    if (track >= int( _trackList->count() ) ) track = 0;
    _trackList->setCurrentItem(track);
    highlighted(track);
}

//------------------------------------------------------------------------------
// Edit the previous track title.
//------------------------------------------------------------------------------
void InfoWindow::procUpPressed()
{
    int track = _trackList->currentItem();
    track--;
	
	// Wrap around if needed.
    if (track < 0) track = _trackList->count() - 1;
    _trackList->setCurrentItem(track);
    highlighted(track);
}

//------------------------------------------------------------------------------
// Edit the next track title.
//------------------------------------------------------------------------------
void InfoWindow::procDownPressed()
{
    int track = _trackList->currentItem();
    track++;

	// Wrap around if needed.
    if (track >= int( _trackList->count() ) ) track = 0;
    _trackList->setCurrentItem(track);
    highlighted(track);
}

void InfoWindow::append()
{
	int item1 = _trackList->currentItem();
	int item2 = _progList->currentItem();
	
	if ( item1 >= 0 ) {
		if ( _progList->count() > 0 ) {
			_progList->insertItem( _trackList->text( item1 ), item2 + 1 );
			_progList->setCurrentItem( item2 + 1 );
		} else {
			_progList->insertItem( _trackList->text( item1 ) );
			_progList->setCurrentItem( 0 );
		}
	}
}

void InfoWindow::prepend()
{
	int item1 = _trackList->currentItem();
	int item2 = _progList->currentItem();

	if ( item1 >= 0 ) {
		if ( _progList->count() > 0 ) {
			_progList->insertItem( _trackList->text( item1 ), item2 );
			_progList->setCurrentItem( item2 );
		} else {
			_progList->insertItem( _trackList->text( item1 ) );
			_progList->setCurrentItem( 0 );
		}
	}
}

void InfoWindow::remove()
{
	if ( _progList->count() <= 0 ) return;
	
	int item = _progList->currentItem();
	
	if ( item >= 0 ) {
		_progList->removeItem( item );
	}
}

int InfoWindow::nameToTrack( const char* name )
{
	int count = _trackList->count();
	
	for ( int i = 0; i < count; i++ ) {
		if ( !strcmp( name, _trackList->text( i ) ) ) {
			return i + 1;
		}
	}
	
	return -1;
}
