// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef _InfoWindow_H_
#define _InfoWindow_H_

#include <qwidget.h>
#include <qlabel.h>
#include <qlined.h>
#include <qlistbox.h>
#include <qpushbt.h>
#include "CompactDisc.h"

//: A modified QListBox.
//. This variant of a QListBox does not accept the keyboard focus.
class MyListBox : public QListBox {
public:
    //: Create a MyListBox.
    MyListBox(QWidget* parent=0, const char* name=0) : QListBox(parent, name) { setAcceptFocus(0); }
};

//: A modified QLineEdit.
//. This variant of a QLineEdit emits two unique signals when the up and down
//. arrow keys are pressed.
class MyLineEdit : public QLineEdit {
    Q_OBJECT
public:
    //: Create a MyLineEdit.
    MyLineEdit(QWidget* parent=0, const char* name=0) : QLineEdit(parent, name) { }
    
protected:
    //: Process a key press.
    //. This implementation emits an upPressed() signal when the Up Arrow key
    //. is pressed, and a downPressed() signal when the Down Arrow key is
    //. pressed.  All other key press events are passed to the normal QLineEdit
    //. key press event handler.
    virtual void keyPressEvent(QKeyEvent* e);

signals:
    //: Emitted when the Up Arrow key is pressed.
    void upPressed();

    //: Emitted when the Down Arrow key is pressed.
    void downPressed();
};

//: Window for modifying CD information.
//. Provide an interface for viewing, modifying, and saving the current CDs
//. title, artist, and track information.
class InfoWindow : public QWidget {
    Q_OBJECT
	QLabel*             _discTitleLabel;
    QLineEdit*          _discTitle;
	QLabel*             _discArtistLabel;
    QLineEdit*          _discArtist;
	QLabel*             _trackTitleLabel;
    MyLineEdit*         _trackTitle;
	QLabel*             _trackListLabel;
    MyListBox*          _trackList;
	QLabel*             _progListLabel;
	QListBox*           _progList;
	QPushButton*        _append;
	QPushButton*        _prepend;
	QPushButton*        _remove;
    QPushButton*        _save;
	QPushButton*        _dismiss;
	QFrame*             _separator;

    // Dont let the compiler create these...
    InfoWindow(const InfoWindow& infoWindow);
    InfoWindow& operator=(const InfoWindow& infoWindow);

    int nameToTrack( const char* name );
public:
    //: Create an InfoWindow.
    InfoWindow(QWidget* parent = 0, const char* name = 0);
public slots:
    void append();
    void prepend();
    void remove();

    //: Hide the info window.
    void dismiss();

    //: Save the disc info.
    void save();

    //: Configure the InfoWindow for no disc in the CDROM drive.
    void noDisc();

    //: Configure the InfoWindow for the new disc.
    void newDisc(CompactDisc* cd);

    //: Process the highlighting of an item in the track list.
    void highlighted(int index);

    //: Process return key for disc title.
    void procDiscTitle();

    //: Process return key for disc artist.
    void procDiscArtist();

    //: Process return key for track title.
    void procTrackTitle();

    //: Process the pressing of the up arrow key.
    void procUpPressed();

    //: Process the pressing of the down arrow key.
    void procDownPressed();

protected:
	void resizeEvent( QResizeEvent* e );
signals:
    //: Emitted when the disc title has been changed.
    void updateDiscTitle(const char* discTitle);

    //: Emitted when the disc artist has been changed.
    void updateDiscArtist(const char* discArtist);

    //: Emitted when a track title has been changed.
    void updateTrackTitle(int track, const char* trackTitle);

    //: Emitted when the user presses the save button.
    void saveDiscInfo();

    void updateProgram(int length, int* tracks);
};

extern InfoWindow* infoWindow;

#endif
