// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

extern "C" {
#include "xcd.xbm"
}
#include <iostream.h>
#include <qfont.h>
#include <qpainter.h>
#include "Options.h"
#include "MainWindow.h"
#include "PrefsWindow.h"
#include "InfoWindow.h"

//------------------------------------------------------------------------------
// Propogate mouse events to the parent widget, and adjust the x and y
// coordinates appropriately.
//------------------------------------------------------------------------------
void MyLabel::mousePressEvent(QMouseEvent* e)
{
    QMouseEvent* e1 = new QMouseEvent(Event_MouseButtonPress, pos()+e->pos(), e->button(), e->state());
    qApp->sendEvent(parentWidget(), e1);
    delete e1;
}

MyLCDNumber::MyLCDNumber( int size, QWidget* parent ) : QLCDNumber(size, parent)
{
	setFrameStyle( QFrame::Sunken | QFrame::Panel );
}

//------------------------------------------------------------------------------
// Propogate mouse events to the parent widget, and adjust the x and y
// coordinates appropriately.
//------------------------------------------------------------------------------
void MyLCDNumber::mousePressEvent(QMouseEvent* e)
{
    QMouseEvent* e1 = new QMouseEvent(Event_MouseButtonPress, pos()+e->pos(), e->button(), e->state());
    qApp->sendEvent(parentWidget(), e1);
    delete e1;
}

//------------------------------------------------------------------------------
// Create the MainWindow.
//------------------------------------------------------------------------------
MainWindow::MainWindow(QWidget* parent, const char* name)
        : QWidget(parent, name),
          _icon( xcd_width, xcd_height, xcd_bits, TRUE ),
          _step("--"),
          _track("--"),
          _time("--:--"),
          _volume("--"),
          _discTitle("No Disc"),
          _discArtist("No Artist"),
          _trackTitle("No Track"),
          _trackNumber(0),
          _cd( 0 ),
          _status( Error )
{
	// Tiemr for flashing the time during a pause.
	startTimer( 500 );
	
	// Set our icon.
    setIcon( _icon );
	
    // Default fonts.
    QFont   menuFont( "Helvetica", 12, QFont::Bold );
    QFont   labelFont( "Helvetica", 10 );
	QFont   trackFont( "Helvetica", 14, QFont::Bold );
	
	// No resizing allowed.
	setMinimumSize(240, 94);
	setMaximumSize(240, 94);
	resize(240, 94);

	// Create the tracks popup menu.
    _trackMenu = new QPopupMenu;
    _trackMenu->setFont( trackFont );
    connect(_trackMenu, SIGNAL(activated(int)), this, SLOT(procTrackMenu(int)));

    // Create the main menu.
    _mainMenu = new QPopupMenu;
    _mainMenu->setFont(menuFont);
	connect( _mainMenu, SIGNAL( activated( int ) ), this, SLOT( procMainMenu( int ) ) );

    // Create the Commands menu.
    QPopupMenu* commandMenu = new QPopupMenu;
    commandMenu->setFont(menuFont);
    commandMenu->insertItem("Play");
    commandMenu->insertItem("Pause");
    commandMenu->insertItem("Stop");
    commandMenu->insertItem("Eject");
    commandMenu->insertItem("Next Track");
    commandMenu->insertItem("Previous Track");
    _mainMenu->insertItem("Commands", commandMenu);
    connect(commandMenu, SIGNAL(activated(int)), this, SLOT(procCommandsMenu(int)));

    // Create the Time menu.
    QPopupMenu* timeMenu = new QPopupMenu;
    timeMenu->setFont(menuFont);
    timeMenu->insertItem("Track Elapsed", TrackElapsed);
    timeMenu->insertItem("Track Remaining", TrackRemaining);
    timeMenu->insertItem("Disc Elapsed", DiscElapsed);
    timeMenu->insertItem("Disc Remaining", DiscRemaining);
    _mainMenu->insertItem("Time Display", timeMenu);
    connect(timeMenu, SIGNAL(activated(int)), this, SLOT(procTimeMenu(int)));

    // Create the Icon menu.
    QPopupMenu* iconMenu = new QPopupMenu;
    iconMenu->setFont(menuFont);
    iconMenu->insertItem("Time Display", Time);
    iconMenu->insertItem("Track Name", TrackName);
    iconMenu->insertItem("Disc Title", DiscTitle);
    iconMenu->insertItem("Disc Artist", DiscArtist);
    iconMenu->insertItem("Program Name", ProgramName);
    _mainMenu->insertItem("Icon Display", iconMenu);
    connect(iconMenu, SIGNAL(activated(int)), this, SLOT(procIconMenu(int)));

	_mainMenu->insertItem( "Preferences..." );
	_mainMenu->insertItem( "Disc Info..." );

    // Create the Step label and value display.
    _stepLabel = new MyLabel("Step", this);
    _stepLabel->setFont(labelFont);
    _stepLabel->setGeometry(4, 4, 40, 14);

    _stepValue = new MyLCDNumber(2, this);
    _stepValue->display("--");
    _stepValue->setGeometry(_stepLabel->x(), _stepLabel->y()+_stepLabel->height(), 40, 36);

    // Create the Track label and value display.
    _trackLabel = new MyLabel("Track", this);
    _trackLabel->setFont(labelFont);
    _trackLabel->setGeometry(_stepLabel->x()+_stepLabel->width()+4, _stepLabel->y(), 40, 14);

    _trackValue = new MyLCDNumber(2, this);
    _trackValue->display("--");
    _trackValue->setGeometry(_trackLabel->x(), _trackLabel->y()+_trackLabel->height(), 40, 36);

    // Create the Time label and value display.
    _timeLabel = new MyLabel("Track Elapsed", this);
    _timeLabel->setFont(labelFont);
    _timeLabel->setGeometry(_trackLabel->x()+_trackLabel->width()+4, _stepLabel->y(), 100, 14);

    _timeValue = new MyLCDNumber(5, this);
    _timeValue->display("--:--");
    _timeValue->setGeometry(_timeLabel->x(), _timeLabel->y()+_timeLabel->height(), 100, 36);
	
	procTimeMenu( int ( options->timeMode() ) );

    // Create the Volume label and value display.
    _volumeLabel = new MyLabel("Volume", this);
    _volumeLabel->setFont(labelFont);
    _volumeLabel->setGeometry(_timeLabel->x()+_timeLabel->width()+4, _stepLabel->y(), 40, 14);

    _volumeValue = new MyLCDNumber(2, this);
    _volumeValue->display("00");
    _volumeValue->setGeometry(_volumeLabel->x(), _volumeLabel->y()+_volumeLabel->height(), 40, 36);
	
	// Shuffle and repeat icons.
	_shuffle = new QPushButton( "S", this );
	_shuffle->setFont( labelFont );
	_shuffle->setToggleButton( true );
	_shuffle->setOn( options->shuffle() );
	_shuffle->setGeometry( _stepLabel->x(), _stepValue->y()+_stepValue->height()+4, 16, 16 );
	connect( _shuffle, SIGNAL( toggled( bool ) ), this, SLOT( procShuffle( bool ) ) );
	
	_repeat = new QPushButton( "R", this );
	_repeat->setFont( labelFont );
	_repeat->setToggleButton( true );
	_repeat->setOn( options->repeat() );
	_repeat->setGeometry( _stepLabel->x(), _shuffle->y()+_shuffle->height(), 16, 16 );
	connect( _repeat, SIGNAL( toggled( bool ) ), this, SLOT( procRepeat( bool ) ) );

    // Create the Track Name value display.
    _trackName = new MyLabel("", this);
    _trackName->setFont(trackFont);
    _trackName->setGeometry(_repeat->x()+_repeat->width()+4, _stepValue->y()+_stepValue->height()+4, 212, 32);
    _trackName->setAlignment(AlignCenter);
    _trackName->setFrameStyle(QFrame::Box | QFrame::Sunken);
}

//------------------------------------------------------------------------------
// Process all mouse button presses within the MainWindow.  The left button
// pops up the main menu.  The middle button pops up the track menu.  The right
// button does nothing.
//------------------------------------------------------------------------------
void MainWindow::mousePressEvent(QMouseEvent* e)
{
    switch (e->button()) {
        case LeftButton:
            _mainMenu->popup(pos()+e->pos());
            break;
            
        case MidButton:
            _trackMenu->popup(pos()+e->pos());
            break;
    }
}

void MainWindow::timerEvent( QTimerEvent* )
{
	static bool flash;

	if ( _status == Paused ) {
		if ( flash ) {
			_timeValue->display( "     " );
		} else {
			_timeValue->display( _time );
		}
	}
	flash = !flash;
}

//------------------------------------------------------------------------------
// Update the displayed step value.
//------------------------------------------------------------------------------
void MainWindow::changeStep(int step)
{
    if (step < 1) {
        _step = "--"; // Invalid step.
    } else if (step > 99) {
        _step = "00";
    } else {
        _step.sprintf("%02d", step);
    }
    _stepValue->display(_step);
}

//------------------------------------------------------------------------------
// Update the displayed track number and track name.
//------------------------------------------------------------------------------
void MainWindow::changeTrack(int track, const char* name)
{
    _trackNumber = track - 1;
    _trackTitle = name;
    _trackName->setText(_trackTitle);
    if (track < 1) {
        _track = "--"; // Invalid track number.
    } else if (track > 99) {
        _track = "00";
    } else {
        _track.sprintf("%02d", track);
    }
    _trackValue->display(_track);

    if ( options->iconMode() == TrackName ) {
        setIconText( _trackTitle );
    }
}

//------------------------------------------------------------------------------
// Update the displayed time.
//------------------------------------------------------------------------------
void MainWindow::changeTime(int minute, int second)
{
    if ((minute < 0) || (second < 0)) {
        _time = "--:--"; // Invalid time.
    } else {
        _time.sprintf("%02d:%02d", minute, second);
    }
    _timeValue->display(_time);

    if ( options->iconMode() == Time ) {
        QString tmp;
        tmp.sprintf("T%s %s", (const char*)_track, (const char*)_time);
		if ( iconText() != tmp ) {
			setIconText(tmp);
		}
    }
}

//------------------------------------------------------------------------------
// Update the displayed volume.
//------------------------------------------------------------------------------
void MainWindow::changeVolume(int volume)
{
    _volume.sprintf("%02d", volume);
    _volumeValue->display(_volume);
}

//------------------------------------------------------------------------------
// Prepare the MainWindow for there being no disc in the drive.
//------------------------------------------------------------------------------
void MainWindow::noDisc()
{
	_cd = 0;
    _trackMenu->clear();
	_discTitle = "No Disc";
	_discArtist = "";
	redrawIcon();
}

//------------------------------------------------------------------------------
// Prepare the MainWindow for the new disc.
//------------------------------------------------------------------------------
void MainWindow::newDisc(CompactDisc* cd)
{
	_cd = cd;
	
    QString track;
    
    _trackMenu->clear();
	for ( int* i = cd->program()->ordered().first(); i; i = cd->program()->ordered().next() ) {
        track.sprintf("T%02d [%02d:%02d] %s", (*i-1)+cd->id().firstTrack(), cd->id().trackMinute(*i-1), cd->id().trackSecond(*i-1), cd->track(*i-1));
        _trackMenu->insertItem( track );
    }

    _discTitle = cd->title();
    _discArtist = cd->artist();
	
	redrawIcon();
	
    if ( options->iconMode() == DiscTitle ) {
        setIconText(_discTitle);
    } else if ( options->iconMode() == DiscArtist ) {
        setIconText(_discArtist);
    }
}

void MainWindow::changeProgram( int length, int* tracks )
{
	QString track;
	
	_trackMenu->clear();
	for ( int i = 0; i < length; i ++ ) {
		int trk = tracks[i] - 1;
		track.sprintf("T%02d [%02d:%02d] %s", trk+_cd->id().firstTrack(), _cd->id().trackMinute(trk), _cd->id().trackSecond(trk), _cd->track(trk));
        _trackMenu->insertItem( track );
	}
}

//------------------------------------------------------------------------------
// Process a selection from the Commands menu.
//------------------------------------------------------------------------------
void MainWindow::procCommandsMenu(int id)
{
    switch (id) {
        case 0: // Play
            emit play();
            break;

        case 1: // Pause
            emit pause();
            break;

        case 2: // Stop
            emit stop();
            break;

        case 3: // Eject
            emit eject();
            break;

        case 4: // Next Track
            emit nextTrack();
            break;

        case 5: // Previous Track
            emit previousTrack();
            break;
    }
}

//------------------------------------------------------------------------------
// Process a selection from the Time menu.
//------------------------------------------------------------------------------
void MainWindow::procTimeMenu(int id)
{
	options->setTimeMode( TimeMode( id ) );
	changeTimeMode();
}

//------------------------------------------------------------------------------
// Process a selection from the Icon menu.
//------------------------------------------------------------------------------
void MainWindow::procIconMenu(int id)
{
	options->setIconMode( IconMode( id ) );
    changeIconMode();
}

//------------------------------------------------------------------------------
// Process a selection from the Windows menu.
//------------------------------------------------------------------------------
void MainWindow::procMainMenu(int id)
{
    switch (id) {
        case 3: // PrefsWindow
            prefsWindow->show();
            prefsWindow->raise();
            break;

        case 4: // InfoWindow
            infoWindow->show();
            infoWindow->raise();
            break;
    }
}

//------------------------------------------------------------------------------
// Process a selection from the Track menu.
//------------------------------------------------------------------------------
void MainWindow::procTrackMenu(int id)
{
    emit playTrack(id);
}

//------------------------------------------------------------------------------
// Change the icon mode pref.
//------------------------------------------------------------------------------
void MainWindow::changeIconMode()
{
    switch ( options->iconMode() ) {
        case ProgramName:
            setIconText( "xcd" );
            break;

        case DiscTitle:
            setIconText( _discTitle );
            break;

        case DiscArtist:
            setIconText( _discArtist );
            break;

        case TrackName:
            setIconText( _trackTitle );
            break;

        case Time:
            QString tmp;
            tmp.sprintf("T%s %s", (const char*)_track, (const char*)_time);
            setIconText(tmp);
            break;
    }
}

//------------------------------------------------------------------------------
// The discs title has been changed, update our local value.
//------------------------------------------------------------------------------
void MainWindow::changeDiscTitle(const char* title)
{
    _discTitle = title;
	
	redrawIcon();
	
    changeIconMode();
}

//------------------------------------------------------------------------------
// The discs artist has been changed, update our local value.
//------------------------------------------------------------------------------
void MainWindow::changeDiscArtist(const char* artist)
{
    _discArtist = artist;
	
	redrawIcon();
	
    changeIconMode();
}

//------------------------------------------------------------------------------
// On of the discs track titles has changed, update the track list.
//------------------------------------------------------------------------------
void MainWindow::changeTrackTitle(int track, const char* title)
{
    QString tmp = _trackMenu->text(track);
    tmp.truncate(12);
    tmp += title;
    _trackMenu->changeItem((const char*)tmp, track);

    if (track == _trackNumber) {
        _trackName->setText(title);
    }
}

void MainWindow::changeTimeMode()
{
	// Update time display label.
	switch ( options->timeMode() ) {
	 case TrackElapsed:
		_timeLabel->setText( "Track Elapsed" );
		break;
		
	 case TrackRemaining:
		_timeLabel->setText( "Track Remaining" );
		break;
		
	 case DiscElapsed:
		_timeLabel->setText( "Disc Elapsed" );
		break;
		
	 case DiscRemaining:
		_timeLabel->setText( "Disc Remaining" );
		break;
	}
}

void MainWindow::procShuffle( bool b )
{
	options->setShuffle( b );
}

void MainWindow::procRepeat( bool b )
{
	options->setRepeat( b );
}

void MainWindow::changeShuffle()
{
	_shuffle->setOn( options->shuffle() );
}

void MainWindow::changeRepeat()
{
	_repeat->setOn( options->repeat() );
}

void MainWindow::nextTimeMode()
{
	procTimeMenu( ( int ( options->timeMode() ) + 1 ) % 4 );
}

void MainWindow::nextIconMode()
{
	procIconMenu( ( int ( options->iconMode() ) + 1 ) % 5 );
}

void MainWindow::redrawIcon()
{
	QString str( (const char*)_discTitle );
	if ( _discArtist != "" ) {
		str += " - ";
		str += _discArtist;
	}
	setCaption( str );
	
	QBitmap bmap( 64, 64 );
	QPainter p;
	p.begin( &bmap );
	p.eraseRect( 0, 0, 64, 64 );
	p.drawText( 0, 0, 64, 64, AlignHCenter | WordBreak, str );
	p.end();
	setIcon( bmap );
}

void MainWindow::changeStatus( CdromStatus status )
{
	if ( ( status == Paused ) && ( _status != Paused ) ) {
		_timeValue->display( _time );
	}
	_status = status;
}
