// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

extern "C" {
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
}

#include "Options.h"
#include "Cdrom.h"

Options::Options( int argc, char* argv[] )
: _autoPlay( true ), _autoEject( true ), _repeat( false ), _shuffle( false ), _timeMode( TrackRemaining ), _iconMode( Time ), _cdromType( CdromType( 0 ) ),
  _pollTime( 250 ), _cdromPath( "/dev/cdrom" ), _dbasePath( 0 )
{
	_dbasePath = getenv( "HOME" );
	_dbasePath += "/.xcddbase";
	
	load();
	
	for ( int i = 1; i < argc; i++ ) {
		if ( !strcmp( argv[i], "-autoplay" ) ) {
			_autoPlay = true;
		} else if ( !strcmp( argv[i], "-noautoplay" ) ) {
			_autoPlay = false;
		} else if ( !strcmp( argv[i], "-autoeject" ) ) {
			_autoEject = true;
		} else if ( !strcmp( argv[i], "-noautoeject" ) ) {
			_autoEject = false;
		} else if ( !strcmp( argv[i], "-repeat" ) ) {
			_repeat = true;
		} else if ( !strcmp( argv[i], "-norepeat" ) ) {
			_repeat = false;
		} else if ( !strcmp( argv[i], "-shuffle" ) ) {
			_shuffle = true;
		} else if ( !strcmp( argv[i], "-noshuffle" ) ) {
			_shuffle = false;
		} else if ( !strcmp( argv[i], "-time" ) ) {
			_timeMode = stringToTimeMode( argv[++i] );
		} else if ( !strcmp( argv[i], "-icon" ) ) {
			_iconMode = stringToIconMode( argv[++i] );
		} else if ( !strcmp( argv[i], "-poll" ) ) {
			sscanf( argv[++i], "%d", &_pollTime );
		} else if ( !strcmp( argv[i], "-cdrom" ) ) {
			_cdromPath = argv[++i];
		} else if ( !strcmp( argv[i], "-cdtype" ) ) {
			_cdromType = CdromType( 0 );
			i++;
			for ( int j = 0; j < MaxCdromTypes; j++ ) {
				if ( !strcmp( cdrom->typeConstant( j ), argv[i] ) ) {
					_cdromType = CdromType( j );
				}
			}
		} else if ( !strcmp( argv[i], "-dbase" ) ) {
			_dbasePath = argv[++i];
		} else {
			if ( strcmp( argv[i], "-h" ) ) {
				fprintf( stderr, "%s: unrecognized option '%s'\n", argv[0], argv[i] );
			}
			fprintf( stderr, "Available options:\n" );
			fprintf( stderr, "-autoeject     Eject the disc after all tracks have been played.\n" );
			fprintf( stderr, "-noautoeject   Don't autoeject.\n" );
			fprintf( stderr, "-autoplay      Automatically begin playing a newly inserted disc.\n" );
			fprintf( stderr, "-noautoplay    Don't autoplay the disc.\n" );
			fprintf( stderr, "-cdrom <path>  Path to the CDROM device.\n" );
			fprintf( stderr, "-cdtype <type> Device type (" );
			for ( int j = 0; j < MaxCdromTypes - 1; j++ ) {
				fprintf( stderr, "%s, ", cdrom->typeConstant( j ) );
			}
			fprintf( stderr, "%s).\n", cdrom->typeConstant( MaxCdromTypes - 1 ) );
			fprintf( stderr, "-dbase <path>  Path to the CD database.\n" );
			fprintf( stderr, "-h             Print this help.\n" );
			fprintf( stderr, "-icon <mode>   Set what is displayed in the icon (program, disc, artist, track, time).\n" );
			fprintf( stderr, "-poll <msec>   Check the CDROM drive every <msec> milliseconds.\n" );
			fprintf( stderr, "-repeat        After playing each track in the program once, do it again.\n" );
			fprintf( stderr, "-norepeat      Don't repeat.\n" );
			fprintf( stderr, "-shuffle       Randomly reorder the tracks in the program.\n" );
			fprintf( stderr, "-noshuffle     Don't shuffle.\n" );
			fprintf( stderr, "-time <mode>   Set the time display mode (track, track_left, disc, disc_left).\n" );
			fprintf( stderr, "\n" );
			fprintf( stderr, "xcd version 2.0, Copyright (C) 1996 Sean Vyain\n" );
			fprintf( stderr, "xcd comes with ABSOLUTELY NO WARRANTY; for details see the LICENSE file.\n" );
			fprintf( stderr, "This is free software, and you are welcome to redistribute it\n" );
			fprintf( stderr, "under certain conditions; see the LICENSE file for details.\n" );

			exit( 1 );
		}
	}
}

void Options::load()
{
	QString fn = getenv( "HOME" );
	fn += "/.xcdrc";
	
	FILE* fp;
	if ( !(fp = fopen( fn, "r" ) ) ) return;
	
	while ( !feof( fp ) ) {
		QString type = getToken( fp );
		QString value = getToken( fp );
		if ( !strcmp( type, "autoplay" ) ) {
			_autoPlay = atoi( value );
		} else if ( !strcmp( type, "autoeject" ) ) {
			_autoEject = atoi( value );
		} else if ( !strcmp( type, "repeat" ) ) {
			_repeat = atoi( value );
		} else if ( !strcmp( type, "shuffle" ) ) {
			_shuffle = atoi( value );
		} else if ( !strcmp( type, "time" ) ) {
			_timeMode = stringToTimeMode( value );
		} else if ( !strcmp( type, "icon" ) ) {
			_iconMode = stringToIconMode( value );
		} else if ( !strcmp( type, "poll" ) ) {
			_pollTime = atoi( value );
		} else if ( !strcmp( type, "cdrom" ) ) {
			_cdromPath = value;
		} else if ( !strcmp( type, "cdtype" ) ) {
			_cdromType = CdromType( 0 );
			for ( int j = 0; j < MaxCdromTypes; j++ ) {
				if ( !strcmp( cdrom->typeConstant( j ), value ) ) {
					_cdromType = CdromType( j );
				}
			}
		} else if ( !strcmp( type, "dbase" ) ) {
			_dbasePath = value;
		}
	}
}

QString Options::getToken( FILE* fp )
{
	QString tok;
	char ch = 0;
	
	while ( !feof( fp ) && isspace( ch = getc( fp ) ) );
	tok += ch;
	while ( !feof( fp ) && !isspace( ch = getc( fp ) ) ) {
		tok += ch;
	}
	return tok;
}

void Options::save()
{
	QString fn = getenv( "HOME" );
	fn += "/.xcdrc";
	
	FILE* fp;
	if ( !(fp = fopen( fn, "w" ) ) ) return;
	
	fprintf( fp, "verson    2.0\n" );
	fprintf( fp, "autoplay  %d\n", _autoPlay );
	fprintf( fp, "autoeject %d\n", _autoEject );
	fprintf( fp, "repeat    %d\n", _repeat );
	fprintf( fp, "shuffle   %d\n", _shuffle );
	switch ( _timeMode ) {
	 case TrackElapsed:
		fprintf( fp, "time      track\n" );
		break;
		
	 case TrackRemaining:
		fprintf( fp, "time      track_left\n" );
		break;
		
	 case DiscElapsed:
		fprintf( fp, "time      disc\n" );
		break;
		
	 case DiscRemaining:
		fprintf( fp, "time      disc_left\n" );
		break;
	}
	switch ( _iconMode ) {
	 case ProgramName:
		fprintf( fp, "icon      program\n" );
		break;
		
	 case DiscTitle:
		fprintf( fp, "icon      disc\n" );
		break;
		
	 case DiscArtist:
		fprintf( fp, "icon      artist\n" );
		break;
		
	 case TrackName:
		fprintf( fp, "icon      track\n" );
		break;
		
	 case Time:
		fprintf( fp, "icon      time\n" );
		break;
	}
	fprintf( fp, "poll      %d\n", _pollTime );
	fprintf( fp, "cdrom     %s\n", (const char*)_cdromPath );
	fprintf( fp, "cdtype    %s\n", cdrom->typeConstant( _cdromType ) );
	fprintf( fp, "dbase     %s\n", (const char*)_dbasePath );
	fclose( fp );
}

TimeMode Options::stringToTimeMode( const char* str )
{
	if ( !strcmp( str, "track" ) ) {
		return TrackElapsed;
	} else if ( !strcmp( str, "track_left" ) ) {
		return TrackRemaining;
	} else if ( !strcmp( str, "disc" ) ) {
		return DiscElapsed;
	} else if ( !strcmp( str, "disc_left" ) ) {
		return DiscRemaining;
	} else {
		fprintf( stderr, "Invalid time mode '%s'\n", (const char*)str );
		return TrackElapsed;
	}
}

IconMode Options::stringToIconMode( const char* str )
{
	if ( !strcmp( str, "program" ) ) {
		return ProgramName;
	} else if ( !strcmp( str, "disc" ) ) {
		return DiscTitle;
	} else if ( !strcmp( str, "artist" ) ) {
		return DiscArtist;
	} else if ( !strcmp( str, "track" ) ) {
		return TrackName;
	} else if ( !strcmp( str, "time" ) ) {
		return Time;
	} else {
		fprintf( stderr, "Invalid icon mode '%s'\n", (const char*)str );
		return ProgramName;
	}
}
