// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

extern "C" {
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
}
#include <qlabel.h>
#include <qpushbt.h>
#include <qframe.h>
#include "Options.h"
#include "PrefsWindow.h"
#include "Cdrom.h"

//------------------------------------------------------------------------------
// Create the PrefsWindow.
//------------------------------------------------------------------------------
PrefsWindow::PrefsWindow( QWidget* parent, const char* name )
: QWidget( parent, name )
{
    const int pad              = 4;
	const int height           = 24;
	const int checkWidth       = 100;
	const int entryWidth       = 60;
	const int optionWidth      = 120;
	const int optionLabelWidth = 80;
	const int pollLabelWidth   = 90;
	const int pathLabelWidth   = 90;
	const int pathWidth        = 2 * pad + checkWidth + optionWidth + optionLabelWidth - pathLabelWidth;
	const int buttonWidth      = 75;

	setFixedSize( 8 * pad + checkWidth + optionWidth + optionLabelWidth, 16 * pad + 8 * height );
	resize( 8 * pad + checkWidth + optionWidth + optionLabelWidth, 16 * pad + 8 * height );

    setCaption( "XCd: Preferences" );

    // Check boxes
	QFrame* checkFrame = new QFrame( this );
	checkFrame->setFrameStyle( QFrame::Sunken | QFrame::Panel );
	checkFrame->setGeometry( pad, pad, 2 * pad + checkWidth, 5 * pad + 4 * height );
	
    _autoPlay = new QCheckBox( "Auto Play", checkFrame );
	_autoPlay->setChecked( options->autoPlay() );
	_autoPlay->setGeometry( pad, pad, checkWidth, height );
    connect( _autoPlay, SIGNAL( clicked() ), this, SLOT( doUpdateAutoPlay() ) );
    
    _autoEject = new QCheckBox( "Auto Eject", checkFrame );
	_autoEject->setChecked( options->autoEject() );
	_autoEject->setGeometry( pad, 2 * pad + height, checkWidth, height );
    connect( _autoEject, SIGNAL(clicked()), this, SLOT(doUpdateAutoEject()));
	
	_shuffle = new QCheckBox( "Shuffle", checkFrame );
	_shuffle->setChecked( options->shuffle() );
	_shuffle->setGeometry( pad, 3 * pad + 2 * height, checkWidth, height );
    connect( _shuffle, SIGNAL( clicked() ), this, SLOT( doUpdateShuffle() ) );
	
	_repeat = new QCheckBox( "Repeat Disc", checkFrame );
	_repeat->setChecked( options->repeat() );
	_repeat->setGeometry( pad, 4 * pad + 3 * height, checkWidth, height );
    connect( _repeat, SIGNAL( clicked() ), this, SLOT( doUpdateRepeat() ) );
	
	// Option menus
	QFrame* optionFrame = new QFrame( this );
	optionFrame->setFrameStyle( QFrame::Sunken | QFrame::Panel );
	optionFrame->setGeometry( 4 * pad + checkWidth, pad, 3 * pad + optionWidth + optionLabelWidth, 5 * pad + 4 * height );
	
	_timeMode = new QComboBox( optionFrame );
    _timeMode->insertItem( "Track Elapsed" );
    _timeMode->insertItem( "Track Remaining" );
    _timeMode->insertItem( "Disc Elapsed" );
    _timeMode->insertItem( "Disc Remaining" );
	_timeMode->setCurrentItem( options->timeMode() );
	_timeMode->setGeometry( pad, pad, optionWidth, height );
    connect( _timeMode, SIGNAL(activated(int)), this, SLOT(doUpdateTimeMode()));
	
	QLabel* _timeModeLabel = new QLabel( "Time Display", optionFrame );
	_timeModeLabel->setGeometry( 2 * pad + optionWidth, pad, optionLabelWidth, height );
	
	_iconMode = new QComboBox( optionFrame );
    _iconMode->insertItem( "Program Name" );
    _iconMode->insertItem( "Disc Title" );
    _iconMode->insertItem( "Disc Artist" );
    _iconMode->insertItem( "Track Title" );
    _iconMode->insertItem( "Time Display" );
	_iconMode->setCurrentItem( options->iconMode() );
	_iconMode->setGeometry( pad,  2 * pad + height, optionWidth, height );
    connect( _iconMode, SIGNAL(activated(int)), this, SLOT(doUpdateIconMode()));
	
	QLabel* _iconModeLabel = new QLabel( "Icon Display", optionFrame );
	_iconModeLabel->setGeometry( 2 * pad + optionWidth, 2 * pad + height, optionLabelWidth, height );
	
	_cdromType = new QComboBox( optionFrame );
	for ( int i = 0; i < MaxCdromTypes; i++ ) {
		_cdromType->insertItem( cdrom->typeName( i ) );
	}
	_cdromType->setCurrentItem( options->cdromType() );
	_cdromType->setGeometry( pad, 3 * pad + 2 * height, optionWidth, height );
	connect( _cdromType, SIGNAL( activated( int ) ), this, SLOT( doUpdateCdromType() ) );
	
	QLabel* _cdromTypeLabel = new QLabel( "CDROM Type", optionFrame );
	_cdromTypeLabel->setGeometry( 2 * pad + optionWidth, 3 * pad + 2 * height, optionLabelWidth, height );

    // Entry fields
	QFrame* entryFrame = new QFrame( this );
	entryFrame->setFrameStyle( QFrame::Sunken | QFrame::Panel );
	entryFrame->setGeometry( pad, 7 * pad + 4 * height, 6 * pad + checkWidth + optionWidth + optionLabelWidth, 5 * pad + 3 * height );
	
    QLabel* _pollTimeLabel = new QLabel( "Poll Time (ms)", entryFrame );
	_pollTimeLabel->setGeometry( pad, pad, pollLabelWidth, height );
	_pollTimeLabel->setAlignment( AlignRight | AlignVCenter );
    
    _pollTime = new QLineEdit( entryFrame );
	_pollTime->setText( QString().sprintf( "%d", options->pollTime() ) );
	_pollTime->setGeometry( 2 * pad + pollLabelWidth, pad, entryWidth, height );
    connect( _pollTime, SIGNAL(returnPressed()), this, SLOT(doUpdatePollTime()));

    QLabel* _devicePathLabel = new QLabel( "CDROM Device", entryFrame );
	_devicePathLabel->setGeometry( pad, 2 * pad + height, pathLabelWidth, height );
	_devicePathLabel->setAlignment( AlignRight | AlignVCenter );

    _devicePath = new QLineEdit( entryFrame );
	_devicePath->setText( options->cdromPath() );
	_devicePath->setGeometry( 2 * pad + pathLabelWidth, 2 * pad + height, pathWidth, height );
    connect( _devicePath, SIGNAL(returnPressed()), this, SLOT(doUpdateDevicePath()));

    QLabel* _dbasePathLabel = new QLabel( "Database File", entryFrame );
	_dbasePathLabel->setGeometry( pad, 3 * pad + 2 * height, pathLabelWidth, height );
	_dbasePathLabel->setAlignment( AlignRight | AlignVCenter );
	
    _dbasePath = new QLineEdit( entryFrame );
	_dbasePath->setText( options->dbasePath() );
	_dbasePath->setGeometry( 2 * pad + pathLabelWidth, 3 * pad + 2 * height, pathWidth, height );
    connect( _dbasePath, SIGNAL(returnPressed()), this, SLOT(doUpdateDbasePath()));

    // Separator
    QFrame* _separator = new QFrame( this );
    _separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
	_separator->setGeometry( pad, 13 * pad + 7 * height, width() - 2 * pad, pad );
    
    // Buttons
    QPushButton* _save = new QPushButton( "Save", this );
	_save->setGeometry( pad, 15 * pad + 7 * height, buttonWidth, height );
    connect( _save, SIGNAL( clicked() ), options, SLOT( save() ) );

    QPushButton* _dismiss = new QPushButton( "Dismiss", this );
	_dismiss->setGeometry( 2 * pad + buttonWidth, 15 * pad + 7 * height, buttonWidth, height );
    connect( _dismiss, SIGNAL(clicked()), this, SLOT(dismiss()) );
}

//------------------------------------------------------------------------------
// Hide the prefs window.
//------------------------------------------------------------------------------
void PrefsWindow::dismiss()
{
    hide();
}

//------------------------------------------------------------------------------
// Emit updateAutoPlay.
//------------------------------------------------------------------------------
void PrefsWindow::doUpdateAutoPlay()
{
	options->setAutoPlay( _autoPlay->isChecked() );
}

//------------------------------------------------------------------------------
// Emit updateAutoEject.
//------------------------------------------------------------------------------
void PrefsWindow::doUpdateAutoEject()
{
	options->setAutoEject( _autoEject->isChecked() );
}

//------------------------------------------------------------------------------
// Emit updatePlayMode.
//------------------------------------------------------------------------------
void PrefsWindow::doUpdateShuffle()
{
	options->setShuffle( _shuffle->isChecked() );
}

//------------------------------------------------------------------------------
// Emit updateRepeatMode.
//------------------------------------------------------------------------------
void PrefsWindow::doUpdateRepeat()
{
	options->setRepeat( _repeat->isChecked() );
}

//------------------------------------------------------------------------------
// Emit updateTimeMode.
//------------------------------------------------------------------------------
void PrefsWindow::doUpdateTimeMode()
{
	options->setTimeMode( TimeMode( _timeMode->currentItem() ) );
}

//------------------------------------------------------------------------------
// Emit updateIconMode.
//------------------------------------------------------------------------------
void PrefsWindow::doUpdateIconMode()
{
	options->setIconMode( IconMode( _iconMode->currentItem() ) );
}

void PrefsWindow::doUpdateCdromType()
{
	options->setCdromType( CdromType( _cdromType->currentItem() ) );
}

//------------------------------------------------------------------------------
// Emit updatePollTime.
//------------------------------------------------------------------------------
void PrefsWindow::doUpdatePollTime()
{
	options->setPollTime( atoi( _pollTime->text() ) );
}

//------------------------------------------------------------------------------
// Emit updateDevicePath.
//------------------------------------------------------------------------------
void PrefsWindow::doUpdateDevicePath()
{
	options->setCdromPath( _devicePath->text() );
}


//------------------------------------------------------------------------------
// Emit updateDbasePath.
//------------------------------------------------------------------------------
void PrefsWindow::doUpdateDbasePath()
{
	options->setDbasePath( _dbasePath->text() );
}

void PrefsWindow::changeShuffle()
{
	_shuffle->setChecked( options->shuffle() );
}

void PrefsWindow::changeRepeat()
{
	_repeat->setChecked( options->repeat() );
}

void PrefsWindow::changeTimeMode()
{
	_timeMode->setCurrentItem( int ( options->timeMode() ) );
}

void PrefsWindow::changeIconMode()
{
	_iconMode->setCurrentItem( int ( options->iconMode() ) );
}
