// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

extern "C" {
#include <sys/scdisk.h>
}
#include "ScsiCdrom.h"

extern int errno;

ScsiCdrom::ScsiCdrom() : _volume( 32 ), _paused( false )
{
}

bool ScsiCdrom::eject( int fd )
{
    _paused = false;
    return ioctl( fd, DKEJECT, 0 ) == 0;
}

CompactDiscID* ScsiCdrom::id( int fd )
{
    struct cd_audio_cmd cmd;
    
    // Read the Table Of Contents Header.
    cmd.audio_cmds = CD_TRK_INFO_AUDIO;
    cmd.msf_flag = false;
    if ( ioctl( fd, DKAUDIO, &cmd ) != 0 ) {
        return 0;
    }
    
    int firstTrack = cmd.indexing.track_index.first_track;
    int numTracks  = cmd.indexing.track_index.last_track - firstTrack + 1;
	
    int* trackStart = new int[numTracks + 1];
	
    // Get the start time of each track.
    cmd.audio_cmds = CD_GET_TRK_MSF;
    for ( int i = 0; i < numTracks; i++ ) {
        cmd.indexing.track_msf.track = i + firstTrack;
        if ( ioctl( fd, DKAUDIO, &cmd ) ) {
            // Could not get TOC Entry --> noDisc().
            delete [] trackStart;
            return 0;
        }
        trackStart[i] = XcdFrames * ( XcdSecs * cmd.indexing.track_msf.mins + cmd.indexing.track_msf.secs ) + cmd.indexing.track_msf.frames;
    }
    
    // Get the start time of the leadout track.
    // %%% Can't get this on AIX!
    trackStart[numTracks] = XcdFrames;
	
    // Create CompactDiscID, and compare it to the current CompactDiscID.
    CompactDiscID* id = new CompactDiscID( numTracks, firstTrack, trackStart );
    delete [] trackStart;
	
    return id;
}

bool ScsiCdrom::init( int fd )
{
    return stop( fd );
}

bool ScsiCdrom::pause( int fd )
{
    struct cd_audio_cmd cmd;

    _paused = true;
    cmd.audio_cmds = CD_PAUSE_AUDIO;
    return ioctl( fd, DKAUDIO, &cmd ) == 0;
}

bool ScsiCdrom::play( int fd, int track )
{
    _paused = false;
    struct cd_audio_cmd cmd;

    cmd.audio_cmds = CD_PLAY_AUDIO;
    cmd.msf_flag = false;
    cmd.indexing.track_index.first_track = track;
    cmd.indexing.track_index.first_index = 1;
    cmd.indexing.track_index.last_track = track;
    cmd.indexing.track_index.last_index = 1;
    return ioctl( fd, DKAUDIO, &cmd ) == 0;
}

bool ScsiCdrom::resume( int fd )
{
    struct cd_audio_cmd cmd;

    _paused = false;
    cmd.audio_cmds = CD_RESUME_AUDIO;
    return ioctl( fd, DKAUDIO, &cmd ) == 0;
}

bool ScsiCdrom::status( int fd, CdromStatus& status, int& absTime, int& volume )
{
    struct cd_audio_cmd cmd;

    cmd.audio_cmds = CD_INFO_AUDIO;
    cmd.msf_flag = true;
    bool s = ioctl( fd, DKAUDIO, &cmd );

    if ( !s ) {
        switch ( cmd.status ) {
            case CD_NOT_VALID:
            case CD_STATUS_ERROR:
                status = Error;
                break;
                
            case CD_COMPLETED:
                status = Completed;
                break;
                
            case CD_NO_AUDIO:
                status = Stopped;
                break;

            default:
		if ( _paused ) {
                    status = Paused;
		} else {
                    status = Playing;
		}
                break;
        }
        
        absTime = XcdFrames * ( XcdSecs * cmd.indexing.info_audio.current_mins + cmd.indexing.info_audio.current_secs ) + cmd.indexing.info_audio.current_frames;

        volume = cmd.out_port_0_vol / 4;
    }
	
    return s;
}

bool ScsiCdrom::stop( int fd )
{
    struct cd_audio_cmd cmd;

    _paused = false;
    cmd.audio_cmds = CD_STOP_AUDIO;
    return ioctl( fd, DKAUDIO, &cmd ) == 0;
}

bool ScsiCdrom::volume( int fd, int vol )
{
    struct cd_audio_cmd cmd;

    cmd.audio_cmds = CD_SET_VOLUME;
    cmd.volume_type = CD_VOLUME_ALL;
    cmd.all_channel_vol = vol * 4;
    return ioctl( fd, DKAUDIO, &cmd ) == 0;
}
