// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

extern "C" {
#include <sys/types.h>
#include <linux/cdrom.h>
}
#include "IdeCdrom.h"

IdeCdrom::IdeCdrom() : _ejectToggle( false )
{
}

bool IdeCdrom::eject( int fd )
{
	_ejectToggle = !_ejectToggle;
	if ( _ejectToggle ) {
		return ioctl( fd, CDROMEJECT, 0 ) == 0;
	} else {
		return ioctl( fd, CDROMCLOSETRAY, 0 ) == 0;
	}
}

CompactDiscID* IdeCdrom::id( int fd )
{
	// Read the Table Of Contents Header.
    struct cdrom_tochdr tochdr;
    if ( ioctl( fd, CDROMREADTOCHDR, &tochdr ) != 0 ) {
        return 0;
    }

	int firstTrack = tochdr.cdth_trk0;
    int numTracks = tochdr.cdth_trk1 - tochdr.cdth_trk0 + 1;
	
	// Quick sanity check.
	if ( ( numTracks <= 0 ) || ( numTracks >= 99 ) ) {
        return 0;
    }
	
	int* trackStart = new int[numTracks + 1];
	
	// Get the start time of each track.
    struct cdrom_tocentry tocentry;
    tocentry.cdte_format = CDROM_MSF;
    for ( int i = 0; i < numTracks; i++ ) {
        tocentry.cdte_track = i + firstTrack;
        if ( ioctl( fd, CDROMREADTOCENTRY, &tocentry ) ) {
            // Could not get TOC Entry --> noDisc().
            delete [] trackStart;
            return 0;
        }
        trackStart[i] = CD_FRAMES * ( CD_SECS * tocentry.cdte_addr.msf.minute + tocentry.cdte_addr.msf.second ) + tocentry.cdte_addr.msf.frame;
    }

	// Get the start time of the leadout track.
    tocentry.cdte_track = CDROM_LEADOUT;
    if ( ioctl( fd, CDROMREADTOCENTRY, &tocentry ) ) {
        // Could not get TOC Entry --> noDisc().
        delete [] trackStart;
        return 0;
    }
    trackStart[numTracks] = CD_FRAMES * ( CD_SECS * tocentry.cdte_addr.msf.minute + tocentry.cdte_addr.msf.second ) + tocentry.cdte_addr.msf.frame;
	
	// Create CompactDiscID, and compare it to the current CompactDiscID.
    CompactDiscID* id = new CompactDiscID( numTracks, firstTrack, trackStart );
    delete [] trackStart;
	
	return id;
}

bool IdeCdrom::init( int fd )
{
	return stop( fd );
}

bool IdeCdrom::pause( int fd )
{
	return ioctl( fd, CDROMPAUSE, 0 ) == 0;
}

bool IdeCdrom::play( int fd, int track )
{
	struct cdrom_ti ti;
	ti.cdti_trk0 = track;
	ti.cdti_ind0 = 0;
	ti.cdti_trk1 = track;
	ti.cdti_ind1 = 0;
	return ioctl( fd, CDROMPLAYTRKIND, &ti ) == 0;
}

bool IdeCdrom::resume( int fd )
{
	return ioctl( fd, CDROMRESUME, 0 ) == 0;
}

bool IdeCdrom::status( int fd, CdromStatus& status, int& absTime, int& volume )
{
	struct cdrom_subchnl subchnl;
	subchnl.cdsc_format = CDROM_MSF;
	bool s = ioctl( fd, CDROMSUBCHNL, &subchnl );
	
	if ( !s ) {
		_ejectToggle = false;

		switch ( subchnl.cdsc_audiostatus ) {
		 case CDROM_AUDIO_INVALID:
		 case CDROM_AUDIO_ERROR:
			status = Error;
			break;
			
		 case CDROM_AUDIO_PLAY:
			status = Playing;
			break;
			
		 case CDROM_AUDIO_PAUSED:
			status = Paused;
			break;
			
		 case CDROM_AUDIO_COMPLETED:
			status = Completed;
			break;
			
		 case CDROM_AUDIO_NO_STATUS:
			status = Stopped;
			break;
		}
		
		absTime = CD_FRAMES * ( CD_SECS * subchnl.cdsc_absaddr.msf.minute + subchnl.cdsc_absaddr.msf.second ) + subchnl.cdsc_absaddr.msf.frame;

		struct cdrom_volctrl volctrl;

		if ( ioctl( fd, CDROMVOLREAD, &volctrl ) == 0 ) {
			volume = volctrl.channel0 / 4;
		} else {
			volume = 32;
		}
	}
	
	return s;
}

bool IdeCdrom::stop( int fd )
{
	return ioctl( fd, CDROMSTOP, 0 ) == 0;
}

bool IdeCdrom::volume( int fd, int vol )
{
	struct cdrom_volctrl volctrl;
	
	volctrl.channel0 = vol * 4;
    volctrl.channel1 = vol * 4;
    volctrl.channel2 = vol * 4;
    volctrl.channel3 = vol * 4;
    return ioctl( fd, CDROMVOLCTRL, &volctrl ) == 0;
}
