#undef DEBUG_FORKED
#undef DEBUG_CLEANUP
#undef DEBUG_DYN_OVERLAP
#undef DEBUG_READS
/*
 * Copyright: GNU Public License 2 applies
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * CDDA2WAV (C) 1995-1998 Heiko Eissfeldt heiko@colossus.escape.de
 * parts    (C) Peter Widow
 * parts    (C) Thomas Niederreiter
 * parts    (C) RSA Data Security, Inc.
 *
 * last changes:
 *   18.12.93 - first version,	OK
 *   01.01.94 - generalized & clean up HE
 *   10.06.94 - first linux version HE
 *   12.06.94 - wav header alignment problem fixed HE
 *   12.08.94 - open the cdrom device O_RDONLY makes more sense :-)
 *		no more floating point math
 *		change to sector size 2352 which is more common
 *		sub-q-channel information per kernel ioctl requested
 *		doesn't work as well as before
 *		some new options (-max -i)
 *   01.02.95 - async i/o via semaphores and shared memory
 *   03.02.95 - overlapped reading on sectors
 *   03.02.95 - generalized sample rates. all integral divisors are legal
 *   04.02.95 - sun format added
 *              more divisors: all integral halves >= 1 allowed
 *		floating point math needed again
 *   06.02.95 - bugfix for last track and not d0
 *              tested with photo-cd with audio tracks
 *		tested with xa disk 
 *   29.01.96 - new options for bulk transfer
 *   01.06.96 - tested with enhanced cd
 *   01.06.96 - tested with cd-plus
 *   02.06.96 - support pipes
 *   02.06.96 - support raw format
 *   04.02.96 - security hole fixed
 *   22.04.97 - large parts rewritten
 *   28.04.97 - make file names DOS compatible
 *   01.09.97 - add speed control
 *   20.10.97 - add find mono option
 *   Jan/Feb 98 - conversion to use Joerg Schillings SCSI library
 *
 */

#include "config.h"

#include "sndconfig.h"
#if defined (HAVE_UNISTD_H) && (HAVE_UNISTD_H == 1)
#include <sys/types.h>
#include <unistd.h>
#endif

#if defined(_GNU_C_SOURCE) || defined(__USE_GNU)
#define USE_GETOPT_LONG
#endif

#include <standard.h>
#include <stdio.h>
/*#include <stdarg.h>*/
#include <stdlib.h>
#include <string.h>
#if defined (HAVE_STRINGS_H) && (HAVE_STRINGS_H == 1)
#include <strings.h>
#endif
#include <signal.h>
#include <math.h>
#if defined (HAVE_FCNTL_H) && (HAVE_FCNTL_H == 1)
#include <fcntl.h>
#endif
#include <time.h>
#if (defined (HAVE_SYS_TIME_H) || (HAVE_SYS_TIME_H != 1)) && defined (TIME_WITH_SYS_TIME)
#include <sys/time.h>
#endif
#if defined (HAVE_LIMITS_H) && (HAVE_LIMITS_H == 1)
#include <limits.h>
#endif
#if defined (HAVE_SYS_IOCTL_H) && (HAVE_SYS_IOCTL_H == 1)
#include <sys/ioctl.h>
#endif

#if defined (HAVE_SYS_WAIT_H) && (HAVE_SYS_WAIT_H == 1)
#include <sys/wait.h>
#endif
#include <vadefs.h>

#include "semshm.h"	/* semaphore functions */
#include "wav.h"	/* wav file header structures */
#include "sun.h"	/* sun audio file header structures */
#include "raw.h"	/* raw file handling */
#include "interface.h"  /* low level cdrom interfacing */
#include "cdda2wav.h"
#include "resample.h"
#include "toc.h"
#include "setuid.h"
#include "ringbuff.h"
#include "global.h"
#include "mycdrom.h"

#include <scsitransp.h>

static const char * optstring = "D:A:I:O:c:b:r:a:t:i:d:o:n:v:l:E:C:S:p:M:P:smweNqxRBVhFGHTJ";

extern char *optarg;
extern int optind, opterr, optopt;

int main				__PR((int argc, char **argv));
static void RestrictPlaybackRate	__PR((long newrate));
static void output_indices		__PR((FILE *fp, index_list *p));
static int write_info_file		__PR((char *fname_baseval, unsigned int track, unsigned long SamplesDone, int numbered));
static void CloseAudio			__PR((char *fname_baseval, unsigned int track, int bulkflag, int channels_val, unsigned long nSamples));
static void CloseAll			__PR((void));
static void OpenAudio			__PR((char *fname, double rate, long nBitsPerSample, long channels_val, unsigned long expected_bytes));
static void set_offset			__PR((myringbuff *p, int offset));
static int get_offset			__PR((myringbuff *p));
static void usage			__PR((void));
static void init_globals		__PR((void));

#ifdef USE_GETOPT_LONG
#include <getopt.h>	/* for get_long_opt () */
static struct option options [] = {
	{"device",required_argument,NULL,'D'},
	{"auxdevice",required_argument,NULL,'A'},
	{"interface",required_argument,NULL,'I'},
	{"output-format",required_argument,NULL,'O'},
	{"channels",required_argument,NULL,'c'},
	{"bits-per-sample",required_argument,NULL,'b'},
	{"rate",required_argument,NULL,'r'},
	{"divider",required_argument,NULL,'a'},
	{"track",required_argument,NULL,'t'},
	{"index",required_argument,NULL,'i'},
	{"duration",required_argument,NULL,'d'},
	{"offset",required_argument,NULL,'o'},
	{"sectors-per-request",required_argument,NULL,'n'},
	{"buffers-in-ring",required_argument,NULL,'l'},
	{"output-endianess",required_argument,NULL,'E'},
	{"cdrom-endianess",required_argument,NULL,'C'},
	{"verbose-level",required_argument,NULL,'v'},
#ifdef MD5_SIGNATURES
	{"md5",required_argument,NULL,'M'},
#endif
	{"stereo",no_argument,NULL,'s'},
	{"mono",no_argument,NULL,'m'},
	{"wait",no_argument,NULL,'w'},
	{"find-extremes",no_argument,NULL,'F'},
	{"find-mono",no_argument,NULL,'G'},
#ifdef	ECHO_TO_SOUNDCARD
	{"echo",no_argument,NULL,'e'},
#endif
	{"info-only",no_argument,NULL,'J'},
	{"no-write",no_argument,NULL,'N'},
	{"no-infofile",no_argument,NULL,'H'},
	{"quiet",no_argument,NULL,'q'},
	{"max",no_argument,NULL,'x'},

	{"set-overlap",required_argument,NULL,'P'},
	{"speed-select",required_argument,NULL,'S'},

	{"dump-rates",no_argument,NULL,'R'},
	{"bulk",no_argument,NULL,'B'},
	{"deemphasize",no_argument,NULL,'T'},
	{"playback-realtime",required_argument,NULL,'p'},
	{"verbose-SCSI",no_argument,NULL,'V'},
	{"help",no_argument,NULL,'h'},
	{NULL,0,NULL,0}
};

#endif /* USE_GETOPT_LONG */


/* global variables */
global_t global;

/* static variables */
static int (*InitAudioFile) __PR(( int audioflag, long channels,
			    unsigned long rate,
			    long nBitsPerSample,
			    unsigned long expected_bytes));
static int (*ExitAudioFile) __PR(( int audioflag, unsigned long nBytesDone ));
static unsigned long (*GetSndHdrSize) __PR(( void ));

static unsigned long nSamplesDone = 0;

static	int child_pid = 0xff;

static unsigned long nSamplesToDo;
static unsigned int current_track;
static int bulk = 0;


static void RestrictPlaybackRate( newrate )
	long newrate;
{
       global.playback_rate = newrate;

       if ( global.playback_rate < 25 ) global.playback_rate = 25;   /* filter out insane values */
       if ( global.playback_rate > 250 ) global.playback_rate = 250;

       if ( global.playback_rate < 100 )
               global.nsectors = (global.nsectors*global.playback_rate)/100;
}


long SamplesNeeded( amount, undersampling_val)
	long amount;
	long undersampling_val;
{
  long retval = ((undersampling_val * 2 + Halved)*amount)/2;
  if (Halved && (nSamplesToDo & 1))
    retval += 2;
  return retval;
}

#ifdef INFOFILES
static void output_indices(fp, p)
	FILE *fp;
	index_list *p;
{
  int ci = 0;

  fprintf(fp, "Index =");
  if (p == NULL) {
    fprintf(fp, "0\n");
    return;
  }

  while (p != NULL) {
    int frameoff = p->frameoffset;

    ci++;
    if ( ci > 8 && (ci % 8) == 1) fputs("\nIndex = ", fp);
    if (frameoff == -1) {
      fprintf(fp, "unavai, ");
    } else {
      fprintf(fp, "%6d, ", frameoff);
    }
    p = p->next;
  }
  fputs("\n", fp);
}

static int write_info_file(fname_baseval, track, SamplesDone, numbered)
	char *fname_baseval;
	unsigned int track;
	unsigned long int SamplesDone;
	int numbered;
{
  FILE *info_fp;
  char fname[200];
  char datetime[30];
  time_t utc_time;
  struct tm *tmptr;

  /* write info file */
  if (!strcmp(fname_baseval,"standard output")) return 0;

  if (numbered)
    sprintf(fname, "%s_%02u.inf", fname_baseval, track);
  else {
    char *pp;

    strncpy(fname, fname_baseval, sizeof(fname) -1);
    fname[sizeof(fname) -1] = 0;
    pp = strrchr(fname, '.');
    if (pp == NULL) {
      pp = fname + strlen(fname);
    }
    strncpy(pp, ".inf", sizeof(fname) - 1 - (pp - fname));
  }
  info_fp = fopen (fname, "w");
  if (!info_fp)
    return -1;

#ifdef MD5_SIGNATURES
  if (global.md5blocksize)
    MD5Final (global.MD5_result, &global.context);
#endif

  utc_time = time(NULL);
  tmptr = localtime(&utc_time);
  if (tmptr) {
    strftime(datetime, sizeof(datetime), "%x %X", tmptr);
  } else {
    strncpy(datetime, "unknown", sizeof(datetime));
  }
  fprintf(info_fp, "#created by cdda2wav %s-%s %s\n", VERSION, HOST
	  , datetime
	  );
  fprintf(info_fp, "Track = %u"
	  , track
	  );
  if (global.tracktitle[track-1] != NULL) {
    fprintf(info_fp, "Title = '%s'", global.tracktitle[track-1]);
  }
  fprintf(info_fp, 
	  "\nLength = %ld milliseconds\n"
	  , (SamplesDone*10 + 220) /441);	/* in milliseconds */
  fprintf(info_fp, 
	  "ISRC = %s\n"
	  , g_toc[track-1].ISRC);
#ifdef MD5_SIGNATURES
  fprintf(info_fp, 
	  "MD-5 = (%d) %02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\n"
	  , global.md5blocksize
	  , global.MD5_result[0]
	  , global.MD5_result[1]
	  , global.MD5_result[2]
	  , global.MD5_result[3]
	  , global.MD5_result[4]
	  , global.MD5_result[5]
	  , global.MD5_result[6]
	  , global.MD5_result[7]
	  , global.MD5_result[8]
	  , global.MD5_result[9]
	  , global.MD5_result[10]
	  , global.MD5_result[11]
	  , global.MD5_result[12]
	  , global.MD5_result[13]
	  , global.MD5_result[14]
	  , global.MD5_result[15]);
#endif
  fprintf(info_fp, 
	  "Pre-emphasis = %s\n"
	  , g_toc[track-1].bFlags & 1 ? "yes" : "no");
  fprintf(info_fp, 
	  "Channels = %d\n"
	  , g_toc[track-1].bFlags & 8 ? 4 : 2);
  fprintf(info_fp, 
	  "Copy permitted = %s\n"
	  , g_toc[track-1].bFlags & 2 ? "yes" : "no");
  fprintf(info_fp, 
	  "Endianess = %s\n"
	  , global.need_big_endian ? "big" : "little"
	  );
    fprintf(info_fp, "# index list\n");
    output_indices(info_fp, global.trackindexlist[track-1]);
    fprintf(info_fp, 
"Media Catalog Number (UPC/EAN) = %s\nCDDB discid = 0x%08lx\nAlbum title = %s\n"
	  , MCN
	  , (unsigned long) global.cddb_id
	  , global.disctitle != NULL ? global.disctitle : (const unsigned char *)""
	  );
  fclose(info_fp);
  return 0;
}
#endif

static void CloseAudio(fname_baseval, track, bulkflag, channels_val, nSamples)
	char *fname_baseval;
	unsigned int track;
	int bulkflag;
	int channels_val;
	unsigned long nSamples;
{
#ifdef INFOFILES
      /* define sizes in the header of the audio file */
      if (!global.no_infofile)
            write_info_file(fname_baseval,track,nSamples,bulkflag);
#endif

      /* define length */
      ExitAudioFile( global.audio, nSamples*global.OutSampleSize*channels_val );

      close (global.audio);
      global.audio = -1;
}

static unsigned int track = 1;

/* On terminating:
 * define size-related entries in audio file header, update and close file */
static void CloseAll ()
{
  int chld_return_status = 0;
  int amichild;

  /* terminate child process first */
  amichild = child_pid == 0;

  if (amichild) {
#ifdef DEBUG_CLEANUP
fprintf(stderr, "Child terminating, \n");
#endif

    if (global.iloop > 0) {
      /* set to zero */
      global.iloop = 0;
      /*return;*/
    }

    /* switch to original mode and close device */
    EnableCdda ( 0);
    _exit(0);
  }
#ifdef DEBUG_CLEANUP
fprintf(stderr, "Parent terminating, \n");
#endif

#ifdef DEBUG_CLEANUP
fprintf(stderr, "Parent wait for child death, \n");
#endif

  fputs("\n", stderr);

  if (global.have_forked == 1) {

#if defined(HAVE_SYS_WAIT_H) && (HAVE_SYS_WAIT_H == 1)
    /* wait for child to terminate */
    if (0 > wait(&chld_return_status)) {
      perror("");
    } else {
      if (WIFEXITED(chld_return_status)) {
        if (WEXITSTATUS(chld_return_status)) {
          fprintf(stderr, "Child exited with %d\n", WEXITSTATUS(chld_return_status));
        }
      }
      if (WIFSIGNALED(chld_return_status)) {
        fprintf(stderr, "Child exited due to signal %d\n", WTERMSIG(chld_return_status));
      }
      if (WIFSTOPPED(chld_return_status)) {
        fprintf(stderr, "Child is stopped due to signal %d\n", WSTOPSIG(chld_return_status));
      }
    }
#endif

#ifdef DEBUG_CLEANUP
fprintf(stderr, "Parent child death, state:%d\n", chld_return_status);
#endif
  }
  /* do general clean up */

  if (global.audio>=0) {
    if (bulk) {
      /* finish sample file for this track */
      CloseAudio(global.fname_base, current_track, bulk, global.channels,
  	  global.nSamplesDoneInTrack);
    } else {
      /* finish sample file for this track */
      CloseAudio(global.fname_base, track, bulk, global.channels,
  	  (unsigned int) nSamplesToDo);
    }
  }

  /* tell minimum and maximum amplitudes, if required */
  if (global.findminmax) {
    fprintf(stderr, "minimum amplitude :%d/-32768, maximum amplitude :%d/32767\n", 
	global.minamp, global.maxamp);
  }

  /* tell mono or stereo recording, if required */
  if (global.findmono) {
    fprintf(stderr, "Audio samples are originally %s.\n", global.ismono ? "mono" : "stereo");
  }
}


/* report a fatal error, clean up and exit */
#ifdef  PROTOTYPES
void FatalError (const char *szMessage, ...)
#else
void FatalError (szMessage, va_alist)
	const char *szMessage;
	va_dcl
#endif
{
  va_list marker;

#ifdef  PROTOTYPES
  va_start(marker, szMessage);
#else
  va_start(marker);
#endif

  vfprintf (stderr, szMessage, marker);

  va_end(marker);

  if (child_pid == 0) {
    /* kill the parent too */
    kill(getppid(), SIGINT);
  }
  exit (1);
}


/* open the audio output file and prepare the header. 
 * the header will be defined on terminating (when the size
 * is known). So hitting the interrupt key leaves an intact
 * file.
 */
static void OpenAudio (fname, rate, nBitsPerSample, channels_val, expected_bytes)
	char *fname;
	double rate;
	long nBitsPerSample;
	long channels_val;
	unsigned long expected_bytes;
{
  if (global.audio == -1) {

    global.audio = open (fname, O_CREAT | O_WRONLY | O_TRUNC
#ifdef O_NONBLOCK
			 | O_NONBLOCK
#endif
#ifdef SYNCHRONOUS_WRITE
			 | O_SYNC
#endif
		  , 0666);
    if (global.audio == -1) {
      perror("open audio sample file");
      FatalError ("Could not open file %s\n", fname);
    }
  }
  InitAudioFile( global.audio, channels_val, (unsigned long)rate, nBitsPerSample, expected_bytes );

#ifdef MD5_SIGNATURES
  if (global.md5blocksize)
    MD5Init (&global.context);
  global.md5count = global.md5blocksize;
#endif
}


static void set_offset(p, offset)
	myringbuff *p;
	int offset;
{
#ifdef DEBUG_SHM
  fprintf(stderr, "Write offset %d at %p\n", offset, &p->offset);
#endif
  p->offset = offset;
}


static int get_offset(p)
myringbuff *p;
{
#ifdef DEBUG_SHM
  fprintf(stderr, "Read offset %d from %p\n", p->offset, &p->offset);
#endif
  return p->offset;
}


static void usage( )
{
  fprintf( stderr,
"cdda2wav [-c chans] [-s] [-m] [-b bits] [-r rate] [-a divider] [-S speed]\n");
  fprintf( stderr,
"         [-t track[+endtrack]] [-i index] [-o offset] [-d duration] [-F] [-G]\n");
  fprintf( stderr,
"         [-x] [-q] [-w] [-v] [-R] [-P overlap] [-T]\n");
  fprintf( stderr,
"	  [-e] [-n sectors] [-N] [-J] [-H] [-l buffers] [-D device] [-I interface]\n");
  fprintf( stderr,
"         [-O audiotype] [-B] [-E output-endianess] [-A auxdevice] [audiofile]\n");
  fprintf( stderr,
"Version %s-%s\n", VERSION, HOST);
  fprintf( stderr,
"cdda2wav copies parts from audio cd's directly to wav or au files.\n");
  fprintf( stderr,
"It requires a supported cdrom drive to work.\n");
  fprintf( stderr,
"options: -D device   : set the cdrom or scsi device (as Bus,Id,Lun).\n");
  fprintf( stderr,
"         -A auxdevice: set the aux device (typically /dev/cdrom).\n");
  fprintf( stderr,
"         -I interface: specify the interface for cdrom access.\n");
  fprintf( stderr,
"                     : (generic_scsi or cooked_ioctl).\n");
  fprintf( stderr,
"         -c channels : set 1 for mono, 2 or s for stereo (s: channels swapped).\n");
  fprintf( stderr,
"         -s          : set to stereo recording.\n");
  fprintf( stderr,
"         -m          : set to mono recording.\n");
  fprintf( stderr,
"         -x          : set to maximum quality (stereo/16-bit/44.1 KHz).\n");
  fprintf( stderr,
"         -b bits     : set bits per sample per channel (8, 12 or 16 bits).\n");
  fprintf( stderr,
"         -r rate     : set rate in samples per second. -R gives all rates\n");
  fprintf( stderr,
"         -a divider  : set rate to 44100Hz / divider. -R gives all rates\n");
  fprintf( stderr,
"         -R          : dump a table with all available sample rates\n");
  fprintf( stderr,
"         -S speed    : set the cdrom drive to a given speed during reading\n");
  fprintf( stderr,
"         -P sectors  : set amount of overlap sampling (default is 0)\n");
#ifdef NOTYET
  fprintf( stderr,
"         -X          : switch off extra alignment paranoia; implies -Y -Z\n");
  fprintf( stderr,
"         -Y          : switch off scratch detection; implies -Z\n");
  fprintf( stderr,
"         -Z          : switch off scratch reconstruction\n");
  fprintf( stderr,
"         -y threshold: tune scratch detection filter; low values catch\n");
  fprintf( stderr,
"                     : more scratches but may also catch legitimate values\n");
  fprintf( stderr,
"         -z          : disable 'smart' scratch auto-detect and force \n");
  fprintf( stderr,
"                     : paranoia to treat all data as possibly scratched\n\n");
#endif
  fprintf( stderr,
"         -n sectors  : read sectors per request.\n");
  fprintf( stderr,
"         -l buffers  : use a ring buffer with 'l' elements.\n");
  fprintf( stderr,
"         -t track[+end track]\n");
  fprintf( stderr,
"                     : select start track (and optionally end track).\n");
  fprintf( stderr,
"         -i index    : select start index.\n");
  fprintf( stderr,
"         -o offset   : start at 'offset' sectors behind start track/index.\n");
  fprintf( stderr,
"                       one sector equivalents 1/75 second.\n");
  fprintf( stderr,
"         -O audiotype: set wav or sun or cdr (aka raw) audio format. Default is wav.\n");
  fprintf( stderr,
"         -C endianess: set little or big input sample endianess.\n");
  fprintf( stderr,
"         -E endianess: set little or big output sample endianess.\n");
  fprintf( stderr,
"         -d duration : set recording time in seconds or 0 for whole track.\n");
  fprintf( stderr,
"         -B          : record each track into a seperate file.\n");
  fprintf( stderr,
"         -w          : wait for audio signal, then start recording.\n");
  fprintf( stderr,
"         -F          : find extrem amplitudes in samples.\n");
  fprintf( stderr,
"         -G          : find if input samples are mono.\n");
  fprintf( stderr,
"         -T          : undo pre-emphasis in input samples.\n");
#ifdef	ECHO_TO_SOUNDCARD
  fprintf( stderr,
"         -e          : echo audio data to sound device SOUND_DEV.\n");
#endif
  fprintf( stderr,
"         -v level    : print informations on current cd (level: 0-63).\n");
  fprintf( stderr,
"         -N          : no file operation.\n");
  fprintf( stderr,
"         -J          : give disc information only.\n");
  fprintf( stderr,
"         -H          : no info file generation.\n");
#ifdef MD5_SIGNATURES
  fprintf( stderr,
"         -M count    : calculate MD-5 checksum for 'count' bytes.\n");
#endif
  fprintf( stderr,
"         -q          : quiet operation, no screen output.\n");
  fprintf( stderr,
"         -p percent  : play (echo) audio at a new pitch rate.\n");
  fprintf( stderr,
"         -V          : increase verbosity for SCSI commands.\n");
  fprintf( stderr,
"         -h          : this help screen.\n"  );
  fprintf( stderr,
"defaults: %s, %d bit, %g Hz, track 1, no offset, one track,\n",
	  CHANNELS-1?"stereo":"mono", BITS_P_S, 44100.0 / UNDERSAMPLING);
  fprintf( stderr,
"          type %s '%s', don't wait for signal, not quiet, not verbose,\n",
          AUDIOTYPE, FILENAME);
  fprintf( stderr,
"          use %s, device %s, aux %s\n",
	  DEF_INTERFACE, CD_DEVICE, AUX_DEVICE);
  fprintf( stderr,
"parameters: (optional) a file name or - for standard output.\n");
  exit( 1 );
}

static void init_globals()
{
  strncpy(global.dev_name, CD_DEVICE, sizeof(global.dev_name));	/* device name */
  strncpy(global.aux_name, AUX_DEVICE, sizeof(global.aux_name));/* auxiliary cdrom device */
  strncpy(global.fname_base, FILENAME, sizeof(global.fname_base));/* auxiliary cdrom device */
  global.have_forked = 0;	/* state variable for clean up */
  global.audio    = -1;		/* audio file desc */
  global.cooked_fd  = -1;	/* cdrom file desc */
  global.no_file  =  0;		/* flag no_file */
  global.no_infofile  =  0;	/* flag no_infofile */
  global.no_cddbfile  =  0;	/* flag no_cddbfile */
  global.quiet	  =  0;		/* flag quiet */
  global.verbose  =  3 + 32 + 64;	/* verbose level */
  global.scsi_verbose = 0;		/* SCSI verbose level */
  global.sh_bits  =  0;		/* sh_bits: sample bit shift */
  global.Remainder=  0;		/* remainder */
  global.iloop    =  0;		/* todo counter */
  global.SkippedSamples =  0;	/* skipped samples */
  global.OutSampleSize  =  0;	/* output sample size */
  global.channels = CHANNELS;	/* output sound channels */
  global.nSamplesDoneInTrack = 0; /* written samples in current track */
  global.buffers = 2;           /* buffers to use */
  global.nsectors = NSECTORS;   /* sectors to read in one request */
  global.overlap = 1;           /* amount of overlapping sectors */
  global.useroverlap = -1;       /* amount of overlapping sectors user override */
  global.outputendianess = NONE; /* user specified output endianess */
  global.littleendian = -1; /* user specified output endianess */
  global.findminmax  =  0;	/* flag find extrem amplitudes */
#ifdef HAVE_LIMITS_H
  global.maxamp = INT_MIN;	/* maximum amplitude */
  global.minamp = INT_MAX;	/* minimum amplitude */
#else
  global.maxamp = 32767;	/* maximum amplitude */
  global.minamp = -32768;	/* minimum amplitude */
#endif
  global.speed = DEFAULT_SPEED; /* use default */ 
  global.findmono  =  0;	/* flag find if samples are mono */
  global.ismono  =  1;		/* flag if samples are mono */
  global.swapchannels  =  0;	/* flag if channels shall be swapped */
  global.deemphasize  =  0;	/* flag undo pre-emphasis in samples */
  global.playback_rate = 100;   /* new fancy selectable sound output rate */
  global.cddb_id = 0;           /* disc identifying id for CDDB database */
  global.disctitle = NULL;
  global.creator = NULL;
  global.copyright_message = NULL;
  memset(global.tracktitle, 0, sizeof(global.tracktitle));
  memset(global.trackindexlist, 0, sizeof(global.trackindexlist));
}

#if !defined (HAVE_STRCASECMP) || (HAVE_STRCASECMP != 1)
#include <ctype.h>
static int strcasecmp __PR(( const char *s1, const char *s2 ));
static int strcasecmp(s1, s2)
	const char *s1;
	const char *s2;
{
  while (s1 && s2 && tolower(*s1) == tolower(*s2)) {
    s1++;
    s2++;
  }
  if (s1 == NULL && s2 == NULL) return 0;
  if (s1 == NULL) return -1;
  if (s2 == NULL) return +1;
  return tolower(*s1) - tolower(*s2);
}
#endif

#if !defined (HAVE_STRTOUL) || (HAVE_STRTOUL != 1)
static unsigned int strtoul __PR(( const char *s1, char **s2, int base ));
static unsigned int strtoul(s1, s2, base)
        const char *s1;
        char **s2;
	int base;
{
	long retval;

	if (base == 10) {
		/* strip zeros in front */
		while (*s1 == '0')
			s1++;
	}
	if (s2 != NULL) {
		*s2 = astol(s1, &retval);
	} else {
		(void) astol(s1, &retval);
	}

	return (unsigned long) retval; 	
}
#endif

static unsigned long SectorBurst;
#if (SENTINEL > CD_FRAMESIZE_RAW)
error block size for overlap check has to be < sector size
#endif


static void
switch_to_realtime_priority __PR((void));

#ifdef  HAVE_SYS_PRIOCNTL_H

#include <sys/priocntl.h>
#include <sys/rtpriocntl.h>
static void
switch_to_realtime_priority()
{
        pcinfo_t        info;
        pcparms_t       param;
        rtinfo_t        rtinfo;
        rtparms_t       rtparam;
	int		pid;

	pid = getpid();

        /* get info */
        strcpy(info.pc_clname, "RT");
        if (-1 == priocntl(P_PID, pid, PC_GETCID, (void *)&info))
                comerr("Cannot get priority class id priocntl(PC_GETCID)\n");

        movebytes(info.pc_clinfo, &rtinfo, sizeof(rtinfo_t));

        /* set priority not to the max */
        rtparam.rt_pri = rtinfo.rt_maxpri - 2;
        rtparam.rt_tqsecs = 0;
        rtparam.rt_tqnsecs = RT_TQDEF;
        param.pc_cid = info.pc_cid;
        movebytes(&rtparam, param.pc_clparms, sizeof(rtparms_t));
	needroot(0);
        if (-1 == priocntl(P_PID, pid, PC_SETPARMS, (void *)&param))
                comerr("Cannot set priority class parameters priocntl(PC_SETPARMS)\n");
	dontneedroot();
}
#else
#if      defined _POSIX_PRIORITY_SCHEDULING
#include <sched.h>

static void
switch_to_realtime_priority()
{
#ifdef  _SC_PRIORITY_SCHEDULING
	if (sysconf(_SC_PRIORITY_SCHEDULING) == -1) {
		errmsg("WARNING: RR-scheduler not available, disabling.\n");
	} else
#endif
	{
	int sched_fifo_min, sched_fifo_max;
	struct sched_param sched_parms;

	sched_fifo_min = sched_get_priority_min(SCHED_FIFO);
	sched_fifo_max = sched_get_priority_max(SCHED_FIFO);
	sched_parms.sched_priority = sched_fifo_max - 1;
	needroot(0);
	if (-1 == sched_setscheduler(getpid(), SCHED_FIFO, &sched_parms)
		&& global.quiet != 1)
		perror("cannot set posix realtime scheduling policy");
	dontneedroot();
	}
}
#else
static void
switch_to_realtime_priority()
{
}
#endif
#endif

static long lSector;
static long lSector_p2;
static double rate = 44100 / UNDERSAMPLING;
static int bits = BITS_P_S;
static char fname[200];
static char audio_type[10];
static long BeginAtSample;
static unsigned long SamplesToWrite; 
static unsigned minover;
static unsigned maxover;

static unsigned long calc_SectorBurst __PR((void));
static unsigned long calc_SectorBurst()
{
	unsigned long SectorBurstVal;

      	SectorBurstVal = min(global.nsectors,
		(global.iloop + CD_FRAMESAMPLES-1) / CD_FRAMESAMPLES);
 
	if ( lSector+(int)SectorBurst-1 >= lSector_p2 )
		SectorBurstVal = lSector_p2 - lSector;

	return SectorBurstVal;
}

static int do_read __PR((myringbuff *p, unsigned *total_unsuccessful_retries));
static int do_read (p, total_unsuccessful_retries)
	myringbuff	*p;
	unsigned	*total_unsuccessful_retries;
{
      unsigned char *newbuf;
      int offset;
      unsigned int retry_count;
      unsigned int added_size;

      /* how many sectors should be read */
      SectorBurst =  calc_SectorBurst();

#define MAX_READRETRY 12

      retry_count = 0;
      do {
#ifdef DEBUG_READS
	fprintf(stderr, "reading from %lu to %lu, overlap %u\n", lSector, lSector + SectorBurst -1, global.overlap);
#endif
          ReadCdRom(  p->data, lSector, SectorBurst );
          if (NULL ==
                 (newbuf = synchronize( p->data, SectorBurst*CD_FRAMESAMPLES,
                			nSamplesToDo-global.iloop ))) {
	    /* could not synchronize!
	     * Try to invalidate the cdrom cache.
	     * Increase overlap setting, if possible.
	     */	
	    if (global.overlap < global.nsectors - 1) {
	        global.overlap++;
		lSector--;
 		SectorBurst =  calc_SectorBurst();
#ifdef DEBUG_DYN_OVERLAP
		fprintf(stderr, "using increased overlap of %u\n", global.overlap);
#endif
	    } else {
		lSector += global.overlap - 1;
	        global.overlap = 1;
 		SectorBurst =  calc_SectorBurst();
	    }
	  } else
		break;
      } while (++retry_count < MAX_READRETRY);

      if (retry_count == MAX_READRETRY && newbuf == NULL && global.verbose != 0) {
        (*total_unsuccessful_retries)++;
      }

      if (newbuf) {
        offset = newbuf - ((unsigned char *)p->data);
      } else {
        offset = global.overlap * CD_FRAMESIZE_RAW;
      }
      set_offset(p,offset);

      /* how much has been added? */
      added_size = SectorBurst * CD_FRAMESAMPLES - offset/4;

      if (newbuf && nSamplesToDo != global.iloop) {
	minover = min(global.overlap, minover);
	maxover = max(global.overlap, maxover);


	/* should we reduce the overlap setting ? */
	if (offset > CD_FRAMESIZE_RAW && global.overlap > 1) {
#ifdef DEBUG_DYN_OVERLAP
          fprintf(stderr, "decreasing overlap from %u to %u (jitter %d)\n", global.overlap, global.overlap-1, offset - (global.overlap)*CD_FRAMESIZE_RAW);
#endif
	  global.overlap--;
	  SectorBurst =  calc_SectorBurst();
	}
      }

      if (global.iloop >= added_size)
        global.iloop -= added_size;
      else
        global.iloop = 0;

      if (global.verbose)
	fprintf(stderr,
	"\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b%2d/%2d/%2d/%7d %3ld%%",
	minover, maxover, global.overlap,
	newbuf ? offset - global.overlap*CD_FRAMESIZE_RAW : 9999999,
	global.iloop ? (nSamplesToDo-global.iloop)/(nSamplesToDo/100) : 100);

      lSector += SectorBurst - global.overlap;

      return offset;
}

#define left_in_this_track(pos, track) (max(0,(int)g_toc[track].dwStartSector*CD_FRAMESAMPLES - (int)(pos)))

static unsigned long do_write __PR((myringbuff *p));
static unsigned long do_write (p)
	myringbuff	*p;
{
      int current_offset;
      unsigned int Sectors;
      current_offset = get_offset(p);

      /* how many bytes are available */
      Sectors = global.nsectors*CD_FRAMESAMPLES - current_offset/4;
      Sectors = min((nSamplesToDo-nSamplesDone),Sectors);

      /* when track end is reached, close current file and start a new one */
      while ((nSamplesDone < nSamplesToDo) && (Sectors != 0)) {
	long unsigned int how_much = Sectors;
	long unsigned int left_in_track = left_in_this_track(BeginAtSample+nSamplesDone,
							     current_track);

	if (bulk)
	  how_much = min(how_much, left_in_track);


#ifdef MD5_SIGNATURES
	if (global.md5count) {
	  MD5Update (&global.context, ((unsigned char *)p->data) +current_offset, min(global.md5count,how_much));
	  global.md5count -= min(global.md5count,how_much);
	}
#endif
	if ( SaveBuffer ( p->data + current_offset/4,
			 how_much,
			 &nSamplesDone) ) {
	  /* kill child */
	  kill(child_pid, SIGINT);
	  exit(2);
	}
        global.nSamplesDoneInTrack += how_much;
	SamplesToWrite -= nSamplesDone;

	/* move residual samples upto buffer start */
	if (how_much < Sectors) 
		movebytes(
		  (char *)(p->data) + current_offset + how_much*4,
		  (char *)(p->data) + current_offset,
		  (Sectors - how_much) * 4);
	if (left_in_track < Sectors) {

	  if (bulk) {
	    /* finish sample file for this track */
	    CloseAudio(global.fname_base, current_track, bulk, global.channels,
	  	  global.nSamplesDoneInTrack);
          } else if (SamplesToWrite == 0) {
	    /* finish sample file for this track */
	    CloseAudio(global.fname_base, track, bulk, global.channels,
	  	  (unsigned int) nSamplesToDo);
	  }

	  if (global.verbose) {
	    if (global.tracktitle[current_track -1] != NULL) {
	      fprintf( stderr, "  track '%s' successfully recorded\n", global.tracktitle[current_track-1]);
            } else {
	      fprintf( stderr, "  track %2u successfully recorded\n", current_track);
            }
          }

          global.nSamplesDoneInTrack = 0;
	  if ( bulk && SamplesToWrite > 0 ) {
	    if ( !global.no_file ) {

	      /* build next filename */
	      sprintf(fname, "%s_%02u.%s",global.fname_base,current_track+1,
			audio_type);
	      OpenAudio( fname, rate, bits, global.channels,
			(g_toc[current_track].dwStartSector -
			 g_toc[current_track-1].dwStartSector)*CD_FRAMESIZE_RAW);
	    }
	  }
	  current_track++;
	}
	Sectors -= how_much;

      }  /* end while */
      return nSamplesDone;
}

#define PRINT_OVERLAP_INIT \
   if (global.verbose) fprintf(stderr, "overlap:min/max/cur, jitter, percent_done:\n??/??/??/???????   0%%");

#if defined HAVE_FORK_AND_SHAREDMEM
static void forked_read __PR((void));

/* This function does all audio cdrom reads
 * until there is nothing more to do
 */
static void
forked_read()
{
   unsigned total_unsuccessful_retries = 0;

#if !defined(HAVE_SEMGET) || !defined(USE_SEMAPHORES)
   init_child();
#endif

   minover = global.nsectors;

   PRINT_OVERLAP_INIT
   while (global.iloop) { 

      do_read(get_next_buffer(), &total_unsuccessful_retries);

      define_buffer();

   } /* while (global.iloop) */

   if (total_unsuccessful_retries) {
      fprintf(stderr,"%u unsuccessful matches while reading\n",total_unsuccessful_retries);
   }
}

static void forked_write __PR((void));

static void
forked_write()
{

    /* don't need these anymore.  Good security policy says we get rid
       of them ASAP */
    neverneedroot();
    neverneedgroup();

#if defined(HAVE_SEMGET) && defined(USE_SEMAPHORES)
#else
    init_parent();
#endif

    while (nSamplesDone < nSamplesToDo) {

      /* get oldest buffers */
      
      nSamplesDone = do_write(get_oldest_buffer());

      if (nSamplesDone < nSamplesToDo) {
        drop_buffer();
      }

    } /* end while */


    if (global.verbose) {
      if (global.tracktitle[current_track -1] != NULL) {
        fprintf( stderr, "  track %2u '%s' successfully recorded\n",
		current_track, global.tracktitle[current_track-1]);
      } else {
        fprintf( stderr, "  track %2u successfully recorded\n", current_track);
      }
    }
}
#else

/* This function implements the read and write calls in one loop (in case
 * there is no fork/thread_create system call). This means reads and writes
 * have to wait for each other to complete.
 */
static void nonforked_loop __PR((void));

static void
nonforked_loop()
{
    unsigned total_unsuccessful_retries = 0;

    minover = global.nsectors;

    PRINT_OVERLAP_INIT
    while (global.iloop) { 

      do_read(get_next_buffer(), &total_unsuccessful_retries);

      do_write(get_oldest_buffer());

    }

    if (total_unsuccessful_retries) {
      fprintf(stderr,"%u unsuccessful matches while reading\n",total_unsuccessful_retries);
    }

    if (global.verbose) {
      if (global.tracktitle[current_track -1] != NULL) {
        fprintf( stderr, "  track %2u '%s' successfully recorded\n",
		current_track, global.tracktitle[current_track-1]);
      } else {
        fprintf( stderr, "  track %2u successfully recorded\n", current_track);
      }
    }
}
#endif


/* and finally: the MAIN program */
int main( argc, argv )
	int argc;
	char *argv [];
{
  long lSector_p1;
  long sector_offset = 0;
  unsigned long endtrack = 1;
  double rectime = DURATION;
  unsigned int cd_index = 1;
  double int_part;
  int just_the_toc = 0;
  char int_name[100];
  int c;
#ifdef USE_GETOPT_LONG
  int long_option_index=0;
#endif /* USE_GETOPT_LONG */
  int am_i_cdda2wav;
  char * env_p;

  strcpy(int_name, DEF_INTERFACE);
  strcpy(audio_type, AUDIOTYPE);
  save_args(argc, argv);

  /* init global variables */
  init_globals();

  /* When being invoked as list_audio_tracks, just dump a list of
     audio tracks. */
  am_i_cdda2wav = strcmp(argv[0],"list_audio_tracks");
  if (!am_i_cdda2wav) global.verbose = SHOW_JUSTAUDIOTRACKS;

  /* Control those set-id privileges... */
  initsecurity();

  env_p = getenv("CDDA_DEVICE");
  if (env_p != NULL) {
    strncpy( global.dev_name, env_p, sizeof(global.dev_name) );
    global.dev_name[sizeof(global.dev_name)-1]=0;
  }

  /* command options parsing */
#ifdef USE_GETOPT_LONG
  while ( (c = getopt_long (argc,argv,optstring,options,&long_option_index)) 
		!= EOF)
#else
  while ( (c = getopt(argc, argv, optstring)) != EOF )
#endif /* USE_GETOPT_LONG */
	{
    switch (c) {
      case 'D':    /* override device */
        strncpy( global.dev_name, optarg, sizeof(global.dev_name) );
        global.dev_name[sizeof(global.dev_name)-1]=0;
	break;
      case 'A':    /* override device */
        strncpy( global.aux_name, optarg, sizeof(global.aux_name) );
        global.aux_name[sizeof(global.aux_name)-1]=0;
	break;
      case 'I':    /* override interface */
	strncpy( int_name, optarg, sizeof(int_name) );
        int_name[sizeof(int_name)-1]=0;
	break;

      /* the following options are relevant to 'cdda2wav' only */
#ifdef MD5_SIGNATURES
      case 'M':
        if (!am_i_cdda2wav) break;
	global.md5blocksize = strtoul( optarg, NULL, 10 );
        break;
#endif
      case 'O':    /* override audio type */
        if (!am_i_cdda2wav) break;
	strncpy( audio_type, optarg, 4);
        audio_type[sizeof(audio_type)-1]=0;
	break;
      case 'C':    /* override input endianess */
        if (!am_i_cdda2wav) break;
	if (strcasecmp(optarg, "little") == 0) {
	   global.littleendian = 1;
	} else 
	if (strcasecmp(optarg, "big") == 0) {
	   global.littleendian = 0;
	} else 
	if (strcasecmp(optarg, "guess") == 0) {
	   global.littleendian = -2;
	} else {
	   usage();
	}
	break;
      case 'E':    /* override output endianess */
        if (!am_i_cdda2wav) break;
	if (strcasecmp(optarg, "little") == 0) {
	   global.outputendianess = LITTLE;
	} else 
	if (strcasecmp(optarg, "big") == 0) {
	   global.outputendianess = BIG;
	} else {
	   usage();
	}
	break;
      case 'c':    /* override channels */
        if (!am_i_cdda2wav) break;
	if (optarg[0] == 's') {
		global.channels = 2;
		global.swapchannels = 1;
	} else {
		global.channels = strtol( optarg, NULL, 10);
	}
	break;
      case 'S':    /* override drive speed */
        if (!am_i_cdda2wav) break;
	global.speed = strtoul( optarg, NULL, 10);
	break;
      case 'l':    /* install a ring buffer with 'buffers' elements */
        if (!am_i_cdda2wav) break;
        global.buffers = strtoul( optarg, NULL, 10);
        break;
      case 'b':    /* override bits */
        if (!am_i_cdda2wav) break;
	bits = strtol( optarg, NULL, 10);
	break;
      case 'r':    /* override rate */
        if (!am_i_cdda2wav) break;
	rate = strtol( optarg, NULL, 10);
	break;
      case 'a':    /* override rate */
        if (!am_i_cdda2wav) break;
	if (strtod( optarg, NULL ) != 0.0)
/*	    rate = fabs(44100.0 / strtod( optarg, NULL ));*/	/* XXX willst Du negative Parameter erlauben ? */
	    rate = 44100.0 / strtod( optarg, NULL );
	else {
	    fprintf(stderr, "-a requires a nonzero, positive divider.\n");
	    usage();
	}
	break;
      case 't':    /* override start track */
        if (!am_i_cdda2wav) break;
	{
	char * endptr;
	char * endptr2;
	track = strtoul( optarg, &endptr, 10 );
	endtrack = strtoul( endptr, &endptr2, 10 );
	if (endptr2 == endptr)
		endtrack = track;
	break;
	}
      case 'i':    /* override start index */
        if (!am_i_cdda2wav) break;
	cd_index = strtoul( optarg, NULL, 10);
	break;
      case 'd':    /* override recording time */
        if (!am_i_cdda2wav) break;
	rectime = strtod( optarg, NULL );
	break;
      case 'o':    /* override offset */
        if (!am_i_cdda2wav) break;
	sector_offset = strtol( optarg, NULL, 10);
	break;
      case 'n':    /* read sectors per request */
        if (!am_i_cdda2wav) break;
	global.nsectors = strtoul( optarg, NULL, 10);
	break;

       /*-------------- RS 98 -------------*/
      case 'p':    /* specify playback pitch rate */
        global.playback_rate = strtol( optarg, NULL, 10);
        RestrictPlaybackRate( global.playback_rate );
        break;
      case 'P':    /* prevent overlap reading */
        if (!am_i_cdda2wav) break;
	global.useroverlap = strtol( optarg, NULL, 10);
	break;

      case 's':    /* stereo */
        if (!am_i_cdda2wav) break;
	global.channels = 2;
	break;
      case 'm':    /* mono */
        if (!am_i_cdda2wav) break;
	global.channels = 1;
	break;
      case 'x':    /* max */
        if (!am_i_cdda2wav) break;
	global.channels = 2; bits = 16; rate = 44100;
	break;
      case 'w':    /* wait for some audio intensity */
        if (!am_i_cdda2wav) break;
	waitforsignal = 1;
	break;
      case 'F':    /* find extreme amplitudes */
        if (!am_i_cdda2wav) break;
	global.findminmax = 1;
	break;
      case 'G':    /* find if mono */
        if (!am_i_cdda2wav) break;
	global.findmono = 1;
	break;
      case 'e':	   /* echo to sound acrd */
        if (!am_i_cdda2wav) break;
#ifdef	ECHO_TO_SOUNDCARD
	global.echo = 1;
#else
	fprintf(stderr, "There is no sound support compiled into %s.\n",argv[0]);
#endif
	break;	
      case 'v':    /* tell us more */
        if (!am_i_cdda2wav) break;
	global.verbose = strtol( optarg, NULL, 10);
	break;
      case 'q':    /* be quiet */
        if (!am_i_cdda2wav) break;
	global.quiet = 1;
	global.verbose = 0;
	break;
      case 'N':    /* don't write to file */
        if (!am_i_cdda2wav) break;
	global.no_file = 1;
	global.no_infofile = 1;
	global.no_cddbfile = 1;
	break;
      case 'J':    /* information only */
        if (!am_i_cdda2wav) break;
	global.no_file = 1;
	global.no_infofile = 1;
	global.no_cddbfile = 1;
	global.verbose = SHOW_MAX;
	just_the_toc = 1;
	break;
      case 'H':    /* don't write extra files */
        if (!am_i_cdda2wav) break;
	global.no_infofile = 1;
	global.no_cddbfile = 1;
	break;
      case 'B':    /* bulk transfer */
        if (!am_i_cdda2wav) break;
	bulk = 1;
	break;
      case 'T':    /* do deemphasis on the samples */
        if (!am_i_cdda2wav) break;
	global.deemphasize = 1;
	break;
      case 'R':    /* list available rates */
        if (!am_i_cdda2wav) break;
	{ int ii;
	  fprintf(stderr, "Cdda2wav version %s-%s: available rates are:\nRate   Divider      Rate   Divider      Rate   Divider      Rate   Divider\n", VERSION, HOST );
	  for (ii = 1; ii <= 44100 / 880 / 2; ii++) {
	    long i2 = ii;
	    fprintf(stderr, "%-7g  %2ld         %-7g  %2ld.5       ",
                    44100.0/i2,i2,44100/(i2+0.5),i2);
	    i2 += 25;
	    fprintf(stderr, "%-7g  %2ld         %-7g  %2ld.5\n",
                    44100.0/i2,i2,44100/(i2+0.5),i2);
	    i2 -= 25;
	  }
	}
	exit(0);
	break;
      case 'V':
        if (!am_i_cdda2wav) break;
	global.scsi_verbose++;	/* XXX nach open_scsi() scgp->verbose = .. !!!! */
	break;
#ifdef USE_GETOPT_LONG
      case 'h':
	usage();
	break;
#if 0
      case '?':
	fputs ("use cdda2wav --help to get more information.", stderr);
	exit (1);
	break;
#endif
#endif
      default:
	fputs ("use cdda2wav --help to get more information.\n", stderr);
	exit (1);
    }
  }

  /* check all parameters */
  if (global.verbose < 0 || global.verbose > SHOW_MAX) {
    fprintf(stderr, "Error: incorrect verbose level setting: %d\n",global.verbose);
    usage();
  }
  if (global.verbose == 0) global.quiet = 1;

  if ( rectime < 0 ) {
    fprintf(stderr, "Error: incorrect recording time setting: %f\n",rectime);
    usage();
  }

  if ( global.channels != 1 && global.channels != 2 ) {
    fprintf(stderr, "Error: incorrect channel setting: %d\n",global.channels);
    usage();
  }

  if ( bits != 8 && bits != 12 && bits != 16 ) {
    fprintf(stderr, "Error: incorrect bits_per_sample setting: %d\n",bits);
    usage();
  }

  if ( rate < 827.0 || rate > 44100.0 ) {
    fprintf(stderr, "Error: incorrect sample rate setting: %g\n",rate);
    usage();
  }

  int_part = (double)(long) (2*44100.0 / rate);
  
  if (2*44100.0 / rate - int_part >= 0.5 ) {
      int_part += 1.0;
      fprintf( stderr, "Nearest available sample rate is %g Hertz\n",
	      2*44100.0 / int_part);
  }
  Halved = ((int) int_part) & 1;
  rate = 2*44100.0 / int_part;
  undersampling = (int) int_part / 2.0;
  samples_to_do = undersampling;

  if (!strcmp((char *)int_name,"generic_scsi"))
      interface = GENERIC_SCSI;
  else if (!strcmp((char *)int_name,"cooked_ioctl"))
      interface = COOKED_IOCTL;
  else  {
    fprintf(stderr, "Error: incorrect interface setting: %s\n",int_name);
    usage();
  }

  /* check * init audio file */
  if (!strncmp(audio_type,"wav",3)) {
    global.need_big_endian = 0;
    if (global.outputendianess != NONE)
	global.need_big_endian = global.outputendianess == BIG;
    InitAudioFile = InitWav;
    ExitAudioFile = ExitWav;
    GetSndHdrSize = GetWavHdrSize;
  } else if (!strncmp(audio_type, "sun", 3)) {
    /* Enhanced compatibility */
    strcpy(audio_type, "au");

    global.need_big_endian = 1;
    if (global.outputendianess != NONE)
	global.need_big_endian = global.outputendianess == BIG;
    InitAudioFile = InitSun;
    ExitAudioFile = ExitSun;
    GetSndHdrSize = GetSunHdrSize;
  } else if (!strncmp(audio_type, "cdr", 3) || 
             !strncmp(audio_type, "raw", 3)) {
    global.need_big_endian = 1;	/* for cd writer compatibility */
    if (global.outputendianess != NONE)
	global.need_big_endian = global.outputendianess == BIG;
    InitAudioFile = (int (*) __PR((int audio, long channels, unsigned long myrate, long nBitsPerSample,
             unsigned long expected_bytes)))InitRaw;
    ExitAudioFile = (int (*) __PR((int audio, unsigned long nBytesDone))) ExitRaw;
    GetSndHdrSize = GetRawHdrSize;
  } else {
    fprintf(stderr, "Error: incorrect audio type setting: %3s\n", audio_type);
    usage();
  }
  if (global.no_file) global.fname_base[0] = '\0';

  if (!bulk) {
    strcat(global.fname_base, ".");
    strcat(global.fname_base, audio_type);
  }

  /* all options processed. Now a file name may follow */
  if ( optind < argc ) {
    if (!strcmp(argv[optind],"-")) {
      /* pipe mode */
      if (bulk == 1) {
        fprintf(stderr, "bulk mode disabled while outputting to a pipe\n");
        bulk = 0;
      }
      global.audio = dup (fileno(stdout));
      strncpy( global.fname_base, "standard output", sizeof(global.fname_base) );
    } else {
      /* filename given */
      strncpy( global.fname_base, argv[optind], sizeof(global.fname_base)-8 );
    }
    global.fname_base[sizeof(global.fname_base)-1]=0;
  }

  /* setup interface and open cdrom device */
  /* request sychronization facilities and shared memory */
  SetupInterface( );
#if defined(HAVE_SEMGET) && defined(USE_SEMAPHORES)
  atexit ( free_sem );
#endif

  /* get table of contents */
  cdtracks = ReadToc(  g_toc );
  if (cdtracks == 0) {
    fprintf(stderr, "No track in table of contents! Aborting...\n");
    exit(10);
  }

  global.cddb_id = calc_cddb_id();
  have_CD_extra = FixupTOC(cdtracks + 1);

  if ( global.verbose == SHOW_JUSTAUDIOTRACKS ) {
    unsigned int z;

    for (z = 0; z < cdtracks; z++)
	if ((g_toc[z].bFlags & CDROM_DATA_TRACK) == 0)
	  printf("%02d\t%06u\n", g_toc[z].bTrack, g_toc[z].dwStartSector);
    exit(0);
  }

  if ( global.verbose != 0 ) {
    fprintf( stderr, "#Cdda2wav version %s-%s", VERSION, HOST );
#if defined _POSIX_PRIORITY_SCHEDULING || defined HAVE_SYS_PRIOCNTL_H
    fputs( " real time sched.", stderr );
#endif
#if defined ECHO_TO_SOUNDCARD
    fputs( " soundcard support", stderr );
#endif
    fputs( "\n", stderr );
  }

  if ( global.verbose & (SHOW_TOC | SHOW_STARTPOSITIONS) )
    DisplayToc ();

  if (just_the_toc) exit(0);

  /* use global.useroverlap to set our overlap */
  if (global.useroverlap != -1)
	global.overlap = global.useroverlap;

  if (global.nsectors < 1+global.overlap) {
      fprintf(stderr, "Error: overlap (%d) is >= nsectors (%d)\nUse -P to set the overlap sectors and -n to set the nsectors\n",global.overlap, global.nsectors);
      exit(-1);
  }

  /* paranoia reigns supreme */
#if defined(HAVE_SYNC) && (HAVE_SYNC == 1)
  sync();
#endif

  /* try to get some extra kicks */
  needroot(0);
#if defined(HAVE_NICE) && (HAVE_NICE == 1)
  nice(-20);
#endif
  dontneedroot();

  /* switch cdrom to audio mode */
  EnableCdda ( 1);

  atexit ( CloseAll );

  if ( !FirstTrack () )
    FatalError ( "This disk has no audio tracks\n" );

  if ( global.verbose & (SHOW_MCN | SHOW_ISRC) )
    Read_MCN_ISRC();

  /* check if start track is in range */
  if ( track < 1 || track > cdtracks ) {
    fprintf(stderr, "Error: incorrect start track setting: %d\n",track);
    usage();
  }

  /* check if end track is in range */
  if ( endtrack < track || endtrack > cdtracks ) {
    fprintf(stderr, "Error: incorrect end track setting: %ld\n",endtrack);
    usage();
  }

  do {
    lSector = GetStartSector ( track );
    lSector_p1 = GetEndSector ( track ) + 1;

    if ( lSector < 0 ) {
      if ( bulk == 0 ) {
        FatalError ( "track %d not found\n", track );
      } else {
        fprintf(stderr, "Skipping data track %d...\n", track);
	if (endtrack == track) endtrack++;
        track++;
      }
    }
  } while (bulk != 0 && track <= cdtracks && lSector < 0);

  if (cd_index != 1)
    sector_offset += ScanIndices( track, cd_index );
  else if (global.verbose & SHOW_INDICES)
    ScanIndices( track, cd_index );

  lSector += sector_offset;
  /* check against end sector of track */
  if ( lSector >= lSector_p1 ) {
    fputs( "sector offset exceeds track size (ignored)\n", stderr );
    lSector -= sector_offset;
  }

  if ( lSector < 0L ) {
    fputs( "negative start sector! Set to zero.\n", stderr );
    lSector = 0L;
  }

  lSector_p2 = GetLastSectorOnCd( track );
  if (bulk == 1 && track == endtrack && rectime == 0.0)
     rectime = 99999.0;
  if ( rectime == 0.0 ) {
    /* set time to track time */
    nSamplesToDo = (lSector_p1 - lSector) * CD_FRAMESAMPLES;
    rectime = (lSector_p1 - lSector) / 75.0;
    if (CheckTrackrange( track, endtrack) == 1) {
      lSector_p2 = GetEndSector ( endtrack ) + 1;

      if (lSector_p2 >= 0) {
        rectime = (lSector_p2 - lSector) / 75.0;
        nSamplesToDo = (long)(rectime*44100.0 + 0.5);
      } else {
        fputs( "end track is no valid audio track (ignored)\n", stderr );
      }
    } else {
      fputs( "track range does not consist of audio tracks only (ignored)\n", stderr );
    }
  } else {
    /* Prepare the maximum recording duration.
     * It is defined as the biggest amount of
     * adjacent audio sectors beginning with the
     * specified track/index/offset. */

    if ( rectime > (lSector_p2 - lSector) / 75.0 ) {
      rectime = (lSector_p2 - lSector) / 75.0;
      lSector_p1 = lSector_p2;
    }

    /* calculate # of samples to read */
    nSamplesToDo = (long)(rectime*44100.0 + 0.5);
  }

  global.OutSampleSize = (1+bits/12);
  if (nSamplesToDo/undersampling == 0L) {
      fprintf( stderr, "time interval too short. Duration > %g secs!\n", 
	       undersampling/44100.0);
      exit(-2);
  }
  SamplesToWrite = nSamplesToDo*2/(int)int_part;

  init_soundcard(rate, bits);

  if (global.speed != 0 && SelectSpeed != NULL)
     SelectSpeed( global.speed);

  if ( global.verbose & SHOW_SUMMARY ) {
     if ( !waitforsignal ) {
	if (!bulk)
	  fprintf(stderr, "samplefile size will be %lu bytes.\n",
           GetSndHdrSize() + SamplesToWrite*global.OutSampleSize*global.channels  ); 
	else {
	  int tracks_included;
	  tracks_included = GetTrack(
			      (unsigned) (lSector + nSamplesToDo/CD_FRAMESAMPLES -1))
				     - track + 1;
	  fprintf(stderr, "samplefiles size total will be %lu bytes. %d audio tracks\n", 
		  tracks_included * GetSndHdrSize() +
	  SamplesToWrite*global.OutSampleSize*global.channels, tracks_included  ); 
	}
     }
     fprintf (stderr, "recording %.5f seconds %s with %d bits @ %05.1f Hz"
	      ,rectime ,global.channels == 1 ? "mono":"stereo", bits, rate);
     if (!global.no_file && *global.fname_base)
		fprintf(stderr, " ->'%s'...", global.fname_base );
     fputs("\n", stderr);
  }

  current_track = track;

  if ( !global.no_file ) {
    if (bulk)
      sprintf(fname, "%s_%02u.%s",global.fname_base,current_track,audio_type);
    else
      strcpy(fname,global.fname_base);
    OpenAudio( fname, rate, bits, global.channels, 
	      (unsigned)(SamplesToWrite*global.OutSampleSize*global.channels) );
  }

  global.Remainder = (75 % global.nsectors)+1;

  global.sh_bits = 16 - bits;		/* shift counter */

  global.iloop = nSamplesToDo;
  if (Halved && (global.iloop&1))
      global.iloop += 2;

  BeginAtSample = lSector * CD_FRAMESAMPLES;

#if defined(HAVE_SEMGET) && defined(USE_SEMAPHORES)
#else
  init_pipes();
#endif

#if defined(HAVE_FORK_AND_SHAREDMEM)

  /* Everything is set up. Now fork and let one process read cdda sectors
     and let the other one store them in a wav file */

  /* forking */
  child_pid = fork();
  global.have_forked = 1;
#ifdef DEBUG_FORKED
  if (child_pid > 0) fprintf( stderr, "child pid is %d\n", child_pid);
  sleep(10);
#endif

  /*********************** fork **************************************/
  if (child_pid == 0) {
    /* child READER section */

    switch_to_realtime_priority();

    forked_read();
    exit(0);
  } else if (child_pid > 0) {
    /* parent WRITER  section */
    
    forked_write();
    exit(0);
  } else
#else
  /* version without fork */
  {
    global.have_forked = 0;
    switch_to_realtime_priority();

    fprintf(stderr, "a nonforking version is running...\n");
    nonforked_loop();
    exit(0);
  }
#endif

  if (!global.quiet) fputs( "\n", stderr );

  return 0;
}
