#include <sys/types.h>
#include <linux/cdrom.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <stdio.h>
#include "cd.h"

// initialize
cd::cd(void)
    : m_fd(-1), m_mode(NORMAL), m_status(NO_CD_INSIDE),
      m_tracks(0), m_volume(255), m_looping(false), m_random(true),
      m_track(0), m_one_track_only(false), m_ind(0), m_programmed(0)
{
    srand( (unsigned int) time(NULL) );

    // this is clever :)
    m_tr[0]=0;
    open();
    
    read_volume();
    read_status();
}

// close the cd
cd::~cd()
{
    close();
}

// new cd in drive, initialize all data
void cd::new_cd(void)
{
    if (m_status == NO_CD_INSIDE)
        return;

    read_info();
    read_status();
    init_tracks();
}

// read the number of tracks
void cd::read_info(void)
{
    if (m_status == NO_CD_INSIDE)
        return;
    
    struct cdrom_tochdr tinfo;

    int x=ioctl(m_fd, CDROMREADTOCHDR, &tinfo);
    if (x == -1)
        return;

    m_tracks=tinfo.cdth_trk1;
}
    
// open cd-rom device
void cd::open(void)
{
    if (m_fd != -1)
        return;
    
    m_fd=::open("/dev/cdrom", O_RDONLY | O_NONBLOCK);

    if (m_fd == -1) {
        fprintf(stderr, "Cannot open /dev/cdrom in non-blocking mode.\n");
        exit(1);
    }
}

// close cd-rom device
void cd::close(void)
{
    if (m_fd == -1)
        return;
    
    ::close(m_fd);
}

// read current status
// if track changed, play next track. meaning of "next track" varies
// with the mode.
// also detects cd insertions/ejections
void cd::read_status(void)
{
    struct cdrom_subchnl ch;
    ch.cdsc_format=CDROM_MSF;
    
    int x=ioctl(m_fd, CDROMSUBCHNL, &ch);

    // new cd inserted
    if ((m_status == NO_CD_INSIDE) && (x != -1)) {

        // the cd could be playing or paused, but it doesn't matter
        // since new_cd() calls read_status() again which then sets it
        // to the correct value
        m_status=STOPPED;
        new_cd();

        return;
    } else if (x == -1) {

        // cd ejected
        m_status=NO_CD_INSIDE;
        m_tracks=0;
        m_ind=0;
        m_track=0;
        m_programmed=0;
        clear_tracks();
        
        return;
    }

    switch (ch.cdsc_audiostatus) {
    case CDROM_AUDIO_PLAY:
        m_status=PLAYING;
        m_track=ch.cdsc_trk;
        break;
    case CDROM_AUDIO_PAUSED:
        m_status=PAUSED;
        break;
    case CDROM_AUDIO_NO_STATUS:
        m_status=STOPPED;
        break;
    case CDROM_AUDIO_INVALID:
        // how stupid can a drive be to return _INVALID when it's
        // stopped when it could return _NO_STATUS ?? yet many drives
        // are that stupid..
        m_status=STOPPED;
        break;
    case CDROM_AUDIO_COMPLETED:
        if (m_one_track_only)
            m_one_track_only=false;
        else
            next();
        break;
    }
}

// play a single track
void cd::play_one(int track)
{
    m_one_track_only=true;
    play(track);
}

// play the track and continue after it with the current mode
void cd::play(int track)
{
    if (m_status == NO_CD_INSIDE) {
        m_one_track_only=false;
        
        return;
    }

    if ( (track < 1) || (track > m_tracks) ) {
        m_one_track_only=false;
        
        return;
    }

    stop();

    // start spinning the drive.
    // necessary on some drives
    ioctl(m_fd, CDROMSTART);

    struct cdrom_ti ti;

    ti.cdti_trk0=track;
    ti.cdti_trk1=track;
    
    ioctl(m_fd, CDROMPLAYTRKIND, &ti);
}

// continue playing from where we stopped, or if we're at the start of
// the disc from the beginning. if we're at the start and in random
// mode, shuffle the right amount of tracks
void cd::play(void)
{
    if (m_status == NO_CD_INSIDE)
        return;

    if (m_ind < 1) {
        if ((m_mode == PROGRAMMED) && m_random)
            shuffle_tracks(m_programmed);
        else if ((m_mode == NORMAL) && m_random)
            shuffle_tracks(m_tracks);
        m_ind=1;
    }

    // it shouldn't be possible for m_ind to grow beyond its limits

    play(m_tr[m_ind]);
}
    
// stop
void cd::stop(void)
{
    if (m_fd == -1)
        return;

    if (m_status == NO_CD_INSIDE)
        return;
    
    ioctl(m_fd, CDROMSTOP);
}

// pause
void cd::pause(void)
{
    if (m_fd == -1)
        return;

    if (m_status != PLAYING)
        return;
    
    ioctl(m_fd, CDROMPAUSE);
}

// pause
void cd::resume(void)
{
    if (m_fd == -1)
        return;

    if (m_status != PAUSED)
        return;
    
    ioctl(m_fd, CDROMRESUME);
}

// eject
void cd::eject(void)
{
    if (m_fd == -1)
        return;

    stop();
    
    ioctl(m_fd, CDROMEJECT);
}

// set volume
void cd::set_volume(int volume)
{
    if (m_fd == -1)
        return;
    
    m_volume=volume;
    if (m_volume > 255)
        m_volume=255;
    else if (m_volume < 0)
        m_volume=0;
    
    struct cdrom_volctrl volctrl;
    volctrl.channel0=m_volume;
    volctrl.channel1=m_volume;
    
    ioctl(m_fd, CDROMVOLCTRL, &volctrl);
}

// read volume
void cd::read_volume(void)
{
    if (m_fd == -1)
        return;
    
    struct cdrom_volctrl volctrl;
    int x=ioctl(m_fd, CDROMVOLREAD, &volctrl);
    if (x == -1)
        return;

    m_volume=volctrl.channel0;
}

// return volume
int cd::get_volume(void)
{
    return m_volume;
}

// next track
void cd::next(void)
{
    m_ind++;
    
    // have we played through the whole array?
    if ( (m_ind > MAX_TRACKS) || (m_tr[m_ind] == -1) ) {
        tracks_played();
        return;
    } else
        play(m_tr[m_ind]);
}

// clear the track array
void cd::clear_tracks(void)
{
    for (int i=1; i <= MAX_TRACKS; i++)
        m_tr[i]=-1;
}

// set mode
void cd::set_mode(mode m)
{
    m_mode=m;
    init_tracks();
}

// init track array according to current mode
void cd::init_tracks(void)
{
    if (m_status == NO_CD_INSIDE)
        return;

    m_ind=0;
    m_programmed=0;
    clear_tracks();
    if (m_mode == NORMAL) {
        for (int i=1; i <= m_tracks; i++)
            m_tr[i]=i;
    }
}

// shuffle first n entries of m_tr
void cd::shuffle_tracks(int n)
{
    for (int i=1; i <= n; i++) {
        int temp=m_tr[i];
        int t=(rand() % n)+1;
        m_tr[i]=m_tr[t];
        m_tr[t]=temp;
    }
}

// called when all tracks in array are played
// if we're not looping, stop playing.
// if we're looping, start over at the beginning of the array.
void cd::tracks_played(void)
{
    if (!m_looping) {
        stop();

        return;
    }
    
    m_ind=0;
    play();
}

// if we're in PROGRAMMED mode, add a new track to be played
void cd::push_track(int track)
{
    if (m_mode != PROGRAMMED)
        return;

    if (m_programmed >= MAX_TRACKS)
        return;

    if ( (track < 1) || (track > m_tracks) )
        return;
    
    m_programmed++;
    m_tr[m_programmed]=track;
}

// set randomness
void cd::set_random(bool r)
{
    m_random=r;
}
