/* cd-rom.c	Flavio Lerda		97-07-27 */

/*
	cdplayer 0.2 - simple cdplayer with shuffle and database capability
	Copyright (C) 1997,98  Flavio Lerda

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

        To contact the author:
                Flavio Lerda
                Via vittime di Bologna 14
                12026 Piasco CN
                Italy
        or:
                flerda@athena.polito.it
*/   

/*
   Utilities for the manegement of the CD-ROM.
   Inspirated by Linux kernel and the WorkBone
   sources.

   97-07-27: [MAS]	Code written for the first time.
*/

#include "cd-rom.h"
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>
#include <errno.h>
#include <time.h>
#include <malloc.h>
#include <string.h>

/* time to pass during a call to assume that the cdrom is open */
#define CDROM_OPENDELAY	10

static void	msf_to_lba(struct cdrom_msf0 *,int *);
static int	lock(struct cd *,int *);
static int	unlock(struct cd *,int *);

/* initialize the cdrom and return an handle to give to other functions */
struct cd	*cdrom_init(char *dev,int lock) {
	struct cd	*cd;
	int		fd;
	
	if((fd = open(dev,O_RDONLY|O_NONBLOCK)) == -1) {
		perror("open()");
		return NULL;
	}
	
	if((cd = malloc(sizeof(struct cd))) == NULL) {
		close(fd);
		perror("malloc()");
		return NULL;
	}
	
	if((cd->dev = malloc(strlen(dev)+1)) == NULL) {
		close(fd);
		perror("malloc()");
		free(cd);
		return NULL;
	}
	
	strcpy(cd->dev,dev);
	
	if(lock)
		cd->fd = fd;
	else {
		close(fd);
		cd->fd = -1;
	}
	
	return cd;
}

/* delete the handle created */
int	cdrom_done(struct cd *cd) {
	if(!cd) {
		errno = EINVAL;
		return -1;
	}
	
	if(cd->fd != -1)
		if(close(cd->fd) == -1) {
			perror("close()");
			return -1;
		}
	
	free(cd->dev);
	free(cd);
	
	return 0;
}

/* read and allocate a new table of contents */
struct cdrom_toc	*cdrom_toc(struct cd *cd) {
	struct cdrom_toc	*toc;
	struct cdrom_entry	entry;
	struct cdrom_tochdr	cdth;
	struct cdrom_tocentry	cdte;
        char			upc[14];
	int			trk,fd,flag;
	
	fd = lock(cd,&flag);
	
        bzero(upc,14);
	if((toc = malloc(sizeof(struct cdrom_toc))) == NULL) {
		unlock(cd,&flag);
		perror("malloc()");
		return NULL;
	}
	
	if(ioctl(fd,CDROMREADTOCHDR,&cdth) == -1) {
		unlock(cd,&flag);
		perror("ioctl(CDROMREADTOCHDR)");
		free(toc);
		return NULL;
	}
	
	toc->first_track = cdth.cdth_trk0;
	toc->last_track = cdth.cdth_trk1;
	toc->ntracks = cdth.cdth_trk1 - cdth.cdth_trk0 + 1;
	
	if(ioctl(fd,CDROM_GET_UPC,upc) == -1) {
		unlock(cd,&flag);
		perror("ioctl(CDROM_GET_UPC)");
		free(toc);
		return NULL;
	}
	memcpy(toc->upc,upc,14);

	if((toc->entries = malloc(sizeof(struct cdrom_entry) * 
			(toc->ntracks))) == NULL) {
		unlock(cd,&flag);
		perror("malloc()");
		free(toc);
		return NULL;
	}

	for(trk=toc->first_track;trk <= toc->last_track;trk++) {
		cdte.cdte_track = trk;
		cdte.cdte_format = CDROM_MSF;

		if(ioctl(fd,CDROMREADTOCENTRY,&cdte) == -1) {
			unlock(cd,&flag);

			perror("ioctl(CDROMREADTOCENTRY)");
			
			free(toc->entries);
			free(toc);
			
			return NULL;
		}
		
		entry.track = trk;
		entry.audio = cdte.cdte_ctrl & CDROM_DATA_TRACK ? 0 : 1;
		msf_to_lba(&cdte.cdte_addr.msf,&entry.lba_start);
		entry.lba_end = entry.lba_length = 0;
		
		if(trk != toc->first_track) {
			int	last = trk - 1 - toc->first_track;
			
			toc->entries[last].lba_end = entry.lba_start;
			toc->entries[last].lba_length = entry.lba_start - 
				toc->entries[last].lba_start;
		}
		toc->entries[trk-toc->first_track] = entry;
	}
	cdte.cdte_track = CDROM_LEADOUT;
	cdte.cdte_format = CDROM_MSF;
	
	if(ioctl(fd,CDROMREADTOCENTRY,&cdte) == -1) {
		unlock(cd,&flag);

		perror("ioctl(CDROMREADTOCENTRY)");
		
		cdrom_freetoc(toc);
		
		return NULL;
	}
	
	trk = toc->last_track - toc->first_track;

	msf_to_lba(&cdte.cdte_addr.msf,&toc->entries[trk].lba_end);
	
	toc->entries[trk].lba_length = toc->entries[trk].lba_end - 
		toc->entries[trk].lba_start;
	
	toc->lba_start = toc->entries[0].lba_start;
	toc->lba_end = toc->entries[trk].lba_end;
	toc->lba_length = toc->lba_end - toc->lba_start;

	unlock(cd,&flag);
	
	return toc;
}

/* free a table of contents */
int	cdrom_freetoc(struct cdrom_toc *toc) {
	free(toc->entries);
	free(toc);

	return 0;
}

/* get the current status of the cdrom */
int	cdrom_status(struct cd *cd,struct cdrom_status *status) {
	struct cdrom_subchnl	cdsc;
	time_t			t;
	int			fd,flag;

	fd = lock(cd,&flag);

	if(status == NULL)
		return 0;
	
	cdsc.cdsc_format = CDROM_MSF;

	time(&t);	
	if(ioctl(fd,CDROMSUBCHNL,&cdsc) == -1) {
		t = time(NULL) - t;
		if(errno == EIO) {
			unlock(cd,&flag);
			
			if(t >= CDROM_OPENDELAY)
				status->status = CDS_OPEN;
			else
				status->status = CDS_EMPTY;
			status->track = 0;
			status->index = 0;
			status->lba_pos_rel = 0;
			status->lba_pos_abs = 0;

			return 0;
		}	
		unlock(cd,&flag);
		perror("ioctl(CDROMSUBCHNL)");
		return -1;
	}
	
	switch(cdsc.cdsc_audiostatus) {
		case CDROM_AUDIO_PLAY:
			status->status = CDS_PLAY;
			status->track = cdsc.cdsc_trk;
			status->index = cdsc.cdsc_ind;
			msf_to_lba(&cdsc.cdsc_reladdr.msf,&status->lba_pos_rel);
			msf_to_lba(&cdsc.cdsc_absaddr.msf,&status->lba_pos_abs);
			break;
			
		case CDROM_AUDIO_PAUSED:
			status->status = CDS_PAUSE;
			status->track = cdsc.cdsc_trk;
			status->index = cdsc.cdsc_ind;
			status->lba_pos_rel = cdsc.cdsc_reladdr.lba;
			status->lba_pos_abs = cdsc.cdsc_absaddr.lba;
			break;
			
		case CDROM_AUDIO_COMPLETED:
		case CDROM_AUDIO_NO_STATUS:
		default:
			status->status = CDS_STOP;
			status->track = 0;
			status->index = 0;
			status->lba_pos_rel = 0;
			status->lba_pos_abs = 0;
			break;
	}
	unlock(cd,&flag);

	return 0;
}

/* pause, stop, resume, or eject the CDROM */
int	cdrom_pause(struct cd *cd) {
	int	fd,flag;
	
	fd = lock(cd,&flag);

	if(ioctl(fd,CDROMPAUSE) == -1) {
		unlock(cd,&flag);
		perror("ioctl(CDROMPAUSE)");
		return -1;
	}
	unlock(cd,&flag);
	
	return 0;
}

int	cdrom_stop(struct cd *cd) {
	int	fd,flag;
	
	fd = lock(cd,&flag);

	if(ioctl(fd,CDROMSTOP) == -1) {
		unlock(cd,&flag);
		perror("ioctl(CDROMSTOP)");
		return -1;
	}	
	unlock(cd,&flag);
	
	return 0;
}

int	cdrom_resume(struct cd *cd) {
	int	fd,flag;
	
	fd = lock(cd,&flag);

	if(ioctl(fd,CDROMRESUME) == -1) {
		unlock(cd,&flag);
		perror("ioctl(CDROMRESUME)");
		return -1;
	}
	unlock(cd,&flag);
	
	return 0;
}

int	cdrom_close(struct cd *cd) {
	int	fd,flag;
	
	fd = lock(cd,&flag);

	if(ioctl(fd,CDROMCLOSETRAY) == -1) {
		unlock(cd,&flag);
		if(errno == EIO)
			return 0;
		perror("ioctl(CDROMCLOSE)");
		return -1;
	}
	unlock(cd,&flag);
	
	return 0;
}

int	cdrom_eject(struct cd *cd) {
	int	fd,flag;
	
	fd = lock(cd,&flag);

	if(ioctl(fd,CDROMEJECT) == -1) {
		unlock(cd,&flag);
		perror("ioctl(CDROMEJECT)");
		return -1;
	}
	unlock(cd,&flag);
	
	return 0;
}

/* set or get the volume */
int	cdrom_setvolume(struct cd *cd,int left,int right) {
	struct cdrom_volctrl	cdvc;
	int			fd,flag;
	
	fd = lock(cd,&flag);

	cdvc.channel0 = left;
	cdvc.channel1 = right;
	cdvc.channel2 = left;
	cdvc.channel3 = right;
	
	if(ioctl(fd,CDROMVOLCTRL,&cdvc) == -1) {
		unlock(cd,&flag);
		perror("ioctl(CDROMVOLCTRL)");
		return -1;
	}	
	unlock(cd,&flag);
	
	return 0;
}

int	cdrom_getvolume(struct cd *cd,int *pleft,int *pright) {
	struct cdrom_volctrl	cdvc;
	int			fd,flag;
	
	fd = lock(cd,&flag);

	if(ioctl(fd,CDROMVOLREAD,&cdvc) == -1) {
		unlock(cd,&flag);
		perror("ioctl(CDROMVOLREAD)");
		return -1;
	}
	unlock(cd,&flag);

	if(pleft) *pleft = cdvc.channel0;
	if(pright) *pright = cdvc.channel1;
	
	return 0;
}

/* play a given amount of CDROM */
int	cdrom_play(struct cd *cd,int lba_start,int lba_end) {
	struct cdrom_msf	msf;
	int			fd,flag;
	
	fd = lock(cd,&flag);

	lba2msf(lba_start,&msf.cdmsf_min0,&msf.cdmsf_sec0,&msf.cdmsf_frame0);
	lba2msf(lba_end,&msf.cdmsf_min1,&msf.cdmsf_sec1,&msf.cdmsf_frame1);

	if(ioctl(fd,CDROMSTART) == -1) {
		unlock(cd,&flag);
		perror("ioctl(CDROMSTART)");
		return -1;
	}
	
	if(ioctl(fd,CDROMPLAYMSF,&msf) == -1) {
		unlock(cd,&flag);
		perror("ioctl(CDROMPLAYMSF)");
		return -1;
	}
	unlock(cd,&flag);
	
	return 0;
}

/* play a given track */
int	cdrom_playtrack(struct cd *cd,int trk) {
	struct cdrom_ti	cdti;
	int		fd,flag;
	
	fd = lock(cd,&flag);
	
	cdti.cdti_trk0 = cdti.cdti_trk1 = trk;
	cdti.cdti_ind0 = cdti.cdti_ind1 = 0;

	if(ioctl(fd,CDROMPLAYTRKIND,&cdti) == -1) {
		unlock(cd,&flag);
		perror("ioctl(CDROMTRKIND)");
		return -1;
	}
	unlock(cd,&flag);
	
	return 0;
}

/* lock or unlock the cdrom */
int	cdrom_lock(struct cd *cd) {
	return lock(cd,NULL);
}

int	cdrom_unlock(struct cd *cd) {
	return unlock(cd,NULL);
}

int	cdrom_locked(struct cd *cd) {
	if(cd == NULL) {
		errno = EINVAL;
		return -1;
	}

	return cd->fd == -1 ? 0 : 1;
}

/* conversion from/to lba */
void	lba2msf(int lba,u_char *min,u_char *sec,u_char *frame) {
	if(min) *min = lba / 4500;
	if(sec) *sec = (lba / 75) % 60;
	if(frame) *frame = lba % 75;
}

void	msf2lba(u_char min,u_char sec,u_char frame,int *lba) {
	if(lba) *lba = (min * 60 + sec) * 75 + frame;
}

static void	msf_to_lba(struct cdrom_msf0 *msf,int *lba) {
	if(msf)
		msf2lba(msf->minute,msf->second,msf->frame,lba);
}

static int	lock(struct cd *cd,int *flag) {
	if(!cd) {
		errno = EINVAL;
		return -1;
	}
	
	if(cd->fd == -1) {
		if((cd->fd = open(cd->dev,O_RDONLY)) == -1) {
			perror("open()");
			return -1;
		}
		if(flag) *flag = 1;
	} else
		if(flag) *flag = 0;
	
	return cd->fd;
}

static int	unlock(struct cd *cd,int *flag) {
	if(!cd) {
		errno = EINVAL;
		return -1;
	}
	
	if(cd->fd != -1 && (flag == NULL || *flag == 1)) {
		if(close(cd->fd) == -1) {
			perror("close()");
			return -1;
		}
		cd->fd = -1;
		*flag = 0;
	}
	
	return 0;
}

/* cd-rom.c	Flavio Lerda		97-07-27 */
