/* cddb.c		Flavio Lerda		97-07-29 */

/*
	cdplayer 0.2 - simple cdplayer with shuffle and database capability
	Copyright (C) 1997,98  Flavio Lerda

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

        To contact the author:
                Flavio Lerda
                Via vittime di Bologna 14
                12026 Piasco CN
                Italy
        or:
                flerda@athena.polito.it
*/   

/*
   Functions for the cd database.
   
   97-07-29: [MAS]	Code written for the first time.
*/

#include <ctype.h>
#include <malloc.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "strutil.h"
#include "cddb.h"

#define	S_ID		0
#define S_CD		1
#define S_TRACK		2
#define S_FINISHED 	3

#define MAXLEN		4096

#define iscomment()	(line[0] == '#')
#define isempty()	(line[0] == 0)
#define spaces()	{ while(isspace(line[idx])) idx++; }

static void	partial_free(struct cddb *cddb,int status,int track) {
	int	i;
	
	if(!cddb) return;
	
	switch(status) {
		case S_FINISHED:
			cddb_free(cddb);
			break;
			
		case S_TRACK:
			if(cddb->tracks) {
				for(i = 0; i < track; i++) {
					struct cddb_track	*track;
				       
					track = &cddb->tracks[i];

					if(track->name) free(track->name);
					if(track->lyrics) free(track->lyrics);
				}
				free(cddb->tracks);
			}
			if(cddb->title) free(cddb->title);
			if(cddb->artist) free(cddb->artist);

			/* continue with the next case */

		case S_CD:
		case S_ID:
			free(cddb);
			break;
	}
}

static int	getnum(char *line,int *idx) {
	int	value = 0;

	while(isdigit(line[*idx])) {
		value *= 10;
		value += line[(*idx)++] - '0';
	}

	return value;
}

static int	getstring(char *line,int *idx,char *buf) {
	int	cnt = 0;

	if(line[*idx] == '\"') {
		(*idx)++;
		while(line[*idx] != '\"' && line[*idx] != 0)
			buf[cnt++] = line[(*idx)++];
		if(line[(*idx)++] == 0)
			return -1;
	}

	buf[cnt] = 0;

	return 0;
}

static int	getid(char *line,struct cddb *cddb) {
	int	cnt = 0,idx = 0;

	spaces();
	while(cnt < 16)
		if(isdigit(line[idx]) || line[idx] == '-')
			cddb->id.id[cnt++] = line[idx++];
		else
			return -1;
	spaces();
	if(line[idx] != 0 && line[idx] != '#')
		return -1;

	return 0;
}

static int	getcd(char *line,struct cddb *cddb) {
	int	idx = 0;
	char	buf[MAXLEN+1];

	spaces();
	cddb->ntracks = getnum(line,&idx);
	spaces();
	if(line[idx++] != ',')
		return -1;
	spaces();
	if(getstring(line,&idx,buf) == -1)
		return -1;
	if(buf[0] == 0)
		cddb->title = NULL;
	else if((cddb->title = strdup(buf)) == NULL) {
		errno = ENOMEM;
		return -1;
	}
	spaces();
	if(line[idx++] != ',') {
		if(cddb->title) free(cddb->title);
		errno = ENOMEM;
		return -1;
	}
	spaces();
	if(getstring(line,&idx,buf) == -1) {
		if(cddb->title) free(cddb->title);
		return -1;
	}
	if(buf[0] == 0)
		cddb->artist = NULL;
	else if((cddb->artist = strdup(buf)) == NULL) {
		if(cddb->title) free(cddb->title);
		errno = ENOMEM;
		return -1;
	}
	spaces();
	if(line[idx++] != ',') {
		if(cddb->title) free(cddb->title);
		if(cddb->artist) free(cddb->artist);
		errno = ENOMEM;
		return -1;
	}
	spaces();
	cddb->flags = getnum(line,&idx);
	spaces();
	if(line[idx] != 0 && line[idx] != '#') {
		if(cddb->title) free(cddb->title);
		if(cddb->artist) free(cddb->artist);
		errno = ENOMEM;
		return -1;
	}

	if((cddb->tracks = malloc(sizeof(struct cddb_track)*
			cddb->ntracks)) == NULL) {
		if(cddb->title) free(cddb->title);
		if(cddb->artist) free(cddb->artist);
		errno = ENOMEM;
		return -1;
	}

	return 0;
}

static int	gettrack(char *line,struct cddb *cddb,int trk) {	
	int	idx = 0;
	char	buf[MAXLEN+1];

	spaces();
	cddb->tracks[trk].track = getnum(line,&idx);
	spaces();
	if(line[idx++] != ',')
		return -1;
	spaces();
	if(getstring(line,&idx,buf) == -1)
		return -1;
	if(buf[0] == 0)
		cddb->tracks[trk].name = NULL;
	else if((cddb->tracks[trk].name = strdup(buf)) == NULL) {
		errno = ENOMEM;
		return -1;
	}
	spaces();
	if(line[idx++] != ',') {
		if(cddb->tracks[trk].name) free(cddb->tracks[trk].name);
		return -1;
	}
	spaces();
	if(getstring(line,&idx,buf) == -1) {
		if(cddb->tracks[trk].name) free(cddb->tracks[trk].name);
		return -1;
	}
	if(buf[0] == 0)
		cddb->tracks[trk].lyrics = NULL;
	else if((cddb->tracks[trk].lyrics = strdup(buf)) == NULL) {
		if(cddb->tracks[trk].name) free(cddb->tracks[trk].name);
		errno = ENOMEM;
		return -1;
	}
	spaces();
	if(line[idx++] != ',') {
		if(cddb->tracks[trk].name) free(cddb->tracks[trk].name);
		if(cddb->tracks[trk].lyrics) free(cddb->tracks[trk].lyrics);
		return -1;
	}
	spaces();
	cddb->tracks[trk].flags = getnum(line,&idx);
	spaces();
	if(line[idx] != 0 && line[idx] != '#') {
		if(cddb->tracks[trk].name) free(cddb->tracks[trk].name);
		if(cddb->tracks[trk].lyrics) free(cddb->tracks[trk].lyrics);
		return -1;
	}

	return 0;
}

static int	parseline(char *line,struct cddb *cddb,int *status,int *trk) {
	switch(*status) {
		case S_ID:
			if(getid(line,cddb) == -1) {
				free(cddb);
				return -1;
			}
			*status = S_CD;
			break;

		case S_CD:
			if(getcd(line,cddb) == -1)
				return -1;
			*status = S_TRACK;
			break;

		case S_TRACK:
			if(gettrack(line,cddb,*trk) == -1)
				return -1;
			if(++(*trk) == cddb->ntracks)
				*status = S_FINISHED;
			break;

		case S_FINISHED:
			return -1;
	}

	return 0;
}

struct cddb	*cddb_load(struct cddb_id *id) {
	char		*filename,idstr[17],line[MAXLEN+1];
	FILE		*fp;
	int		trk = 0, status = S_ID;
	struct cddb	*cddb;

	memcpy(idstr,id,16);
	idstr[16] = 0;

	filename = cddb_file(idstr);

	if((fp = fopen(filename,"r")) == NULL) {
		free(filename);
		return NULL;
	}
	free(filename);

	if((cddb = malloc(sizeof(struct cddb))) == NULL) {
		errno = ENOMEM;
		return NULL;
	}

	while(readline(fp,line,MAXLEN))
		if(!iscomment() && !isempty())
			if(parseline(line,cddb,&status,&trk) == -1) {
				partial_free(cddb,status,trk);
				return NULL;
			}
	
	if(status != S_FINISHED) {
		partial_free(cddb,status,trk);
		return NULL;
	}
	
	if(memcmp(id,&cddb->id,sizeof(struct cddb_id)) != 0) {
		partial_free(cddb,status,trk);
		return NULL;
	}
	
	fclose(fp);

	return cddb;
}

struct cddb	*cddb_new(struct cddb_id *id,int n,int f) {
	struct cddb	*cddb;
	int		trk;

	if((cddb = malloc(sizeof(struct cddb))) == NULL) {
		errno = ENOMEM;
		return NULL;
	}

	cddb->ntracks = n;
	cddb->id = *id;
	cddb->title = NULL;
	cddb->artist = NULL;
	cddb->flags = 0;
	
	if((cddb->tracks = malloc(sizeof(struct cddb_track)*n)) == NULL) {
		free(cddb);
		errno = ENOMEM;
		return NULL;
	}
	
	for(trk = 0; trk < n; trk++) {
		cddb->tracks[trk].track = f+trk;
		cddb->tracks[trk].name = NULL;
                cddb->tracks[trk].lyrics = NULL;
		cddb->tracks[trk].flags = 0;
	}
	
	return cddb;
}

int	cddb_save(struct cddb *cddb) {
	char	*filename,idstr[17];
	FILE	*fp;
	int	trk;
	
	if(cddb == NULL)
		return -1;
	
	memcpy(idstr,cddb->id.id,16);
	idstr[16] = 0;
	
	filename = cddb_file(idstr);

	if((fp = fopen(filename,"w")) == NULL) {
		free(filename);
		return -1;
	}
	free(filename);

	for(trk = 0; trk < 16; trk++)
		putc(cddb->id.id[trk],fp);
	putc('\n',fp);
	fprintf(fp,"%d, \"",cddb->ntracks);
	if(cddb->title)
		fprintf(fp,"%s",cddb->title);
	fprintf(fp,"\", \"");
	if(cddb->artist)
		fprintf(fp,"%s",cddb->artist);
	fprintf(fp,"\", %d\n",cddb->flags);
		
	for(trk = 0; trk < cddb->ntracks; trk++) {
           	struct cddb_track	*track;
                
                track = &cddb->tracks[trk];
		fprintf(fp,"%d, \"",track->track);
		if(track->name)
			fprintf(fp,"%s",track->name);
                fprintf(fp,"\", \"");
                if(track->lyrics)
                   	fprintf(fp,"%s",track->lyrics);
		fprintf(fp,"\", %d\n",track->flags);
	}
	
	fclose(fp);
	
	return 0;
}

void	cddb_free(struct cddb *cddb) {
	int	trk;
	
	if(cddb == NULL)
		return;
	
	for(trk = 0; trk < cddb->ntracks; trk++) 
		if(cddb->tracks[trk].name) free(cddb->tracks[trk].name);
	free(cddb->tracks);
	free(cddb);
}

char	*cddb_trackname(struct cddb *cddb,int track) {
	int	trk;
	
	if(cddb == NULL)
		return NULL;
	
	for(trk = 0; trk < cddb->ntracks; trk++)
		if(cddb->tracks[trk].track == track)
			return cddb->tracks[trk].name;
	
	return NULL;
}

int	cddb_trackflags(struct cddb *cddb,int track) {
	int	trk;
	
	if(cddb == NULL)
		return 0;
	
	for(trk = 0; trk < cddb->ntracks; trk++)
		if(cddb->tracks[trk].track == track)
			return cddb->tracks[trk].flags;
	
	return 0;
}

char	*cddb_file(char *fn)
{
	char *home, *filename;

	home = getenv("HOME");

	if((filename = malloc(strlen(home)+strlen(CDDB_DIR)+strlen(fn)+3)) == NULL) {
		errno = ENOMEM;
		return NULL;
	}

	sprintf(filename,"%s/%s/%s",home,CDDB_DIR,fn);
	
	return filename;
}

char	*cddb_lyrics(struct cddb *cddb, int idx)
{
	char *h, *filename;
	char *a, *t, *n;
	int len;
	
	if((a = cddb_artist(cddb)) == NULL) return NULL;
	if((t = cddb_title(cddb)) == NULL) return NULL;
	if((n = cddb->tracks[idx].name) == NULL) return NULL;

	h = getenv("HOME");

	len = strlen(h)+strlen(CDDB_DIR)+strlen(a)+strlen(t)+strlen(n)+9;

	if((filename = malloc(len)) == NULL) {
		errno = ENOMEM;
		return NULL;
	}

	sprintf(filename,"%s/%s/%s/%s/%s.txt",h,CDDB_DIR,a,t,n);
	
	return filename;
}

/* cddb.c		Flavio Lerda		97-07-29 */
