/* cdplayer.c		Flavio Lerda		97-07-29 */

/*
	cdplayer 0.2 - simple cdplayer with shuffle and database capability
	Copyright (C) 1997,98  Flavio Lerda

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

        To contact the author:
                Flavio Lerda
                Via vittime di Bologna 14
                12026 Piasco CN
                Italy
        or:
                flerda@athena.polito.it
*/   

/*
   CD Player program: handle cd database and shuffle.
   
   97-07-29: [MAS]	Code written for the first time.
   97-08-01: [MAS]	Added code for command line arguments.
   97-08-02: [MAS]	Added repeat disk option.
   97-08-11: [MAS]	Added libterm code.
   97-08-30: [MAS]	Added playcd, track and quit options.
   98-06-26: [MAS]	Conversion to ncurses.
*/

#include "cd-rom.h"
#include "cddb.h"
#include "screen.h"
#include "strutil.h"
#include "lyrics.h"
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <getopt.h>
#include <regex.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <signal.h>

#define	CDROM	"/dev/cdrom"		/* Default device */

/* Define the commands */
#define	CMD_NOTHING		0
#define CMD_PLAY		1
#define CMD_PLAY_CD		2
#define CMD_PLAY_TRACK		3
#define CMD_PLAY_PROGRAM	4
#define CMD_STOP		5
#define CMD_PAUSE		6
#define CMD_RESUME		7
#define CMD_INTERACTIVE		8
#define CMD_KILL		9
#define CMD_NUM			10

/* Define the options */
#define OPT_NONE		0x00
#define OPT_SHUFFLE		0x01
#define OPT_REPEAT		0x02
#define OPT_DONTSKIP		0x04
#define OPT_EJECT		0x08
#define OPT_ID			0x10
#define OPT_LIST		0x20
#define OPT_QUIT		0x40
#define OPT_TRACE		0x80
#define OPT_ALL			0xFF
#define OPT_NUM			8

static int	incompatibility[OPT_NUM] = {
	OPT_ID|OPT_LIST|OPT_TRACE,
	OPT_ID|OPT_LIST|OPT_TRACE,
	OPT_ID|OPT_LIST|OPT_TRACE,
	OPT_NONE,
	OPT_SHUFFLE|OPT_REPEAT|OPT_DONTSKIP,
	OPT_SHUFFLE|OPT_REPEAT|OPT_DONTSKIP,
	OPT_ID|OPT_LIST|OPT_TRACE,
	OPT_SHUFFLE|OPT_REPEAT|OPT_DONTSKIP|OPT_EJECT|OPT_ID|OPT_LIST|OPT_QUIT
};

static int	options_table[OPT_NUM] = {
	OPT_SHUFFLE,
	OPT_REPEAT,
	OPT_DONTSKIP,
	OPT_EJECT,
	OPT_ID,
	OPT_LIST,
	OPT_QUIT,
	OPT_TRACE
};

static int	validoptions[CMD_NUM] = {
	OPT_EJECT|OPT_ID|OPT_LIST|OPT_TRACE,
	OPT_SHUFFLE|OPT_REPEAT|OPT_DONTSKIP|OPT_EJECT|OPT_QUIT,
	OPT_REPEAT|OPT_EJECT|OPT_QUIT,
	OPT_REPEAT|OPT_DONTSKIP|OPT_EJECT|OPT_QUIT,
	OPT_SHUFFLE|OPT_REPEAT|OPT_DONTSKIP|OPT_EJECT|OPT_QUIT,
	OPT_EJECT,
	OPT_NONE,
	OPT_NONE,
	OPT_ALL,
	OPT_ALL
};
	
#define	isset(x)		((options & (x)) == (x))
#define set(x)			(options |= (x))
#define unset(x)		(options &= ~(x))

/* Color schemes for the ncurses */
#define DEFAULT			0
#define TITLES			1
#define SONG_TITLE		2
#define SONG_TIMES		3

static color attrs[] = {
	{ 3, 3, 3, 3 },
	{ COLOR_RED, COLOR_BLACK, A_BOLD, A_BOLD },
	{ COLOR_BLUE, COLOR_BLACK, A_BOLD, A_BOLD },
	{ COLOR_WHITE, COLOR_BLACK, A_BOLD, A_UNDERLINE }
};

/* Color schemes for the interactive program */
#define CLOSEBUTTON			1
#define TITLEBAR			2
#define INFOBAR				3
#define INFORMATION			4
#define STATUSBAR			5
#define SONGS_ACTIVE			6
#define SONGS_ACTIVE_SELECTED		7
#define SONGS_INACTIVE			8
#define SONGS_INACTIVE_SELECTED		9
#define LYRICS_ACTIVE			10
#define LYRICS_ACTIVE_SELECTED		11
#define LYRICS_INACTIVE			12
#define LYRICS_INACTIVE_SELECTED	13

static color interactives[] = {
	{ 13, 13, 13, 13 },
	{ COLOR_BLACK, COLOR_WHITE, A_NORMAL, A_REVERSE },	/* CLOSEBUTTON */
	{ COLOR_WHITE, COLOR_BLUE, A_BOLD, A_BOLD },		/* TITLEBAR */
	{ COLOR_BLACK, COLOR_CYAN, A_NORMAL, A_NORMAL },	/* INFOBAR */
	{ COLOR_RED, COLOR_CYAN, A_NORMAL, A_UNDERLINE },	/* INFORMATION */
	{ COLOR_BLACK, COLOR_WHITE, A_NORMAL, A_REVERSE },	/* STATUSBAR */
	{ COLOR_BLUE, COLOR_BLACK, A_BOLD, A_NORMAL},		/* SONGS_ACTIVE */
	{ COLOR_CYAN, COLOR_BLACK, A_BOLD, A_REVERSE },		/* SONGS_ACTIVE_SELECTED */
	{ COLOR_WHITE, COLOR_BLACK, A_NORMAL, A_NORMAL},	/* SONGS_INACTIVE */
	{ COLOR_WHITE, COLOR_BLACK, A_BOLD, A_BOLD },		/* SONGS_INACTIVE_SELECTED */
	{ COLOR_YELLOW, COLOR_BLUE, A_BOLD, A_NORMAL},		/* LYRICS_ACTIVE */
	{ COLOR_WHITE, COLOR_BLUE, A_BOLD, A_REVERSE },		/* LYRICS_ACTIVE_SELECTED */
	{ COLOR_WHITE, COLOR_BLUE, A_NORMAL, A_NORMAL},		/* LYRICS_INACTIVE */
	{ COLOR_BLACK, COLOR_BLUE, A_BOLD, A_BOLD }		/* LYRICS_INACTIVE_SELECTED */
};

/* Some global flags. Used for help, version and license options */
static int	showhelp = FALSE;
static int	showversion = FALSE;
static int	showlicense = FALSE;
static int	command = CMD_NOTHING;
static int	options = OPT_NONE;
static char	*pname = NULL;
static char	*device = NULL;

/* Common variables */
static struct cd		*cd = NULL;
static struct cdrom_toc		*toc = NULL;
static struct cddb		*cddb;
static char			*track = NULL;
static char			*program = NULL;
static struct cdrom_status	status;
static int			current = 0;
static int			playing = FALSE;
static int			stopped = FALSE;
static int			shuffling = FALSE;
static int			last = -1;
static lyrics_line		**song_lyrics = NULL;
static int			active_line = 0;
static int			active_length = 0;
static int			first_line = 0;
static int			active_win = 0;
static int			lba_next = 0;
#define MSF(m,s,f)		( ( (m) * 60 + (s) ) * 75 + (f) )
static int			lba_pos[32] = {
	MSF(0,	11,	0),
	MSF(0,	16,	0),
	MSF(0,	22,	0),
	MSF(0,	25,	0),
	MSF(0,	31,	0),
	MSF(0,	37,	0),
	MSF(0,	44,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
	MSF(6,	0,	0),
};

/* Windows used in the interactive mode */
static WINDOW			*titlebar;
static WINDOW			*infobar;
static WINDOW			*statusbar;
static WINDOW			*songs;
static WINDOW			*lyrics;

/* Banner to write at startup */
static char	*banner[] = {
	"CDPlayer 0.2 - Simple CD player with database and shuffle functions",
	"Copyright (C) 1997,98 Flavio Lerda",
	"",
	"This software is free, and you are welcome to redistribute it under",
	"the term of GNU General Public License. Type `cdplayer --license'",
	"to view the full license.",
	"",
	NULL
};

/* The program license */
static char	*license[] = {
	"CDPlayer 0.2 - Simple CD player with database and shuffle functions",
	"Copyright (C) 1997,98 Flavio Lerda",
	"",
	"This program is free software; you can redistribute it and/or modify",
	"it under the terms of the GNU General Public License as published by",
	"the Free Software Foundation; either version 2 of the License, or",
	"(at your option) any later version.",
	"",
	"This program is distributed in the hope that it will be useful,",
	"but WITHOUT ANY WARRANTY; without even the implied warranty of",
	"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the",
	"GNU General Public License for more details.",
	"",
	"You should have received a copy of the GNU General Public License",
	"along with this program; if not, write to the Free Software",
	"Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.",
	"",
	"To contact the author via electronic mail write to:",
	"    <flerda@athena.polito.it>",
	"or via paper mail write to:",
	"    Flavio Lerda",
	"    via vittime di Bologna 14",
	"    12026 Piasco CN",
	"    Italy",
	NULL
};

/* Help of the program */
static char	*help[] = {
	"usage: cdplayer [COMMAND] [OPTIONS]... [DEVICE]",
	"Play the CD in the device DEVICE (/dev/cdrom by default).",
	"The default command is nothing.",
	"",
	"Commands:",
	"  --play, -p            plays the CD",
	"  --cd, -c              plays the entire CD as a whole",
	"  --track, -t TRACK     plays track TRACK",
	"  --program, -r PROGRAM plays the program PROGRAM",
	"  --stop, -s            stops the drive motor",
	"  --pause, -u           pauses playing",
	"  --resume, -e          resumes playing",
	"",
	"Options:",
	"  --shuffle, -S         plays track in random order",
	"  --repeat, -R          repeats playing endless",
	"  --dontskip, -D        doesn't skip tracks",
	"  --eject, -E           ejects the CD before exiting",
	"  --id, -I              shows the ID of a CD",
	"  --list, -L            lists the tracks of a CD",
	"  --quit, -Q            quit after starting CD",
	"  --trace, -T           trace the current state of CD",
	"",
	NULL
};

/* Struct to translate long options to short ones or set flags */
static struct option long_options[] = {
	/* Special switches */
	{ "help", no_argument, &showhelp, TRUE },
	{ "version", no_argument, &showversion, TRUE },
	{ "license", no_argument, &showlicense, TRUE },
	/* Command switches */
	{ "play", no_argument, NULL, 'p' },
	{ "cd", no_argument, NULL, 'c' },
	{ "track", required_argument, NULL, 't' },
	{ "program", required_argument, NULL, 'r' },
	{ "stop", no_argument, NULL, 's' },
	{ "pause", no_argument, NULL, 'u' },
	{ "resume", no_argument, NULL, 'e' },
	{ "interactive", no_argument, NULL, 'i' },
	{ "kill", no_argument, NULL, 'k' },
	/* Option switches */
	{ "shuffle", no_argument, NULL, 'S' },
	{ "repeat", no_argument, NULL, 'R' },
	{ "dontskip", no_argument, NULL, 'D' },
	{ "eject", no_argument, NULL, 'E' },
	{ "id", no_argument, NULL, 'I' },
	{ "list", no_argument, NULL, 'L' },
	{ "quit", no_argument, NULL, 'Q' },
	{ "trace", no_argument, NULL, 'T' },
	{ NULL, 0, NULL, 0 }
};

/* Keyboard mask */
#define	KB_QUIT		0x01
#define KB_NEXT		0x02
#define KB_PAUSE	0x04
#define KB_RESUME	0x08
#define SHOW_LYRICS	0x10

/* Utility functions */
static void	die(char *);
static void	printerror(char *);
static void	printincompatible(char *,char *);
static void	showstrlist(char **);
static void	print_lba(int);
static void	wprint_lba(WINDOW *, int);
static int	isnull(char *);
static void	genid(struct cddb_id *);
static int	shuffle(int *,int);
static int	showstatus(int);
static void	showplaying(void);
static int	setcommand(int);
static int	setoption(int);
static char	*cmdname(int);
static char	*optname(int);
static int	opencd(int);
static int	closecd(void);
static int	readtoc(int);
static int	freetoc(void);
static int	readcddb(void);
static int	freecddb(void);
static int	strtoi(char *,int *);
static int	splitargs(char *,char ***,char);
static int	addtracks(char *,int *,int *,int *,int);
static int	audiotracks(void);
static void	print_titles(void);
static int	running(void);
static int	get_running_pid(void);
static void	finished(void);
static void	printpid(int);
static void	show_titlebar(void);
static void	show_infobar(void);
static void	show_statusbar(void);
static void	show_songs(void);
static void	show_lyrics(void);
static int	handle_keys(void);
static int	uniform(int,int);
static void	edit_song(void);
static void	sighandler(int);

/* Command functions */
static int	play(void);
static int	play_cd(void);
static int	play_track(void);
static int	play_program(void);
static int	stop(void);
static int	nothing(void);
static int	pause_cd(void);
static int	resume_cd(void);
static int	interactive(void);
static int	kill_running(void);

/* Main program body */
int main(int argc,char *argv[]) {
	int	c,opt = OPT_NONE;
	
	/* stores the name of the program */
	pname = argv[0];
	
	/* set a signal handler */
	signal(SIGTERM, sighandler);

	/* Parse the command line options and set command and options */
	while((c = getopt_long(argc,argv,"pct:r:sueikSRDEILQT",long_options,NULL)) != EOF)
		switch(c) {
			case 0:
				break;
				
			case 'p':
				if(!setcommand(CMD_PLAY)) return 1;
				break;
				
			case 'c':
				if(!setcommand(CMD_PLAY_CD)) return 1;
				break;
				
			case 't':
				if(!setcommand(CMD_PLAY_TRACK)) return 1;
				track = optarg;
				break;
				
			case 'r':
				if(!setcommand(CMD_PLAY_PROGRAM)) return 1;
				program = optarg;
				break;
				
			case 's':
				if(!setcommand(CMD_STOP)) return 1;
				break;
				
			case 'u':
				if(!setcommand(CMD_PAUSE)) return 1;
				break;
				
			case 'e':
				if(!setcommand(CMD_RESUME)) return 1;
				break;
				
			case 'i':
				if(!setcommand(CMD_INTERACTIVE)) return 1;
				break;
				
			case 'k':
				if(!setcommand(CMD_KILL)) return 1;
				break;
				
			case 'S':
				opt |= OPT_SHUFFLE;
				break;
				
			case 'R':
				opt |= OPT_REPEAT;
				break;
				
			case 'D':
				opt |= OPT_DONTSKIP;
				break;
				
			case 'E':
				opt |= OPT_EJECT;
				break;
				
			case 'I':
				opt |= OPT_ID;
				break;
				
			case 'L':
				opt |= OPT_LIST;
				break;
				
			case 'Q':
				opt |= OPT_QUIT;
				break;
				
			case 'T':
				opt |= OPT_TRACE;
				break;
				
			default:
				printerror(NULL);
				return 1;
		}
	
	/* Check if there is the device name */
	if(optind < argc)
		if(optind == argc - 1)
			device = argv[optind];
		else {
			/* Only one device name on the command line */
			printerror("too many arguments");
			return 1;
		}
	else
		device = CDROM;
	
	/* Set the options now that the command is given */
	for(c = 0; c < OPT_NUM; c++)
		if(options_table[c] & opt)
			if(!setoption(options_table[c]))
				return FALSE;
	
	/* The license options has priority on all the others */
	if(showlicense) {
		showstrlist(license);
		return 0;
	}

	/* Then come the help screen */
	if(showhelp) {
		showstrlist(help);
		return 0;
	}
	
	/* Follows the version informations */
	if(showversion) {
		printf("CDPlayer 0.2\n");
		return 0;
	}
	
	/* Show the banner with the program copyright */
	showstrlist(banner);
		
	/* Initialize the random number generator */
	srand(time(NULL));
	
	/* Call the function for the given command */
	switch(command) {
		case CMD_PLAY:
			if(!play())
				return 1;
			break;
			
		case CMD_PLAY_CD:
			if(!play_cd())
				return 1;
			break;
			
		case CMD_PLAY_TRACK:
			if(!play_track())
				return 1;
			break;
			
		case CMD_PLAY_PROGRAM:
			if(!play_program())
				return 1;
			break;
			
		case CMD_STOP:
			if(!stop())
				return 1;
			break;
			
		case CMD_NOTHING:
			if(!nothing())
				return 1;
			break;
			
		case CMD_PAUSE:
			if(!pause_cd())
				return 1;
			break;
			
		case CMD_RESUME:
			if(!resume_cd())
				return 1;
			break;
			
		case CMD_INTERACTIVE:
			if(!interactive())
				return 1;
			break;
			
		case CMD_KILL:
			if(!kill_running())
				return 1;
			break;

		default:
			/* This shouldn't have happened */
			printerror("Internal error 01");
			return 1;
	}

	return 0;
}

/* System utility functions */
static void	die(char *msg) {
	screen_done();
	fprintf(stderr,"%s: ",pname);
	perror(msg);
	exit(1);
}

static void	printerror(char *message) {
	if(message) fprintf(stderr,"%s: %s\n",pname,message);
	fprintf(stderr,"try `%s --help' for more information\n",pname);
}

static void	printincompatible(char *s1,char *s2) {
	fprintf(stderr,"%s: Can't use %s and %s together\n",pname,s1,s2);
	fprintf(stderr,"try `%s --help' for more information\n",pname);
}

/* Utility functions */
static void	showstrlist(char **s) {
	if(!s) return;

	for(; *s; s++)
		fprintf(stderr,"%s\n",*s);
}

static void	print_lba(int lba) {
	u_char  m,s;

	lba2msf(lba,&m,&s,NULL);
	printf("%d:%02d",m,s);
}

static void	wprint_lba(WINDOW *w, int lba) {
	u_char  m,s;

	lba2msf(lba,&m,&s,NULL);
	screen_wprintf(w, "%d:%02d",m,s);
}

static int	isnull(char *s) {
	for(; *s; s++)
		if(*s != '0')
			return FALSE;

	return TRUE;
}

static void	genid(struct cddb_id *id) {
	int	i,n,lba;

	n = toc->ntracks;

	lba = toc->lba_length + n;

	for(i = 0;i < n;i++) lba += toc->entries[i].lba_length;

	if(toc->upc[0] != 0 && !isnull(toc->upc)) {
		int	l = strlen(toc->upc);

		for(i = 0; i < 13; i++)
			if(13 - i > l)
				id->id[i] = '0';
			else
				id->id[i] = toc->upc[l - 13 + i];

		id->id[13] = '-';	
		for(i = 14;i < 16;i++,lba /= 10)
			id->id[i] = '0' + abs(lba % 10);
	} else
		for(i = 15;i >= 0;i--,lba /= 10)
			id->id[i] = '0' + abs(lba % 10);
}

static int	shuffle(int *sequence,int ntracks) {
	int	*shuffled;
	int	idx;
	
	if(!sequence || !ntracks) return FALSE;

	if((shuffled = malloc(sizeof(int) * ntracks)) == NULL)
		return FALSE;
	
	for(idx = 0; idx < ntracks; idx++)
		shuffled[idx] = -1;
	
	for(idx = 0; idx < ntracks; idx++) {
		int	j;
		
		j = uniform(0, ntracks-1);
		while(shuffled[j] != -1)
			j = (j+1) % ntracks;
		shuffled[j] = sequence[idx];
	}
	
	for(idx = 0; idx < ntracks; idx++)
		sequence[idx] = shuffled[idx];
	
	free(shuffled);
		
	return TRUE;
}

static int	showstatus(int mask) {
	int current;
		
	/* Get the current status */
	if(cdrom_status(cd,&status) == -1)
		die("cdrom_status()");

	for(current = 0; current < toc->ntracks; current++)
		if(cddb->tracks[current].track == status.track)
			break;
		
	if(current == toc->ntracks) current = -1;

	if((mask & SHOW_LYRICS) == 0 || 
			song_lyrics == NULL || 
			current == -1 || 
			song_lyrics[current] == NULL) {
		switch(status.status) {
			case CDS_PLAY:
				showplaying();
				break;

			case CDS_PAUSE:
				screen_printf("Paused");
				break;

			case CDS_STOP:
				screen_printf("Stopped");
				break;

			case CDS_OPEN:
				screen_printf("Open");
				break;

			case CDS_EMPTY:
				screen_printf("Empty");
				break;
		}

		/* Clear until end of line */
		screen_clreol();

		/* Do down and then up one line */
		screen_printf("\n");
		screen_moverel(0, -1);
	} else {
		if(lba_pos[lba_next] <= status.lba_pos_rel) {
			lyrics_line *l = song_lyrics[current];
			int idx;

			for(idx = lba_next; idx > 0; idx++)
				if(l)
					l = l->ll_next;

			do {
				if(l) {
					printf("%s\n", l->ll_data);
					l = l->ll_next;
				}
				lba_next++;
			} while(lba_pos[lba_next] <= status.lba_pos_rel);
		}
	}

	/* Check if the song is finished */
	if(status.status != CDS_STOP) {
		struct timeval	tv;
		fd_set		rfds;
		int		wait;

		if(status.status == CDS_PAUSE) {
			/* wait a millisecond */
			tv.tv_sec = 0;
			tv.tv_usec = 1000;
		} else {
			/* wait for one second */
			tv.tv_sec = 1;
			tv.tv_usec = 0;
		}

		do {
			FD_ZERO(&rfds);
			FD_SET(0,&rfds);

			select(1,&rfds,NULL,NULL,&tv);
			wait = FALSE;

			if(screen_kbhit())
				switch(screen_getkey()) {
					case 'q': /* quit */
					case 'Q':
						if(mask & KB_QUIT) return KB_QUIT;
						break;

					case 'n': /* next */
					case 'N':
						if(mask & KB_NEXT) return KB_NEXT;
						break;

					case 'p': /* pause */
					case 'P':
						if(mask & KB_PAUSE) return KB_PAUSE;
						break;

					case 'r': /* resume */
					case 'R':
						if(mask & KB_RESUME) return KB_RESUME;
						break;

					default:
						wait = TRUE;
						break;
				}
		} while(wait);
	}

	return EOF;
}

static void	showplaying(void) {
	char	*name;
	int	i;
	int	lba_length = -1;

	/* Get the name of the track */
	name = cddb_trackname(cddb,status.track);

	/* Get the lenght of the track */
	for(i = 0; i < toc->ntracks; i++)
		if(toc->entries[i].track == status.track) {
			lba_length = toc->entries[i].lba_length;
			break;
		}

	/* Show the current position and the time till end */
	screen_printf("Playing ");
	if(name) {
		screen_printf("\"");
		screen_attr(SONG_TITLE);
		screen_printf("%s",name);
		screen_attr(DEFAULT);
		screen_printf("\"");
	} else {
		screen_attr(SONG_TITLE);
		screen_printf("#%02d",status.track);
		screen_attr(DEFAULT);
	}

	screen_printf(" ");
	screen_attr(SONG_TIMES);
	wprint_lba(stdscr, status.lba_pos_rel);
	screen_attr(DEFAULT);
	if(lba_length != -1) {
		screen_printf(" - ");
		screen_attr(SONG_TIMES);
		wprint_lba(stdscr, lba_length - status.lba_pos_rel);
		screen_attr(DEFAULT);
	}
}

static int	setcommand(int cmd) {
	if(command != CMD_NOTHING) {
		printincompatible(cmdname(cmd),cmdname(command));
		return FALSE;
	}
	command = cmd;
	return TRUE;
}

static int	setoption(int opt) {
	int	j;

	for(j = 0; j < OPT_NUM; j++)
		if(options_table[j] == opt)
			break;

	if((incompatibility[j] & options) != 0) {
		int	i;

		for(i = 1;i; i <<= 1)
			if((incompatibility[j] & i) == i)
				if((options & i) == i)
					break;

		printincompatible(optname(opt),optname(i));
		return FALSE;
	}

	if((validoptions[command] & opt) != opt) {
		printincompatible(cmdname(command),optname(opt));
		return FALSE;
	}

	set(opt);

	return TRUE;
}

static char	*cmdname(int cmd) {
	static char	*names[CMD_NUM] = {
		"no command",
		"--play",
		"--cd",
		"--track",
		"--program",
		"--stop",
		"--pause",
		"--resume",
		"--interactive",
		"--kill"
	};

	if(cmd >= 0 && cmd < CMD_NUM)	
		return names[cmd];

	return NULL;
}

static char	*optname(int opt) {
	static char	*names[OPT_NUM] = {
		"--shuffle",
		"--repeat",
		"--dontskip",
		"--eject",
		"--id",
		"--list",
		"--quit"
	};
	int	i;

	for(i = 0; i < OPT_NUM; i++)
		if(options_table[i] == opt)
			return names[i];

	return NULL;
}

static int	opencd(int check) {
	cd = cdrom_init(device,TRUE);

	if(cd == NULL) return FALSE;

	cdrom_status(cd,&status);

	if(check) {
		if(status.status == CDS_OPEN) {
			printerror("cdrom drive open");
			return FALSE;
		}
		if(status.status == CDS_EMPTY) {
			printerror("cdrom drive empty");
			return FALSE;
		}
	}

	return TRUE;
}

	static int	closecd(void) {
		if(cdrom_done(cd) == -1)
			return FALSE;

		return TRUE;
	}

	static int	readtoc(int check) {
		toc = cdrom_toc(cd);

		if(toc == NULL) return FALSE;

		if(check)
			if(!audiotracks()) {
				printerror("not audio cdrom in drive");
				return FALSE;
			}

		return TRUE;
	}

	static int	freetoc(void) {
		if(cdrom_freetoc(toc) == -1)
			return FALSE;

		return TRUE;
	}

	static int	readcddb(void) {
		struct cddb_id	id;
		genid(&id);
		if((cddb = cddb_load(&id)) == NULL) {
			cddb = cddb_new(&id,toc->ntracks,toc->first_track);
		if(cddb == NULL) return FALSE;
		cddb_save(cddb);
	}
	
	return TRUE;
}

static int	freecddb(void) {
	cddb_free(cddb);
	
	return TRUE;
}

static int	strtoi(char *str,int *pint) {
	int	n = 0;
	int	sign = +1;
	
	while(isspace(*str))
		str++;
	
	if(*str == '+' || *str == '-') {
		if(*str == '-') sign = -1;
		str++;
	}
	
	while(isdigit(*str)) {
		n *= 10;
		n += *str - '0';
		str++;
	}
	
	while(isspace(*str))
		str++;
	
	if(*str == 0) {
		if(pint) *pint = n;
		return TRUE;
	}
	
	return FALSE;
}

static int	splitargs(char *line,char ***args,char sep) {
	int	cnt = 2,idx;
	
	for(idx = 0; line[idx]; idx++)
		if(line[idx] == sep)
			cnt++;
	
	if((*args = malloc(sizeof(char *) * cnt)) == NULL) return 0;
	
	(*args)[0] = line;
	cnt = 1;
	for(idx = 0; line[idx]; idx++)
		if(line[idx] == sep) {
			line[idx] = 0;
			(*args)[cnt++] = line + idx + 1;
		}
	(*args)[cnt] = NULL;
	
	return cnt;
}

static int	addtracks(char *track,int *present,int *sequence,int *ntracks,int add) {
	int	trk;

	/* check if it's the number of a track */
	if(!strtoi(track,&trk)) {
		regex_t	regexp;
		
		if(regcomp(&regexp,track,REG_NOSUB) != 0) {
			printerror("error in track regular expressionv");
			return FALSE;
		}
		
		if(isset(OPT_DONTSKIP)) {
			int	i;
			
			for(i = 0; i < cddb_ntracks(cddb); i++)
				if(cddb->tracks[i].name)
					if(regexec(&regexp,cddb->tracks[i].name,0,NULL,0) == 0) {
						trk = cddb->tracks[i].track;
						if(!present[trk-toc->first_track]) {
							if(add) sequence[*ntracks] = trk;
							(*ntracks)++;
							present[trk-toc->first_track] = TRUE;
						}
					}
		} else {
			int	i;
			
			for(i = 0; i < cddb_ntracks(cddb); i++)
				if(cddb->tracks[i].name)
					if((cddb->tracks[i].flags & CDDB_TF_SKIP) != CDDB_TF_SKIP)
						if(regexec(&regexp,cddb->tracks[i].name,0,NULL,0) == 0) {
							trk = cddb->tracks[i].track;
							if(!present[trk-toc->first_track]) {
								if(add) sequence[*ntracks] = trk;
								(*ntracks)++;
								present[trk-toc->first_track] = TRUE;
							}
						}
		}

		regfree(&regexp);
	} else {
		if(trk < toc->first_track || trk > toc->last_track) {
			printerror("track out of range");
			return FALSE;
		}

		if(!isset(OPT_DONTSKIP))
			if((cddb_trackflags(cddb,trk) & CDDB_TF_SKIP) == CDDB_TF_SKIP) {
				printerror("track skipped");
				return FALSE;
			}

		if(!present[trk-toc->first_track]) {
			if(add) sequence[*ntracks] = trk;
			(*ntracks)++;
			present[trk-toc->first_track] = TRUE;
		}
	}

	return TRUE;
}

static int	audiotracks(void) {
	int	i;
	
	for(i = 0; i < toc->ntracks; i++)
		if(!toc->entries[i].audio)
			return FALSE;
	
	return TRUE;
}

static void	print_titles(void)
{
	screen_printf("Title : ");
	screen_attr(TITLES);
	if(cddb_title(cddb))
		screen_printf("%s",cddb_title(cddb));
	else
		screen_printf("-- unnamed --");
	screen_attr(DEFAULT);
	screen_printf("\nArtist: ");
	screen_attr(TITLES);
	if(cddb_artist(cddb)) 
		screen_printf("%s",cddb_artist(cddb));
	else
		screen_printf("-- unnamed --");
	screen_attr(DEFAULT);
	screen_printf("\n\n");
}

static int	running(void)
{
	return open("/var/tmp/cdplayer.pid", O_WRONLY | O_CREAT | O_EXCL, 0600);
}

static int	get_running_pid(void)
{
	FILE *fp;
	int pid;
	
	fp = fopen("/var/tmp/cdplayer.pid", "r");
	fscanf(fp, "%d", &pid);
	fclose(fp);
	
	return pid;
}

static void	finished(void)
{
	unlink("/var/tmp/cdplayer.pid");
}

static void	printpid(int fd)
{
	FILE *fp = fdopen(fd, "w");
	
	fprintf(fp, "%d\n", getpid());
	fclose(fp);
	close(fd);
}

static void	show_titlebar(void)
{
	screen_wattr(titlebar, TITLEBAR);
	screen_wfill(titlebar, ' ');
	screen_wattr(titlebar, CLOSEBUTTON);
	screen_wprintf(titlebar, "   ");
	screen_wattr(titlebar, TITLEBAR);
	screen_wprintf(titlebar, "  CDPlayer 0.2 (c) 1997,98 Flavio Lerda");
	wrefresh(titlebar);
}

static void	show_infobar(void)
{
	screen_wattr(infobar, INFOBAR);
	screen_wfill(infobar, ' ');
	screen_wmove(infobar, 0, 0);
	screen_wprintf(infobar, " Title : ");
	screen_wattr(infobar, INFORMATION);
	if(cddb_title(cddb))
		screen_wprintf(infobar, " %s ",cddb_title(cddb));
	else
		screen_wprintf(infobar, " -- unnamed -- ");
	screen_wmove(infobar, 0, 1);
	screen_wattr(infobar, INFOBAR);
	screen_wprintf(infobar, " Artist: ");
	screen_wattr(infobar, INFORMATION);
	if(cddb_artist(cddb)) 
		screen_wprintf(infobar, " %s ",cddb_artist(cddb));
	else
		screen_wprintf(infobar, " -- unnamed -- ");
	screen_wrefresh(infobar);
}

static void	show_statusbar(void)
{
	char	*name;
	int	i;
	int	lba_length = -1;

	screen_wattr(statusbar, STATUSBAR);
	screen_wfill(statusbar, ' ');
	screen_wmove(statusbar, 1, 0);
	switch(status.status) {
		case CDS_PLAY:
			/* Get the name of the track */
			name = cddb_trackname(cddb,status.track);

			/* Get the lenght of the track */
			for(i = 0; i < toc->ntracks; i++)
				if(toc->entries[i].track == status.track) {
					lba_length = toc->entries[i].lba_length;
					break;
				}

			/* Show the current position and the time till end */
			screen_wprintf(statusbar, "Playing ");
			if(name) {
				screen_wprintf(statusbar, "\"");
				screen_wprintf(statusbar, "%s",name);
				screen_wprintf(statusbar, "\"");
			} else
				screen_wprintf(statusbar, "#%02d",status.track);

			screen_wprintf(statusbar, " ");
			wprint_lba(statusbar, status.lba_pos_rel);
			if(lba_length != -1) {
				screen_wprintf(statusbar, " - ");
				wprint_lba(statusbar, lba_length - status.lba_pos_rel);
			}
			break;

		case CDS_PAUSE:
			screen_wprintf(statusbar, "Paused");
			break;

		case CDS_STOP:
			screen_wprintf(statusbar, "Stopped");
			break;

		case CDS_OPEN:
			screen_wprintf(statusbar, "Open");
			break;

		case CDS_EMPTY:
			screen_wprintf(statusbar, "Empty");
			break;
	}
	screen_wrefresh(statusbar);
}

static void	show_songs(void)
{
	int i;

	if(active_win == 0)
	/* 1 */
	{
	/* 1 */
		screen_wattr(songs, SONGS_ACTIVE);
	/* 1
	else
		screen_wattr(songs, SONGS_INACTIVE);
	*/
	screen_wfill(songs, ' ');
	for(i = 0; i < cddb_ntracks(cddb); i++) {
		char *name;
		int trk, flg;

		name = cddb->tracks[i].name;
		trk = cddb->tracks[i].track;
		flg = cddb->tracks[i].flags;

		screen_wmove(songs, 0, i);
		
		if(active_win == 0)
			if(i == current)
				screen_wattr(songs, SONGS_ACTIVE_SELECTED);
			else
				screen_wattr(songs, SONGS_ACTIVE);
		else
			if(i == current)
				screen_wattr(songs, SONGS_INACTIVE_SELECTED);
			else
				screen_wattr(songs, SONGS_INACTIVE);

		if(trk == status.track)
			screen_waddch(songs, '>');
		else
			screen_waddch(songs, ' ');

		if((flg & CDDB_TF_SKIP) != CDDB_TF_SKIP)
			screen_waddch(songs, 'P');
		else
			screen_waddch(songs, ' ');
		
		if(song_lyrics[i])
			screen_waddch(songs, 'L');
		else
			screen_waddch(songs, ' ');

		screen_wprintf(songs, " [%d] %s", trk, name);
	}
	screen_wrefresh(songs);
	/* 1 */
	}
	/* 1 */
}

static void	show_lyrics(void)
{
	if(active_win == 1) {
		screen_wattr(lyrics, LYRICS_ACTIVE);
		lyrics_wprint(lyrics, song_lyrics[current], first_line, active_line, LYRICS_ACTIVE_SELECTED, LYRICS_ACTIVE);
		/* 1
	} else {
		screen_wattr(lyrics, LYRICS_INACTIVE);
		lyrics_wprint(lyrics, song_lyrics[current], first_line, -1, LYRICS_INACTIVE_SELECTED, LYRICS_INACTIVE);
	}
		*/
	screen_wrefresh(lyrics);
	/* 1 */
	}
	/* 1 */
}

static int	handle_keys(void)
{
	struct timeval tv;
	int wait, trk;
	fd_set rfds;

	trk = cddb->tracks[current].track;

	tv.tv_sec = 1;
	tv.tv_usec = 0;

	do {
		FD_ZERO(&rfds);
		FD_SET(0,&rfds);

		select(1,&rfds,NULL,NULL,&tv);
		wait = FALSE;

		if(screen_kbhit())
			switch(screen_getkey()) {
				case ' ':
					playing = FALSE;
					if(status.status == CDS_PLAY && trk == status.track) {
						if(cdrom_stop(cd) == -1)
							die("cdrom_stop()");
					} else {
						if(cdrom_playtrack(cd,trk) == -1)
							die("cdrom_playtrack()");
					}
					break;

				case 'r':
					clearok(curscr, TRUE);
					wnoutrefresh(stdscr);
					wnoutrefresh(titlebar);
					wnoutrefresh(infobar);
					wnoutrefresh(statusbar);
					wnoutrefresh(songs);
					wnoutrefresh(lyrics);
					doupdate();
					break;
					

				case 'E':
					edit_song();
					show_lyrics();
					clearok(curscr, TRUE);
					wnoutrefresh(stdscr);
					wnoutrefresh(titlebar);
					wnoutrefresh(infobar);
					wnoutrefresh(statusbar);
					wnoutrefresh(songs);
					wnoutrefresh(lyrics);
					doupdate();
					break;
					
				case 'e':
					playing = FALSE;
					if(cdrom_eject(cd) == -1)
						die("cdrom_eject()");
					break;

				case 'P':
					shuffling = FALSE;
					playing = TRUE;
					last = -1;
					break;

				case 'S':
					shuffling = TRUE;
					playing = TRUE;
					last = -1;
					break;
					
				case 's':
					playing = FALSE;
					if(cdrom_stop(cd) == -1)
						die("cdrom_stop()");
					break;

				case 'p':
					if((cddb->tracks[current].flags & CDDB_TF_SKIP) == CDDB_TF_SKIP)
						cddb->tracks[current].flags &= ~CDDB_TF_SKIP;
					else
						cddb->tracks[current].flags |= CDDB_TF_SKIP;
					break;

				case 'Q':
					playing = TRUE;
					return FALSE;
					
				case 'q':
					playing = FALSE;
					return FALSE;
					
				case 'h':
				case KEY_LEFT:
					if(active_win != 0) {
						active_win = 0;
						show_songs();
						/* 1 */
						redrawwin(songs);
						/* 1 */
					}
					break;
					
				case 'l':
				case KEY_RIGHT:
					if(active_win != 1 && song_lyrics[current]) {
						active_win = 1;
						active_length = lyrics_length(song_lyrics[current]);
						show_lyrics();
						/* 1 */
						redrawwin(lyrics);
						/* 1 */
					}
					break;

				case 'j':
				case KEY_DOWN:
					if(active_win == 0) {
						if(current+1 < cddb_ntracks(cddb)) {
							current++;
							first_line = active_line = 0;
							active_length = lyrics_length(song_lyrics[current]);
						} else
							beep();
					} else {
						if(active_line < LINES-5)
							active_line++;
						else if(active_line + first_line < active_length)
							first_line++;
						else
							beep();
					}
					break;

				case 'k':
				case KEY_UP:
					if(active_win == 0) {
						if(current > 0) {
							current--;
							first_line = active_line = 0;
							active_length = lyrics_length(song_lyrics[current]);
						} else
							beep();
					} else {
						if(active_line > 0)
							active_line--;
						else if(first_line > 0)
							first_line--;
						else
							beep();
					}
					break;

				case 'J':
				case KEY_END:
					if(active_win == 0) {
						if(current != cddb_ntracks(cddb)-1) {
							current = cddb_ntracks(cddb)-1;
							first_line = active_line = 0;
							active_length = lyrics_length(song_lyrics[current]);
						}
					} else {
						active_line = LINES-5;
						first_line = active_length - active_line;
					}
					break;

				case 'K':
				case KEY_HOME:
					if(active_win == 0) {
						if(current != 0) {
							current = 0;
							first_line = active_line = 0;
							active_length = lyrics_length(song_lyrics[current]);
						}
					} else {
						active_line = first_line = 0;
					}
					break;

				default:
					beep();
					wait = TRUE;
					break;
			}
	} while(wait);

	return TRUE;
}

static int	uniform(int min,int max)
{
	return min + (int)(1.0*(max-min+1)*rand()/(RAND_MAX+1.0));
}

static void	edit_song(void)
{
	char *fn = cddb_lyrics(cddb, current);
	int pid;
	
	if(fn == NULL) beep();
	
	reset_shell_mode();
	if((pid = fork()) == -1)
		beep();
	else if(pid == 0) {
		execl("/usr/bin/vi", "vi", fn, NULL);
	} else
		wait(NULL);
	reset_prog_mode();
	
	if(song_lyrics[current] != NULL) 
		lyrics_unload(song_lyrics[current]);
	song_lyrics[current] = lyrics_load(fn);
	active_line = first_line = 0;
	active_length = lyrics_length(song_lyrics[current]);
}

static void	sighandler(int sig)
{
	signal(sig, sighandler);
	
	cdrom_stop(cd);
	stopped = TRUE;
}

/* Command functions */
static int	play(void) {
	int	tracks = 0,*sequence,idx,n;

	if(!opencd(TRUE)) return FALSE;
	if(!readtoc(TRUE)) return FALSE;
	if(!readcddb()) return FALSE;

	n = cddb_ntracks(cddb);
	song_lyrics = (lyrics_line **)malloc(sizeof(lyrics_line) * n);
	if(song_lyrics == NULL) return FALSE;
	for(idx = 0; idx < n; idx++) {
		char *fn = cddb_lyrics(cddb, idx);
		
		if(fn) {
			song_lyrics[idx] = lyrics_load(fn);
			free(fn);
		} else
			song_lyrics[idx] = NULL;
	}

	if(isset(OPT_DONTSKIP))
		tracks = toc->ntracks;
	else 
		for(idx = toc->first_track; idx <= toc->last_track; idx++)
			if((cddb_trackflags(cddb,idx) & CDDB_TF_SKIP) != CDDB_TF_SKIP)
				tracks++;

	if((sequence = malloc(sizeof(int) * tracks)) == NULL) {
		printerror("can't allocate enough memory");
		return FALSE;
	}

	if(isset(OPT_DONTSKIP))
		for(idx = 0; idx < tracks; idx++)
			sequence[idx] = idx + toc->first_track;
	else {
		int	i = 0;

		for(idx = toc->first_track; idx <= toc->last_track; idx++)
			if((cddb_trackflags(cddb,idx) & CDDB_TF_SKIP) != CDDB_TF_SKIP)
				sequence[i++] = idx;
	}

	/* Shuffle if necessary */
	if(isset(OPT_SHUFFLE))
		if(!shuffle(sequence,tracks))
			return FALSE;

	if(!isset(OPT_QUIT)) {
		/* Initialize ncurses */
		screen_init();
		screen_colors(attrs);
		screen_attr(DEFAULT);

		/* Clear the screen and write the title and the artist */
		screen_clear();
		print_titles();

		/* Repeat to infinite if repeat flag is set, one else */
		do {
			int	x;

			for(x = 0;x < tracks;x++) {
				char	*name;
				int	trk;

				trk = sequence[x];

				/* Get the name of the track */
				name = cddb_trackname(cddb,trk);

				/* Play the track */
				if(cdrom_playtrack(cd,trk) == -1)
					die("cdrom_playtrack()");

				do {
					int	c = showstatus(SHOW_LYRICS|KB_QUIT|KB_NEXT|(status.status != CDS_PAUSE ? KB_PAUSE : KB_RESUME));

					switch(c) {
						case KB_QUIT:
							/* stop the CD */
							cdrom_stop(cd);

							/* we lay about it, but who cares ? */
							status.status = CDS_STOP;
							x = tracks;
							options &= ~OPT_REPEAT;
							break;

						case KB_NEXT:
							/* stop the CD, so we can go on with the next track */
							cdrom_stop(cd);
							status.status = CDS_STOP;

						case KB_PAUSE:
							/* pause the CD */
							cdrom_pause(cd);
							break;

						case KB_RESUME:
							/* resume playing */
							cdrom_resume(cd);
							break;
					}
				} while(status.status != CDS_STOP);

				screen_printf("Played ");
				if(name)
					screen_printf("\"%s\"",name);
				else
					screen_printf("#%02d",trk);
				screen_clreol();
				screen_printf("\n");
			}
		} while(isset(OPT_REPEAT));

				/* clear the screen and show the banner again */
				screen_clear();
				screen_done();
				showstrlist(banner);
	} else {
		int pid, fd;
	       
		if((fd = running()) == -1) {
			fprintf(stderr,"%s: batch process already running\n",pname);
			fprintf(stderr,"try `%s --help' for more information\n",pname);

			return FALSE;
		}

		if((pid = fork()) == -1)
			die("fork()");

		if(pid == 0) {
			printpid(fd);

			/* Repeat to infinite if repeat flag is set, one else */
			do {
				int	x;

				for(x = 0;x < tracks && !stopped;x++) {
					int	trk;

					trk = sequence[x];

					/* Play the track */
					if(cdrom_playtrack(cd,trk) == -1)
						die("cdrom_playtrack()");

					do {
						sleep(5);
						if(cdrom_status(cd,&status) == -1)
							die("cdrom_status()");
					} while(status.status != CDS_STOP);
				}
			} while(isset(OPT_REPEAT) && !stopped);

			finished();
		} else {
			for(idx = 0; idx < n; idx++)
				if(song_lyrics[idx] != NULL)
					lyrics_unload(song_lyrics[idx]);
			free(song_lyrics);

			free(sequence);

			if(!freecddb()) return FALSE;
			if(!freetoc()) return FALSE;
			if(!closecd()) return FALSE;

			return TRUE;
		}
	}

	for(idx = 0; idx < n; idx++)
		if(song_lyrics[idx] != NULL)
			lyrics_unload(song_lyrics[idx]);
	free(song_lyrics);

	free(sequence);

	if(isset(OPT_EJECT))
		if(cdrom_eject(cd) == -1)
			die("cdrom_eject()");

	if(!freecddb()) return FALSE;
	if(!freetoc()) return FALSE;
	if(!closecd()) return FALSE;

	return TRUE;
}

static int	play_cd(void) {
	if(!opencd(TRUE)) return FALSE;
	if(!readtoc(TRUE)) return FALSE;
	if(!readcddb()) return FALSE;
	
	printf("Playing CD...\n");
	if(!isset(OPT_QUIT)) {
		screen_init();
		screen_colors(attrs);
		screen_attr(DEFAULT);
		
		/* Clear the screen and write the title and the artist */
		screen_clear();
		print_titles();

		do {
			if(cdrom_play(cd,toc->lba_start,toc->lba_end) == -1)
				die("cdrom_play()");
			do {
				int	c = showstatus(KB_QUIT|(status.status != CDS_PAUSE ? KB_PAUSE : KB_RESUME));

				switch(c) {
					case KB_QUIT:
						/* stop the CD */
						cdrom_stop(cd);

						/* we lay about it, but who cares ? */
						status.status = CDS_STOP;
						options &= ~OPT_REPEAT;
						break;
						
					case KB_PAUSE:
						/* pause the CD */
						cdrom_pause(cd);
						break;
						
					case KB_RESUME:
						/* resume playing */
						cdrom_resume(cd);
						break;
				}
			} while(status.status != CDS_STOP);
		} while(isset(OPT_REPEAT));
		screen_clear();
		screen_done();
		showstrlist(banner);
	} else {
		int pid, fd;
	       
		if((fd = running()) == -1) {
			fprintf(stderr,"%s: batch process already running\n",pname);
			fprintf(stderr,"try `%s --help' for more information\n",pname);

			return FALSE;
		}

		if((pid = fork()) == -1)
			die("fork()");

		if(pid == 0) {
			printpid(fd);

			if(isset(OPT_REPEAT)) {
				do {
					if(cdrom_play(cd,toc->lba_start,toc->lba_end) == -1)
						die("cdrom_play()");
					do {
						sleep(5);
						if(cdrom_status(cd,&status) == -1)
							die("cdrom_status()");
					} while(status.status != CDS_STOP);
				} while(isset(OPT_REPEAT) && !stopped);
			} else
				if(cdrom_play(cd,toc->lba_start,toc->lba_end) == -1)
					die("cdrom_play()");
			finished();
		} else {
			if(!freecddb()) return FALSE;
			if(!freetoc()) return FALSE;
			if(!closecd()) return FALSE;

			return TRUE;
		}
	}

	if(isset(OPT_EJECT))
		if(cdrom_eject(cd) == -1)
			die("cdrom_eject()");

	if(!freecddb()) return FALSE;
	if(!freetoc()) return FALSE;
	if(!closecd()) return FALSE;

	return TRUE;
}

static int	play_track(void) {
	char	*name;
	int	trk = -1;

	if(!opencd(TRUE)) return FALSE;
	if(!readtoc(TRUE)) return FALSE;
	if(!readcddb()) return FALSE;
	
	/* check if it's the number of a track */
	if(!strtoi(track,&trk)) {
		regex_t	regexp;
		
		if(regcomp(&regexp,track,REG_NOSUB) != 0) {
			printerror("error in track regular expressionv");
			return FALSE;
		}
		
		if(isset(OPT_DONTSKIP)) {
			int	i;
			
			for(i = 0; i < cddb_ntracks(cddb); i++)
				if(cddb->tracks[i].name)
					if(regexec(&regexp,cddb->tracks[i].name,0,NULL,0) == 0) {
						trk = cddb->tracks[i].track;
						break;
					}
		} else {
			int	i;

			for(i = 0; i < cddb_ntracks(cddb); i++)
				if(cddb->tracks[i].name)
					if((cddb->tracks[i].flags & CDDB_TF_SKIP) != CDDB_TF_SKIP)
						if(regexec(&regexp,cddb->tracks[i].name,0,NULL,0) == 0) {
							trk = cddb->tracks[i].track;
							break;
						}
		}

		regfree(&regexp);
	}

	if(trk == -1) {
		printerror("track not found.");
		return FALSE;
	} else {
		if(trk < toc->first_track || trk > toc->last_track) {
			printerror("track out of range");
			return FALSE;
		}

		if(!isset(OPT_DONTSKIP))
			if((cddb_trackflags(cddb,trk) & CDDB_TF_SKIP) == CDDB_TF_SKIP) {
				printerror("track skipped");
				return FALSE;
			}
	}

	printf("Playing ");

	/* Get the name of the track */
	name = cddb_trackname(cddb,trk);

	if(name)
		printf("\"%s\"",name);
	else
		printf("#%02d",trk);

	printf("...\n");

	if(!isset(OPT_QUIT)) {
		screen_init();
		screen_colors(attrs);
		screen_attr(DEFAULT);

		/* Clear the screen and write the title and the artist */
		screen_clear();
		print_titles();

		do {
			if(cdrom_playtrack(cd,trk) == -1)
				die("cdrom_playtrack()");
			do {
				int	c = showstatus(KB_QUIT|(status.status != CDS_PAUSE ? KB_PAUSE : KB_RESUME));

				switch(c) {
					case KB_QUIT:
						/* stop the CD */
						cdrom_stop(cd);

						/* we lay about it, but who cares ? */
						status.status = CDS_STOP;
						options &= ~OPT_REPEAT;
						break;

					case KB_PAUSE:
						/* pause the CD */
						cdrom_pause(cd);
						break;
						
					case KB_RESUME:
						/* resume playing */
						cdrom_resume(cd);
						break;
				}
			} while(status.status != CDS_STOP);
		} while(isset(OPT_REPEAT));
		screen_clear();
		screen_done();
		showstrlist(banner);
	} else {
		int pid, fd;
	       
		if((fd = running()) == -1) {
			fprintf(stderr,"%s: batch process already running\n",pname);
			fprintf(stderr,"try `%s --help' for more information\n",pname);

			return FALSE;
		}

		if((pid = fork()) == -1)
			die("fork()");

		if(pid == 0) {
			printpid(fd);

			if(isset(OPT_REPEAT)) {
				do {
					do {
						sleep(5);
						if(cdrom_status(cd,&status) == -1)
							die("cdrom_status()");
					} while(status.status != CDS_STOP);
				} while(isset(OPT_REPEAT) && !stopped);
			} else
				if(cdrom_playtrack(cd,trk) == -1)
					die("cdrom_playtrack()");
			finished();
		} else {
			if(!freecddb()) return FALSE;
			if(!freetoc()) return FALSE;
			if(!closecd()) return FALSE;

			return TRUE;
		}
	}

	if(isset(OPT_EJECT))
		if(cdrom_eject(cd) == -1)
			die("cdrom_eject()");

	if(!freecddb()) return FALSE;
	if(!freetoc()) return FALSE;
	if(!closecd()) return FALSE;

	return TRUE;
}

static int	play_program(void) {
	int	nargs,ntracks = 0,*sequence = NULL,n = 0,i;
	int	*present;
		char	**args;

		if(!opencd(TRUE)) return FALSE;
		if(!readtoc(TRUE)) return FALSE;
		if(!readcddb()) return FALSE;
	
	if((present = malloc(sizeof(int) * toc->ntracks)) == NULL) {
		printerror("can't allocate enough memory");
		return FALSE;
	}
	
	for(i = 0; i < toc->ntracks; i++)
		present[i] = FALSE;

	nargs = splitargs(program,&args,',');
	
	for(i = 0; i < nargs; i++)
		if(!addtracks(args[i],present,sequence,&ntracks,FALSE))
			return FALSE;
	
	if(ntracks == 0) {
		printerror("no tracks in program");
		return FALSE;
	}
	
	printf("%d tracks in program.\n",ntracks);
	
	if((sequence = malloc(sizeof(int) * ntracks)) == NULL) {
		printerror("can't allocate enough memory");
		return FALSE;
	}
	
	for(i = 0; i < toc->ntracks; i++)
		present[i] = FALSE;

	for(i = 0; i < nargs; i++) {
		if(!addtracks(args[i],present,sequence,&n,TRUE))
			return FALSE;
	}

	if(isset(OPT_SHUFFLE))
		if(!shuffle(sequence,ntracks))
			return FALSE;

	free(present);

	if(!isset(OPT_QUIT)) {
		/* Initialize ncurses */
		screen_init();
		screen_colors(attrs);
		screen_attr(DEFAULT);

		/* Clear the screen and write the title and the artist */
		screen_clear();
		print_titles();

		for(i = 0; i < ntracks; i++) {
			char	*name;

			screen_printf("Adding ");
			name = cddb_trackname(cddb,sequence[i]);
			if(name)
				screen_printf("\"%s\"",name);
			else
				screen_printf("#%02d",sequence[i]);
			screen_printf(".\n");
		}
		screen_printf("\n");

		/* Repeat to infinite if repeat flag is set, one else */
		do {
			int	x;

			for(x = 0;x < ntracks;x++) {
				char	*name;
				int	trk;

				trk = sequence[x];

				/* Get the name of the track */
				name = cddb_trackname(cddb,trk);

				/* Play the track */
				if(cdrom_playtrack(cd,trk) == -1)
					die("cdrom_playtrack()");

				do {
					int	c = showstatus(KB_QUIT|KB_NEXT|(status.status != CDS_PAUSE ? KB_PAUSE : KB_RESUME));

					switch(c) {
						case KB_QUIT:
							/* stop the CD */
							cdrom_stop(cd);

							/* we lay about it, but who cares ? */
							status.status = CDS_STOP;
							x = ntracks;
							options &= ~OPT_REPEAT;
							break;

						case KB_NEXT:
							/* stop the CD, so we can go on with the next track */
							cdrom_stop(cd);
							status.status = CDS_STOP;

						case KB_PAUSE:
							/* pause the CD */
							cdrom_pause(cd);
							break;

						case KB_RESUME:
							/* resume playing */
							cdrom_resume(cd);
							break;
					}
				} while(status.status != CDS_STOP);

				screen_printf("Played ");
				if(name)
					screen_printf("\"%s\"",name);
				else
					screen_printf("#%02d",trk);
				screen_clreol();
				screen_printf("\n");
			}
		} while(isset(OPT_REPEAT));

		/* clear the screen and show the banner again */
		screen_clear();
		screen_done();
		showstrlist(banner);
	} else {
		int pid, fd;

		if((fd = running()) == -1) {
			fprintf(stderr,"%s: batch process already running\n",pname);
			fprintf(stderr,"try `%s --help' for more information\n",pname);

			return FALSE;
		}

		if((pid = fork()) == -1)
			die("fork()");

		if(pid == 0) {
			printpid(fd);

			/* Repeat to infinite if repeat flag is set, one else */
			do {
				int	x;

				for(x = 0;x < ntracks && !stopped;x++) {
					int	trk;

					trk = sequence[x];

					/* Play the track */
					if(cdrom_playtrack(cd,trk) == -1)
						die("cdrom_playtrack()");

					do {
						sleep(5);
						if(cdrom_status(cd,&status) == -1)
							die("cdrom_status()");
					} while(status.status != CDS_STOP);
				}
			} while(isset(OPT_REPEAT) && !stopped);
			finished();
		} else {
			free(sequence);

			if(!freecddb()) return FALSE;
			if(!freetoc()) return FALSE;
			if(!closecd()) return FALSE;

			return TRUE;
		}
	}

	free(sequence);

	if(isset(OPT_EJECT))
		if(cdrom_eject(cd) == -1)
			die("cdrom_eject()");

	if(!freecddb()) return FALSE;
	if(!freetoc()) return FALSE;
	if(!closecd()) return FALSE;
	
	return TRUE;
}

static int	stop(void) {
	if(!opencd(TRUE)) return FALSE;

	printf("Stopping CD...\n");
	if(cdrom_stop(cd) == -1)
		die("cdrom_stop()");
	
	if(isset(OPT_EJECT))
		if(cdrom_eject(cd) == -1)
			die("cdrom_eject()");
	
	if(!closecd()) return FALSE;
	
	return TRUE;
}

static int	nothing(void) {
	int	check;
	
	check = isset(OPT_ID) || isset(OPT_LIST) || isset(OPT_TRACE);
	
	if(!opencd(check)) return FALSE;
	if(check) {
		if(!readtoc(TRUE)) return FALSE;
		if(!readcddb()) return FALSE;
	}
	
	if(isset(OPT_ID)) {
		int	i;
		
		printf("ID: ");
		for(i = 0; i < 16; i++)
			putchar(cddb->id.id[i]);
		printf("\n\n");
	}
	
	if(isset(OPT_LIST)) {
		int	i;
		
		printf("Title : ");
		if(cddb_title(cddb))
			printf("%s",cddb_title(cddb));
		else
			printf("-- unnamed --");
		printf("\nArtist: ");
		if(cddb_artist(cddb)) 
			printf("%s",cddb_artist(cddb));
		else
			printf("-- unnamed --");
		printf("\n\n");
		
		for(i = 0; i < cddb_ntracks(cddb); i++) {
			char	*name;
			int	length = -1,j,trk;
		
			name = cddb->tracks[i].name;
			trk = cddb->tracks[i].track;
			printf("%d: ",trk);
			if(name)
				printf("\"%s\"",name);
			else
				printf("-- unnamed --");
			for(j = 0; j < toc->ntracks; j++)
				if(toc->entries[j].track == trk) {
					length = toc->entries[j].lba_length;
					break;
				}
			if(length != -1) {
				printf(" ");
				print_lba(length);
			}
			if((cddb->tracks[i].flags & CDDB_TF_SKIP) == CDDB_TF_SKIP)
				printf(" (SKIP)");
			printf("\n");
		}
		printf("\n");
	}

	if(isset(OPT_TRACE)) {
		screen_init();
		screen_colors(attrs);
		screen_attr(DEFAULT);

		/* Clear the screen and write the title and the artist */
		screen_clear();
		print_titles();

		do {
			int	c = showstatus(KB_QUIT|(status.status != CDS_PAUSE ? KB_PAUSE : KB_RESUME));

			switch(c) {
				case KB_QUIT:
					/* we lay about it, but who cares ? */
					status.status = CDS_STOP;
					options &= ~OPT_REPEAT;
					break;

				case KB_PAUSE:
					/* pause the CD */
					cdrom_pause(cd);
					break;

				case KB_RESUME:
					/* resume playing */
					cdrom_resume(cd);
					break;
			}
		} while(status.status != CDS_STOP);

		screen_clear();
		screen_done();
		showstrlist(banner);
	}

	if(isset(OPT_EJECT))
		if(cdrom_eject(cd) == -1)
			die("cdrom_eject()");
	
	if(check) {
		if(!freecddb()) return FALSE;
		if(!freetoc()) return FALSE;
	}
	if(!closecd()) return FALSE;
	
	return TRUE;
}

static int	pause_cd(void) {
	if(!opencd(TRUE)) return FALSE;
	
	if(cdrom_pause(cd) == -1)
		die("cdrom_pause()");
	
	if(!closecd()) return FALSE;
	
	return TRUE;
}

static int	resume_cd(void) {
	if(!opencd(TRUE)) return FALSE;
	
	if(cdrom_resume(cd) == -1)
		die("cdrom_resume()");
	
	if(!closecd()) return FALSE;
	
	return TRUE;
}

static int	interactive(void)
{
	int finished = FALSE;
	int n, i;

	if(!opencd(TRUE)) return FALSE;
	if(!readtoc(TRUE)) return FALSE;
	if(!readcddb()) return FALSE;
	
	n = cddb_ntracks(cddb);
	song_lyrics = (lyrics_line **)malloc(sizeof(lyrics_line) * n);
	if(song_lyrics == NULL) return FALSE;
	for(i = 0; i < n; i++) {
		char *fn = cddb_lyrics(cddb, i);
		
		if(fn) {
			song_lyrics[i] = lyrics_load(fn);
			free(fn);
		} else
			song_lyrics[i] = NULL;
	}

	screen_init();
	screen_colors(interactives);

	titlebar = screen_newwin(0, 0, COLS, 1);
	infobar = screen_newwin(0, 1, COLS, 2);
	statusbar = screen_newwin(0, LINES-1, COLS, 1);
	/* 1 */
	songs = screen_newwin(0, 3, COLS, LINES-4);
	lyrics = screen_newwin(0, 3, COLS, LINES-4);
	/* 1 */
	/* 1
	songs = screen_newwin(1, 3, 29, LINES-4);
	lyrics = screen_newwin(30, 3, COLS-30, LINES-4);
	*/
	
	show_titlebar();
	show_infobar();

	while(!finished) {
		if(cdrom_status(cd,&status) == -1)
			die("cdrom_status()");

		if(status.status == CDS_STOP)
			if(playing) {
				int *prg, ntrk, i, trk;
				
				prg = (int *)malloc(cddb_ntracks(cddb));
				ntrk = 0;
				
				for(i = 0; i < cddb_ntracks(cddb); i++)
					if((cddb->tracks[i].flags & CDDB_TF_SKIP) != CDDB_TF_SKIP)
						prg[ntrk++] = cddb->tracks[i].track;
				
				if(shuffling)
					trk = prg[uniform(0, ntrk-1)];
				else {
					if(last == -1)
						i = 0;
					else {
						for(i = 0; i < ntrk; i++)
							if(prg[i] == last) {
								i++;
								break;
							}
						i %= ntrk;
					}

					trk = prg[i];
				}
				free(prg);

				last = trk;

				if(cdrom_playtrack(cd,trk) == -1)
					die("cdrom_playtrack()");
			}
		
		show_songs();
		show_lyrics();
		show_statusbar();
		finished = !handle_keys();
	}
	
	screen_done();
	
	if(playing)
		if(cdrom_stop(cd) == -1)
			die("cdrom_stop()");

	for(i = 0; i < n; i++)
		if(song_lyrics[i] != NULL)
			lyrics_unload(song_lyrics[i]);
	free(song_lyrics);

	cddb_save(cddb);

	if(!freecddb()) return FALSE;
	if(!freetoc()) return FALSE;
	if(!closecd()) return FALSE;
	
	return TRUE;
}

static int	kill_running(void)
{
	if(!running()) {
		fprintf(stderr,"%s: batch process not running\n",pname);
		fprintf(stderr,"try `%s --help' for more information\n",pname);
		finished();

		return FALSE;
	}
	
	kill(get_running_pid(), SIGTERM);
	
	return TRUE;
}

/* cdplayer.c		Flavio Lerda		97-07-29 */
