/* This file is part of TCD 2.0.
   cdrom.c - CD DEVICE INTERFACE MODULE

   All the functions that start with tcd_ are here, and aren't
   (shouldn't be...) dependent on any user interface.

   Copyright (C) 1997-98 Tim P. Gerla <timg@means.net>
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
                   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
                                   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
                                               
   Tim P. Gerla
   RR 1, Box 40
   Climax, MN  56523
   timg@means.net
*/

#define DATABASE_SUPPORT

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>
//#include <linux/ucdrom.h> // WILL BE OBSOLETE SOON

#include "../config.h"
#include "cdrom.h"
#include "cddb.h"

#define FALSE (0)
#define TRUE (!FALSE)

int tcd_init_disc( cd_struct *cd )
{
	char tcd_dir[60];
	struct stat fs;
	struct cdrom_volctrl vol;

	tcd_opencddev( cd );

#ifdef DATABASE_SUPPORT
	strcpy( tcd_dir, getenv("HOME"));
	strcat( tcd_dir, "/.tcd/" );
	if( stat( tcd_dir, &fs ) < 0 )
	{
		if( errno == ENOENT )
		{
			fprintf( stderr, "Can't open \'%s\'\n", tcd_dir );
			fprintf( stderr, "TCD will now attempt to create %s to store it's files.\n", tcd_dir);
			if( mkdir(tcd_dir,S_IRWXU) )
			{
				/* FIXME, let it continue, but without database support. */
				fprintf( stderr,"Error creating TCD directory!\n" );
				perror( "mkdir" );	
				return -1;
			}
			fprintf( stderr,"Succesful.\n" );
		}
		else {
			fprintf( stderr,"Error opening \'~/.tcd/\'. Please correct permissions.\n" );
			perror( "stat" );
		}
	}
#endif
	
	if( !cd->err ) tcd_readtoc(cd);
#ifdef DATABASE_SUPPORT
	if( !cd->err ) tcd_readdiskinfo(cd);
#endif
	if( cd->err )
		return(-1);
	cd->ejected=FALSE;

#ifdef TCD_CHANGER_ENABLED
	/* FIXME set capability flag here for CHANGER */
	cd->nslots = ioctl( cd->cd_dev, CDROM_CHANGER_NSLOTS );
#else
	cd->nslots = 0;
#endif
	ioctl( cd->cd_dev, CDROMVOLREAD, &vol );
	cd->volume = vol.channel0;
	
	return(0);
}

int tcd_close_disc( cd_struct *cd )
{
	close( cd->cd_dev );
	return 0;
}

int tcd_readtoc( cd_struct *cd )
{
	int tmp,i;
	int delsecs;

	cd->err = FALSE;
	cd->isplayable=FALSE;
	
	tmp = ioctl( cd->cd_dev, CDROMREADTOCHDR, &cd->tochdr );
	if( tmp < 0 )
	{
		strcpy( cd->errmsg, "Can't read disc." );
		cd->err = TRUE;
		return(-1);		
	}
	
	cd->first_t = cd->tochdr.cdth_trk0;
	cd->last_t = cd->tochdr.cdth_trk1;


	/* Ah ha! I was forgetting to read the leadout track! 
	   My timings were all screwed up. Thanks to the cdtool
	   source, I figured it out.
	*/
	cd->old_id = 0; /* Prevent errors :P */

    	cd->trk[cd->last_t+1].toc.cdte_track = CDROM_LEADOUT;
        cd->trk[cd->last_t+1].toc.cdte_format = CDROM_MSF;
	tmp = ioctl(cd->cd_dev, CDROMREADTOCENTRY, &cd->trk[cd->last_t+1].toc );
	if( tmp < 0 )
	{
		strcpy( cd->errmsg, "Can't read disc." );
                cd->err = TRUE;
         	return(-1);
        }                                         

	for( i = cd->first_t; i <= cd->last_t; i++ )
	{
		cd->trk[i].toc.cdte_track = i;
                cd->trk[i].toc.cdte_format = CDROM_MSF;

                tmp = ioctl( cd->cd_dev, CDROMREADTOCENTRY, &cd->trk[i].toc );		
                if( tmp < 0 )
		{
			strcpy( cd->errmsg, "Can't read disc." );
                  	cd->err = TRUE;
			return(-1);
                }

		cd->trk[i].length = cd->trk[i].toc.cdte_addr.msf.minute * 60 +
                        	    cd->trk[i].toc.cdte_addr.msf.second;
                cd->trk[i].start = cd->trk[i].length * 75 + 
                                   cd->trk[i].toc.cdte_addr.msf.frame;
		
		/* Must not change */
		cd->old_id += cd->trk[i].length * 
			       (cd->trk[cd->last_t+1].toc.cdte_addr.msf.minute*60+
		                cd->trk[cd->last_t+1].toc.cdte_addr.msf.second);

		/* Set up the default playlist */
		cd->playlist[i] = i;
	}
	cd->playlist[i] = -1;
	
	for( i = cd->first_t; i <= cd->last_t; i ++ )
	{
		/* Taken from cdtool...Thanks Thomas I.! */
		delsecs = cd->trk[i+1].toc.cdte_addr.msf.minute * 60
		+ cd->trk[i+1].toc.cdte_addr.msf.second
		- cd->trk[i].toc.cdte_addr.msf.minute * 60
		- cd->trk[i].toc.cdte_addr.msf.second;

		cd->trk[i].tot_min = delsecs / 60;
		cd->trk[i].tot_sec = delsecs - (delsecs/60)*60;

		strcpy( cd->trk[tmp].name, "(unknown)" );
		cd->trk[tmp].titled = FALSE;
	}	
	cd->trk[cd->last_t+1].titled=TRUE;

#ifdef DATABASE_SUPPORT
	cd->cddb_id = cddb_discid(cd);
#endif

	cd->isplayable=TRUE;
	return tmp;
}

void tcd_gettime( cd_struct *cd )
{
	int result;
	struct cdrom_volctrl vol;
	cd->err = FALSE;
        
        cd->sc.cdsc_format = CDROM_MSF;
        	
	result = ioctl( cd->cd_dev, CDROMSUBCHNL, &cd->sc );
	if( result < 0 )
	{
		strcpy( cd->errmsg, "Can't read disc." );
               	cd->err = TRUE;
		return;
        }

	vol.channel0 = cd->volume;
	vol.channel1 = vol.channel2 = vol.channel3 = vol.channel0; 
	ioctl( cd->cd_dev, CDROMVOLCTRL, &vol );
	ioctl( cd->cd_dev, CDROMVOLREAD, &vol );
	cd->volume = vol.channel0;

	cd->cur_t = cd->sc.cdsc_trk;

	cd->cur_pos_abs = cd->sc.cdsc_absaddr.msf.minute * 60 +
         cd->sc.cdsc_absaddr.msf.second;
        cd->cur_frame = cd->cur_pos_abs * 75 + cd->sc.cdsc_absaddr.msf.frame;
        
	cd->cur_pos_rel = (cd->cur_frame - cd->trk[cd->cur_t].start) / 75;
	
	if (cd->cur_pos_rel < 0)
        	cd->cur_pos_rel = -cd->cur_pos_rel;
        
  	if (cd->cur_pos_rel > 0 && (result = cd->cur_pos_rel % 60) == cd->t_sec)
      		return;

        cd->t_sec = result;
        cd->t_min = cd->cur_pos_rel / 60;
        
        cd->cd_sec = cd->cur_pos_abs % 60;
	cd->cd_min = cd->cur_pos_abs / 60;

	/* Update cd->trk.status */
	for( result = cd->first_t; result <= cd->last_t; result++ )
	{
		cd->trk[result].status = 0;
		if( (result == cd->cur_t) && cd->sc.cdsc_audiostatus == CDROM_AUDIO_PLAY )
			cd->trk[result].status = TRK_PLAYING;
		if( result == cd->repeat_track )
			cd->trk[result].status = TRK_REPEAT;
		if( cd->trk[result].toc.cdte_ctrl == CDROM_DATA_TRACK )
			cd->trk[result].status = TRK_DATA;
	}			
#ifdef TCD_CHANGER_ENABLED
	cd->cur_disc = ioctl( cd->cd_dev, CDROM_SELECT_DISC, CDSL_CURRENT );
#endif
}
	                                   
int tcd_playtracks( cd_struct *cd, int start_t, int end_t )
{
	int tmp;
	cd->err = FALSE;
	cd->isplayable=FALSE;
	cd->isdisk=FALSE;
	
	if( cd->trk[start_t].status == TRK_DATA )
		return -1;
	
        cd->ti.cdti_trk0 = start_t;   /* start track */
   	cd->ti.cdti_ind0 = 0;   /* start index */
        cd->ti.cdti_trk1 = end_t;  /* end track */
        cd->ti.cdti_ind1 = 0;   /* end index */
                                
        tmp = ioctl( cd->cd_dev, CDROMPLAYTRKIND, &cd->ti );
   	if( tmp < 0 )
	{
		strcpy( cd->errmsg, "Error playing disc" );
               	cd->err = TRUE;
		return(-1);
        }
   	cd->isplayable=TRUE;                                                 
	cd->isdisk=TRUE;
   	return tmp;
}       

int tcd_play_seconds( cd_struct *cd, long int offset )
{
	struct cdrom_msf msf;
	int tmp;
	cd->err = FALSE;
	cd->isplayable=FALSE;
	cd->isdisk=FALSE;

	/* got subchannel? */
	msf.cdmsf_sec0 = cd->sc.cdsc_absaddr.msf.second+offset;
	msf.cdmsf_min0 = cd->sc.cdsc_absaddr.msf.minute;
	msf.cdmsf_frame0 = cd->sc.cdsc_absaddr.msf.frame;
	msf.cdmsf_min1 = cd->trk[cd->last_t+1].toc.cdte_addr.msf.minute;
	msf.cdmsf_sec1 = cd->trk[cd->last_t+1].toc.cdte_addr.msf.second;
	msf.cdmsf_frame1 = cd->trk[cd->last_t+1].toc.cdte_addr.msf.frame;
	if( msf.cdmsf_sec0 > 60 && (offset<0) )
	{
		msf.cdmsf_sec0 = 60-abs(offset);
		msf.cdmsf_min0--;
	}
	
        tmp = ioctl( cd->cd_dev, CDROMPLAYMSF, &msf );
   	if( tmp < 0 )
	{
		strcpy( cd->errmsg, "Error playing disc." );
               	cd->err = TRUE;
		return(-1);
        }
   	cd->isplayable=TRUE;                                                 
	cd->isdisk=TRUE;
   	return tmp;
}       

int tcd_ejectcd( cd_struct *cd )
{
	int tmp;
	cd->err = FALSE;
	if( cd->ejected == FALSE )
	        tmp = ioctl( cd->cd_dev, CDROMEJECT );
	else
		tmp = ioctl( cd->cd_dev, CDROMCLOSETRAY );
   	if( tmp < 0 )
	{
		strcpy( cd->errmsg, "Can't eject disc." );
               	cd->err = TRUE;
		return(-1);
        }
	cd->isplayable=FALSE;
	cd->isdisk=FALSE;

	cd->ejected = ~cd->ejected;

   	return 0;
}       

int tcd_stopcd( cd_struct *cd )
{
	int tmp;
	cd->err = FALSE;

        tmp = ioctl( cd->cd_dev, CDROMSTOP );
   	if( tmp < 0 )
	{
		strcpy( cd->errmsg, "Can't stop disc." );
               	cd->err = TRUE;
		return(-1);
        }
   	return tmp;
}       

int tcd_pausecd( cd_struct *cd )
{
	int tmp;
	cd->err = FALSE;
	
	if( cd->sc.cdsc_audiostatus==CDROM_AUDIO_PAUSED )
	{       
		tmp = ioctl( cd->cd_dev, CDROMRESUME );
		if( tmp < 0 )
		{
			strcpy( cd->errmsg, strerror( errno ) );
                        cd->err = TRUE;
                        return(-1);
		}
	        return tmp;
	}	        
	else
	{
		tmp = ioctl( cd->cd_dev, CDROMPAUSE );
		if( tmp < 0 )
		{
			strcpy( cd->errmsg, strerror( errno ) );
                        cd->err = TRUE;
                        return(-1);
		}
	        return tmp;
	}
}

int tcd_change_disc( cd_struct *cd, int disc )
{
#ifdef TCD_CHANGER_ENABLED
	int tmp, fd;
	cd->err = FALSE;
	fd = open( cd->cdpath, O_RDONLY | O_NONBLOCK );

        tmp = ioctl( fd, CDROM_SELECT_DISC, disc );
	if( tmp < 0 )
		fprintf( stdout, "ioctl: %s\n", strerror(errno) );	
	/* Don't be noisy if there's an error */

	close(fd);
   	return tmp;
#else
	fprintf( stderr, "tcd_change_disc called, but changer support isn't compiled in. Ickyblah.\n" );
	return 0;
#endif
}
	                   
void tcd_opencddev( cd_struct *cd )
{
	cd->err = FALSE;
	cd->isdisk=FALSE;

	if( cd->cd_dev < 0 )
		close( cd->cd_dev );
                                                
	cd->cd_dev = open( cd->cdpath, O_RDONLY | O_NONBLOCK );
 
        if( cd->cd_dev < 0 )
        {
		strcpy( cd->errmsg, "Can't open drive." );
		cd->err = TRUE;
                return;
        }
	cd->isdisk=TRUE;
}

int tcd_readdiskinfo( cd_struct *cd )
{
	struct stat fs;
	int i, res;
	FILE *fp;
	char fn[60];
	char artist[DISC_INFO_LEN], album[DISC_INFO_LEN];
	char tcd_dir[128];
	
	strcpy( tcd_dir, getenv("HOME"));
        strcat( tcd_dir, "/.tcd/" );
                 
	sprintf( fn, "%s%08lx", tcd_dir, cd->cddb_id );
	fp = fopen(fn, "r");	
	if( fp != NULL ) // Does the new format exist?
	{
		fclose(fp);
		if( (res = tcd_readcddb( cd, fn ))<0 )
		{
			fprintf( stderr, "tcd_readcddb returned an error, %d\n", res );
			sleep(2);
			return -1;
		}
		return 0;
	}
	else
	{
		fprintf( stderr,"Warning, can't open new format \'%s\', trying old.\n", fn );
		sprintf( fn, "%s%ld.tcd", tcd_dir, cd->old_id );
		fprintf( stderr, "opening...%s\n", fn );
		if( !stat(fn, &fs) )
		{
			fp = fopen( fn, "r" );
			fgets( artist, 40, fp );
			fgets( album, 40, fp );
			artist[strlen(artist)-1] = 0;
			album[strlen(album)-1] = 0;
			
			strcpy( cd->dtitle, "" );
			strcat( cd->dtitle, artist );
			strcat( cd->dtitle, " / " );
			strcat( cd->dtitle, album );
				
			for( i = cd->first_t; i <= cd->last_t; i++ )
			{
				fgets( cd->trk[i].name, 40, fp );
			        cd->trk[i].name[strlen(cd->trk[i].name)-1] = 0;
				cd->trk[i].titled = TRUE;
			}
			sprintf( fn, "%s%08lx", tcd_dir, cd->cddb_id );
			tcd_writecddb( cd, fn );
			fclose(fp);
			return 0;
		}	                                                                                       			
		else 
		{		
			fprintf( stderr,"Warning, can't open \'%s\' \n", fn );
			strcpy( cd->dtitle, "Unknown / Unknown" );
		
			for( i = cd->first_t; i <= cd->last_t; i++ )
			{
				sprintf( cd->trk[i].name, "Track %d", i );
				cd->trk[i].titled = FALSE;
			}
                        sprintf( fn, "%s%08lx", tcd_dir, cd->cddb_id );
                        tcd_writecddb( cd, fn );
			return 0;
        	}
	}
	fclose(fp);
	return -1;
}

void tcd_writediskinfo( cd_struct *cd )
{
	char fn[60];
	char home[60];

	strcpy( home, getenv("HOME"));
	sprintf( fn, "%s/.tcd/%08lx", home, cd->cddb_id );
	
        if( tcd_writecddb(cd, fn) < 0 )
        {
        	perror( "tcd_writecddb" );
        	exit(-1);
        }

	cd->needs_dbwrite=FALSE;        
	return;
}					

