/*
   DAGRAB - dumps digital audio from cdrom to riff wave files

   (C) 1998,2000 Marcello Urbani <murbani@libero.it>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   */
/*
 *
 * Changes:
 *
 * - Initial release 0.1
 *   02/10/98
 *
 * - Added some options and fixed a few bugs
 *   02/18/98
 *
 * - Added speed and rest time diagnostics for single track
 *   04/02/98 Sascha Schumann <sas@schell.de>
 *
 * - Added -m -e -s -p functions, extended track info
 *   04/05/98 Sascha Schumann <sas@schell.de>
 *
 * - release 0.2
 *   04/23/98
 *
 * - changed my e-mail address - the old one expired :(
 *   07/??/98
 *
 * - Added cddb support, switches -S and -N, plus -D -P and -H
 *   07/30/98
 * 
 * - release 0.3
 *   08/25/98
 * 
 * - fixed a few bugs, mostly fixed by various people that mailed me
 *   I apologize for the lateness of this release.
 * 
 * - release 0.3.1
 *   06/24/99
 * 
 * - changed my e-mail address - the old one is expired (I forgot 
 *   to do this in the previous release)
 * 
 * - changed the default CDDB address to a working one (www.cddb.com)
 * 
 * - release 0.3.2
 *   10/10/99
 *
 * - modern kernels need O_NONBLOCK set when opening the cdrom device.
 *   (1999-10-19 dsembr01@slug.louisville.edu)
 *
 * - better error reporting when some of the system calls, or other
 *   functions that set errno, fail.
 *   (1999-10-19 dsembr01@slug.louisville.edu)
 *
 * - changed the default CDDB address to a free one (freedb.freedb.org)
 *   as suggested by Darren Stuart Embry (dsembr01@slug.louisville.edu)
 *
 * - release 0.3.3
 *   22/10/99
 *
 * - 16/02/00 fixed a few CDDB bugs 
 * 
 * - changed opt_overlap default to a more reasonable value (at least i hope so)
 *
 * - allowed disabling of jitter correction thru -o 0
 *
 * - added CDDB data based variables in filenames and commands
 * 
 * - added two scripts to automate mp3 encoding
 *
 * -added a spec file
 *
 * -release 0.3.5 19/02/00
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <dirent.h>
#include <netdb.h>
#include <unistd.h>
#include <pwd.h>
#include <ctype.h>
#define __need_timeval   /* needed by glibc */
#include <time.h>
#include <linux/cdrom.h>
#ifdef USE_UCDROM
#include <linux/ucdrom.h>
#endif
#include <sys/vfs.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#define CDDEVICE "/dev/cdrom"
#define N_BUF 8
#define OVERLAP 2
#define KEYLEN 12
#define OFS 12
#define RETRYS 40
#define IFRAMESIZE (CD_FRAMESIZE_RAW/sizeof(int))
#define BLEN 255
#define D_MODE 0660
#define CDDB_MAX  65535
#define CDDB_PORT 888
#define CDDB_PATH "/usr/lib/X11/xmcd/cddb"
#define CDDB_HOST "freedb.freedb.org"
//#define USE_CDDB (opt_save||opt_name)
#define USE_CDDB opt_cddb
#define PROGNAME "dagrab"
#define VERSION "0.3.5"
#define KW_TRACK 0
#define KW_FULLD 1
#define KW_AUTHOR 2
#define KW_NUM 3
#define KW_DISK 4
#define KW_MAX 4
#define TRUE 1
#define FALSE 0

struct cd_trk_list{
  int min;
  int max;
  int *starts;
  char *types;
  char *cddb;		// complete cddb entry
  int cddb_size;
  char *gnr;		// category; NULL if not obtained via cddbp
};
struct kword{
  char *kw;
  int idx;
  char *desc;
};

typedef unsigned short Word;

struct Wavefile{
  char Rid[4];
  unsigned Rlen;	/*0x24+Dlen*/
  char Wid[4];
  char Fid[4];
  unsigned Flen;
  Word tag;
  Word channel;
  unsigned sample_rate;
  unsigned byte_rate;
  Word align;
  Word sample;
  char Did[4];
  unsigned Dlen;
};

static int cdrom_fd;
static char *progname;
static int opt_blocks=N_BUF;
static int opt_keylen=KEYLEN;
static int opt_overlap=OVERLAP;
static int opt_retrys=RETRYS;
static int opt_ofs=OFS;
static int opt_ibufsize=0;
static int opt_bufsize=0;
static int opt_bufstep=0;
static int opt_verbose=0;
static int opt_chmod = D_MODE;
static int opt_spchk=0;
static unsigned opt_srate = 44100;	// still unused
static int opt_mono=0;
static int retries=0;

static int opt_cddb=0;
static char *opt_cddb_path=CDDB_PATH;
static int opt_cddb_port=CDDB_PORT;
static char *opt_cddb_host=CDDB_HOST;
static int opt_save=0;			// save cddb info
static int opt_name=0;			// use cddb name for wavs

struct kword kwords[]={
  {"@TRK",KW_TRACK,"Track name"},
  {"@FDS",KW_FULLD,"Full disk name (usually author/title)"},
  {"@AUT",KW_AUTHOR,"Disk author (guessed)"},
  {"@NUM",KW_NUM,"Track number"},
  {"@DIS",KW_DISK,"Disk name (guessed)"}
};


struct Wavefile cd_newave(unsigned size)
{
  struct Wavefile dummy={{'R','I','F','F'},0x24+size,{'W','A','V','E'},
			 {'f','m','t',' '},0x10,1,2,44100,4*44100,4,16,
			 {'d','a','t','a'},size };
  /*dummy.Dlen=size;
		  dummy.Rlen=0x24+size;*/
  dummy.sample_rate = opt_srate;
  dummy.channel = 2 - opt_mono;
  dummy.byte_rate = opt_srate << dummy.channel;
  dummy.align = dummy.channel * dummy.sample >> 3;
  dummy.Dlen >>= opt_mono;
  return dummy;
}

char *resttime(int sec)
{
  static char buf[BLEN+1];
  snprintf(buf, BLEN, "%02d:%02d:%02d", sec/3600, (sec/60)%60, sec%60);
  return buf;
}

int cd_get_tochdr(struct cdrom_tochdr *Th)
{
	return ioctl(cdrom_fd,CDROMREADTOCHDR,Th);
}

int cd_get_tocentry(int trk,struct cdrom_tocentry *Te,int mode)
{
  Te->cdte_track=trk;
  Te->cdte_format=mode;
  return ioctl(cdrom_fd,CDROMREADTOCENTRY,Te);
}

void cd_read_audio(int lba,int num,char *buf)
	/* reads num CD_FRAMESIZE_RAW sized
	   sectors in buf, starting from lba*/
	/*NOTE: if num>CDROM_NBLOCKS_BUFFER as defined in ide_cd.c (8 in linux 2.0.32)
	  jitter correction may be required inside the block. */					   
{
	struct cdrom_read_audio ra;

	ra.addr.lba=lba;
	ra.addr_format=CDROM_LBA;
	ra.nframes=num;
	ra.buf=buf;
	if(ioctl(cdrom_fd,CDROMREADAUDIO,&ra)){
		/*fprintf(stderr,"%s: read raw ioctl failed \n",progname);*/
		fprintf(stderr,"\n%s: read raw ioctl failed at lba %d length %d: %s\n",
				progname,lba,num,strerror(errno));
		exit(1);
	}
}


// CDDB section
static int cddb_sock;

int cddb_sum(int n)
{
	int	ret;
	n/=75; /*elimina i frames*/
	ret = 0;
	while (n > 0) {
		ret = ret + (n % 10);
		n = n / 10;
	}
	return (ret);
}
unsigned long cddb_discid(struct cd_trk_list *tl){
	int i,n,t,st;
	st=n=0;
	for (i=tl->min;i<=tl->max;i++) 
	   n+=cddb_sum(tl->starts[i-tl->min]+CD_MSF_OFFSET); 
	t=(tl->starts[tl->max-tl->min+1]-tl->starts[0])/75;
	return (n % 0xff) << 24 | t << 8 | (tl->max-tl->min+1);
}
//int make_socket (unsigned short int port,unsigned int addr)
int make_socket (unsigned short int port,struct hostent *addr)
{
  int sock;
  struct sockaddr_in name;
  /* Create the socket. */
  sock = socket (PF_INET, SOCK_STREAM, 0);
  if (sock < 0)
    {
      perror ("dagrab");
      exit (EXIT_FAILURE);
    }
  /* Give the socket a name. */
  name.sin_family = AF_INET;
  name.sin_port = htons (port);
//  name.sin_addr.s_addr = htonl (addr);
  memcpy(&name.sin_addr,addr->h_addr,addr->h_length);
  if(connect(sock,(struct sockaddr *) &name,sizeof(name))){
      perror ("dagrab");
      exit (EXIT_FAILURE);
  };
  return sock;
}

void cddb_sendcmd(char *c)
{
//  printf("--> %s",c);
  write(cddb_sock,c,strlen(c));
}

int cddb_getresp(char *c, int s)
{
  int a;
  fd_set re,wr,ex;
  struct timeval to;
  to.tv_sec=0;
  to.tv_usec=200000;
  FD_ZERO(&re);
  FD_ZERO(&wr);
  FD_ZERO(&ex);
  FD_SET(cddb_sock,&re);
  select(cddb_sock+1,&re,&wr,&ex,&to);
  a=read(cddb_sock,c,s);if(a>0)c[a]=0;
//  printf("<-- %s",c);
  if(c[0]!='2') return -1;else return a;
}

int cddb_getresp2(char *c, int s)
{
  int a;
  fd_set re,wr,ex;
  struct timeval to;
  to.tv_sec=0;
  to.tv_usec=200000;
  FD_ZERO(&re);
  FD_ZERO(&wr);
  FD_ZERO(&ex);
  FD_SET(cddb_sock,&re);
  select(cddb_sock+1,&re,&wr,&ex,&to);
  a=read(cddb_sock,c,s);if(a>0)c[a]=0;
  return a;
}


int cddb_handshake(void){
  struct passwd	*pw;
  char *un,*hn,hn1[100];
  struct hostent *he;
  char buf[2001];
  struct hostent *hent;
  /* get the address */
  hent=gethostbyname(opt_cddb_host);
  if(hent==NULL) return 1;
  cddb_sock=make_socket(opt_cddb_port,hent);
  if(cddb_getresp(buf,2000)<0) return 1;
  /* Get login name from the password file if possible */
  if ((pw = getpwuid(getuid())) != NULL)
    un=pw->pw_name;
  else
    /* Try the LOGNAME environment variable */
    if ((un = (char *) getenv("LOGNAME")) == NULL)
      /* Try the USER environment variable */
      un = (char *) getenv("USER");
  if(un==NULL) un="unknown";
  if((gethostname(hn1,99)<0)||((he=gethostbyname(hn1))==NULL)||(he->h_name==NULL))
    hn="unknown";
  else 
    hn=(char *)he->h_name;
  sprintf(buf,"cddb hello %s %s %s %s\n",un,hn,PROGNAME,VERSION);
  cddb_sendcmd(buf);
  if(cddb_getresp(buf,2000)<0) return 1;
  return 0;
}

int cddb_query(char *qs,char **id,char **gnr){
  char buf[2001],*p,*q;
  sprintf(buf,"cddb query %s\n",qs);
  cddb_sendcmd(buf);
  if(cddb_getresp(buf,2000)<0)return 1;
  p=buf+4;
  if (buf[1]=='1'){
     fprintf(stderr,"%s: cddb inexact matches found, picking first\n",PROGNAME);
     p=strchr(buf,'\n')+1;
  }
  q=strchr(p,' ');*q=0;*gnr=strdup(p);p=q+1;
  q=strchr(p,' ');*q=0;*id=strdup(p);p=q+1;
  if ((buf[1]=='1')&&(p[strlen(p)-3]!='.')) 
    do cddb_getresp(buf,2000);while (buf[strlen(buf)-3]!='.');
  return 0;
}

int cddb_getdesc(char *gnr,char *id,char *data,int len){
  char buf[100];
  int i,a;
  char *q,*p,*t=malloc(len);
  if(t==NULL) return -1;
  sprintf(buf,"cddb read %s %s\n",gnr,id);
  cddb_sendcmd(buf);
  t[0]=0;a=i=0;
  if((i=cddb_getresp(t+a,len-a))<0) {
    return -1;}
  a=i;
  while (t[a-3]!='.'){
    i=cddb_getresp2(t+a,len-a);
    if(i<0) 
      return -1;
    a+=i;
  }
  t[a-3]=0;
  cddb_sendcmd("quit\n");
  close(cddb_sock);
//printf("%s",t);
  q=data;p=t;
  while(*p++!='\n');
  //while ((*q=*p++)!=0) if(*q!='\r')q++;
  while (*p) if((*q=*p++)!='\r')q++;
  free(t);
  return q-data;
}

int cddb_gettitle(char *data,char *title,int n){
  char *p=data,*q=title;
  int i;
  for (i=0;i<=n;i++) {p=strstr(p,"TITLE");if(p==NULL) return 1;else p+=2;};
  p=strchr(p,'=')+1;
  while (p && *p!='\n' && *p!=0) *q++=*p++;
  if(!p||(*p==0)) return 1;
  /*
  for (i=0;i<=n;i++) {p=strstr(p,"EXT");if(p==NULL) return 1;else p+=2;};
  p=strchr(p,'=')+1;
  if (*p!='\n' && *p!=0){
    *q++=' ';*q++='-';*q++=' ';
    while (*p!='\n' && *p!=0) *q++=*p++;
  };
  if(*p==0) return 1;
  */
  *q=0;
  while (--q>title && !isalnum(*q) && 
         *q!='-' && *q!='_' && *q!=' ' && *q!='_' && *q!='/') *q=0;
  if (q<=title) return 1;
  return 0;
}

int cddb_check(struct cd_trk_list *tl,char *cddb)
{
  char title[200];
  int i;
  for (i=tl->min-1;i<=tl->max;i++)
  {
    if (cddb_gettitle(cddb,title,1+i-tl->min)) return 1;
  }
  return 0;
}
char *cddb_getdir(){
  static int f=1;
  static char path[500];
  if(f){
    if(getenv("XMCD_LIBDIR")==NULL) strcpy(path,opt_cddb_path);
    else sprintf(path,"%s/cddb",getenv("XMCD_LIBDIR"));
    f=0;
  }
  return path;
}
// returns: -1 on error, 0 for cddb query, 1 for file match
int cddb_main(struct cd_trk_list *tl)
{
  FILE *f;
  DIR *d;
  struct dirent *e;
  char *id2,*p,*cddb,*loc;
  int i,cddbs,locs;
  char id[12];
  char *path;
  char path2[500];
  cddb=malloc(CDDB_MAX);
  loc=malloc(CDDB_MAX);
  if (cddb==NULL||loc==NULL){
    return -1;
  }
  loc[0]=cddb[0]=0;
  sprintf(id,"%lx",cddb_discid(tl));
  if(!opt_save){
    path=cddb_getdir();
    if((d=opendir(path))!=NULL){
      while((e=readdir(d))!=NULL){
        sprintf(path2,"%s/%s/%s",path,e->d_name,id);
        f=fopen(path2,"r");
        if(f!=NULL) {
          locs=fread(loc,1,CDDB_MAX,f);
//  	  if(cddb_check(tl))return -1;
//          return 1;
	  break;
        }
      };
    }
  }
  if(!opt_save && !cddb_check(tl,loc)){
    tl->cddb=loc;
    tl->cddb_size=locs;
    free(cddb);
    return 1;
  }
  else{ 
    if(USE_CDDB){
      if(cddb_handshake()){
        fprintf(stderr,"%s: error in cddb handshaking\n",PROGNAME);
        free(cddb);free(loc);
        return -1;
      }
      p=path2;p+=sprintf(p,"%s %d",id,tl->max-tl->min+1);
      for(i=tl->min;i<=tl->max;i++) 
        p+=sprintf(p," %d",tl->starts[i-tl->min]+CD_MSF_OFFSET);
      p+=sprintf(p," %d\n",(tl->starts[tl->max-tl->min+1]-tl->starts[0])/75);
      if(!cddb_query(path2,&id2,&tl->gnr)){
        if ((cddbs=cddb_getdesc(tl->gnr,id2,cddb,CDDB_MAX))>=0){
            //if(cddb_check(tl))return -1;
            //return 0;
            if(!cddb_check(tl,cddb)){
              tl->cddb=cddb;
              tl->cddb_size=cddbs;
	      free(loc);
	      return 0;
	    }
        } 
        //else return -1;
      }
    }
    if(!cddb_check(tl,loc)){
      tl->cddb=loc;
      tl->cddb_size=locs;
      free(loc);
      return 1;
    }
  }
  free(cddb);free(loc);
  return -1;
}

//end CDDB section

void ExpandVar (int kw,int lowcase,char *out,int tn,
                struct cd_trk_list *tl,int escape)
{
  char tmp[BLEN+1];
  char *p,*q;
  switch(kw){
  case KW_TRACK:
    cddb_gettitle(tl->cddb,out,tn);
    break;
  case KW_FULLD: 
    cddb_gettitle(tl->cddb,out,0);
    break;
  case KW_AUTHOR: 
    cddb_gettitle(tl->cddb,tmp,0);
    if ((p=strchr(tmp,'/'))!=NULL){
      *p=0;
      strcpy(out,tmp);
    }
    else *tmp=0;
    break;
  case KW_NUM :
    sprintf (out,"%02d",tn);
    break;
  case KW_DISK: 
    cddb_gettitle(tl->cddb,tmp,0);
    if ((p=strchr(tmp,'/'))!=NULL){
      while (*(++p)==' ' && *p);
      strcpy(out,p);
    }
    else strcpy (out,tmp);
    break;
  default: *out=0;break;
  };
  p=out+strlen(out)-1;
  while (p>out && *p==' ')*(p--)=0;
  if (lowcase){
    /* Remove "/" , useful from filenames got via cddb query */
    for(p = out; (p = strchr(p, '/')); p++) *p = '-';
    for(p = out; *p !=0; p++) if (strchr("\"' `",*p)) *p= '_';
  }
  if (escape){
    for(p=out,q=tmp;*p!=0;p++){
      if (!isalnum(*p)){*(q++)='\\';};
      *(q++)=*p;
    };
    *q=0;
    strcpy(out,tmp);
  }
};
char *SearchKw(char *str,char *kw)
{
  char *s1,*s2;
  char lc[10];
  strcpy(lc,kw);
  s1=lc;
  while (*s1)
    *(s1++)=tolower(*s1);
  s1=strstr(str,kw);
  s2=strstr(str,lc);
  if (s2==NULL) return (s1);
  if ((s1==NULL) ||(s2<s1)) return s2;
  else return s1;
};

void ExpandTempl (char *templ,char *out,int tn,
		  struct cd_trk_list *tl,int escape)
{
  char *tpidx,*outidx,*tmp,*tmp2;
  char varval[2000];
  int i,vart,len;
  tpidx=templ;
  outidx=out;
  while (tpidx!=NULL && *tpidx!=0){
    vart=-1;
    tmp=NULL;
    tmp2=NULL;
    for (i=0;i<=KW_MAX;i++){
      tmp=SearchKw(tpidx,kwords[i].kw);
      if (tmp!=NULL && (tmp<tmp2 || tmp2==NULL)){
	tmp2=tmp;
	vart=i;
      }
    }
    /*ho trovato la variabile*/
    if (vart>=0){
      len=tmp2-tpidx;
      ExpandVar(kwords[vart].idx,islower(*(tmp2+1)),varval,tn,tl,escape);
      strncpy(outidx,tpidx,len);
      outidx+=len;
      tpidx=tmp2+strlen(kwords[vart].kw);
      strcpy(outidx,varval);
      outidx+=strlen(varval);
    }
    else{
      strcpy (outidx,tpidx);
      tpidx=NULL;
    }
  }
};


int cd_getinfo(char *cd_dev,struct cd_trk_list *tl)
{
	int i;
	struct cdrom_tochdr Th;
	struct cdrom_tocentry Te;

	if ((cdrom_fd=open(cd_dev,O_RDONLY|O_NONBLOCK))==-1){
		fprintf(stderr,"%s: error opening device %s\n",progname,cd_dev);
		exit(1);
	}
	if(cd_get_tochdr(&Th)){
		fprintf(stderr,"%s: read TOC ioctl failed: %s\n",progname,strerror(errno));
		exit(1);
	}
	tl->min=Th.cdth_trk0;tl->max=Th.cdth_trk1;
	if((tl->starts=(int *)malloc((tl->max-tl->min+2)*sizeof(int)))==NULL){
		fprintf(stderr,"%s: list data allocation failed\n",progname);
		exit(1);
	}
	if((tl->types=(char *)malloc(tl->max-tl->min+2))==NULL){
		fprintf(stderr,"%s: list data allocation failed\n",progname);
		exit(1);
	}

	for (i=tl->min;i<=tl->max;i++)
	{
		if(cd_get_tocentry(i,&Te,CDROM_LBA)){
			fprintf(stderr,"%s: read TOC entry ioctl failed: %s\n",
				progname,strerror(errno));
			exit(1);
		}
		tl->starts[i-tl->min]=Te.cdte_addr.lba;
		tl->types[i-tl->min]=Te.cdte_ctrl&CDROM_DATA_TRACK;
	}
	i=CDROM_LEADOUT;
	if(cd_get_tocentry(i,&Te,CDROM_LBA)){
		fprintf(stderr,"%s: read TOC entry ioctl failed: %s\n",progname,strerror(errno));
		exit(1);
	}
	tl->starts[tl->max-tl->min+1]=Te.cdte_addr.lba;
	tl->types[tl->max-tl->min+1]=Te.cdte_ctrl&CDROM_DATA_TRACK;
	
        i=cddb_main(tl);
	if(i==-1) {
	  fprintf(stderr,"%s: error retrieving cddb data\n",PROGNAME);
	  opt_save=opt_name=0;
	  tl->cddb=NULL;tl->gnr=NULL;opt_cddb=0;
	};
	if(i==1) tl->gnr=NULL;
	return 0;
}

void cd_disp_TOC(struct cd_trk_list *tl)
{
	int i, len;
	char title[200];
	if(USE_CDDB){
	  cddb_gettitle(tl->cddb,title,0);
	  printf("\nDISK: %s\n\n",title);
	}
	printf("%5s %8s %8s %5s %9s %4s","track","start","length","type",
			"duration", "MB");
	if(USE_CDDB) printf(" Title");
	printf("\n");
	for (i=tl->min;i<=tl->max;i++)
	{
		len = tl->starts[i + 1 - tl->min] - tl->starts[i - tl->min];
		printf("%5d %8d %8d %5s %9s %4d",i,
			tl->starts[i-tl->min]+CD_MSF_OFFSET, len,
			tl->types[i-tl->min]?"data":"audio",resttime(len / 75),
			(len * CD_FRAMESIZE_RAW) >> (20 + opt_mono));
		if(USE_CDDB){
		  cddb_gettitle(tl->cddb,title,1+i-tl->min);
		  printf(" %s",title);
		}
		printf("\n");
	}
	printf("%5d %8d %8s %s\n",CDROM_LEADOUT,
			tl->starts[i-tl->min]+CD_MSF_OFFSET,"-","leadout");
	printf("\nCDDB DISCID: %lx\n",cddb_discid(tl));
}
int cd_jc1(int *p1,int *p2) //nuova
	/* looks for offset in p1 where can find a subset of p2 */
{
	int *p,n;

	p=p1+opt_ibufsize-IFRAMESIZE-1;n=0;
	while(n<IFRAMESIZE*opt_overlap && *p==*--p)n++;
	if (n>=IFRAMESIZE*opt_overlap)	/* jitter correction is useless on silence */ 
	{
		n=(opt_bufstep)*CD_FRAMESIZE_RAW;
	}
	else			/* jitter correction */
	{
		n=0;p=p1+opt_ibufsize-opt_keylen/sizeof(int)-1;
		while((n<IFRAMESIZE*(1+opt_overlap)) && memcmp(p,p2,opt_keylen))
		  {p--;n++;};
/*		  {p-=6;n+=6;}; //should be more accurate, but doesn't work well*/
		if(n>=IFRAMESIZE*(1+opt_overlap)){		/* no match */
			return -1;
		};
		n=sizeof(int)*(p-p1);
	}
	return n;
}

int cd_jc(int *p1,int *p2)
{
	int n,d;
	n=0;
	if (opt_overlap==0) return (opt_bufstep)*CD_FRAMESIZE_RAW;
	do
		d=cd_jc1(p1,p2+n);
	while((d==-1)&&(n++<opt_ofs));n--;
	if (n<0)n=0;
	if (d==-1) return (d);
	else return (d-n*sizeof(int));
}

int check_for_space(char *path, int space)
{
	struct statfs buffs;
	struct stat buf;

	if(!stat(path, &buf) && !statfs(path, &buffs) &&
			(buffs.f_bavail * buf.st_blksize) >= space) return 1;
	return 0;
}

void cd_track_name(char *name,struct cd_trk_list *tl, int tn,char *basename)
{
  // char exec[BLEN+1],*slashp;
  if(opt_name){
    //  cddb_gettitle(tl->cddb,exec,1+tn-tl->min);   
    /* Remove "/" from filenames might used in cddb query */
    // for(slashp = exec; (slashp = strchr(slashp, '/')); slashp++) 
    //*slashp = '-';
    //snprintf(name,BLEN,"%s.wav",exec);
    ExpandTempl (basename,name,tn,tl,FALSE);
  }
  else snprintf(name,BLEN,basename,tn);
}

int cd_read_track(char *basename,int tn,struct cd_trk_list *tl,
		char *filter)
{
	int buf1[opt_ibufsize];		
	int buf2[opt_ibufsize];
	short buf3[opt_ibufsize];
	int *p1,*p2,*p;
	char nam[BLEN+1], exec[BLEN+1],exec2[BLEN+1];
	struct Wavefile header;
	int fd,bytes,i,n,q,space;
	int bcount, sc, missing, speed = 0, ldp, now;

	if(tn<tl->min || tn>tl->max) return (-1);
	space = ((tl->starts[tn-tl->min+1]-tl->starts[tn-tl->min]) * 
			CD_FRAMESIZE_RAW) >> opt_mono;
	cd_track_name(nam,tl,tn,basename);

	if(tl->types[tn-tl->min]){
		fprintf(stderr,"Track %d is not an audio track\n",tn);
		return 1;
	}
	else printf("Dumping track %d: lba%7d to lba%7d (needs %d MB)\nOutput file is: %s\n",tn,
			tl->starts[tn-tl->min],tl->starts[tn-tl->min+1]-1, space>>20,nam);
	tn-=tl->min;
	if ((fd=open(nam,O_TRUNC|O_CREAT|O_WRONLY|O_APPEND))==-1){
		fprintf(stderr,"%s: error opening wave file %s: %s\n",progname,nam,strerror(errno));
		exit(1);
	};
	if(fchmod(fd,opt_chmod)==-1){
		fprintf(stderr,"%s: error changing mode of wave file %s: %s\n",progname,nam,strerror(errno));
		exit(1);
	}
	if(opt_spchk && !check_for_space(nam, space)) {
		close(fd);
		unlink(nam);
		fprintf(stderr, "Not enough free space on disk for track %d\n",
				tn + tl->min);
		return 1;
	}
	header=cd_newave((tl->starts[tn+1]-tl->starts[tn])*CD_FRAMESIZE_RAW);
	if(write(fd,&header,sizeof(header))==-1){
		fprintf(stderr,"%s: error writing wave file %s: %s\n",progname,nam,strerror(errno));
		exit(1);
	};
	/* main loop */
	bytes=0;p1=buf1;p2=buf2;q=0;
	cd_read_audio(tl->starts[tn],opt_blocks,(char *)p1);

	bcount = ldp = now = 0; sc = time((time_t*)0) - 1;
	for(i=tl->starts[tn]+opt_bufstep;i<tl->starts[tn+1];i+=opt_bufstep){
		/* jitter correction; a=number of bytes written */
		if(opt_verbose){
			now = time((time_t*) 0);
			if(ldp < now) {
				ldp = now;
				speed = bcount / (now - sc);
				printf("Reading block %7d - %2.2fx speed",i, (float) speed/75);
				if(speed > 0) {
					missing = (tl->starts[tn+1] - i) / speed;
					printf(" - %s left\r", resttime(missing));
				} else {
					printf("                \r");
				}
				fflush(stdout);
			}
			bcount += opt_bufstep;
		}
		q=0;
		if (tl->starts[tn+1]>i+1){
		    do {
			    if((i+opt_blocks)<tl->starts[tn+1])
				cd_read_audio(i,opt_blocks,(char *)p2);
			    else
				cd_read_audio(i,tl->starts[tn+1]-i,(char *)p2);
    		    }while (((n=cd_jc(p1,p2))==-1)&&(q++<opt_retrys)&&++retries);
		}
		else{
		    n=0;
		};
		if(n==-1)
		{
			n=opt_bufstep*CD_FRAMESIZE_RAW;
			printf ("jitter error near block %d                                \n",i);
		};
		if(bytes+n>(tl->starts[tn+1]-tl->starts[tn])*CD_FRAMESIZE_RAW){
			n=(tl->starts[tn+1]-tl->starts[tn])*CD_FRAMESIZE_RAW-bytes;
		}
		if (n>0){
			if(opt_mono) {
				register int c, d;
				for(c = 0; c < (n>>2); c++) {
					d = p1[c];
					buf3[c] = ((short)(d&65535) + (short)(d>>16)) >> 1;
				}
				write(fd,buf3,n>>1);
			} else if(write(fd,p1,n)==-1){
				fprintf(stderr,"%s: error writing wave file %s: %s\n",
					progname,nam,strerror(errno));
				exit(1);
			};
			bytes+=n;
		}
		else{
			/*printf("errore!!\n");*/
			break;
		}
		p=p1;p1=p2;p2=p;
	}
	/* dump last bytes */
	if (bytes<(tl->starts[tn+1]-tl->starts[tn])*CD_FRAMESIZE_RAW){
		n=(tl->starts[tn+1]-tl->starts[tn])*CD_FRAMESIZE_RAW-bytes;
		if(write(fd,p1,n)==-1){
			fprintf(stderr,"%s: error writing wave file %s: %s\n",progname,nam,strerror(errno));
			exit(1);
		};
		bytes+=n;
	}
	close (fd);
	if(opt_verbose) {
		printf("Track %d dumped at %2.2fx speed in %s            \n", 
				tn + tl->min, (float)speed/75, 
				speed==0?"":resttime(bcount/speed));
	}
	if(filter[0]) {
		snprintf(exec, BLEN, filter, nam);
		if (opt_cddb||opt_name) {
		  ExpandTempl (exec,exec2,tn + tl->min,tl,TRUE);
		  system(exec2);
		}
		else system(exec);
	}
	return 0;
}
void usage(void)
{
  int i;
  fprintf(stderr,
	  "\n%s v%s - dumps digital audio from cdrom to riff wave files\n"
	  "Usage:\tdagrab [options] [track list]\nOptions:\n"
	  "\t-h \t\t: show this message\n\t-i \t\t: display track list\n"
	  "\t-d device\t: set cdrom device (default=%s)\n"
	  "\t-a\t\t: dump all tracks (ignore track list)\n"
	  "\t-v\t\t: verbose execution\n\t-f file\t\t: set output file name\n"
	  "\t\t\t\tembed %%02d for track numbering (no CDDB mode)\n\t\t\t\t(or variables in CDDB mode)\n"
	  "\t-o overlap\t: sectors overlap for jitter correction, default %d\n"
	  "\t-n sectors\t: sectors per request (default %d)\n"
	  "\t\t\t warning: a number greater than that defined in your CD\n"
	  "\t\t\t driver may cause unreported jitter correction failures \n"
	  "\t-k key_length\t: size of key to match for jitter correction, default %d\n"
	  "\t-r retries\t: read retries before reporting a jitter error (%d)\n"
	  "\t-t offset\t: number of offsets to search for jitter correction (%d)\n"
	  "\t-m mode  \t: default mode for files, will be chmod to mode (0%o)\n"
	  "\t-e command\t: executes command for every copied track\n"
	  "\t\t\t\tembed %%s for track's filename \n\t\t\t\t(or variables in CDDB mode)\n"
	  "\t-s \t\t: enable free space checking before dumping a track\n"
	  "\t-p \t\t: mono mode\n"
	  "\t-C\t\t: enables cddbp\n"
	  "\t-S\t\t: save cddb data in local database, forces -C\n"
	  "\t-N\t\t: use cddb name, changes behavior of -f's parameter\n"
	  "\t-H host\t\t: cddbp server, default %s\n"
	  "\t-P port\t\t: cddbp port number default %d\n"
	  "\t-D dir\t\t: base of local cddb database\n",
	  PROGNAME,VERSION,CDDEVICE,OVERLAP,N_BUF,KEYLEN,
	  RETRYS,OFS,D_MODE,CDDB_HOST,CDDB_PORT);
  printf("\n\tVariables: (use lowcase values for removing slashes)\n");
  for (i=0;i<=KW_MAX;i++){
    printf ("\t\t%s %s\n",kwords[i].kw,kwords[i].desc);
  }
  exit(0);
}

#define CPARG(str) strncpy((str),optarg,BLEN); (str)[BLEN]=0

int main(int ac,char **av)
{
	int i,l,disp_TOC=0;
	char c;
	int all_tracks=0;
	struct cd_trk_list tl;
	char cd_dev[BLEN+1]=CDDEVICE;
	char basename[BLEN+1]="";
	char filter[BLEN+1] = "";
	char path[500];
	FILE *f;

	progname=av[0];
	optind=0;
	while((c=getopt(ac,av,"d:f:n:o:k:r:t:m:e:H:P:D:pshaivCSN"))!=EOF){
		switch(c){
			case 'h':usage();break;
			case 'd':CPARG(cd_dev);break;
			case 'f':CPARG(basename);break;
			case 'a':all_tracks=1;break;
			case 'i':disp_TOC=1;break;
			case 'n':opt_blocks=atoi(optarg);break;
			case 'o':opt_overlap=atoi(optarg);break;
			case 'k':opt_keylen=atoi(optarg);break;
			case 'v':opt_verbose=1;break;
			case 'r':opt_retrys=atoi(optarg);break;
			case 't':opt_ofs=atoi(optarg);break;
			case 'm':opt_chmod=strtol(optarg,(char**)0,8);break;
			case 'e':CPARG(filter);break;
			case 's':opt_spchk=1;break;
			case 'C':opt_cddb=1;break;
			case 'S':opt_save=1;opt_cddb=1;break;
			case 'N':opt_name=1;break;
			case 'D':opt_cddb_path=strdup(optarg);break;
			case 'P':opt_cddb_port=atoi(optarg);break;
			case 'H':opt_cddb_host=strdup(optarg);break;
			case 'p':opt_mono=1;break;
		}
	}
	if (basename[0]==0){
	  if (opt_name){
	    strcpy(basename,"@num-@trk.wav");
	  }
	  else{
	    strcpy(basename,"track%02d.wav");
	  }
	}
	if(opt_blocks<4){
		opt_blocks=4;if(opt_verbose)
			fprintf(stderr,"sectors per request too low,setting to 4\n");
	};
	if(opt_blocks>200){
		opt_blocks=200;if(opt_verbose)
			fprintf(stderr,"sectors per request too high,setting to 200\n");
	};
	if(opt_overlap>opt_blocks-2){
		opt_overlap=opt_blocks-2;if(opt_verbose)
			fprintf(stderr,"overlap too high,setting to (sectors per request-2)\n");
	};
	if(opt_overlap<1){
		opt_overlap=0;if(opt_verbose)
			fprintf(stderr,"jitter correction disadled\n");
	};
	if(opt_keylen<4){
		opt_keylen=4;if(opt_verbose)
			fprintf(stderr,"key too short,setting to 4\n");
	};
	if(opt_keylen>400){
		opt_keylen=400;if(opt_verbose)
			fprintf(stderr,"key too long,setting to 400\n");
	};
	if(opt_retrys>1000){
		opt_retrys=1000;if(opt_verbose)
			fprintf(stderr,"retrys too high,setting to 1000\n");
	};
	if(opt_retrys<1){
		opt_retrys=1;if(opt_verbose)
			fprintf(stderr,"retrys too low,setting to 1\n");
	};
	if(opt_ofs>256){
		opt_retrys=256;if(opt_verbose)
			fprintf(stderr,"offset too high,setting to 256\n");
	};
	if(opt_ofs<1){
		opt_ofs=1;if(opt_verbose)
			fprintf(stderr,"offset too low,setting to 1\n");
	};
	if(opt_chmod & ~07777) {
		opt_chmod=0660; if(opt_verbose)
			fprintf(stderr, "strange chmod value, setting to 0660\n");
	}
	opt_bufsize=CD_FRAMESIZE_RAW * opt_blocks;
	opt_ibufsize=opt_bufsize/sizeof(int);
	opt_bufstep=opt_blocks-opt_overlap;
	if((optind==ac)&&!all_tracks&&!opt_save) {
		if(disp_TOC){
			if(cd_getinfo(cd_dev,&tl))
				exit(1);
			cd_disp_TOC(&tl);
			exit(0);
		}
		else usage();
	};

	if(cd_getinfo(cd_dev,&tl)){
	  exit(1);
	}
	if(disp_TOC)cd_disp_TOC(&tl);
	if(opt_save && (tl.cddb!=NULL)){
	  if(tl.gnr==NULL) 
	    fprintf(stderr,"%s: entry found in local database not saved\n",PROGNAME);
	  else{
	    sprintf(path,"%s/%s",cddb_getdir(),tl.gnr);
	    mkdir(path,0777);
	    sprintf(path,"%s/%s/%lx",cddb_getdir(),tl.gnr,cddb_discid(&tl));
	    if((f=fopen(path,"w"))==NULL){
	      perror ("dagrab");
	    }else{
	      fwrite(tl.cddb,1,tl.cddb_size,f);
	      fclose(f);
	    }
	  }
	};
	if(opt_verbose) fprintf(stderr,
			"sectors %3d overlap %3d key length %3d retrys %4d offset %3d\n",
			opt_blocks,opt_overlap,opt_keylen,opt_retrys,opt_ofs);
	if(all_tracks){
		printf("Dumping all tracks\n");
		for(i=tl.min;i<=tl.max;i++){
			cd_read_track(basename,i,&tl,filter);
		}
	}
	else
	{
		for(i=optind;i<ac;i++)
		{
			l=atoi(av[i]);
			if((l>=tl.min)&&(l<=tl.max)) {
				cd_read_track(basename,l,&tl,filter);
			}
		}
		
	}
	if(opt_verbose) {
		printf("Total retries for jitter correction: %d\n", retries);
	}
	printf("Done!                 \n");
	exit(0);
}
