package Jcd;

/**
  Jcd - Java CD Audio Player
  Copyright (c) 1996.   Michael Hamilton (michael@actrix.gen.nz).
  All rights reserved.  See the README for full details
 */

import java.io.*;
import java.net.*;  
import java.lang.*;
import java.util.*;
import Jcd.JcdProps;

class CddbClient {

  static final boolean DEBUG = JcdProps.getBoolean("jcd.debug", false); 

				// Command response - first digit
  public static final int INFORMATIONAL    = 100;
  public static final int VALID            = 200;
  public static final int VALID_SO_FAR     = 300;
  public static final int VALID_NOT_DOABLE = 400;
  public static final int INVALID          = 500;
  
				// Command response - second digit
  public static final int READY       =  0;
  public static final int MORE_OUTPUT = 10;
  public static final int MORE_INPUT  = 20;
  public static final int CLOSING     = 30;

				// Server sign on response code
  public static final int READ_WRITE_OK  = 200;
  public static final int READ_ONLY_OK   = 201;
  public static final int NO_ACCESS      = 432;
  public static final int TOO_MANY_USERS = 433;
  public static final int LOAD_TOO_HIGH  = 434;

				// Query responses
  public static final int FOUND_EXACT_MATCH     = 200;
  public static final int FOUND_INEXACT_MATCHES = 211;
  public static final int FOUND_NO_MATCHES      = 202;
  public static final int FOUND_BAD_DB_ENTRY    = 203;
  public static final int FOUND_NO_HANDSHAKE    = 409; //Haven't called connect

				// Read respose
  public static final int READ_FOUND_ENTRY      = 210;
  public static final int READ_ENTRY_NOT_FOUND  = 401;
  public static final int READ_NO_HANDSHAKE     = 409; //Haven't called connect
  
  Socket serverSocket;

  DataInputStream fromServer;
  PrintStream       toServer;

  boolean read_write_access = true;

  int status;

  String server_name;
  int port_number;

  public CddbClient(String server, int port)
  {
    server_name = server;
    port_number = port;
  }

  void connect() throws CddbException 
  {
    String str;

    try {
      serverSocket = new Socket(server_name, port_number);
      fromServer = new DataInputStream(serverSocket.getInputStream());
      toServer = new PrintStream(serverSocket.getOutputStream());
    }
    catch (IOException socketFailure) {
      throw new CddbException(this, socketFailure.getMessage());
    }

    try {
      str = fromServer.readLine();
    }
    catch (IOException initFailure) {
      throw new CddbException(this, initFailure.getMessage());
    }
    if (str == null) {
      throw new CddbException(this, "Read null initialisation from server.");
    }

    if (DEBUG) System.out.println(str);

    status = extractStatus(str);
    
    switch (status) {
    case READ_WRITE_OK:
      read_write_access = true;
      break;
    case READ_ONLY_OK:
      read_write_access = false;
      break;
    default:
      throw new CddbException(this, status, "Can't get access to server.");
    }

    toServer.println("cddb hello" + 
		     " " + System.getProperty("user.name") + 
		     " " + CddbClient.hostname() +
		     " " + JcdProps.get("jcd.version", JcdProps.JCD_VERSION)
		     );
      
    try {
      str = fromServer.readLine();
    }
    catch (IOException helloFailure) {
      throw new CddbException(this, helloFailure.getMessage());
    }

    if (DEBUG) System.out.println(str);

    if (str == null) {
      throw new CddbException(this, "Failed to read hello response from server."); 
    }
    status = extractStatus(str);
    if (status != VALID) {
      throw new CddbException(this, status, "Cddb hello: " + str); 
    }
  }

  CddbEntry read(String category, String discID) throws CddbException
  {
    // Fetch the details for a particular entry.
    StringBuffer request = new StringBuffer(100);

    request.append("cddb read " +
		   category     + " " +
		   discID);

    toServer.println(request.toString());
    
    try {
      int status;

      String response = fromServer.readLine();
      if (DEBUG) System.out.println(response);

      status = extractStatus(response);

      if (status == READ_FOUND_ENTRY) {
	StringBuffer entry_buffer = new StringBuffer(100);
	response = fromServer.readLine();
	while (response.compareTo(".") != 0) {
	  if (DEBUG) System.out.println(response);
	  entry_buffer.append(response);
	  entry_buffer.append("\n");
	  response = fromServer.readLine();
	}
	return new CddbEntry(category, discID, entry_buffer.toString());
      }
      else {
	throw new CddbException(this, "Cddb Read: " + response);
      }
      
    }
    catch (IOException IOFailure) {
      throw new CddbException(this, "Cddb Read: " + IOFailure.getMessage());
    }

  }

  Vector query(String discID, int trackLength[]) throws CddbException
  {
    // A query can return one or more possible matches. 
    int lengthInSeconds = trackLength[0] / Drive.FRAMES_PER_SECOND;
    StringBuffer request = new StringBuffer(100);

    request.append("cddb query " + 
		   discID + " " +
		   (trackLength.length - 1));
    
    for (int i = 1; i <= trackLength.length - 1; i++) {
      request.append(" " + trackLength[i]);
    }
    
    request.append(" " + lengthInSeconds);
    
    if (DEBUG) System.out.println(request);

    toServer.println(request.toString()); // Query the server.

    try {
      String response = fromServer.readLine();

      int status = extractStatus(response);
      int ct_index, id_index, tt_index;
      CddbMatch match;
      Vector result;

      if (DEBUG) System.out.println(response);

      switch (status) {

      case FOUND_EXACT_MATCH:
	result = new Vector(1);
	ct_index = response.indexOf(' ') + 1;
	id_index = response.indexOf(' ', ct_index) + 1; 
	tt_index = response.indexOf(' ', id_index) + 1; 
	match = new CddbMatch(response.substring(ct_index, id_index - 1),
			      response.substring(id_index, tt_index - 1),
			      response.substring(tt_index));
	result.addElement((Object) match);
	return result;

      case FOUND_INEXACT_MATCHES:
	result = new Vector(10);

	for (response = fromServer.readLine();
	     response.compareTo(".") != 0;
	     response = fromServer.readLine()) {
	  if (DEBUG) System.out.println(response);
	  id_index = response.indexOf(' ') + 1; 
	  tt_index = response.indexOf(' ', id_index) + 1; 
	  match = new CddbMatch(response.substring(0, id_index - 1),
				response.substring(id_index, tt_index - 1),
				response.substring(tt_index));
	  result.addElement((Object) match);
	}
	return result;

      case FOUND_NO_MATCHES:
	return null;
      case FOUND_BAD_DB_ENTRY:
      case FOUND_NO_HANDSHAKE:
      default:
	throw new CddbException(this, "Cddb query: " + response);
      }
    }
    catch (IOException IOFailure) {
      throw new CddbException(this, "Cddb query:" + IOFailure.getMessage());
    }
  }

  void closeConnection() throws CddbException
  {
    String str;
    int status;

    try {
      toServer.println("quit");
      str = fromServer.readLine();
      if (DEBUG) System.out.println(str);
      status = extractStatus(str);
      serverSocket.close();
    }
    catch (IOException IOFailure) {
      throw new CddbException(this, IOFailure.getMessage());
    }

    if (status != (VALID + CLOSING)) {
      throw new CddbException(this, status, "Bad status on closing connection");
    }
  } 

  int extractStatus(String str) {
    try {
      return Integer.parseInt(str.substring(0,3));
    }
    catch (NumberFormatException e) {
      return 0;
    }
  }

  String serverDetails()
  {
    return "Cddb server " + server_name + ":" + port_number;
  }

  static String hostname() 
  {
    String hostname;
    try {
      hostname = InetAddress.getLocalHost().getHostName();
    }
    catch (UnknownHostException e) {
      hostname = "UnknownHost";
    }
    return JcdProps.get("cddb.client_hostname", hostname);
  }
}
  

class CddbMatch {		// A match structure - for appending to
				// a list (vector) of possible matches.
  static final boolean DEBUG = JcdProps.getBoolean("jcd.debug", false); 

  String category;
  String cddbID;
  String title;

  CddbMatch(String category, String id, String title) 
  {
    this.category = category;
    cddbID = id;
    this.title = title;
    if (DEBUG) System.out.println("New match " + category + "." + id + "." + title);
  }
}

