package Jcd;

/**
  Jcd - Java CD Audio Player
  Copyright (c) 1996.   Michael Hamilton (michael@actrix.gen.nz).
  All rights reserved.  See the README for full details
 */

import java.io.*;
import java.net.*;  
import java.lang.*;
import java.util.*;
import Jcd.JcdProps;

class CddbEntry {

  static final boolean DEBUG = JcdProps.getBoolean("jcd.debug", false); 

  public  static final String CACHE_DIR_NAME =
    JcdProps.get("cddb.directory", 
		 System.getProperty("user.home") +
		 File.separator +
		 "CDDB");

  static final String NEWLINE             = "\n";
  static final String COMMENT             = "#";
  static final String INDENT              = "    ";
  static final String XMCD_FIRST_LINE     = "# xmcd";
  static final String TRACK_FRAME_OFFSETS = "# Track frame offsets:";
  static final String DISC_LENGTH         = "# Disc length:";
  static final String REVISION            = "# Revision:";
  static final String SUBMITTED_VIA       = "# Submitted via:";

  static final String DISCID    = "DISCID";
  static final String DTITLE    = "DTITLE";
  static final String TTITLE    = "TTITLE";
  static final String EXTD      = "EXTD";
  static final String EXTT      = "EXTT";
  static final String PLAYORDER = "PLAYORDER";

  static final String CATEGORIES[] = {
    "blues",
    "classical",
    "country",
    "data",
    "folk",
    "jazz",
    "misc",
    "newage",
    "reggae",
    "rock",
    "soundtrack"    
  };

  String category;		
  String cddbID;		// Main ID - arbitrary choice based on
				// how it was looked up.
  String cddbIDaliases[];	// ID's for various pressings
  String discArtist;
  String discTitle;
  String discExtendedInfo;	// Could be more than one line.
  String trackTitle[];		// Each track title.
  String trackExtendedInfo[];
  int playOrder[];

  int trackFrameOffsets[];
  int discLengthSeconds;

  int revision;
  String submittedVia;

  boolean changed = true;

  CddbEntry(String category,
	    String discID,
	    String momento) throws CddbException
  {				// recreate an entry from its text momento
    String line;
    
    this.category = category;
    cddbID = discID;
    fromMomento(momento);	// Restore state from string.
    // System.out.println(makeMomento());
  }

  CddbEntry(String discID) throws IOException, CddbException
  {				// recreate an entry from its text momento
    File dir = new File(CACHE_DIR_NAME);

    if (dir.exists()) {
      String categories[] = dir.list();

      for (int i = 0; i < categories.length; i++) {
	if (DEBUG) 
	  System.out.println(CACHE_DIR_NAME + File.separator + 
			     categories[i] + "." + discID);
	
	File cdfile = new File(CACHE_DIR_NAME + File.separator + categories[i],
			       discID);
	
	if (cdfile.exists() && cdfile.canRead()) {
	  DataInputStream cdstream = 
	    new DataInputStream(new FileInputStream(cdfile));
	  StringBuffer momentoBuffer = new StringBuffer(2000);
	  
	  this.category = categories[i];
	  cddbID = discID;
	  
	  for (;;) {
	    String line = cdstream.readLine();
	    if (line == null) {
	      break;
	    }
	    momentoBuffer.append(line);
	    momentoBuffer.append("\n");
	  }
	  fromMomento(momentoBuffer.toString());	// Restore state from string.
	  return;
	}
      }
    }
    throw new CddbException("File not found");
  }


  CddbEntry(String discID,
	    int trackLength[])
  {				// recreate an entry from its text momento
    int numTracks = trackLength.length - 1; // Zero is the leadout - skip it
    cddbID = "" + discID;
    discLengthSeconds = trackLength[0] / Drive.FRAMES_PER_SECOND;
    category = "misc";
    discTitle  = "";
    discArtist = "";
    discExtendedInfo = "";
    trackTitle = new String[numTracks];
    trackExtendedInfo = new String[numTracks];
    trackFrameOffsets = new int[numTracks];
    cddbIDaliases = new String[1];
    cddbIDaliases[0] = "" + discID;
    for (int i = 0; i < numTracks; i++) {
      trackTitle[i] = "";
      trackExtendedInfo[i] =  "";
      trackFrameOffsets[i] = trackLength[i+1]; // Zero is the leadout - skip it
    }
    revision = 0;
    cddbID = discID;
  }


  void setCategory(String newValue) 
  {
    changed = true;
    category = newValue;
  }

  void setArtist(String newValue) 
  {
    changed = true;
    discArtist = newValue;
  }

  void setTitle(String newValue) 
  {
    changed = true;
    discTitle = newValue;
  }

  void setExtendedInfo(String newValue) 
  {
    changed = true;
    discExtendedInfo = newValue;
  }

  void setTrackTitle(int i, String newValue)
  {
    changed = true;
    trackTitle[i] = newValue;
  }

  void setTrackExtendedInfo(int i, String newValue)
  {
    changed = true;
    trackExtendedInfo[i] = newValue;
  }

  void save() throws IOException
  {
    save(cddbID);
  }

  void save(String alias) throws IOException
  {
    String basename = CACHE_DIR_NAME + File.separator + category;
    File dir = new File(basename);
    dir.mkdirs();

    if (changed) {
      submittedVia = JcdProps.get("jcd.version", JcdProps.JCD_VERSION) +
	" " + System.getProperty("user.name") + 
	" " + CddbClient.hostname();
      revision++;
    }

    String fname = basename + File.separator + cddbID;
    PrintStream pstream = new PrintStream(new FileOutputStream(fname));

    boolean new_alias = true;
    for (int i = 0; i < cddbIDaliases.length; i++) {
      if (cddbIDaliases[i].compareTo(alias) == 0) {
	new_alias = false;
	break;
      }
    }
				// The real save...
				// Save the momemto to the file
    pstream.print(makeMomento(new_alias ? alias : null));
    pstream.close();
    changed = false;

				// Set links
    for (int i = 0; i < cddbIDaliases.length; i++) {
      if (cddbIDaliases[i].compareTo(cddbID) != 0) {
	String link_to = basename + File.separator + cddbIDaliases[i];
	if (!(new File(link_to).exists())) {
	  link_cmd(cddbID, link_to);
	}
      }
    }
    if (new_alias) {
      String link_to = basename + File.separator + alias;
      if (!(new File(link_to).exists())) {
	link_cmd(cddbID, link_to);
      }
    }
  }

  protected void link_cmd(String from, String to) throws IOException
  {
    String link_cmd = "/bin/ln -s " + from + " " + to;
    Runtime.getRuntime().exec(link_cmd);
  }

  void fromMomento(String momento) throws CddbException
  {
    int num_tracks = 0;
    int n;
				// Break into lines
    StringTokenizer line_enum = new StringTokenizer(momento, NEWLINE);
    String line;		// Each line.

    StringBuffer buffer = new StringBuffer(1000);
    
    line = line_enum.nextToken();

    if (!line.startsWith(XMCD_FIRST_LINE)) {
      throw new CddbException("Invalid Entry Header: " + line);
    }

    line = line_enum.nextToken();
				// Parse the comment header.
    while (line.startsWith(COMMENT)) {
      if (line.startsWith(TRACK_FRAME_OFFSETS)) {

	buffer.setLength(0);

	for (line = line_enum.nextToken(); 
	     line.trim().compareTo("#") != 0 && line_enum.hasMoreTokens(); 
	     line = line_enum.nextToken()) {
	  buffer.append(line);
	}
	
	StringTokenizer fnum_enum = 
	  new StringTokenizer(buffer.toString(), COMMENT);
	String str;
	num_tracks = fnum_enum.countTokens();
	trackFrameOffsets = new int[num_tracks];
	trackTitle        = new String[num_tracks];
	trackExtendedInfo = new String[num_tracks];
	
	for (int i = 0; i < num_tracks; i++) {
	  str = (String) fnum_enum.nextToken();
	  trackFrameOffsets[i] = Integer.parseInt(str.trim());
	}
      }
      else if (line.startsWith(DISC_LENGTH)) {
	discLengthSeconds = 
	  Integer.parseInt(line.substring(DISC_LENGTH.length(),
					  line.indexOf(' ',
						       DISC_LENGTH.length() +1)
					  ).trim());
      }
      else if (line.startsWith(REVISION)) {
	revision = 
	  Integer.parseInt(line.substring(REVISION.length()).trim());
      }
      else if (line.startsWith(SUBMITTED_VIA)) {
	submittedVia = line.substring(SUBMITTED_VIA.length()).trim();
      }
      line = line_enum.nextToken();
    }
				// Parse disc ID's
    buffer.setLength(0);
    while (line.startsWith(DISCID)) {
      buffer.append(",");
      buffer.append(line.substring(DISCID.length() + 1));
      line = line_enum.nextToken();
    }
    StringTokenizer aliases_enum = 
      new StringTokenizer(buffer.toString().substring(1), ",");
    n = aliases_enum.countTokens();
    cddbIDaliases = new String[n];
    for (int i = 0; i < n; i++) {
      cddbIDaliases[i] = aliases_enum.nextToken();
    }
				// Disc Title lines
    discTitle = "";
    while (line.startsWith(DTITLE)) {
      discTitle = discTitle + line.substring(DTITLE.length() + 1);
      line = line_enum.nextToken();
    }
				// Track title lines
    while (line.startsWith(TTITLE)) {
      int tnum = Integer.parseInt(line.substring(TTITLE.length(),
						line.indexOf('=')));
      trackTitle[tnum] = 
	((trackTitle[tnum] == null) ? "" : trackTitle[tnum]) +
	line.substring(line.indexOf('=') + 1);
      line = line_enum.nextToken();
    }
				// Extended disc info
    discExtendedInfo = "";
    while (line.startsWith(EXTD)) {
      discExtendedInfo = discExtendedInfo + line.substring(EXTD.length() + 1);
      line = line_enum.nextToken();
    }
				// Extended track info
    while (line.startsWith(EXTT)) {
      int tnum = Integer.parseInt(line.substring(EXTT.length(),
						 line.indexOf('=')));
      trackExtendedInfo[tnum] = 
	(trackExtendedInfo[tnum] == null ? "" : trackExtendedInfo[tnum]) +
	line.substring(line.indexOf('=') + 1);
      line = line_enum.nextToken();
    }
				// Play order lines
    buffer.setLength(0);
    while (line_enum.hasMoreTokens() && line.startsWith(PLAYORDER)) {
      buffer.append(",");
      buffer.append(line.substring(PLAYORDER.length() + 1));
    }
    if (buffer.length() > 1) {
      StringTokenizer order_enum = 
	new StringTokenizer(buffer.toString().substring(1), ",");
      n = order_enum.countTokens();
      playOrder = new int[n];
      for (int i = 0; i < n; i++) {
	playOrder[i] = Integer.parseInt(order_enum.nextToken().trim());
      }
    }
    int slash_pos = discTitle.indexOf('/');
    if (slash_pos >= 0) {
      String artist = discTitle.substring(0, slash_pos).trim(); 
      String title = discTitle.substring(slash_pos + 1).trim();
      discArtist = artist;
      discTitle = title;
    }
    else {
      discArtist = "";
    }
  }

  String makeMomento()
  {
    return makeMomento(null);
  }

  String makeMomento(String newAlias)
  {				// create a text string momento for the entry
				// that can be written to a file.

				// Needs to enforce cddb 80 col limit
    StringBuffer momento = new StringBuffer(4000);
    StringBuffer line = new StringBuffer(80);

    momento.append(XMCD_FIRST_LINE); momento.append(NEWLINE);
    momento.append(COMMENT); momento.append(NEWLINE);
    momento.append(TRACK_FRAME_OFFSETS); momento.append(NEWLINE);
    for (int i = 0; i < trackFrameOffsets.length; i++) {
      momento.append(COMMENT); 
      momento.append(INDENT); 
      momento.append(trackFrameOffsets[i]);
      momento.append(NEWLINE);
    }
    momento.append(COMMENT); momento.append(NEWLINE);
    
    momento.append(DISC_LENGTH); 
    momento.append(" ");
    momento.append(discLengthSeconds); 
    momento.append(" seconds"); 
    momento.append(NEWLINE);

    momento.append(COMMENT); momento.append(NEWLINE);

    momento.append(REVISION); 
    momento.append(" ");
    momento.append(revision); 
    momento.append(NEWLINE);

    momento.append(SUBMITTED_VIA);
    momento.append(" ");
    momento.append(submittedVia); 
    momento.append(NEWLINE);
    
    momento.append(COMMENT); momento.append(NEWLINE);
    
    MomentoLine id_line = new MomentoLine(DISCID);
    id_line.appendToken(cddbIDaliases[0]);
    for (int i = 1; i < cddbIDaliases.length; i++) {
      id_line.append(","); 
      id_line.appendToken(cddbIDaliases[i]);
    }
    if (newAlias != null) {
      if (cddbIDaliases.length > 0) {
	id_line.append(","); 
      }
      id_line.appendToken(newAlias);
    }
    momento.append(id_line.toString());
    momento.append(NEWLINE);
    
    MomentoLine dtitle_line = new MomentoLine(DTITLE);
    dtitle_line.appendString(discArtist + "/" + discTitle);
    momento.append(dtitle_line.toString());
    momento.append(NEWLINE);
    
    MomentoLine ttitle_line;
    for (int i = 0; i < trackTitle.length; i++) {
      ttitle_line = new MomentoLine(TTITLE + i);
      ttitle_line.appendString(trackTitle[i]);
      momento.append(ttitle_line.toString());
      momento.append(NEWLINE);
    }
    
    MomentoLine extd_line = new MomentoLine(EXTD);
    extd_line.appendString(discExtendedInfo);
    momento.append(extd_line.toString());
    momento.append(NEWLINE);

    MomentoLine extt_line;
    for (int i = 0; i < trackExtendedInfo.length; i++) {
      extt_line = new MomentoLine(EXTT + i);
      extt_line.appendString(trackExtendedInfo[i]);
      momento.append(extt_line.toString());
      momento.append(NEWLINE);
    }

    MomentoLine po_line = new MomentoLine(PLAYORDER);
    if (playOrder != null) {
      po_line.appendToken("" + playOrder[0]);
      for (int i = 0; i < playOrder.length; i++) {
	po_line.append(","); 
	po_line.appendToken("" + playOrder[i]);
      }
    }
    momento.append(po_line.toString());
    momento.append(NEWLINE);
    return momento.toString();
  }

  static public String timeToString(int t) 
  {
    t /= Drive.FRAMES_PER_SECOND;

    StringBuffer time = new StringBuffer(10);
    int mins = t / 60;
    int secs = t % 60;
    if (mins < 10) {
      time.append("0");
    }
    time.append(Integer.toString(mins));
    time.append(":");
    if (secs < 10) {
      time.append("0");
    }
    time.append(Integer.toString(secs));

    return time.toString(); 
  }
}


