package Jcd;

/**
  Jcd - Java CD Audio Player
  Copyright (c) 1996.   Michael Hamilton (michael@actrix.gen.nz).
  All rights reserved.  See the README for full details
 */

import java.io.*;
import java.util.*;
import java.awt.*;
import Jcd.SmartDrive;
import Jcd.Monitor;
import Jcd.DiscInfo;
import Jcd.JcdProps;
import Jcd.Form;

class DiscEdit extends Frame {

  // Independent track info display thread.  
  // Updates itself once a second.

  public  static final int FIELD_LEN = 40;

  public CddbEntry cddbEntry = null;

  protected DiscInfo discInfo;

  protected Form  form;

  protected TextField artistField    = new TextField(FIELD_LEN);
  protected TextField discTitleField = new TextField(FIELD_LEN);

  protected TextArea trackArea = new TextArea(15, FIELD_LEN);

  protected Choice categoryChoice = new Choice();

  protected MenuItem fileSave;
  protected MenuItem fileClose;

  protected SubPanel shortcuts = new SubPanel();
  protected SmallButton numberButton = new SmallButton("Numbers");
  protected SmallButton saveButton   = new SmallButton("Save");
  protected SmallButton closeButton  = new SmallButton("Close");

  protected boolean showNumbers = 
    JcdProps.getBoolean("jcd.show_numbers", false);
  
  public DiscEdit(DiscInfo info) 
  {
    super("Jcd Disc Edit");

    setBackground(Color.lightGray);

    setResizable(true);

    discInfo = info;

    form = new Form();

    setLayout(new GridBagLayout());

    setMenuBar(new MenuBar());
    getMenuBar().add(createFileMenu());

    add(form);

    form.addXY(new Label("Artist", Label.RIGHT), 1, 1);
    form.addXY(artistField, 2, 1, 2, 1);

    form.addXY(new Label("Title", Label.RIGHT), 1, 2);
    form.addXY(discTitleField, 2, 2, 2, 1);

    form.addXY(new Label("Category", Label.RIGHT), 1, 3);
    form.addXY(categoryChoice, 2, 3);

    for (int i = 0; i < CddbEntry.CATEGORIES.length; i++) {
      categoryChoice.addItem(CddbEntry.CATEGORIES[i]);
    }

    form.addXY(new Label("Tracks", Label.RIGHT), 1, 4);
    form.addXY(trackArea, 2, 4, 2, 15);

    setCddbEntry(info.cddbEntry);

    shortcuts.add(numberButton);
    shortcuts.add(saveButton);
    shortcuts.add(closeButton);

    form.addXY(shortcuts, 2, 20, 1, 1);

    pack();			// Resize to fit.

    show();			// Let the user have it!
  }

  public void dismiss()
  {
    hide();
    dispose();
  }

  public boolean action(Event e, Object arg)
  {
    if (e.target == fileClose || e.target == closeButton) {
      dismiss();
    }
    else if (e.target == fileSave || e.target == saveButton) {
      if (copyFromForm()) {
	try {
	  cddbEntry.save();
	}
	catch (IOException badIO) {
	  new ErrorMessage(this, 
			   "Failed to save Disc Info.",
			   "" + badIO);
	}
	discInfo.setCddbEntry(cddbEntry);
      }
    }
    else if (e.target == numberButton) {
      showNumbers = !showNumbers;
      copyToForm();
    }
    return true;
  }

  public boolean handleEvent(Event e) 
  {
    if (e.id == Event.WINDOW_DESTROY && e.target == this) {
      dismiss();
    }

    return super.handleEvent(e);
  }

  public void setCddbEntry(CddbEntry entry)
  {
    if (entry == null ||
	cddbEntry == null ||
	entry.cddbID.compareTo(cddbEntry.cddbID) != 0) {
      Monitor monitor = discInfo.cdPlayer.monitor;

      cddbEntry = entry;
      
      if (cddbEntry == null) {
	cddbEntry = new CddbEntry(monitor.cddbID, monitor.trackLength);
      }
      copyToForm();
    }
  }

  protected void copyToForm()
  {
    StringBuffer trackText = new StringBuffer(2000);
    artistField.setText(cddbEntry.discArtist);
    discTitleField.setText(cddbEntry.discTitle);
    for (int i = 0; i < CddbEntry.CATEGORIES.length; i++) {
      if (CddbEntry.CATEGORIES[i].compareTo(cddbEntry.category) == 0) {
	categoryChoice.select(i);
      }
    }
    for (int i = 0; i < cddbEntry.trackTitle.length; i++) {
      if (showNumbers) {
	trackText.append(prefix(i));
      }
      trackText.append(cddbEntry.trackTitle[i]);
      trackText.append("\n");
    }
    trackArea.setText(trackText.toString());
  }

  protected boolean copyFromForm()
  {
    String tmp;
    tmp = artistField.getText().trim();
    if (tmp.length() > 0) {
      cddbEntry.setArtist(tmp);
    }
    else {
      new ErrorMessage(this, 
		       "Artist cannot be blank.",
		       "Please enter the disc's artist.");
      return false;
    }

    tmp = discTitleField.getText().trim();
    if (tmp.length() > 0) {
      cddbEntry.setTitle(tmp);
    }
    else {
      new ErrorMessage(this, 
		       "Title cannot be blank.",
		       "Please enter the disc's title.");
      return false;
    }

    cddbEntry.setCategory(categoryChoice.getSelectedItem());

    StringTokenizer tr = new StringTokenizer(trackArea.getText(), "\n", false);

    int numTracks =  cddbEntry.trackTitle.length;
    int numTokens = tr.countTokens();
    if (numTokens == numTracks) {
      for (int i = 0; i < numTracks; i++) {
	String text = tr.nextToken();
	if (showNumbers) {
	  String pre = prefix(i);
	  if (text.startsWith(pre)) { 
	    text = text.substring(pre.length());
	  }
	}
	cddbEntry.setTrackTitle(i, text);
      }
    }
    else if (numTokens < numTracks) {
      new ErrorMessage(this,
		       "Too few (" + numTokens + ") track titles.",
		       "Should be " + numTracks + ".");
      return false;
    }
    else {
      new ErrorMessage(this,
		       "Too many (" + numTokens + ") track titles.",
		       "Should be " + numTracks + ".");
      return false;
    }
    return true;
  }

  protected Menu createFileMenu()
  {
    Menu fileMenu = new Menu("File");
    fileSave = new MenuItem("Save");
    fileClose = new MenuItem("Close");
    fileMenu.add(fileSave);
    fileMenu.add(fileClose);
    return fileMenu;
  }

  protected String prefix(int i)
  {
    return ((i < 9) ? "0" : "") + (i + 1) + ". ";
    // + CddbEntry.timeToString(cddbEntry.trackFrameOffsets[i]) + " ";
  }

}

class ErrorMessage extends Dialog {

  Form form;

  public ErrorMessage(Frame parent, String message1, String message2)
  {
    super(parent, "Disc Edit Error", false); 
    setLayout(new GridBagLayout());

    form = new Form();
    
    form.addCenter(new Label(message1));
    form.addCenter(new Label(message2));

    form.addCenter(new Button("Dismiss"));

    add(form);
    pack();
    show();
  }

  public boolean action(Event e, Object arg)
  {
    if (e.target instanceof Button) {
      hide();
      dispose();
      return true;
    }
    return false;
  }
}
