package Jcd;

/**
  Jcd - Java CD Audio Player
  Copyright (c) 1996.   Michael Hamilton (michael@actrix.gen.nz).
  All rights reserved.  See the README for full details
 */

import java.io.*;
import java.util.*;
import java.awt.*;
import Jcd.SmartDrive;
import Jcd.Monitor;
import Jcd.JcdProps;

class DiscInfo extends Form implements Observer {

  // Independent track info display thread.  
  // Updates itself once a second.

  static final boolean DEBUG = JcdProps.getBoolean("jcd.debug", false); 

  private static final String FONTNAME  = "Helvetica";
  private static final int    TEXTSIZE =  10;
  private static final Font   TEXTFONT = new Font(FONTNAME, Font.BOLD, TEXTSIZE);

  public CddbEntry cddbEntry = null;

  public boolean remoteLookup = JcdProps.getBoolean("cddb.remote",   true);
  public boolean autosave     = JcdProps.getBoolean("cddb.autosave", true);

  protected Label discTitle;
  protected Label trackTitle;

  protected SmartDrive cdPlayer;

  protected Frame frame;

  protected DiscEdit editor;

  public DiscInfo(SmartDrive drive, Frame frame) 
  {
    cdPlayer = drive;
    this.frame = frame;

				// Hack - force the size.
    discTitle  = new Label("                                                    ");
    trackTitle = new Label("                                                    ");
    discTitle.setFont(TEXTFONT);
    trackTitle.setFont(TEXTFONT);
    addDown(discTitle,  GridBagConstraints.HORIZONTAL, new Insets(0,0,0,0));
    addDown(trackTitle, GridBagConstraints.HORIZONTAL, new Insets(0,0,0,0));

    cdPlayer.monitor.addObserver(this);
  }

  public boolean handleEvent(Event e) 
  {
    if (e.target == editor && e.id == Event.WINDOW_DESTROY) {
      editor = null;
    }
    return super.handleEvent(e);
  }

  public synchronized void update(Observable o, Object arg) {
    Monitor monitor = (Monitor) o;
    if (monitor.status != Drive.STATUS_INVALID) {
      if (monitor.cdChanged) {
	cddbQuery(monitor);
      }
      if (monitor.trackChanged) {
	if (cddbEntry != null) {
	  if (editor != null) {
	    editor.setCddbEntry(cddbEntry);
	  }
	  discTitle.setText(cddbEntry.discArtist + "/" + cddbEntry.discTitle);
	  // zero origin
	  if (monitor.currentTrack > 0) {
	    trackTitle.setText(cddbEntry.trackTitle[monitor.currentTrack - 1]);
	  }
	  else { // Not audio CD, ie Data CDROM
	    discTitle.setText("");
	  }
	}
      }
    }
  }

  public synchronized void setCddbEntry(CddbEntry entry)
  {
    cddbEntry = entry;
    if (cddbEntry != null) {
      discTitle.setText(cddbEntry.discArtist + "/" + cddbEntry.discTitle);
      trackTitle.setText(cddbEntry.trackTitle[0]);
    }
    else {
      discTitle.setText("");
      trackTitle.setText("");
    }
  }

  void edit() 
  {
    editor = new DiscEdit(this);
  }

  void cddbQuery(Monitor monitor)
  {
    setCddbEntry(null);
    try {			// Try local system
      setCddbEntry(new CddbEntry(monitor.cddbID));
      return;
    }
    catch (IOException badfile) {
      discTitle.setText("Not found in local database.");
				// Continue;
    }
    catch (CddbException ignore) {
      discTitle.setText("Not found in local database.");
				// Continue;
    }

    if (remoteLookup) {		// This thread completes asynchronously.
      new RemoteQueryThread(this, monitor.cddbID, monitor.trackAddress);
    }
  }
  
  void remoteQueryCompleted(String id, CddbEntry entry)
  {
    setCddbEntry(entry);
    if (autosave && entry != null) {
      try {
	entry.save(id);
      }
      catch (IOException ioerror) {
	trackTitle.setText("Error " + ioerror);
      }
    }
  }
}

class RemoteQueryThread extends Thread
{
  // Remote queries may take a while - so do them in a separate thread.

  public String cddbServer    = JcdProps.get("cddb.server", "sunsite.unc.edu");
  public int cddbServerPort   = JcdProps.getInt("cddb.port", 888);

  DiscInfo parent; 
  Monitor monitor;
  String cddbID;
  int trackAddress[];

  RemoteQueryThread(DiscInfo parent, String id, int addresses[])
  {
    this.parent = parent;
    cddbID = id;
    trackAddress = addresses;
    start();
  }

  public void run() 
  {
    CddbEntry result = null;
    try {			// Try remote server
      parent.trackTitle.setText("Searching " + cddbServer + ":" + cddbServerPort);
	
      CddbClient cddb = new CddbClient(cddbServer, cddbServerPort);
      cddb.connect();
      Vector v = cddb.query(cddbID, trackAddress);
      
      if (v != null) {
	parent.trackTitle.setText("" + v.size() +
			   (v.size() != 1 ? " matches at " : " match at ") +
			   cddbServer);
	
	CddbMatch selected = null;
	if (v.size() == 1) {
	  selected = (CddbMatch) v.elementAt(0);
	}
	else {
	  ChooseEntry choiceWindow = new ChooseEntry(parent.frame, v);
	  selected = choiceWindow.selected;
	}
	if (selected != null) {
	  parent.trackTitle.setText("Retrieving " + selected.title);
	  result = cddb.read(selected.category, selected.cddbID);
	}
      }
      else {
	parent.trackTitle.setText("Not found at " + cddbServer);
      }
      cddb.closeConnection();
    }
    catch (CddbException c) {
      parent.trackTitle.setText("Lookup error " + c);
    }
    parent.remoteQueryCompleted(cddbID, result);
  }
}

class ChooseEntry extends Dialog {

  static final boolean DEBUG = JcdProps.getBoolean("jcd.debug", false); 

  Form form = new Form();
  CheckboxGroup choiceGroup = new CheckboxGroup();
  Checkbox choiceBox[];
  Button cancel = new Button("Cancel");
  Vector choices;
  
  boolean madeChoice = false;
  CddbMatch selected = null;

  public ChooseEntry(Frame parent, Vector elements)
  {
    super(parent, "Cddb Query Results", false); // Not Modal
    setLayout(new GridBagLayout());
    choices = elements;
    choiceBox = new Checkbox[choices.size()];

    form.addCenter(new Label("Please select the most appropriate match."));

    for (int i = 0; i < choices.size(); i++) {
      if (DEBUG) System.out.println("new choice");
      choiceBox[i] = 
	new Checkbox(((CddbMatch) (choices.elementAt(i))).category + ": " +
		     ((CddbMatch) (choices.elementAt(i))).title,
		     choiceGroup,
		     false);
      form.addDown(choiceBox[i]);
    }
    form.addCenter(cancel);
    add(form);
    pack();
    show();
    while (madeChoice == false) {
      try {
	Thread.sleep(500);
      }
      catch (InterruptedException e) { 
      }
    }
  }

  public boolean action(Event e, Object arg)
  {
    if (e.target instanceof Checkbox) {
      for (int i = 0; i < choices.size(); i++) {
	if (DEBUG) System.out.println("checking " + i);
	if (choiceBox[i].getState()) {
	  selected = (CddbMatch) choices.elementAt(i);
	  hide();
	  dispose();
	  madeChoice = true;
	  return true;
	}
      }
    } 
    else if (e.target == cancel) {
      selected = null;
      hide();
      dispose();
      madeChoice = true;
      return true;
    }
    return false;
  }
}

