#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <linux/cdrom.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include "Jcd_Drive.h"

/**
  Jcd - Java CD Audio Player
  Copyright (c) 1996.   Michael Hamilton (michael@actrix.gen.nz).
  All rights reserved.  See the README for full details
 */

/*
 * Ignoring supporting multi-disc players
 * for now, but the java class allows for it in the future.
 *
 * Converted to JNI (Apr 2000)
 */

#ifdef TRUE
#undef TRUE
#undef FALSE
#endif

#define TRUE  1
#define FALSE 0

#define FRAMES_PER_SECOND Jcd_Drive_FRAMES_PER_SECOND		
#define MAX_DEVICE_LEN 512
#define MAX_ERROR_LEN 512

#define FRAME_ADDRESS(mins, secs, frames) \
  (((mins) * FRAMES_PER_SECOND * 60) + ((secs) * FRAMES_PER_SECOND) + (frames))

static int debug;


jclass getClass(JNIEnv *env, jobject obj)
{
  static jclass myClass = NULL;
  if (myClass == NULL) {
    myClass = (*env)->GetObjectClass(env, obj);
  }
}

/*
 * Macro that produces a int getter and setter.  See example that follows.
 */

#define GETSETINT(jname) \
static jfieldID fieldID_ ## jname = NULL; \
jint get_ ## jname (JNIEnv *env, jobject obj) \
{ \
  if (debug) fprintf(stderr, "Get %s.\n", #jname); \
  if (fieldID_ ## jname == NULL) \
  { \
    fieldID_ ## jname  = (*env)->GetFieldID(env, (*env)->GetObjectClass(env, obj), #jname, "I"); \
    if (fieldID_ ## jname == NULL) { \
      fprintf(stderr, "Coudn't get field ID for %s.\n", #jname); \
    } \
  } \
  if (debug) fprintf(stderr, "Got %s.\n", #jname); \
  return (*env)->GetIntField(env, obj, fieldID_ ## jname); \
} \
 \
void set_ ## jname (JNIEnv *env, jobject obj, jint value) \
{ \
  if (debug) fprintf(stderr, "Set %s.\n", #jname); \
  if (fieldID_ ## jname == NULL) \
  { \
    if (debug) fprintf(stderr, "Get ID for %s %d %d.\n", #jname, env, obj); \
    fieldID_ ## jname = (*env)->GetFieldID(env, (*env)->GetObjectClass(env, obj), #jname, "I"); \
    if (fieldID_ ## jname == NULL) { \
      fprintf(stderr, "Coudn't get field ID for %s.\n", #jname); \
    } \
  } \
  if (debug) fprintf(stderr, "Set %s=%d.\n", #jname, value); \
  (*env)->SetIntField(env, obj, fieldID_ ## jname, value); \
} 

/*
 * The following code is an example of what the GETSETINT macro expands
 * to
 */

#if EXAMPLE_ONLY
static jfieldID fieldID_fd  = NULL ;
jint get_fd  (JNIEnv *env, jobject obj) { 
  if (fieldID_fd  == NULL ) { 
    fieldID_fd   = (*env)->GetFieldID(env,
                                      getClass(env, obj), "fd", "I"); 
    if (fieldID_fd == NULL) { 
      fprintf(stderr, "Coudn't get field ID for %s.\n", "fd"); 
    } 
  } 
  return (*env)->GetIntField(env, obj, fieldID_fd );
} 

void set_fd  (JNIEnv *env, jobject obj, jint value) {
  if (fieldID_fd  == NULL ) {
    fieldID_fd  = (*env)->GetFieldID(env, getClass(env, obj), "fd", "I");
    if (fieldID_fd == NULL) { 
      fprintf(stderr, "Coudn't get field ID for %s.\n", "fd"); 
    } 
  } 
  (*env)->SetIntField(env, obj, fieldID_fd , value); 
} 
#endif

/* 
 * Generate getters and setters for a Drive java object.
 */

GETSETINT(fd)
GETSETINT(current_track)
GETSETINT(current_index)
GETSETINT(current_address)
GETSETINT(number_of_tracks)
GETSETINT(audio_status)
GETSETINT(device_flags)


jint get_device_name (JNIEnv *env, jobject obj, char *devname) { 
  static jfieldID fieldID = NULL ;
  jstring jstr;
  const char *value;
  if (fieldID == NULL ) { 
    fieldID = (*env)->GetFieldID(env,
                                 (*env)->GetObjectClass(env, obj),
                                 "device_name", 
                                 "Ljava/lang/String;"); 
  } 
  jstr = (*env)->GetObjectField(env, obj, fieldID);
  value = (*env)->GetStringUTFChars(env, jstr, NULL);
  strncpy(devname, value, MAX_DEVICE_LEN);
  if (strlen(value) > MAX_DEVICE_LEN) {
    devname[MAX_DEVICE_LEN] = '\0';
  }
  (*env)->ReleaseStringUTFChars(env, jstr, value);
}

static void yield_player(JNIEnv *env, jobject obj);
static void take_player(JNIEnv *env, jobject obj);

static void take_player(JNIEnv *env, jobject obj)
{
  /*
   * If not already open, open the device.
   */
  if (get_fd(env, obj) == -1) {
    char device_name[MAX_DEVICE_LEN + 1];
    get_device_name(env, obj, device_name);
    if (device_name == '\0') {
      char *device = getenv("CDROM");
      if (device == NULL) {
	strcpy(device_name, "/dev/cdrom");
      }
      set_device_flags(env, obj, 
                       get_device_flags(env, obj) | ( (getenv("SBPCD") != NULL) ?
                                                      Jcd_Drive_FLAG_STOP_PLAY : 
                                                      Jcd_Drive_FLAG_NONE)); 
    }
    set_fd(env, obj, open(device_name, O_RDONLY));

    if (debug)
      fprintf(stderr, "Openned %s flags=%d fd=%d\n ", 
	      device_name, 
	      get_device_flags(env, obj),
	      get_fd(env, obj));
  }
}


static void yield_player(JNIEnv *env, jobject obj)
{
  /*
   * Close the device, making it avialable to others.
   */
  if (get_fd(env, obj) != -1) {
    close(get_fd(env, obj));
  }

  set_fd(env, obj,  -1);
  set_current_track(env, obj,  1);
  set_current_index(env, obj,  1);
  set_number_of_tracks(env, obj,  0);
  set_audio_status(env, obj,  Jcd_Drive_STATUS_INVALID);

  if (debug)
    fprintf(stderr, "closed cd device\n");
}

static jint new_status(JNIEnv *env, jobject obj)
{
  /* 
   * Try to obtain the device and query its status.
   * Could cause the tray to close with some drivers.
   */
  struct cdrom_subchnl ch;
  long stat;

  set_current_track(env, obj,  1);
  set_current_index(env, obj,  1);
  set_current_address(env, obj,  0);


  if (get_fd(env, obj) == -1) { /* See if there's a new CD */
    take_player(env, obj);
    if (get_fd(env, obj) != -1) {
      struct cdrom_tochdr tinfo;
      stat = ioctl(get_fd(env, obj), CDROMREADTOCHDR, &tinfo);
      if (stat == -1) {
	yield_player(env, obj);
	set_audio_status(env, obj,  Jcd_Drive_STATUS_INVALID);
	return (jint) get_audio_status(env, obj);
      }
      set_number_of_tracks(env, obj,  tinfo.cdth_trk1);
      if (debug)
	fprintf(stderr, "number_of_tracks=%d\n", get_number_of_tracks(env, obj));
    }
  }

  ch.cdsc_format = CDROM_MSF;
    
  stat = ioctl(get_fd(env, obj), CDROMSUBCHNL, &ch);
  if (stat == -1) {		/* Assume no CD in drive */
    yield_player(env, obj);
    set_audio_status(env, obj,  Jcd_Drive_STATUS_INVALID);
    return (jint) get_audio_status(env, obj);
  }

  set_current_track(env, obj,  ch.cdsc_trk);
  set_current_index(env, obj,  ch.cdsc_ind);

  set_current_address(env, obj,  FRAME_ADDRESS(ch.cdsc_absaddr.msf.minute,
						 ch.cdsc_absaddr.msf.second,
						 ch.cdsc_absaddr.msf.frame));

  set_audio_status(env, obj,  ch.cdsc_audiostatus);

  /* fprintf(stderr,"status=0x%x\n", audio_status); */
  return (jint) get_audio_status(env, obj);
}

static int is_open(JNIEnv *env, jobject obj)
{
  if (get_fd(env, obj) == -1) {
    /* SignalError(0, "Jcd/DriveException", "Drive device not open."); */
    return FALSE;
  }
  return TRUE;
}

static int is_available_now(JNIEnv *env, jobject obj)
{
  new_status(env, obj);  
  return is_open(env, obj);
}

void Java_Jcd_Drive_initDrive(JNIEnv *env, jobject obj)
{
  fprintf(stderr, "Initializing cd drive...\n");
  debug = getenv("JCD_DEBUG") != NULL;
  new_status(env, obj);
}

jint Java_Jcd_Drive_status(JNIEnv *env, jobject obj)
{
  if (get_fd(env, obj) == -1) {
    return get_audio_status(env, obj);
  }
  else {
    return new_status(env, obj);
  }
}

jint Java_Jcd_Drive_currentTrack(JNIEnv *env, jobject obj) 
{
  if (!is_available_now(env, obj)) { return 1; }
  return get_current_track(env, obj);
}

jint Java_Jcd_Drive_currentIndex(JNIEnv *env, jobject obj) 
{				/* Must call current track first */
  if (!is_available_now(env, obj)) { return 1; }
  return get_current_index(env, obj);
}


jint Java_Jcd_Drive_currentAddress(JNIEnv *env, jobject obj) 
{
  if (!is_available_now(env, obj)) { return 0; }
  return get_current_address(env, obj);
}


jint Java_Jcd_Drive_numberOfTracks(JNIEnv *env, jobject obj)
{
  if (!is_open(env, obj)) { return 0; }
  return get_number_of_tracks(env, obj);
}


jint Java_Jcd_Drive_trackAddress(JNIEnv *env, jobject obj, jint track)
{
  struct cdrom_tocentry tocentry;

  if (!is_open(env, obj)) { return 0; }
  tocentry.cdte_track = track;
  tocentry.cdte_format = CDROM_MSF;
  if ((ioctl(get_fd(env, obj), CDROMREADTOCENTRY, &tocentry)) == -1) {
    if (debug)
      fprintf(stderr, "tae=%d\n", track);
    SignalError(0, "Jcd/TrackAddressException", strerror(errno));
    return 0;
  }
  return FRAME_ADDRESS(tocentry.cdte_addr.msf.minute,
		       tocentry.cdte_addr.msf.second,
		       tocentry.cdte_addr.msf.frame);
}

jint Java_Jcd_Drive_trackLength(JNIEnv *env, jobject obj, jint n)
{
  int starts_at = Java_Jcd_Drive_trackAddress(env, obj, n);
  int start_of_next =
    (n >= get_number_of_tracks(env, obj)) 
    ? Java_Jcd_Drive_cdEndAddress(env, obj) 
    : Java_Jcd_Drive_trackAddress(env, obj, n + 1);
  return start_of_next - starts_at;
}


jint Java_Jcd_Drive_cdEndAddress(JNIEnv *env, jobject obj)
{
  if (!is_open(env, obj)) { return 0; }
  return Java_Jcd_Drive_trackAddress(env, obj, CDROM_LEADOUT);
}

static int cddbSum(int n) 
{				/* Sum the digits */
  int s = 0;
  while (n != 0) {
    s += n % 10;
    n = n / 10; 
  }
  return s;
}

jstring Java_Jcd_Drive_cddbID(JNIEnv *env, jobject obj)
{
  /* xjcd cddb address - see http://sunsite.unc.edu~/cddb/xjcd/ */
  char id[10] = "00000000\0\0";
  int i;
  int t = 0;
  int n = 0;

  if (!is_open(env, obj)) { return (*env)->NewStringUTF(env, id); }
  for (i = 1; i <= get_number_of_tracks(env, obj); i++) {
    n += cddbSum(Java_Jcd_Drive_trackAddress(env, obj, i) / FRAMES_PER_SECOND);
    t += (Java_Jcd_Drive_trackLength(env, obj, i) / FRAMES_PER_SECOND) ;
  }
  i = ((n % 0xff) << 24 | t << 8 | (get_number_of_tracks(env, obj)));  
  sprintf(id, "%08x", i);
  return (*env)->NewStringUTF(env, id);
}

void Java_Jcd_Drive_play(JNIEnv *env, jobject obj, 
			jint start_track, 
			jint start_index,
			jint end_track,
			jint end_index)
{
  struct cdrom_ti ti;
  long stat;

  set_current_track(env, obj,  1);

  if (!is_available_now(env, obj)) { return; }

  if (debug)
    fprintf(stderr, "play %d %d %d\n", start_track, end_track, get_number_of_tracks(env, obj));

  if (end_track == 0) {
    end_track = get_number_of_tracks(env, obj); /* Kludge for first call to play. */
  }

  if (start_track < 1 || start_track > get_number_of_tracks(env, obj)) {
    SignalError(0, "Jcd/DriveException", "Play: start track out of range.");
    return;
  }

  if (end_track < start_track || end_track > get_number_of_tracks(env, obj)) {
    SignalError(0, "Jcd/DriveException", "Play: end track out of range.");
    return;
  }

  if (!is_open(env, obj)) {
    return;
  }
    
  if ((get_device_flags(env, obj) & Jcd_Drive_FLAG_STOP_PLAY)) {
    Java_Jcd_Drive_stop(env, obj);	/* Must issue stop before play. */
  }

  if (debug)
    fprintf(stderr, "going to play %d\n", start_track);

  ti.cdti_trk0 = start_track;
  ti.cdti_ind0 = start_index;
  ti.cdti_trk1 = end_track;
  ti.cdti_ind1 = end_index;

  if (ti.cdti_ind1 == 0) {
    /* Doesn't seem to be a way of specifying end index.  But this
     * seems to work.  
     */
    ti.cdti_ind1 = get_number_of_tracks(env, obj);
  }

  stat = ioctl(get_fd(env, obj), CDROMPLAYTRKIND, &ti);
  if (stat < 0) {
    SignalError(0, "Jcd/PlayException", strerror(errno));
    return;
  }

  set_current_track(env, obj,  start_track);

  if (debug)
    fprintf(stderr, "playing %d\n", get_current_track(env, obj));
}


void Java_Jcd_Drive_stop(JNIEnv *env, jobject obj)
{
  if (!is_available_now(env, obj)) { return; }
    
  if (ioctl(get_fd(env, obj), CDROMSTOP) == -1) {
    SignalError(0, "Jcd/StopException", strerror(errno));
  }
}


void Java_Jcd_Drive_pause(JNIEnv *env, jobject obj)
{
  if (!is_available_now(env, obj)) { return; }
    
  if (get_audio_status(env, obj) != Jcd_Drive_STATUS_PLAY) {
    SignalError(0, "Jcd/PauseException", "Pause: drive isn't playing.");
    return;
  }
    
  if (ioctl(get_fd(env, obj), CDROMPAUSE) == -1) {
    SignalError(0, "Jcd/PauseException", strerror(errno));
  }
}


void Java_Jcd_Drive_resume(JNIEnv *env, jobject obj)
{
  if (!is_available_now(env, obj)) { return; }
    
  if (get_audio_status(env, obj) != Jcd_Drive_STATUS_PAUSED) {
    SignalError(0, "Jcd/ResumeException", "Resume: drive isn't paused.");
    return;
  }

  if (ioctl(get_fd(env, obj), CDROMRESUME) == -1) {
    SignalError(0, "Jcd/ResumeException", strerror(errno));
  }
}


void Java_Jcd_Drive_eject(JNIEnv *env, jobject obj)
{
  long retract = get_fd(env, obj) == -1;

  if (!is_available_now(env, obj)) { return; }

  if (retract) {
    return;
  }
    
  Java_Jcd_Drive_stop(env, obj);
    
  if (ioctl(get_fd(env, obj), CDROMEJECT) == -1) {
    SignalError(0, "Jcd/EjectException", strerror(errno));
  }

  yield_player(env, obj);
}


jint Java_Jcd_Drive_volume(JNIEnv *env, jobject obj)
{
  struct cdrom_volctrl vol;

  if (!is_available_now(env, obj)) { return; }

  if (ioctl(get_fd(env, obj), CDROMVOLREAD, &vol) == -1) {
    SignalError(0, "Jcd/VolumeException", strerror(errno));
    return 0;
  }
  return vol.channel0;
}

void Java_Jcd_Drive_setVolume(JNIEnv *env, jobject obj, jint volume)
{
  struct cdrom_volctrl vol;

  if (!is_available_now(env, obj)) { return; }

  if (volume < 0 || volume > 255) {
    SignalError(0, "Jcd/SetVolumeException", "Volume out of range.");
    return;
  }

  vol.channel0 = volume;
  vol.channel1 = volume;
  vol.channel2 = volume;
  vol.channel3 = volume;

  if (ioctl(get_fd(env, obj), CDROMVOLCTRL, &vol) == -1) {
    SignalError(0, "Jcd/SetVolumeException", strerror(errno));
  }
}


jstring Java_Jcd_Drive_productCode(JNIEnv *env, jobject obj)
{
  char upc[9] = "00000000\0";

  if (!is_available_now(env, obj)) { return (*env)->NewStringUTF(env, upc); }
    
  if (ioctl(get_fd(env, obj), CDROM_GET_UPC, upc) == -1) {
    SignalError(0, "Jcd/ProductCodeException", strerror(errno));
  }
  upc[8] = '\0';
  return (*env)->NewStringUTF(env, upc);
}

