#include <sys/types.h>
#include <dmedia/cdaudio.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <errno.h>
#include "Jcd_Drive.h"

/**
  Jcd - Java CD Audio Player
  Copyright (c) 1996.   Michael Hamilton (michael@actrix.gen.nz).
  All rights reserved.  See the README for full details
 */

/*
 * This code could be far more integrated into the Java code, but
 * I may want to used it with python (my prefered langauge), so
 * I'm avoiding this for now.  Ignoring supporting multi-disc players
 * for now, but the java class allows for it in the future.
 */

#ifdef TRUE
#undef TRUE
#undef FALSE
#endif

#define TRUE  1
#define FALSE 0

#define CDROM_LEADOUT 0

/*
 * The following are normally be by "javah Jcd.Drive" using Sun's JDK.
 * However kaffeh doesn't produce them - so we'll have to it ourselves.
 */
#ifndef Jcd_Drive_FRAMES_PER_SECOND
#  define Jcd_Drive_FRAMES_PER_SECOND 75L

#  define Jcd_Drive_STATUS_INVALID 0L
#  define Jcd_Drive_STATUS_PLAY 17L
#  define Jcd_Drive_STATUS_PAUSED 18L
#  define Jcd_Drive_STATUS_COMPLETED 19L
#  define Jcd_Drive_STATUS_ERROR 20L
#  define Jcd_Drive_STATUS_NOSTATUS 21L

#  define Jcd_Drive_FLAG_NONE 0L
#  define Jcd_Drive_FLAG_STOP_PLAY 1L
#endif

/*
 * Javah and Kaffeh differ over the type of long
 */
#ifdef Jcd_Drive_FRAMES_PER_SECOND
#  define Java_Int long		/* javah */
#else
#  define Java_Int jint		/* kaffeh */
#endif


#define FRAMES_PER_SECOND Jcd_Drive_FRAMES_PER_SECOND		
#define MAX_DEVICE_LEN 512
#define MAX_ERROR_LEN 512
#define MAX_DRIVES 100

#define FRAME_ADDRESS(mins, secs, frames) \
  (((mins) * FRAMES_PER_SECOND * 60) + ((secs) * FRAMES_PER_SECOND) + (frames))

static int debug;

static void yield_player(struct HJcd_Drive *drive);
static void take_player(struct HJcd_Drive *drive);

CDPLAYER *drive_ptr[MAX_DRIVES] = { NULL } ;

static Java_Int cd_length = 0;

static void take_player(struct HJcd_Drive *drive)
{
  int i, fd;
  /*
   * If not already open, open the device.
   */
  if (unhand(drive)->fd == -1) {
    char device_name[MAX_DEVICE_LEN + 1];
    javaString2CString(unhand(drive)->device_name,device_name,MAX_DEVICE_LEN);
    if (device_name == '\0') {
      char *device = getenv("CDROM");
      if (device == NULL) {
	strcpy(device_name, "/dev/cdrom");
      }
    }

    unhand(drive)->fd = -1;

    for (i = 0; i < MAX_DRIVES; i++) {
      if (drive_ptr[i] == NULL) {
	unhand(drive)->fd = i;
	break;
      }
    }
    
    if (unhand(drive)->fd != -1) {
      drive_ptr[unhand(drive)->fd] = 
	CDopen(strcmp(device_name,"NULL") == 0 ? NULL : device_name, "r");
      if (drive_ptr[unhand(drive)->fd] == NULL) {
	unhand(drive)->fd = -1;
      }
    }

    if (debug)
      fprintf(stderr, "Openned %s flags=%d fd=%d\n ", 
	      device_name, 
	      unhand(drive)->device_flags,
	      unhand(drive)->fd);
  }
}


static void yield_player(struct HJcd_Drive *drive)
{

}

static Java_Int new_status(struct HJcd_Drive *drive)
{
 
  static const int state_trans[] = {
    Jcd_Drive_STATUS_ERROR,
    Jcd_Drive_STATUS_INVALID,
    Jcd_Drive_STATUS_COMPLETED,
    Jcd_Drive_STATUS_PLAY,    
    Jcd_Drive_STATUS_PAUSED,
    Jcd_Drive_STATUS_INVALID,
    Jcd_Drive_STATUS_INVALID,
  };
  
  int stat;
  CDSTATUS cdstatus;
  /* 
   * Try to obtain the device and query its status.
   * Could cause the tray to close with some drivers.
   */
  unhand(drive)->current_track   = 1;
  unhand(drive)->current_index   = 1;
  unhand(drive)->current_address = 0;


  if (unhand(drive)->fd == -1) { /* See if there's a new CD */
    take_player(drive);
    if (unhand(drive)->fd == -1) {
      return Jcd_Drive_STATUS_INVALID;
    }
  }
  stat = CDgetstatus(drive_ptr[unhand(drive)->fd], &cdstatus);

  unhand(drive)->current_track = cdstatus.track;
  unhand(drive)->current_index = 1;
  unhand(drive)->number_of_tracks = cdstatus.last;

  cd_length = FRAME_ADDRESS(cdstatus.total_min,
			    cdstatus.total_sec,
			    cdstatus.total_frame);

  unhand(drive)->current_address = FRAME_ADDRESS(cdstatus.abs_min,
						 cdstatus.abs_sec,
						 cdstatus.abs_frame);

  unhand(drive)->audio_status = state_trans[cdstatus.state];

  /* fprintf(stderr,"status=0x%x\n", audio_status); */
  return (Java_Int) unhand(drive)->audio_status;
}

static int is_open(struct HJcd_Drive *drive)
{
  if (unhand(drive)->fd == -1) {
    /* SignalError(0, "Jcd/DriveException", "Drive device not open."); */
    return FALSE;
  }
  return TRUE;
}

static int is_available_now(struct HJcd_Drive *drive)
{
  new_status(drive);  
  return is_open(drive);
}

void Jcd_Drive_initDrive(struct HJcd_Drive *drive)
{
  fprintf(stderr, "Initializing cd drive...\n");
  debug = getenv("JCD_DEBUG") != NULL;
  new_status(drive);
}

Java_Int Jcd_Drive_status(struct HJcd_Drive *drive)
{
  if (unhand(drive)->fd == -1) {
    return unhand(drive)->audio_status;
  }
  else {
    return new_status(drive);
  }
}

Java_Int Jcd_Drive_currentTrack(struct HJcd_Drive *drive) 
{
  if (!is_available_now(drive)) { return 1; }
  return unhand(drive)->current_track;
}

Java_Int Jcd_Drive_currentIndex(struct HJcd_Drive *drive) 
{				/* Must call current track first */
  if (!is_available_now(drive)) { return 1; }
  return unhand(drive)->current_index;
}


Java_Int Jcd_Drive_currentAddress(struct HJcd_Drive *drive) 
{
  if (!is_available_now(drive)) { return 0; }
  return unhand(drive)->current_address;
}


Java_Int Jcd_Drive_numberOfTracks(struct HJcd_Drive *drive)
{
  if (!is_open(drive)) { return 0; }
  return unhand(drive)->number_of_tracks;
}


Java_Int Jcd_Drive_trackAddress(struct HJcd_Drive *drive, Java_Int track)
{
  CDTRACKINFO trackinfo;

  if (!is_open(drive)) { return 0; }

  if (track == CDROM_LEADOUT) {
    new_status(drive); 
    return cd_length;
  }

  CDgettrackinfo(drive_ptr[unhand(drive)->fd], track, &trackinfo);

  return FRAME_ADDRESS(trackinfo.start_min,
		       trackinfo.start_sec,
		       trackinfo.start_frame);
}

Java_Int Jcd_Drive_trackLength(struct HJcd_Drive *drive, Java_Int n)
{
  CDTRACKINFO trackinfo;

  if (!is_open(drive)) { return 0; }

  if (n == CDROM_LEADOUT) {
    new_status(drive); 
    return cd_length;
  }

  CDgettrackinfo(drive_ptr[unhand(drive)->fd], n, &trackinfo);

  return FRAME_ADDRESS(trackinfo.total_min,
		       trackinfo.total_sec,
		       trackinfo.total_frame);
}


Java_Int Jcd_Drive_cdEndAddress(struct HJcd_Drive *drive)
{
  if (!is_open(drive)) { return 0; }
  return Jcd_Drive_trackAddress(drive, CDROM_LEADOUT);
}

static int cddbSum(int n) 
{				/* Sum the digits */
  int s = 0;
  while (n != 0) {
    s += n % 10;
    n = n / 10; 
  }
  return s;
}

struct Hjava_lang_String *Jcd_Drive_cddbID(struct HJcd_Drive *drive)
{
  /* xjcd cddb address - see http://sunsite.unc.edu~/cddb/xjcd/ */
  char id[10] = "00000000";
  int i;
  int t = 0;
  int n = 0;

  if (!is_open(drive)) { return makeJavaString(id, 8); }
  for (i = 1; i <= unhand(drive)->number_of_tracks; i++) {
    n += cddbSum(Jcd_Drive_trackAddress(drive, i) / FRAMES_PER_SECOND);
    t += (Jcd_Drive_trackLength(drive, i) / FRAMES_PER_SECOND) ;
  }
  i = ((n % 0xff) << 24 | t << 8 | (unhand(drive)->number_of_tracks));  
  sprintf(id, "%08x", i);
  return makeJavaString(id, 8);
}

void Jcd_Drive_play(struct HJcd_Drive *drive, 
			Java_Int start_track, 
			Java_Int start_index,
			Java_Int end_track,
			Java_Int end_index)
{
  long stat;

  unhand(drive)->current_track = 1;

  if (!is_available_now(drive)) { return; }

  if (debug)
    fprintf(stderr, "play %d %d %d\n", start_track, end_track, unhand(drive)->number_of_tracks);

  if (end_track == 0) {
    end_track = unhand(drive)->number_of_tracks; /* Kludge for first call to play. */
  }

  if (start_track < 1 || start_track > unhand(drive)->number_of_tracks) {
    SignalError(0, "Jcd/DriveException", "Play: start track out of range.");
    return;
  }

  if (end_track < start_track || end_track > unhand(drive)->number_of_tracks) {
    SignalError(0, "Jcd/DriveException", "Play: end track out of range.");
    return;
  }

  if (!is_open(drive)) {
    return;
  }
    
  if ((unhand(drive)->device_flags & Jcd_Drive_FLAG_STOP_PLAY)) {
    Jcd_Drive_stop(drive);	/* Must issue stop before play. */
  }

  if (debug)
    fprintf(stderr, "going to play %d\n", start_track);

  if (end_track == start_track + 1) {
    CDplaytrack(drive_ptr[unhand(drive)->fd], start_track, 1);
  }
  else {
    stat = CDplay(drive_ptr[unhand(drive)->fd], start_track, 1);
  }
  
  if (stat == 0) {
    SignalError(0, "Jcd/PlayException", strerror(errno));
    return;
  }

  unhand(drive)->current_track = start_track;

  if (debug)
    fprintf(stderr, "playing %d\n", unhand(drive)->current_track);
}


void Jcd_Drive_stop(struct HJcd_Drive *drive)
{
  if (!is_available_now(drive)) { return; }
    
  if (CDstop(drive_ptr[unhand(drive)->fd]) == 0) {
    SignalError(0, "Jcd/StopException", strerror(errno));
  }
}


void Jcd_Drive_pause(struct HJcd_Drive *drive)
{
  if (!is_available_now(drive)) { return; }
    
  if (unhand(drive)->audio_status != Jcd_Drive_STATUS_PLAY) {
    SignalError(0, "Jcd/PauseException", "Pause: drive isn't playing.");
    return;
  }
    
  if (CDtogglepause(drive_ptr[unhand(drive)->fd]) == 0) {
    SignalError(0, "Jcd/PauseException", strerror(errno));
  }
}


void Jcd_Drive_resume(struct HJcd_Drive *drive)
{
  if (!is_available_now(drive)) { return; }
    
  if (unhand(drive)->audio_status != Jcd_Drive_STATUS_PAUSED) {
    SignalError(0, "Jcd/ResumeException", "Resume: drive isn't paused.");
    return;
  }

  if (CDtogglepause(drive_ptr[unhand(drive)->fd]) == 0) {
    SignalError(0, "Jcd/ResumeException", strerror(errno));
  }
}


void Jcd_Drive_eject(struct HJcd_Drive *drive)
{
  long retract = unhand(drive)->fd == -1;

  if (!is_available_now(drive)) { return; }

  if (retract) {
    return;
  }
    
  Jcd_Drive_stop(drive);
    
  if (CDeject(drive_ptr[unhand(drive)->fd]) == 0) {
    SignalError(0, "Jcd/EjectException", strerror(errno));
  }

  yield_player(drive);
}


Java_Int Jcd_Drive_volume(struct HJcd_Drive *drive)
{
  CDVOLUME cdvolume;

  if (CDgetvolume(drive_ptr[unhand(drive)->fd], &cdvolume) == 0) {
    SignalError(0, "Jcd/VolumeException", strerror(errno));
    return 0;
  }
  
  return cdvolume.chan0;
}

void Jcd_Drive_setVolume(struct HJcd_Drive *drive, Java_Int volume)
{
  CDVOLUME cdvolume;

  cdvolume.chan0 = volume;
  cdvolume.chan1 = volume;
  cdvolume.chan2 = volume;
  cdvolume.chan3 = volume;

  if (CDsetvolume(drive_ptr[unhand(drive)->fd], &cdvolume) == 0) {
    SignalError(0, "Jcd/SetVolumeException", strerror(errno));
  }
  
}


struct Hjava_lang_String * Jcd_Drive_productCode(struct HJcd_Drive *drive)
{
  char upc[9] = "00000000";

  return makeJavaString(upc, 8);
}

