package Jcd;
  
/**
  Jcd - Java CD Audio Player
  Copyright (c) 1996.   Michael Hamilton (michael@actrix.gen.nz).
  All rights reserved.  See the README for full details
 */

import java.lang.*;
import java.util.*;
import Jcd.Drive;
import Jcd.JcdProps;

class Monitor extends Observable implements Runnable {

  // Updates once a second - calls client object's cdStatusEvent()
  // methods, so they can reflect the updates.

  // The following public status info is available for the client to use.

  static final boolean DEBUG = JcdProps.getBoolean("jcd.debug", false); 

  public Drive cdPlayer;

  public int status = Drive.STATUS_INVALID;

  public int currentTrack = 0;
  public int currentIndex = 0;
  public int currentAddress = 0; 
  public int trackStartAddress = 0; 
  public int trackElapsed = 0;	
  
  public boolean cdChanged = false;
  public boolean trackChanged = false;

  public int cdEndAddress = -1; 
  public int numberOfTracks = 0;

				// Info orgin [1], [0] is total for CD.
  public int trackAddress[] = {0};
  public int trackEndAddress[] = {0};
  public int trackLength[] = {0};

  public String cddbID = "00000000";

  protected int lastTrack = 0;


  protected Thread updateThread = null;

  public Monitor(Drive cdPlayer) {
    this.cdPlayer = cdPlayer;
  }

  public void run() {

    for (;;) {
      int sleep = 1000;
      synchronized (cdPlayer) {
	updateCdInfo();
	setChanged();		// Force notifyObservers() to do its thing.
	notifyObservers();
      }
      try {
	Thread.sleep(sleep); 
      }
      catch (InterruptedException e) {
      }
    }
  }

  public void start() {
    if (updateThread == null) {
      updateThread = new Thread(this);
      updateThread.setPriority(1);
      updateThread.start();
    }
  }
  
  public void stop() {
    if (updateThread != null && updateThread.isAlive()) {
      updateThread.stop();
    }
    updateThread = null;
  }

  protected void updateCdInfo()
  {
    try {
      cdChanged = false;
      trackChanged = false;
      status = cdPlayer.status();
      if (status != Drive.STATUS_INVALID) {
	int cdEnd = cdPlayer.cdEndAddress();

	if (cdEnd != cdEndAddress) { 
	  // Assumes we never get two cd's the
	  // exact same length in a row!  Breaks if we do!

	  cdEndAddress = cdEnd;
	  cdChanged = true;
	  trackChanged = true;
	  
	  cddbID = cdPlayer.cddbID();
	  if (DEBUG) System.out.println("CD Changed " + cddbID);

	  numberOfTracks = cdPlayer.numberOfTracks();
	  
	  trackAddress    = new int[numberOfTracks + 1];
	  trackEndAddress = new int[numberOfTracks + 1];
	  trackLength     = new int[numberOfTracks + 1];
	  
	  for (int i = 1; i <= numberOfTracks; i++) {
	    trackAddress[i]    = cdPlayer.trackAddress(i);
	    // System.out.print(" " + trackAddress[i]);
	    trackEndAddress[i] = cdPlayer.trackEndAddress(i);
	    trackLength[i]     = cdPlayer.trackLength(i);
	  }
	  trackAddress[0]    = cdPlayer.trackAddress(Drive.LEAD_OUT);
	  trackEndAddress[0] = cdPlayer.trackEndAddress(Drive.LEAD_OUT);
	  trackLength[0]     = cdPlayer.trackLength(Drive.LEAD_OUT);
	  // System.out.println(" " + (trackEndAddress[0] / Drive.FRAMES_PER_SECOND));
	}

	currentTrack      = cdPlayer.currentTrack();
	if (currentTrack > numberOfTracks) { // Fishy - probably at end of CD.
	  currentTrack = numberOfTracks; // Prevent subscript out of bounds.
	}
	if (currentTrack == 0) { // Just in case
	  currentTrack = 1;
	}
	currentIndex      = cdPlayer.currentIndex();
	currentAddress    = cdPlayer.currentAddress();
	trackStartAddress = cdPlayer.trackAddress(currentTrack);
	trackElapsed      = currentAddress - trackStartAddress;
      if (trackElapsed < 0) {
        trackElapsed    = 0;
      }
	
	if (currentTrack != lastTrack) {
	  trackChanged = true;
	  lastTrack = currentTrack;
	}
      }
    }
    catch (DriveException e) {
      System.out.println("Monitor Exception " + e);
    }
  }
}
