package Jcd;

/**
  Jcd - Java CD Audio Player
  Copyright (c) 1996.   Michael Hamilton (michael@actrix.gen.nz).
  All rights reserved.  See the README for full details
 */

import java.io.*;
import java.util.*;
import java.awt.*;

import Jcd.Program;
import Jcd.Display;
import Jcd.Controls;
import Jcd.DiscInfo;
import Jcd.SmartDrive;

public class Player extends Frame { 		// Main window.

  static final String LIB_DIR = JcdProps.get("jcd.lib", "/usr/local/lib/jcd") + "/";

  static final String ICON_GIF_FILE  = LIB_DIR + "/images/jcd.gif";

  static final String DEFAULT_DEVICE = "/dev/cdrom";
				// Arch and OS dependent native module name.
  static final String DEFAULT_MODULE = LIB_DIR 
    + "/native/"
    + System.getProperty("os.arch")
    + "-"
    + System.getProperty("os.name")
    + "/Jcd_Drive.so";

  SmartDrive cdPlayer;		// Hardware interface.
  Program program;

  Form form = new Form();

  Display  display;		// Components of the interface.
  DiscInfo discInfo;
  Controls controls;

  protected Scrollbar volume = new Scrollbar(Scrollbar.HORIZONTAL,
					     127, 10, 0, 255);

  protected SubPanel shortcuts = new SubPanel();

  protected SmallButton programButton = new SmallButton("Program");
  protected SmallButton singleButton  = new SmallButton("Cont.");
  protected SmallButton timeButton    = new SmallButton("Time");
  protected SmallButton exitButton    = new SmallButton("Exit");


  protected MenuItem fileDiscEdit = new MenuItem("Edit");
  protected MenuItem fileProgram  = new MenuItem("Program");
  protected MenuItem fileExit     = new MenuItem("Exit");

  protected CheckboxMenuItem optionAutosave  = new CheckboxMenuItem("Autosave");
  protected CheckboxMenuItem optionRemote    = new CheckboxMenuItem("Remote lookup");

  protected MenuItem showTrackTime = new MenuItem("Track elapsed");
  protected MenuItem showTrackRem  = new MenuItem("Track remaining");
  protected MenuItem showDiscRem   = new MenuItem("Disc remaining");

  protected MenuItem playCont   = new MenuItem("Continuously");
  protected MenuItem playSingle = new MenuItem("Single tracks");

  protected MenuItem aboutJcd =  new MenuItem("About Jcd");

  // 
  // 1. Connect to a cdrom device.
  // 2. Create the interface: 
  //      the display-panel;
  //      the control-panel;
  // 3. Establish the panel components as clients of the monitor.
  //

  public Player() 
  {
    super(JcdProps.get("jcd.version", JcdProps.JCD_VERSION));

    setBackground(Color.lightGray);

    setIconImage(Toolkit.getDefaultToolkit().getImage(ICON_GIF_FILE));

    setResizable(true);

    String device = JcdProps.get("jcd.device", Player.DEFAULT_DEVICE);
    String module = JcdProps.get("jcd.module", Player.DEFAULT_MODULE);
    int flags     = JcdProps.getInt("jcd.device_flags", 0);

    setLayout(new GridBagLayout());

    setMenuBar(new MenuBar());
    getMenuBar().add(createFileMenu());
    getMenuBar().add(createOptionsMenu());
    getMenuBar().setHelpMenu(createHelpMenu());
    
    cdPlayer = new SmartDrive(device, module, flags);

    display  = new Display(cdPlayer);
    discInfo = new DiscInfo(cdPlayer, this);
    controls = new Controls(cdPlayer, display);

    form.addCenter(discInfo, GridBagConstraints.HORIZONTAL);
    form.addCenter(display);
    form.addCenter(controls);

    form.addDown(volume, GridBagConstraints.HORIZONTAL, new Insets(10,10,2,10));

    shortcuts.add(programButton);
    shortcuts.add(singleButton);
    shortcuts.add(timeButton);
    shortcuts.add(exitButton);

    form.addCenter(shortcuts);

    add(form);

    pack();			// Resize to fit.

    show();			// Let the user have it!

				// Now everyone is ready...
    cdPlayer.monitor.start();	// ...start processing events
  }

  public boolean action(Event e, Object arg)
  {
    if (e.target == programButton || e.target == fileProgram) {
      if (program != null) {
	program.dismiss();
      }
      program = new Program(cdPlayer);
    }
    else if (e.target == singleButton) {
      setSingleMode(!cdPlayer.singleMode);
    }
    else if (e.target == playCont) {
      setSingleMode(false);
    }
    else if (e.target == playSingle) {
      setSingleMode(true);
    }
    else if (e.target == timeButton) {
      display.setTimeMode((display.timeMode + 1) % Display.NUM_TIME_MODES);
    }
    else if (e.target == showTrackTime) {
      display.setTimeMode(0);
    }
    else if (e.target == showTrackRem) {
      display.setTimeMode(1);
    }
    else if (e.target == showDiscRem) {
      display.setTimeMode(2);
    }
    else if (e.target == exitButton || e.target == fileExit) {
      System.exit(0);
    }
    else if (e.target == fileDiscEdit) {
      discInfo.edit();
    }
    else if (e.target == optionAutosave) {
      discInfo.autosave = !discInfo.autosave;
      optionAutosave.setState(discInfo.autosave);
      // System.out.println("autosave toggle " + discInfo.autosave);
    }
    else if (e.target == optionRemote) {
      discInfo.remoteLookup = !discInfo.remoteLookup;
      optionRemote.setState(discInfo.remoteLookup); 
      // System.out.println("Remote lookup toggle " + discInfo.remoteLookup);
    }
    else if (e.target == aboutJcd) {
      new AboutJcd(this);
    }
    else {
      return false;
    }
    return true;  
  }

  public boolean handleEvent(Event e) 
  {
    try {
      if (e.id == Event.WINDOW_DESTROY && e.target == this) {
        System.exit(0);
      } else if (e.target == volume) {
	int new_vol;
	switch (e.id) {
	case Event.SCROLL_LINE_UP:
	case Event.SCROLL_PAGE_UP:
	case Event.SCROLL_LINE_DOWN:
	case Event.SCROLL_PAGE_DOWN:
	case Event.SCROLL_ABSOLUTE:
	  new_vol = ((Integer) e.arg).intValue();
	  cdPlayer.setVolume(new_vol);
	  return true;
	default:
	  break;
	}
      }
    }
    catch (DriveException except) {
      System.out.println("Exception " + except);
    }
    return super.handleEvent(e);
  }

  protected void setSingleMode(boolean state) 
  {
    cdPlayer.setSingleMode(state);
    singleButton.setLabel(state ? "Single" : "Cont.");
  }

  protected Menu createFileMenu()
  {
    Menu fileMenu = new Menu("File");
    fileMenu.add(fileDiscEdit);
    fileMenu.add(fileProgram);
    fileMenu.addSeparator();
    fileMenu.add(fileExit);
    return fileMenu;
  }

  protected Menu createOptionsMenu()
  {
    Menu optionMenu = new Menu("Options");

    Menu optionTime = new Menu("Show time");
    Menu optionPlay = new Menu("Play");

    optionAutosave.setState(true);
    optionMenu.add(optionAutosave);
    optionRemote.setState(true);
    optionMenu.add(optionRemote);

    optionMenu.addSeparator();
    optionMenu.add(optionTime);

    optionMenu.addSeparator();
    optionMenu.add(optionPlay);

    optionTime.add(showTrackTime);
    optionTime.add(showTrackRem);
    optionTime.add(showDiscRem);

    optionPlay.add(playCont);
    optionPlay.add(playSingle);

    return optionMenu;
  }

  protected Menu createHelpMenu()
  {
    Menu helpMenu = new Menu("Help");
    helpMenu.add(aboutJcd);
    return helpMenu;
  }

  public void set_program(Program p)
  {
    program = p;
  }
}


