package Jcd;

/**
  Jcd - Java CD Audio Player
  Copyright (c) 1996.   Michael Hamilton (michael@actrix.gen.nz).
  All rights reserved.  See the README for full details
 */

import java.io.*;
import java.util.*;
import java.awt.*;

import Jcd.Drive;
import Jcd.Monitor;

class Program extends Frame implements Observer {

  // Independent status display thread.  
  // Updates itself once a second.

  private static final int INITIAL_BUTTONS = 20; // Num Track buttons
  private static final int LISTING_SIZE = 26;    // Field width
  private static final int GRID_SIZE = 8;        // Button Grid

  private static final int ADD_MODE = 0;
  private static final int DEL_MODE = 1;
  private static final int PLAY_MODE  = 2; 

  protected static final boolean selectCurrent = JcdProps.getBoolean("jcd.hilight_program_step", true);

  protected SmartDrive cdPlayer;

  protected LEDfield programListing;

  protected Form     form;
  protected SubPanel trackPanel;
  protected SubPanel buttonPanel;
  protected SubPanel loadSavePanel;

  protected SmallButton editButton;
  protected SmallButton shuffleButton;
  protected SmallButton resetButton;
  protected SmallButton clearButton;
  protected SmallButton dismissButton;

  protected SmallButton loadButton;
  protected SmallButton saveButton;


  protected int mode = 0; // 0 => add, 1 => delete, 2 => play

  public Program(SmartDrive drive) 
  {
    super("Jcd Program");

    int n;

    setBackground(Color.lightGray);

    cdPlayer = drive;

    form = new Form();

    setLayout(new GridBagLayout());

    add(form);

    programListing = new LEDfield("Current Program", 
				  Program.LISTING_SIZE); 

    programListing.setEditable(false);

    form.addCenter(programListing, GridBagConstraints.HORIZONTAL);

    trackPanel = new SubPanel();
    form.addCenter(trackPanel);

    editButton    = new SmallButton("Add");
    shuffleButton = new SmallButton("Shuffle");
    resetButton   = new SmallButton("Reset");
    clearButton   = new SmallButton("Clear");
    dismissButton = new SmallButton("Dismiss");

    buttonPanel = new SubPanel();
    
    buttonPanel.add(editButton);
    buttonPanel.add(shuffleButton);
    buttonPanel.add(resetButton);
    buttonPanel.add(clearButton);
    buttonPanel.add(dismissButton);

    form.addCenter(buttonPanel);

    loadButton = new SmallButton("Load");
    saveButton = new SmallButton("Save");

    loadSavePanel = new SubPanel();

    loadSavePanel.add(loadButton);
    loadSavePanel.add(saveButton);

    form.addCenter(loadSavePanel);

    n = cdPlayer.monitor.numberOfTracks;
    updateTrackPanel(n > 0 ? n : Program.INITIAL_BUTTONS);

    cdPlayer.monitor.addObserver(this);

    show();
  }

  public void shuffle()
  {
    int n = cdPlayer.monitor.numberOfTracks;
    Vector choices = new Vector(n);
    Random random = new Random();

    cdPlayer.tracksToPlay.clear();

    for (int i=1; i <= n; i++) { // Make a list of all choices.
      choices.addElement(new Integer(i));
    }
    
    for (int i=1; i <= n; i++) { 
				// Remove choices at random until none
				// are left.
      int which = (int) (random.nextFloat() * (choices.size()));
      cdPlayer.tracksToPlay.addTrack(((Integer) choices.elementAt(which)).intValue());
      choices.removeElementAt(which);
    }
    displayProgram();
  }

  public void dismiss()
  {
    hide();
    dispose();
  }

  public void update(Observable o, Object arg)
  {
    if (cdPlayer.monitor.cdChanged) {
      updateTrackPanel(cdPlayer.monitor.numberOfTracks);
    }
    if (cdPlayer.monitor.status == Drive.STATUS_PLAY
	&& cdPlayer.monitor.trackChanged) { // track changed?
      displayProgram();
    }
  }
  
  public boolean action(Event e, Object arg) 
  {
    showError("Current Program");
    if (e.target == editButton) {
      setEditMode();
    }
    else if (e.target == dismissButton) {
      dismiss();
    }
    else if (e.target == resetButton) {
      cdPlayer.tracksToPlay.reset();
      displayProgram();
    }
    else if (e.target == clearButton) {
      cdPlayer.tracksToPlay.clear();
      displayProgram();
    }
    else if (e.target == shuffleButton) {
      shuffle();
      displayProgram();
    }
    else if (e.target == loadButton) {
      try {
        String filename = 
            System.getProperty("user.home") +
            "/JcdPlaylistDatabase/" +
            cdPlayer.cddbID() + ".txt";
      
        try {
          cdPlayer.monitor.updateCdInfo();
          cdPlayer.tracksToPlay.load(filename);
          displayProgram();
        }
        catch (IOException ioe) {
          showError(ioe.getMessage());
        }
      }
      catch (DriveException dex) {
        showError(dex.getMessage());
      }
    }
    else if (e.target == saveButton) {
      try {
        String filename = 
            System.getProperty("user.home") +
            "/JcdPlaylistDatabase/" +
            cdPlayer.cddbID() + ".txt";

        try {
          cdPlayer.monitor.updateCdInfo();
          cdPlayer.tracksToPlay.save(filename);
          displayProgram();
        } 
        catch (IOException ioe) {
          showError(ioe.getMessage());
        }
      }
      catch (DriveException dex) {
        showError(dex.getMessage());
      }
    }
    else if (e.target instanceof Button) {
				// Any other kind of button is a track
				// button.
      if (mode == ADD_MODE) {
	cdPlayer.tracksToPlay.addTrack(Integer.parseInt(((Button) e.target).getLabel()));
      }
      else if (mode == DEL_MODE) {
	cdPlayer.tracksToPlay.removeTrack(Integer.parseInt(((Button) e.target).getLabel()));
      }
      else if (mode == PLAY_MODE) {
	int t = Integer.parseInt(((Button) e.target).getLabel());
	cdPlayer.tracksToPlay.skipTo(t);
	try {
	  cdPlayer.play(t);
	}
	catch (DriveException except) {
	    System.out.println("Exception " + except);
	}
      }
      displayProgram();
    }
    else {
      return false;
    }
    return true;
  }

  public boolean handleEvent(Event e)
  {
    if (e.id == Event.WINDOW_DESTROY && e.target == this) {
      dismiss();
    }
    return super.handleEvent(e);
  }

  protected void displayProgram()
  {

    String str =
      cdPlayer.monitor.cdChanged ? "[]" : cdPlayer.tracksToPlay.toString();

    programListing.setText(str.substring(1, str.length() - 1));
    // System.out.println("-->" + str);
				// Highlight current track
    if (selectCurrent) {
      if (cdPlayer.tracksToPlay.atStart()
	  || cdPlayer.tracksToPlay.atEnd()) {
	programListing.select(0,0); // No current track
      }
      else {			// Find it by counting comma's.
	int start = 0;
	int end;
	for (int i = 0; i < cdPlayer.tracksToPlay.index; i++) {
	  start = str.indexOf(',', start) + 1;
	}
	end = str.indexOf(',', start + 1) - 1;
	if (end <= 0) {
	  end = str.length() - 1;
	}
	programListing.select(start,end);
      }
    }
  }

  protected void setEditMode()
  {
    mode++;
    if (mode > PLAY_MODE) {
      mode = ADD_MODE;
    }
    switch (mode) {
    case ADD_MODE:
      editButton.setLabel("Add ");
      break;
    case DEL_MODE:
      editButton.setLabel("Del ");
      break;
    case PLAY_MODE:
      editButton.setLabel("Play");
      break;
    }
  }

  protected void updateTrackPanel(int n)
  {

    Button trackButton;

    if (n != trackPanel.countComponents()) {
      int prev_n = trackPanel.countComponents();
      Component button[] = trackPanel.getComponents();
      trackPanel.setLayout(new GridLayout(n / Program.GRID_SIZE + 1,
					  Program.GRID_SIZE));
      for (int i = n; i < prev_n; i++) {
	trackPanel.remove(button[i]);
      }
      for (int i = prev_n; i < n; i++) {
	trackPanel.add(new Button(Integer.toString(i + 1)));
      }
    }
    displayProgram();
    pack();
  }

  protected void showError(String err)
  {
    programListing.setLabel(err);
  }
}
