package Jcd;

/**
  Jcd - Java CD Audio Player
  Copyright (c) 1996.   Michael Hamilton (michael@actrix.gen.nz).
  All rights reserved.  See the README for full details
 */

import java.io.*;
import java.util.*;
import java.awt.*;
import Jcd.Drive;
import Jcd.Monitor;
import Jcd.JcdProps;

class SmartDrive extends Drive implements Observer {

  static final boolean DEBUG = JcdProps.getBoolean("jcd.debug", false); 

  public Monitor monitor;

  public TrackList tracksToPlay = null;
  public int program_pos = 0;
  public boolean singleMode = false;


  public SmartDrive(String device, String module, int flags) 
  {
    super(device, module, flags);
    setUp();
  }

  protected synchronized void setUp()
  {
    monitor = new Monitor(this);
    monitor.addObserver(this);
    tracksToPlay = new TrackList(30);
  }

  public synchronized void setSingleMode(boolean state)
  {
    singleMode = state;
  }

  public synchronized void play(int track) throws DriveException
  {
    if (singleMode) {
      single(track);
    } 
    else {
      super.play(track);
    }
  }

  public synchronized void eject() throws DriveException
  {
    super.eject();
  }

  public synchronized boolean startPlaying() throws DriveException
  {
    if (status() == Drive.STATUS_PAUSED) {
      resume();
    }
    else if (!tracksToPlay.isEmpty()) {
      if (tracksToPlay.atEnd()) {
	tracksToPlay.reset();
      }
      play(tracksToPlay.nextTrack());
    }
    else {
      play(1);
    }
    return true;
  }

  public synchronized boolean next() throws DriveException
  {
    if (tracksToPlay.isEmpty()) {
      int track = currentTrack();
      int max = numberOfTracks();
      if (track >= max) {
	return false;
      }
      play(track + 1);
    }
    else {
      int next = tracksToPlay.nextTrack();
      if (next <= 0) {
	return false;
      }
      play(next);
    }
    return true;
  }

  public synchronized boolean prev() throws DriveException
  {
    if (tracksToPlay.isEmpty()) {
      int track = currentTrack();
      if (track <= 1) {
	return false;
      }
      play(track - 1);
    }
    else {
      int next = tracksToPlay.prevTrack();
      if (next <= 0) {
	return false;
      }
      play(next);
    }
    return true;
  }

  public synchronized int remaining() 
  {				// Should this method belong to Monitor?
    int result = 0;
    if (tracksToPlay.isEmpty()) {
      result = monitor.cdEndAddress - monitor.currentAddress;
    }
    else {			// Should really do some caching here.
				// But no serious impact on CPU, so lets be
				// lazy for now.
      int index = 0;
      Enumeration iter = tracksToPlay.elements();
      while (iter.hasMoreElements()) {
	int track = ((Integer) iter.nextElement()).intValue();
	if (index >= tracksToPlay.index 
	    && track < monitor.trackLength.length) {
	  result += monitor.trackLength[track];
	}
	index++;
      }
      if (!tracksToPlay.atStart() && !tracksToPlay.atEnd()) {
	result -= (monitor.currentAddress - monitor.trackStartAddress);
      }
    }
    return result;
  }

  public synchronized int togglePause() throws DriveException
  {
    if (status() == Drive.STATUS_PAUSED) {
      resume();
      return Drive.STATUS_PLAY;
    }
    pause();
    return Drive.STATUS_PAUSED;
  }

  public synchronized void update(Observable o, Object arg)
  {
    try {
      if (monitor.status != Drive.STATUS_INVALID) {
	if (monitor.cdChanged) {

	  if (DEBUG) {
	    System.out.println("New CD");
	    System.out.println("Id=" + productCode());
	  }
	  tracksToPlay.clear();
	}
	else if (monitor.status == Drive.STATUS_PLAY) {

	  int track = monitor.currentTrack;
	  int tend =  monitor.trackEndAddress[track];

	  if (monitor.currentAddress >= tend - 210) { // Close to end of track?

	    // Poll frequently so we don't miss the event. 
            // Allow for some drives stopping just before end and others
            // resetting to zero
            int address = currentAddress();
            int lastAddress = 0; 
	    while (address < tend 
		   && monitor.status == Drive.STATUS_PLAY
                   && address > lastAddress) {
              lastAddress = address;
	      try {
		Thread.sleep(100);	   // Sleep 100 msec's.
	      }
	      catch (InterruptedException e) {
	      }
	    }

	    if (!tracksToPlay.atStart()) { // Programme play
	      programEndOfTrack();
	    }
	    else {			   // Normal play
	      normalEndOfTrack(track);
	    }

	  }
	}
      }
    }
    catch (DriveException e) {
      System.out.println("Exception: " + e);
    }
  }

  protected synchronized void programEndOfTrack()
  {
    try {
      int next = tracksToPlay.nextTrack(); 
      if (tracksToPlay.atEnd()) {
	if (DEBUG) System.out.println("end");
	stop();
	tracksToPlay.reset();
      }
      else if (singleMode) {
	stop();
      }
      else {
	if (DEBUG) System.out.println("next");
	play(next);
      }
    }
    catch (DriveException e) {
      System.out.println("Program end of track exception: " + e);
    }
  }
  
  protected void normalEndOfTrack(int track)
  {					   
    // If during the track we switched play modes, ensure the correct
    // behaviour occurs.

    try {
      if (singleMode) {
	stop();
      }
      else if (track < monitor.numberOfTracks) {
	play(track + 1);
      }
    }
    catch (DriveException e) {
      System.out.println("Normal-Play end of track exception: " + e);
    }      
  }
}


