/* Program version 0.94, released January 13, 2003, Copyright Nathanael Hoyle.  Please read both README and LICENSE files. */

/*#include <linux/types.h>*/
#include <linux/cdrom.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <unistd.h>
#include <errno.h>

#include "cdstatus.h"

extern int errno;
extern track_listing trackinfo[100];
extern char album_name[256];
extern char artist_name[256];
extern unsigned int cddb_tracks;

/* create output file, and handle --format string */
int openOuputFile(cdstatus_args *,int);

int main(int argc, char *argv[]){
   int status;
   int type;
   cdstatus_args cdsargs;


   /*take care of these up front */
   if(argc==2){
      if(!strcmp(argv[1],"--version")){
         printf("\nCD Drive Status and CD Analyzer / Reader by Nathanael Hoyle\n");
         printf("Version %s. Built at %s on %s.\n\n",VERSION,__TIME__,__DATE__);
         return 0;
      }
      else if(!strcmp(argv[1],"--help")){
         printHelp();
         return 0;
      }
   }

   /* set default options and override as specified */
   status=handleArgs(argc, argv, &cdsargs);

   if(status!=0){
      return status;
   }

   if(!cdsargs.silent)
   {
      printf("\nCD Drive Status and CD Analyzer / Reader by Nathanael Hoyle\n");
      printf("Version %s. Built at %s on %s.\n\n",VERSION,__TIME__,__DATE__);
   }

   type=basicInfo(&cdsargs);
   if((type==CDS_AUDIO)||(type==CDS_MIXED)){
      process_audio(&cdsargs);
   }

   return 0;
}

int handleArgs(int argc, char *argv[], cdstatus_args *cdsargs){
   int arg;

   memset((void *)cdsargs,0,(size_t)sizeof(*cdsargs));
   strcpy(cdsargs->outputdir,".");
   strcpy(cdsargs->basename,"track");
   strcpy(cdsargs->drivename,"/dev/cdrom");
   cdsargs->waveout=1;
   cdsargs->read_chunk_size=1;
   cdsargs->max_retries=3;
   cdsargs->cddb=0;

   for(arg=1;arg<argc;++arg){
      if((!strcmp(argv[arg],"--version"))||(!strcmp(argv[arg],"--help"))){
         printf("Sorry, %s is an exclusive option and may not be",argv[arg]);
         printf(" used in conjunction with other options.\n");
         return -1;
      }
      else if(!strcmp(argv[arg],"--reset")){
         cdsargs->reset=1;
      }
      else if((!strcmp(argv[arg],"--readtest")) ||
         (!strcmp(argv[arg],"--rip")) ){
         cdsargs->readtest=1;
      }
      else if(!strcmp(argv[arg],"--start")){
         cdsargs->start=atoi(argv[++arg]);
      }
      else if(!strcmp(argv[arg],"--stop")){
         cdsargs->stop=atoi(argv[++arg]);
      }
      else if(!strcmp(argv[arg],"--retries")){
         cdsargs->max_retries=atoi(argv[++arg]);
      }
      else if(!strcmp(argv[arg],"--waveout")){
         cdsargs->readtest=1;
         cdsargs->waveout=1;
      }
      else if(!strcmp(argv[arg],"--rawout")){
         cdsargs->readtest=1;
         cdsargs->waveout=0;
      }
      else if(!strcmp(argv[arg],"--drive")){
         strncpy(cdsargs->drivename,argv[++arg],254);
      }
      else if(!strcmp(argv[arg],"--outputdir")){
         strncpy(cdsargs->outputdir,argv[++arg],254);
      }
      else if(!strcmp(argv[arg],"--basename")){
         strncpy(cdsargs->basename,argv[++arg],254);
      }
      else if(!strcmp(argv[arg],"--readchunk")){
         cdsargs->read_chunk_size=atoi(argv[++arg]);
      }
      else if(!strcmp(argv[arg],"--cddb")){
         cdsargs->cddb=1;
      }
      else if(!strcmp(argv[arg],"--format")){
         cdsargs->format_on=1;
         strncpy(cdsargs->format,argv[++arg],511);
      }
      else if(!strcmp(argv[arg],"--nomangle")){
         cdsargs->noMangle=1;
      }
      else if(!strcmp(argv[arg],"--silent")){
          cdsargs->silent=1;
      }
      else{
         printf("Invalid option selected.  Please verify options.\n");
         printf("Invalid option passed: %s\n\n",argv[arg]);
         return -1;
      }

   }

   if(cdsargs->reset){
      if(cdsargs->readtest){
         printf("Error, cannot use data extraction and --reset.\n");
         return -1;
      }
      else{
         return doReset(cdsargs);
      }
   }

   return 0;
}

int process_audio(const cdstatus_args *cdsargs){
   int drive;
   struct cdrom_read_audio cdra;
   struct cdrom_tochdr cdtochdr;
   cd_toc_info cdtocinfo[100];
   int status;
   int counter,counter2,readframes,retry;
   FILE * audio_out;

   /* abused, misused, and reused int for all sorts of values that need to be
      written to file and hence need a variable*/
   double progress_tick;
   double progress_threshold;
   int start_track,end_track;
   unsigned int discid;

   drive = open(cdsargs->drivename, O_RDONLY | O_NONBLOCK);
   if(drive==-1){
      printf("Error opening drive %s.\n",cdsargs->drivename);
      return -1;
   }

   if(readTOC(drive,cdtocinfo,&cdtochdr,cdsargs->silent)!=0){
      printf("Error reading TOC.  Exiting.");
      exit(-1);
   }

   if(cdsargs->cddb){
      discid=calcDiscId(cdtochdr.cdth_trk1,(const cd_toc_info *)cdtocinfo);
      if(!cdsargs->silent)
      {
         printf("CDDB disc id: %08x\n",discid);
      }

      /*do cddb query*/
      cddb_query(cdtochdr.cdth_trk1,(const cd_toc_info *)cdtocinfo,cdsargs->noMangle);

      if(!cdsargs->silent)
      {
         printf("Track listing found for cd %s by %s:\n",album_name,artist_name);
         for(counter=1;counter<=cddb_tracks;++counter){
            printf("Track %d: ",counter);
            if(trackinfo[counter].artist[0]!='\0'){
               printf("%s by %s\n",trackinfo[counter].title,trackinfo[counter].artist);
            } else {
               printf("%s\n",trackinfo[counter].title);
            }
         }
      }
   }

   if(cdsargs->readtest){
      if(cdsargs->start<=1){
         start_track=cdtochdr.cdth_trk0;
      } else {
         start_track=cdsargs->start;
      }
      if((cdsargs->stop>=cdtochdr.cdth_trk1)||(cdsargs->stop==0)){
         end_track=cdtochdr.cdth_trk1;
      } else {
         end_track=cdsargs->stop;
      }

      if(!cdsargs->silent)
      {
        if(start_track==end_track){
             printf("\n\nBeginning digital audio extraction of track %d.\n",start_track);
          } else {
             printf("\n\nBeginning digital audio extraction of tracks");
             printf(" %d through %d.\n",start_track,end_track);
          }
      }

      for(counter=start_track;counter<=end_track;++counter){
         if((cdtocinfo[counter].data)&&(!cdsargs->silent)){
            printf("Skipping track %d.  (Data track).\n",counter);
            continue;
         }
         cdra.buf=malloc(CD_FRAMESIZE_RAW*(cdsargs->read_chunk_size));

         if(cdra.buf==NULL){
            printf("Unable to allocate enough memory for buffer to hold cd data");
            printf(" (cdra.buf). Try decreasing cdsargs->read_chunk_size and");
            printf(" try again.i\n");
            return -1;
         }
         if(!cdsargs->silent)
         {
             printf("Beginning track %d: %s.  Progress:\n",counter,trackinfo[counter].title);
         }

         audio_out=(FILE *)openOutputFile(cdsargs,counter);

         cdra.nframes=cdsargs->read_chunk_size;
         cdra.addr_format=CDROM_MSF;

         if(counter==cdtochdr.cdth_trk1){
            readframes=cdtocinfo[0].frame_global- \
               cdtocinfo[counter].frame_global;
         }
         else{
            readframes=cdtocinfo[counter+1].frame_global- \
               cdtocinfo[counter].frame_global;
         }

         cdra.addr.msf.minute=cdtocinfo[counter].min;
         cdra.addr.msf.second=cdtocinfo[counter].sec;
         cdra.addr.msf.frame=cdtocinfo[counter].frame_local;

         progress_threshold=((double)readframes/(double)cdsargs-> \
            read_chunk_size)/(double)75;
         progress_tick=0.0;

         if(cdsargs->waveout){
            writeWavHeader(readframes,audio_out);
         }
         /* counter2 represents position of next frame to be read */
         for(counter2=0;counter2<readframes;counter2+=(cdsargs->read_chunk_size)){
            if((cdsargs->read_chunk_size+counter2)>readframes){
               cdra.nframes=readframes-counter2;
            }

       status=ioctl(drive,CDROMREADAUDIO,&cdra);
       if(status==-1){
               if(errno==EOPNOTSUPP){
                  printf("Error, the drive does not support reading ");
                  printf("in this mode.\n");
                  printf("You will not be able to use the --readtest option ");
                  printf("with this drive, sorry.");
                  return -2;
               }
               if(!cdsargs->silent)
               {
                   printf("\nError trying to read track %d at ",counter);
                   printf("minute: %d, second: %d, frame %d",cdra.addr.msf.minute,\
                      cdra.addr.msf.second,cdra.addr.msf.frame);
                   printf(strerror(errno));
                   printf("\n");
                   fflush(stdout);
               }
               for(retry=1;(retry<=cdsargs->max_retries)&&(status==-1);retry++){
                  if(!cdsargs->silent)
                  {
                      printf("Retrying...\n");
                  }
                  status=ioctl(drive,CDROMREADAUDIO,&cdra);
               }
               if(status==-1){
                  if(!cdsargs->silent)
                  {
                      printf("All %d retries failed. Expect tiny skip in audio.\n", cdsargs->max_retries);
                  }
               }
               else{
                  if(!cdsargs->silent)
                  {
                      printf("Succeeded on retry %d.\n",retry-1);
                  }
               }
            fflush(stdout);
            }
            if(status!=-1){
               fwrite((const void *)cdra.buf,(size_t)CD_FRAMESIZE_RAW, \
                  (size_t)cdra.nframes,audio_out);
            }

            cdra.addr.msf.frame+=cdra.nframes;
            if(cdra.addr.msf.frame>=CD_FRAMES){
               cdra.addr.msf.second+=cdra.addr.msf.frame/CD_FRAMES;
               cdra.addr.msf.frame=cdra.addr.msf.frame%CD_FRAMES;
               if(cdra.addr.msf.second>=60){
                  cdra.addr.msf.minute+=cdra.addr.msf.second/60;
                  cdra.addr.msf.second=cdra.addr.msf.second%60;
               }
            }
            ++progress_tick;
            if((progress_tick>progress_threshold)&&(!cdsargs->silent)){
               printf("#");
               fflush(stdout);
               progress_tick-=progress_threshold;
             }
         }
         fclose(audio_out);
         free(cdra.buf);
         if(!cdsargs->silent)
         {
             printf("\nTrack %d completed.\n",counter);
         }
      }
   }

   return 0;
}

int doReset(const cdstatus_args *cdsargs){
   int drive;
   int status;

   printf("Attempting to reset drive.\n");
   drive = open(cdsargs->drivename, O_RDONLY | O_NONBLOCK);
   if(drive==-1){
      printf("Error opening drive device.\n");
      return 1;
   }
   status=ioctl(drive,CDROMRESET);
   if(status==-1){
      if(errno==EACCES){
         printf("Permission denied error received.  Note that most systems\n");
         printf("will require root access to reset a drive.\n\n");
      }
      else{
         printf(strerror(errno));
      }
      printf("Error on resetting drive.  Assume reset failed.\n");
   }
   else{
      printf("Drive successfully reset.\n");
   }
   return 0;
}

int basicInfo(const cdstatus_args *cdsargs){
   int status,type,drive;

   drive = open(cdsargs->drivename, O_RDONLY | O_NONBLOCK);
   if(drive==-1){
      printf("Error opening drive %s.\n",cdsargs->drivename);
      exit(-1);
   }

   status=ioctl(drive,CDROM_DRIVE_STATUS);
   if(status==-1){
      printf("Error getting drive status (while performing ioctl).\n");
      exit(1);
   }

   switch(status){
      case CDS_NO_INFO:
         printf("No information is available (i.e. this function is not\n");
         printf("implemented on the drive.\n");
         exit(0);
      case CDS_NO_DISC:
         printf("No disc is in the drive.\n");
         exit(0);
      case CDS_TRAY_OPEN:
         printf("The drive tray is open (or no disc is inserted).  Please close it then retry.\n");
         exit(0);
      case CDS_DRIVE_NOT_READY:
         printf("The drive is not ready.\n");
         exit(0);
      case CDS_DISC_OK:
         if(!cdsargs->silent)
         {
             printf("The drive reads disc ok. Attempting to retrieve disc info.\n");
         }
         break;
      default:
         printf("Error: Invalid status code returned: %d",status);
         exit(status);
   }

   type=ioctl(drive,CDROM_DISC_STATUS);

   if(type==-1){
      printf("Unexpected error reading disc info from drive. Exiting.\n");
      exit(1);
   }

   switch(type){
      case CDS_NO_INFO:
         printf("Unable to read disc type info.  Function not implemented.\n");
         exit(0);
      case CDS_NO_DISC:
         printf("\"No disc in drive\" status returned.  Unusual since\n");
         printf("disc presence and drive readiness was just verified.\n");
         exit(0);
      case CDS_AUDIO:
         if(!cdsargs->silent)
         {
             printf("Disc appears to be an audio cd.\n");
         }
         break;
      case CDS_DATA_1:
         printf("Disc appears to be a DATA MODE 1 cd.\n");
         break;
      case CDS_DATA_2:
         printf("Disc appears to be a DATA MODE 2 cd.\n");
         break;
      case CDS_XA_2_1:
         printf("Disc appears to be of type XA 2 1\n");
         break;
      case CDS_XA_2_2:
         printf("Disc appears to be of type XA 2 2\n");
         break;
      case CDS_MIXED:
         if(!cdsargs->silent)
         {
             printf("Disc appears to be MIXED MODE. Will try to gather info on");
             printf("contained audio tracks.\n");
         }
         break;
      default:
         printf("Unknown type returned: %d.  Possible error occured.",type);
   }
   close(drive);
   return type;
}

void printHelp(){

   printf("Usage summary (see README file for full details):\n\n");
   printf("--version          displays version information and then exits\n\n");

   printf("--help             displays this usage summary and exits\n\n");

   printf("--reset            resets the drive internally.  normally\n");
   printf("                   requires root privileges\n\n");

   printf("--drive /dev/path/ overrides the default cdrom drive path of\n");
   printf("                   /dev/cdrom\n\n");

   printf("--outputdir /path/ specifies a directory for file output other\n");
   printf("                   than the default of the current working\n");
   printf("                   directory\n\n");

   printf("--retries X        overrides the default number of retries for\n");
   printf("                   failed reads. Default is 3\n\n");

   printf("--start X          specifies a track to start audio extraction\n");
   printf("                   from. Default is 1\n\n");

   printf("--stop X           specifies a track to stop audio extraction\n");
   printf("                   at. Default is last track on cd\n\n");

   printf("--readtest\n");
   printf("--rip              perform a digital audio extraction (rip) of\n");
   printf("                   tracks. --rip is an alias for --readtest and\n");
   printf("                   has the same function\n\n");

   printf("--waveout          now redundant, because .wav file output is\n");
   printf("                   the default, this also sets --readtest\n\n");

   printf("--rawout           specifies the old headerless files for\n");
   printf("                   output, which will contain exactly the data\n");
   printf("                   on the cd.  not playable by most music\n");
   printf("                   programs without passing a lot of audio info\n");
   printf("                   (2 channel, 44.1khz 16-bit signed, etc)\n\n");

   printf("--basename name    specifies to create file names of the form\n");
   printf("                   nameXX.wav where XX is the track number,\n");
   printf("                   instead of the default of trackXX.wav.\n\n");

   printf("--readchunk X      specifies to read X frames of audio data at\n");
   printf("                   a time, instead of 1 by 1.  this will\n");
   printf("                   increase performance up to a point.  the\n");
   printf("                   ideal value is different for every system.\n");
   printf("                   there is a direct correlation between higher\n");
   printf("                   values and higher memory usage.  also,\n");
   printf("                   beyond a certain point there is no\n");
   printf("                   improvement.  this gives up some of the\n");
   printf("                   fine-grained error/scratch-detection that\n");
   printf("                   cdstatus was designed for, so this still\n");
   printf("                   defaults to 1, but you now have the option.\n\n");

   printf("--cddb             instruct cdstatus to calculate the cddb disc\n");
   printf("                   id and contact the remote cddb database server\n");
   printf("                   to look up the album information. see --format\n\n");

   printf("--format           used to specify filename format for output files.\n");
   printf("                   only valid when used with --cddb.  syntax is:\n\n");
   printf("                   #Artist#      -> Becomes artist name\n");
   printf("                   #Album#       -> Becomes name of cd\n");
   printf("                   #TrkNum#      -> Becomes track number, like 01\n");
   printf("                   #TrkName#     -> Becomes track name\n\n");
   printf("                   see README for example.  Other characters are \n");
   printf("                   allowed freeform.\n");

   return;
}


void writeWavHeader(int readframes,FILE *audio_out){
   long int chunksize;
   typedef struct _wavHeader
   {
      int32_t RIFF_header;
      int32_t total_size;
      int32_t WAVE;
      int32_t fmt;
      int32_t subchunk_size;
      int16_t audio_format;
      int16_t number_channels;
      int32_t sampling_rate;
      int32_t byte_rate;
      int16_t block_align;
      int16_t bits_per_sample;
   } wavHeader;

   wavHeader wHeader;
   /* "RIFF" */
   wHeader.RIFF_header=0x46464952;

   chunksize=readframes*CD_FRAMESIZE_RAW;
   wHeader.total_size=chunksize+sizeof(wavHeader);

   /* "WAVEfmt " */
   wHeader.WAVE=0x45564157;
   wHeader.fmt=0x20746D66;

   wHeader.subchunk_size=16;
   wHeader.audio_format=1;
   wHeader.number_channels=2;
   wHeader.sampling_rate=44100;
   wHeader.byte_rate=176400;
   wHeader.block_align=4;
   wHeader.bits_per_sample=16;

   fwrite((const void *)&wHeader,sizeof(wavHeader),(size_t)1,audio_out);
   fprintf(audio_out,"data");
   fwrite((const void *)&chunksize,sizeof(long int),(size_t)1,audio_out);

   return;
}

int readTOC(int drive,cd_toc_info *cdtocinfo, struct cdrom_tochdr *cdtochdr,int silent){
   struct cdrom_tocentry cdtocentry;
   struct cdrom_volctrl cdvolume;
   int counter;
   int status;


   if(!silent)
   {
       printf("Reading table of contents.\n");
   }
   status=ioctl(drive,CDROMREADTOCHDR,cdtochdr);
   if(status==-1){
      printf("Unable to read TOC.\n");
      printf(strerror(errno));
      return -1;
   }

   if(!silent)
   {
       printf("Disc appears to have %d tracks. Querying for track info.\n", \
      cdtochdr->cdth_trk1);
   }

   for(counter=cdtochdr->cdth_trk0;counter<=cdtochdr->cdth_trk1;++counter){
      cdtocentry.cdte_format=CDROM_MSF;
      cdtocentry.cdte_track=counter;
      status=ioctl(drive,CDROMREADTOCENTRY,&cdtocentry);
      if(status==-1){
         printf("Error querying track info for track %d\n",counter);
         printf(strerror(errno));
      }
      else{
         cdtocinfo[counter].min=cdtocentry.cdte_addr.msf.minute;
         cdtocinfo[counter].sec=cdtocentry.cdte_addr.msf.second;
         cdtocinfo[counter].frame_local=cdtocentry.cdte_addr.msf.frame;
         cdtocinfo[counter].frame_global=cdtocinfo[counter].frame_local;
         (cdtocinfo[counter].frame_global)+=((cdtocinfo[counter].min) \
            *CD_FRAMES*60);
         (cdtocinfo[counter].frame_global)+=((cdtocinfo[counter].sec) \
            *CD_FRAMES);
         /*cdtocinfo[counter].frame_global-=CD_MSF_OFFSET;*/
         cdtocinfo[counter].data=(cdtocentry.cdte_ctrl&CDROM_DATA_TRACK);

         if(!silent)
         {
            printf("Track %d starts at: minute: %d, second: %d, frame: %d; global frame: %d\n", \
               counter,cdtocinfo[counter].min, cdtocinfo[counter].sec, \
               cdtocinfo[counter].frame_local, cdtocinfo[counter].frame_global);
         }
      }
   }

   if(!silent)
   {
       printf("Gathering info on leadout track.\n");
   }
   cdtocentry.cdte_format=CDROM_MSF;
   cdtocentry.cdte_track=CDROM_LEADOUT;

   status=ioctl(drive,CDROMREADTOCENTRY,&cdtocentry);
   if(status==-1){
      printf("Error querying track info for leadout track.\n");
      printf(strerror(errno));
   }
   else{
      cdtocinfo[0].min=cdtocentry.cdte_addr.msf.minute;
      cdtocinfo[0].sec=cdtocentry.cdte_addr.msf.second;
      cdtocinfo[0].frame_local=cdtocentry.cdte_addr.msf.frame;
      cdtocinfo[0].frame_global=cdtocinfo[0].frame_local;
      (cdtocinfo[0].frame_global)+=((cdtocinfo[0].min)*CD_FRAMES*60);
      (cdtocinfo[0].frame_global)+=((cdtocinfo[0].sec)*CD_FRAMES);
      /*cdtocinfo[0].frame_global-=CD_MSF_OFFSET;*/

      if(!silent)
      {
         printf("Leadout track starts at: minute: %d, second: %d, frame: %d; global frame: %d\n", \
         cdtocinfo[0].min, cdtocinfo[0].sec,cdtocinfo[0].frame_local,cdtocinfo[0].frame_global);
      }
   }

   if(!silent)
   {
      printf("Querying drive volume settings:\n");
      status=ioctl(drive,CDROMVOLREAD,&cdvolume);
      if(status==-1){
         printf("Unable to read volume settings. Error: %s\n",strerror(errno));
      }
      else{
         printf("Volume settings for drive:\n");
         printf("Channel 1: %d, Channel 2: %d, Channel 3: %d, Channel 4: %d\n", \
            cdvolume.channel0,cdvolume.channel1,cdvolume.channel2, \
            cdvolume.channel3);
      }
   }

   return 0;
}

int openOutputFile(const cdstatus_args *cdsargs,int counter){
   char filename[1024];
   char filename2[1024];
   char scratch[3];
   char * token;
   char last_token[1024];
   char syscmd[1024];
   char pathname[1024];
   int index=0;
   struct stat stat_info;

   pathname[0]='\0';

    if(cdsargs->cddb==0)
    {
       sprintf(filename,"%s%02d", cdsargs->basename,counter);
    }
    else
    {

        /* If you're using --cddb and/or --format WITHOUT .wav output, you're either crazy
        * or doing something really complicated, but I'll support your nutso self */
        if(!cdsargs->format_on){
        sprintf(filename,"%s/%s/%s_-_%s_-_%02d_-_%s",artist_name,album_name, \
            artist_name,album_name,counter,trackinfo[counter].title);
        }

   /* Parse format string */
   else{
      filename[0]='\0'; 

      while(index<strlen(cdsargs->format)){
         if(cdsargs->format[index]!='#'){
            strncat(filename,&(cdsargs->format[index]),1);
            ++index;
         }
         else{
            if(!strncasecmp(&(cdsargs->format[index]),"#artist#",strlen("#artist#"))){
               strcat(filename,artist_name);
               index+=strlen("#artist#");
            }
            else if(!strncasecmp(&(cdsargs->format[index]),"#album#",strlen("#album#"))){
               strcat(filename,album_name);
               index+=strlen("#album#");
            }
            else if(!strncasecmp(&(cdsargs->format[index]),"#trknum#",strlen("#trknum#"))){
               sprintf(scratch,"%02d",counter);
               strcat(filename,scratch);
               index+=strlen("#trknum#");
            }
            else if(!strncasecmp(&(cdsargs->format[index]),"#trkname#",strlen("#trkname#"))){
               strcat(filename,trackinfo[counter].title);
               index+=strlen("#trkname#");
            }
         }
      }
   }
    }

   if(cdsargs->waveout){
      strcat(filename,".wav");
   }
   else{
      strcat(filename,".raw");
   }

   sprintf(filename2,"%s/%s",cdsargs->outputdir,filename);
   strcpy(filename,filename2);

   token=strtok(filename2,"/");
   strcpy(last_token,token);
   token=strtok(NULL,"/");
    do
    {
        /*No more directories in path, create and return file*/
        if(token==NULL)
        {
#ifdef DEBUG
            printf("Creating %s.\n",filename);
#endif
            return (int)fopen(filename,"w");
        }
        else
        {

            strcat(pathname,last_token);
            /*make another directory, maybe*/
            if(stat(pathname,&stat_info))
            {
                if(errno==ENOENT)
                {
#ifdef DEBUG
                    printf("%s directory does not exist, creating.\n",pathname);
#endif
                    sprintf(syscmd,"mkdir \"%s\"",pathname);
                    system(syscmd);
                }
             }
            else
            {
                if(stat_info.st_mode&S_IFDIR)
                {
#ifdef DEBUG
                    printf("%s directory is already present, not creating.\n",pathname);
#endif
                }
                else if(stat_info.st_mode&S_IFREG)
                {
                    printf("directory component \'%s\' is a file. Please correct.  Exiting.\n",pathname);
                    exit(1);
                }
            }
        }
	    strcat(pathname,"/");
        strcpy(last_token,token);
        token=strtok(NULL,"/");
    }while(1);
}


