#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>

#include "cdstatus.h"

/* to get started, i hard-coded the hostname.  need to use a file to store
   site info in, and get updated site lists to use hosts from and such */

#define CDDB_SITE "us.freedb.org"
#define CDDB_PORT 8880
#define BUFFSIZE 4096

extern int errno;
extern int h_errno;

char album_name[256];
char artist_name[256];
track_listing trackinfo[99];
unsigned int cddb_tracks;

void cddb_query(int tracks, const cd_toc_info* cdtocinfo, int noMangle){
   /* disc related */
   int discid;
   char category[64];
   int index;
   int choice;

   /* network related */
   int client_sock;
   struct sockaddr_in dest_addr;
   struct hostent* host_info;
   void * buffer;
   int result;
   char *scratchbuffer;
   char *token;
   char *token2;
   void * backup_buffer;

   discid=calcDiscId(tracks, cdtocinfo);

   for(index=0;index<99;++index){
      trackinfo[index].artist[0]='\0';
      trackinfo[index].title[0]='\0';
   }

   buffer=calloc((size_t)1,(size_t)BUFFSIZE);
   if(buffer==NULL){
      printf("Unable to allocate buffer for connection with cddb site.\n");
      printf("Lookup aborted.\n");
      return;
   }
#ifdef DEBUG
   else{
      printf("Buffer allocated successfully.\n");
   }
#endif
   scratchbuffer=calloc(1,256);
   if(scratchbuffer==NULL){
      printf("Unable to allocate scratch buffer.\n");
      printf("Lookup aborted.\n");
      return;
   }
#ifdef DEBUG
   else{
      printf("scratch buffer allocated successfully.\n");
   }
#endif

   client_sock=socket(AF_INET,SOCK_STREAM,0);
   if(client_sock==-1){
      printf("Error, unable to create client socket for CDDB lookup.\n");
      printf("Lookup will not be performed.\n");
      return;
   }

   dest_addr.sin_family=AF_INET;
   dest_addr.sin_port=htons(CDDB_PORT);

   host_info=gethostbyname(CDDB_SITE);
   if(host_info==NULL){
      printf("Unable to look up the host address of the CDDB site %s.\n", \
         CDDB_SITE);
      if(h_errno==HOST_NOT_FOUND){
         printf("Host not found.\n");
      }
      return;
   }

   memcpy(&(dest_addr.sin_addr.s_addr),host_info->h_addr,host_info->h_length);
   memset(&(dest_addr.sin_zero),'\0',8);

   result=connect(client_sock,(struct sockaddr *)&dest_addr, \
      sizeof(struct sockaddr));
   if(result==-1){
      printf("Unable to connect to remote cddb host.  Aborting lookup.\n");
      printf("Error: %s\n",strerror(errno));
      return;
   }

#ifdef DEBUG
   else{
      printf("Connected to cddb lookup server.\n");
   }
#endif

   result=hearken(client_sock,buffer);
   if((result!=200)&&(result!=201)){
      printf("CDDB server is either busy (max users) or undergoing ");
      printf("maintenance.  Please try a different server, or try later.\n");
      return;
   }

   memset(buffer,0,BUFFSIZE);
   sprintf((char *)buffer,"cddb hello %s %s cdstatus %s\n",getenv("USER"), \
      getenv("HOSTNAME"), VERSION);
   send(client_sock,buffer,strlen(buffer),0);

   result=hearken(client_sock,buffer);
   if(result!=200){
      printf("Error while \"handshaking\" with server.  If problem persists ");
      printf("please try a different server.  If problem still persists, ");
      printf("please contact the developer.\n");
      return;
   }

   memset(buffer,0,BUFFSIZE);
   strcpy((char *)buffer, "proto\n");
   send(client_sock,buffer,strlen(buffer),0);

   result=hearken(client_sock,buffer);
   if(result!=200){
      printf("The server does not appear to handle the basic cddb protocol ");
      printf("correctly.  Strongly recommend changing cddb servers.\n");
      printf("Aborting.\n");
      return;
   }

   sscanf((const char *)buffer, \
      "200 CDDB protocol level: current %d, supported %*d",&result);
#ifdef DEBUG
   printf("Current protocol version: %d\n",result);
#endif
   if(result<5){
      sscanf((const char *)buffer, \
         "200 CDDB protocol level: current %*d, supported %d",&result);
#ifdef DEBUG
      printf("Max protocol version: %d\n",result);
#endif
      if(result<5){
         printf("Server does not support recent protocol updates.  Will ");
         printf("continue, but new features will not be available.\n");
      }
      else {
         memset(buffer,0,BUFFSIZE);
         strcpy((char *)buffer,"proto 5\n");
         send(client_sock,buffer,strlen(buffer),0);
         result=hearken(client_sock,buffer);
         if(result!=201){
            printf("Error negotiating protocol level.");
            printf(" Remaining at old level.\n");
         }
#ifdef DEBUG
         else{
            printf("Protocol changed to 5.\n");
         }
#endif
      }
   }

   memset(buffer,0,BUFFSIZE);
   sprintf((char *)buffer,"cddb query %08x %d",discid,tracks);
   for(index=1;index<=tracks;++index){
      sprintf(scratchbuffer," %d",cdtocinfo[index].frame_global);
      strcat((char *)buffer,scratchbuffer);
   }
   sprintf(scratchbuffer," %d\n",cdtocinfo[0].frame_global/75);
   strcat((char *)buffer,scratchbuffer);

   send(client_sock,buffer,strlen(buffer),0);
   
   result=hearken(client_sock,buffer);
   switch(result){
      case 200:
         printf("Found exact match: %s\n",(char *)buffer);
         token=strtok((char *)buffer," ");
         token=strtok(NULL," ");
         strcpy(category,token);
         token=strtok(NULL," ");
         sscanf(token,"%x",&discid);
         break;
      case 211:
         printf("Found several inexact matches.  Please select a number from the");
         printf(" list below.\n");
         break;
      case 210:
         printf("Found more than one exact match.  Please select a number from ");
         printf("the list below.\n");
         break;
      case 202:
         printf("No match was found.  Please note that cdstatus does not ");
         printf("currently support cddb entry submission.  Normal file names");
         printf(" will be used.\n");
         return;
      case 403:
         printf("The server reports that this database entry is corrupt.\n");
         printf("This is an ugly and unfortunate server-side error.  Normal");
         printf(" file names will be used.\n");
         return;
      case 409:
         printf("The server is being either forgetful or haughty.  Then again,");
         printf(" it's probably just broken.  Try a different server.\n");
         return;
   }
   if((result==211)||(result==210)){
      backup_buffer=calloc(1,BUFFSIZE);
      if(backup_buffer==NULL){
         printf("Error, unable to allocate buffer for parsing entries.\n");
         return;
      }
      memcpy(backup_buffer,buffer,BUFFSIZE);
      token=strtok((char *)buffer,"\n");
      index=1;
      while((token!=NULL)&&(strcmp(token,".") )){
         token=strtok(NULL,"\n");
         if(strncmp(token,".",1)){
            printf("%d. %s\n",index,token);
         }
         else{
            break;
         }
         ++index;
      }
      fflush(stdout);
      choice=getchar()-48;
      memcpy(buffer,backup_buffer,BUFFSIZE);
      free(backup_buffer);

      token=strtok((char *)buffer,"\n");
      index=1;
      while((token!=NULL)&&(strcmp(token,".") )){
         token=strtok(NULL,"\n");
         if(strncmp(token,".",1)){
            if(index==choice){
               break;
            }
         }
         ++index;
      }
      strcpy(buffer,token);
      token=strtok((char *)buffer," ");
      strcpy(category,token);
      token=strtok(NULL," ");
      sscanf(token,"%x",&discid);

   }

   sprintf((char *)buffer,"cddb read %s %08x\n",category,discid);
   send(client_sock,buffer,strlen(buffer),0);
   result=hearken(client_sock,buffer);

   switch (result){
      case 401:
         printf("The server reports that the entry does not exist, despite ");
         printf("having just listed it.  Please try a different server.\n");
         return;
      case 402:
         printf("General server error.  Please try a different server.\n");
         return;
      case 403:
         printf("The server reports that this database entry is corrupted.\n");
         printf("Please try a different server.\n");
         return;
      case 409:
         printf("The server indicates that the client has not established a ");
         printf("proper server connection, despite having done so.  Please ");
         printf("try a different server.\n");
         return;
      case 210:
         break;
      default:
         printf("The server has returned an error code that does not comply ");
         printf("with the published cddb protocol.  Please try a different ");
         printf("server.\n");
         return;
   }

   /* parse and handle the returned information */
   /*discard the already used status line*/
   token=strtok((char *)buffer,"\n");

   token=strtok(NULL,"\n");
   while(token!=NULL){
      if(token[0]=='#'){
         token=strtok(NULL,"\n");
         continue;
      }
      else if(token[0]=='.'){
         break;
      }
#ifdef DEBUG
      printf("Lineitem: %s\n",token);
#endif
      /* actual handling of data lines */
      if(!strncmp(token,"DTITLE=",7)){
         /* artist and album names */
         token2=strchr(token,'/');
         --token2;
         strncpy(artist_name,token+7,token2-(token+7));
/*         printf("Artist name: %s\n",artist_name);*/
         token2+=3;
         strncpy(album_name,token2,strlen(token2)-1);
         if(strchr(album_name,'\n')!=NULL){
            *strchr(album_name,'\n')='\0';
         }
/*         printf("Album name: %s\n",album_name);*/
      }
      else if(!strncmp(token,"TTITLE",6)){
         /* track listing data */
         ++cddb_tracks;
         sscanf(token,"TTITLE%d=",&index);
         ++index;
         if(strchr(token,'/')==NULL){
            strncpy(trackinfo[index].title,strchr(token,'=')+1,strlen(strchr(token,'=')+1)-1);
            /*printf("Track %d: %s\n",index,trackinfo[index].title);*/
         }
         else {
            token2=strchr(token,'/');
            --token2;
            strncpy(trackinfo[index].artist,strchr(token,'=')+1, \
               token2-(strchr(token,'=')+1));
            token2+=3;
            strncpy(trackinfo[index].title,token2,strlen(token2)-1);
            /*printf("Track %d: %s by %s",index,trackinfo[index].title, \
               trackinfo[index].artist);*/
            fflush(stdout);
         }
      }
      token=strtok(NULL,"\n");
   }

   strcpy((char *)buffer,"QUIT\n");
   send(client_sock,buffer,strlen(buffer),0);

   close(client_sock);
   free(scratchbuffer);
   free(buffer);

   /* beautify names to make them easy on the filesystem */
   if(!noMangle)
   {
      makePrettyString(album_name);
      makePrettyString(artist_name);
      for(index=1;index<=tracks;++index){
         makePrettyString(trackinfo[index].artist);
         makePrettyString(trackinfo[index].title);
      }
   }

}

/* make names safe for filesystem */
char * makePrettyString(char *incoming){
   int index;
   int place;
   if(strlen(incoming)==0){
      return incoming;
   }
   for(index=0;index<strlen(incoming);++index){
      if(incoming[index]==' '){
         incoming[index]='_';
      }
   }

   for(index=0;index<strlen(incoming);++index){
      /* this mess checks to see if the character is a-z or A-Z or 0-9*/
      /* for those of you not familiar, the |0x20 will convert a uppercase */
      /* letter to lowercase.  bit 5 gets set (look at an ascii chart) */
      if((( (incoming[index]|0x20)<'a')||((incoming[index]|0x20)>'z'))&&((incoming[index]<'0')||(incoming[index]>'9'))&&incoming[index]!='_'){
         for(place=index;place<=strlen(incoming);++place){
            incoming[place]=incoming[place+1];
         }
         --index;
      }
   }
   return incoming;

}

/* based on the sample code on www.freedb.org.  Re-writen to inline
 * the second function and to eliminate a couple useless variables.
 * thank the documentation on that site for my ability to code the
 * cddb lookup code.  it's very well and clearly written,
 * especially the info on the server protocol
 */
unsigned int calcDiscId(int tracks, const cd_toc_info* cdtocinfo){
   int i;
   int t=0;
   int n=0;
   int j=0;
   unsigned int id;

   for(i=1;i<=tracks;++i){
      j=((cdtocinfo[i].min*60)+cdtocinfo[i].sec);
      while(j>0){
         n+=(j%10);
         j/=10;
      }
   }
   t=((cdtocinfo[0].min*60)+cdtocinfo[0].sec) -((cdtocinfo[1].min \
      *60) + cdtocinfo[1].sec);

   /*printf("Disc length is %d seconds.\n",t);*/
   id=((n%0xff)<<24)|t<<8|tracks;

   return id;

}

int hearken(int client_sock, void *buffer){
   int data_size;
   int result;

   memset(buffer,0,BUFFSIZE);
   data_size=recv(client_sock,buffer,BUFFSIZE-1,0);
   if(data_size==-1){
      printf("Error while receiving data from cddb site. Aborting Lookup.\n");
      printf("Error: %s\n",strerror(errno));
      return -1;
   }

#ifdef DEBUG
   else {
      printf("Received %d bytes of data from server: \n%s\n",data_size, \
         (char *)buffer);
   }
#endif

   sscanf((const char *)buffer,"%d",&result);
   return result;
}

