
#include "job_control.h"

void calc_stat( _main_data *main_data, _stat *stat,
                unsigned current, int cur_track, int cur_type, int ops )
{
	static unsigned int total_wav_length, total_mp3_length;
	static unsigned int total_wav_length_remain, saved_total_wav_length_remain;
	static unsigned int total_mp3_length_remain, saved_total_mp3_length_remain;
	static float wav_ratio;
	static float mp3_ratio;
	static time_t session_start_time;
	static time_t start_time;
	static time_t pause_time;
	static float temp_ratio;
	static unsigned prev_length_processed;
	static int count;
	time_t cur_time;
	unsigned length_processed;
	int track, type;
	float total_time, temp;

	switch ( ops )
	{
		case CALC_START_SESSION :
			/* Reset stat structure */
			memset( stat, 0, sizeof( stat ) );

			wav_ratio = config.wav_ratio;
			mp3_ratio = config.mp3_ratio;

			total_wav_length_remain = 0;
			total_mp3_length_remain = 0;
			stat->total_progress = 0;

			track = -1;
			type = WAV;
			while ( find_next_job( main_data, track, type, &track, &type ) >= 0 )
			{
				if ( type == WAV )
					total_wav_length_remain += main_data->track[ track ].length;
				else
					total_mp3_length_remain += main_data->track[ track ].length;
			}
			total_wav_length = total_wav_length_remain;
			total_mp3_length = total_mp3_length_remain;
			saved_total_wav_length_remain = total_wav_length_remain;
			saved_total_mp3_length_remain = total_mp3_length_remain;

			session_start_time = time( NULL );
			return;

		case CALC_START :
			prev_length_processed = 0;
			stat->time_elapsed = 0;
			stat->type = cur_type;
			stat->track = cur_track;
			if ( cur_type == WAV )
				strcpy( stat->dest_file_name,
        				file_name_without_path( main_data->track[ cur_track ].wav_file ) );
			else
			{
				strcpy( stat->src_file_name,
        				file_name_without_path( main_data->track[ cur_track ].wav_file ) );
				strcpy( stat->dest_file_name,
        				file_name_without_path( main_data->track[ cur_track ].mp3_file ) );
			}
			count = 0;
			start_time = time( NULL );
			return;

		case CALC_UPDATE :
			cur_time = time( NULL );
			stat->time_elapsed = cur_time - start_time;
			stat->total_time_elapsed = cur_time - session_start_time;

			length_processed = current;

			stat->progress = ( float )length_processed
                 			/ main_data->track[ cur_track ].length;

			temp_ratio = ( float )( stat->time_elapsed ) / length_processed;

			if ( count++ >= COUNT_BEFORE_GET_AVG )
			{
				if ( cur_type == WAV )
					wav_ratio = ( temp_ratio + 2 * wav_ratio ) / 3;
				else
					mp3_ratio = ( temp_ratio + 2 * mp3_ratio ) / 3;
			}
			else
			{
				if ( cur_type == WAV )
					temp_ratio = wav_ratio;
				else
					temp_ratio = mp3_ratio;
			}

			stat->time_remain =
  			( main_data->track[ cur_track ].length - length_processed ) * temp_ratio;

			if ( cur_type == WAV )
				total_wav_length_remain -= length_processed - prev_length_processed;
			else
				total_mp3_length_remain -= length_processed - prev_length_processed;

			stat->total_time_remain =
  			total_wav_length_remain * wav_ratio
  			+ total_mp3_length_remain * mp3_ratio;

			/* Total progress */
			total_time = total_wav_length * wav_ratio + total_mp3_length * mp3_ratio;
			temp = ( total_time - stat->total_time_remain ) / total_time;

			if ( temp >= stat->total_progress )
				stat->total_progress = temp;
			else
				if ( ( temp - stat->total_progress ) / temp >= 0.05 )
					stat->total_progress = temp;

			prev_length_processed = length_processed;
			return;

		case CALC_PAUSE :
			pause_time = time( NULL );
			return;

		case CALC_CONT :
			pause_time = time( NULL ) - pause_time;
			session_start_time += pause_time;
			start_time += pause_time;
			return;

		case CALC_STOP :
			if ( cur_type == WAV )
			{
				saved_total_wav_length_remain -= main_data->track[ cur_track ].length;
				total_wav_length_remain = saved_total_wav_length_remain;
			}
			else
			{
				saved_total_mp3_length_remain -= main_data->track[ cur_track ].length;
				total_mp3_length_remain = saved_total_mp3_length_remain;
			}
			return;

		case CALC_STOP_SESSION :
			config.wav_ratio = wav_ratio;
			config.mp3_ratio = mp3_ratio;
			memset( stat, 0, sizeof( stat ) );
			return;
	}
}

int lock_file( char *file_name, int is_temp )
{
	int fd;
	char *temp;
	char buf[ MAX_FILE_NAME_LENGTH ];

	fd = open( file_name, O_CREAT | O_EXCL, 00777 );

	if ( fd < 0 && errno != ENOENT )
	{
		if ( config.ask_when_file_exists == FALSE || is_temp == TRUE )
		{
			/* Prepend config.prepend_char until we succeed open() */
			while ( fd < 0 && errno != ENOENT )
			{
				if ( strlen( file_name ) >
    					MAX_FILE_NAME_LENGTH + MAX_FILE_PATH_LENGTH - 2 )
					return - 1;
				temp = file_name_without_path( file_name );
				strcpy( buf, temp );
				temp[ 0 ] = config.prepend_char;
				strcpy( temp + 1, buf );

				fd = open( file_name, O_CREAT | O_EXCL, 00777 );
			}
		}
		else
		{
			do
			{
				if ( dialog_handler( WIDGET_CREATE, FALSE, DL_OVERWRITE_CONFIRM,
                 					FALSE, file_name_without_path( file_name ),
                 					NULL, 0 ) == TRUE )
					/* Just return. Cdparanoia or 8hz-mp3 will overwrite on it */
					return 0;

				/* Let's ask the user what s/he wants */
				temp = file_name_without_path( file_name );
				strcpy( buf, temp );
				if ( dialog_handler( WIDGET_CREATE, TRUE, DL_ENTER_FILE_NAME,
                 					TRUE, buf, buf, sizeof( buf ) - 1 ) == FALSE )
					/* The user does not want to continue. return error */
					return - 1;
				strcpy( temp, buf );
				fd = open( file_name, O_CREAT | O_EXCL, S_IWUSR | S_IRUSR );
			}
			while ( fd < 0 && errno != ENOENT );
		}
	}
	close( fd );
	return 0;
}

void unlock_file( char *file_name )
{
	remove( file_name );
}

void job_starter( _main_data *main_data )
{
	int i, track, type, is_temp;

	/* Sync main_data structure with the data user has entered */
	select_frame_handler( SF_SYNC_MAIN_DATA, 0, main_data );

	/* Reset exist flags */
	for ( i = 0; i < main_data->num_tracks; i++ )
	{
		if ( main_data->track[ i ].make_wav )
			main_data->track[ i ].wav_exist = FALSE;
		if ( main_data->track[ i ].make_mp3 )
			main_data->track[ i ].mp3_exist = FALSE;
	}
	/* Lock target files */
	track = -1;
	type = WAV;
	while ( find_next_job( main_data, track, type, &track, &type ) >= 0 )
	{
		if ( type == WAV )
		{
			is_temp = FALSE;
			if ( main_data->track[ track ].make_wav == FALSE )
			{
				is_temp = TRUE;
				strcpy( main_data->track[ track ].wav_file,
        				get_default_file_name( type, track ) );
			}
			if ( config.auto_append_extension == TRUE )
				auto_append_extension( main_data->track[ track ].wav_file, WAV );
			if ( lock_file( main_data->track[ track ].wav_file, is_temp ) < 0 )
				return;
		}
		else
		{
			if ( config.auto_append_extension == TRUE )
				auto_append_extension( main_data->track[ track ].mp3_file, MP3 );
			if ( lock_file( main_data->track[ track ].mp3_file, FALSE ) < 0 )
				return;
		}
	}
	/* Destroy select frame & change main window button state */
	select_frame_handler( WIDGET_DESTROY, 0, NULL );
	main_window_handler( MW_MODE_STATUS, NULL, NULL );
	/* Start Job */
	job_controller( JC_START, main_data );
}

void job_finisher( _main_data *main_data )
{
	int i;
	char dirstring[MAX_COMMAND_LENGTH];
	char cmdstring[MAX_COMMAND_LENGTH];

	/* Clean up */
	for ( i = 0; i < main_data->num_tracks; i++ )
	{
		if ( main_data->track[ i ].wav_exist == FALSE )
			remove( main_data->track[ i ].wav_file );
		else if ((config.cddb_config.make_directories == TRUE) && (main_data->disc_artist[0] != 0))
		{
				/* create directory hierarchy */
				snprintf(dirstring, MAX_COMMAND_LENGTH, "%s/%s - %s/", config.wav_path, main_data->disc_artist, main_data->disc_title);
				snprintf(cmdstring, MAX_COMMAND_LENGTH , "mkdir \"%s\"", dirstring);
				system(cmdstring);
				
				/* move mp3 files in directory hierarchie */
				snprintf(cmdstring, MAX_COMMAND_LENGTH, "mv \"%s\" \"%s\"", main_data->track[i].wav_file, dirstring);
				system( cmdstring );
		}
		main_data->track[ i ].make_wav = FALSE;

		if ( main_data->track[ i ].mp3_exist == FALSE )
			remove( main_data->track[ i ].mp3_file );
		else if ((config.cddb_config.make_directories == TRUE) && (main_data->disc_artist[0] != 0))
		{
				/* create directory hierarchy */
				snprintf(dirstring, MAX_COMMAND_LENGTH, "%s/%s - %s/", config.mp3_path, main_data->disc_artist, main_data->disc_title);
				snprintf(cmdstring, MAX_COMMAND_LENGTH, "mkdir \"%s\"", dirstring);
				system(cmdstring);
				
				/* move mp3 files in directory hierarchie */
				snprintf(cmdstring, MAX_COMMAND_LENGTH, "mv \"%s\" \"%s\"", main_data->track[i].mp3_file, dirstring);
				system( cmdstring );
		}
		main_data->track[ i ].make_mp3 = FALSE;
	}

	/* Clear status bar */
	main_window_handler( MW_CLEAR_STATUSBAR, NULL, NULL );
	/* Create select frame */
	select_frame_handler( WIDGET_CREATE, 0, main_data );
	main_window_handler( MW_MODE_SELECT, NULL, NULL );

}

int find_next_job( _main_data *main_data,
                   int cur_track, int cur_type,
                   int *next_track, int *next_type )
{
	int flag, track;

	if ( cur_track != -1
    		&& cur_type == WAV
    		&& main_data->track[ cur_track ].make_mp3 == TRUE )
	{
		*next_track = cur_track;
		*next_type = MP3;
		return 0;
	}

	/* Find next track */
	flag = 1;
	track = cur_track + 1;

	while ( track < main_data->num_tracks && flag )
	{
		if ( main_data->track[ track ].make_wav
    			|| main_data->track[ track ].make_mp3 )
			flag = 0;
		else
			track++;
	}

	if ( flag )
		return - 1;
	else
	{
		if ( main_data->track[ track ].make_mp3
    			&& main_data->track[ track ].wav_exist == TRUE
    			&& config.make_mp3_from_existing_wav == TRUE )
		{
			*next_track = track;
			*next_type = MP3;
		}
		else
		{
			*next_track = track;
			*next_type = WAV;
		}
	}
	return 0;
}

int job_controller_timeout( gpointer data )
{
	int ops;
	static int timer;

	ops = ( int )data;

	switch ( ops )
	{
		case JC_T_START :
			timer = gtk_timeout_add( JC_TIMEOUT,
                         			job_controller_timeout, ( gpointer )JC_T_UPDATE );
			return TRUE;

		case JC_T_UPDATE :
			job_controller( JC_UPDATE, NULL );
			return TRUE;

		case JC_T_STOP :
			gtk_timeout_remove( timer );
			return TRUE;
	}
	/* Just to avoid warning msg */
	return FALSE;
}

void job_controller( int ops, _main_data *main_data )
{
	static int cur_track = -1, cur_type = -1;
	static pid_t pg_pid = -1;
	static pid_t pi_pid = -1;
	static int read_fd;
	static _stat stat;
	static _main_data *saved_main_data;
	unsigned current;
	double progress;
	char msg[ MAX_PLUGIN_OUTPUT_LENGTH ];
	char *str;
	int temp;

	switch( ops )
	{
		case JC_START :
			saved_main_data = main_data;

			if ( cur_track != -1 )
			{
				err_handler( JOB_IN_PROGRESS_ERR, NULL );
				job_finisher( main_data );
				return;
			}

			if ( find_next_job( main_data, cur_track, cur_type,
                				&cur_track, &cur_type ) < 0 )
			{
				err_handler( NOTHING_TO_DO_ERR, NULL );
				job_finisher( main_data );
				return;
			}

			calc_stat( main_data, &stat, current,
           			cur_track, cur_type, CALC_START_SESSION );
			calc_stat( main_data, &stat, current,
           			cur_track, cur_type, CALC_START );

			// start ripping/encoding
			if ( start_ripping_encoding( cur_type,
                         				main_data->track[ cur_track ].begin,
                         				main_data->track[ cur_track ].length,
                         				cur_track,
                         				main_data->track[ cur_track ].wav_file,
                         				main_data->track[ cur_track ].mp3_file,
                         				&pg_pid, &pi_pid,
                         				&read_fd ) < 0 )
			{
				job_finisher( main_data );
				return;
			}

			/* Create wav/mp3 status frame */
			wm_status_frame_handler( WIDGET_CREATE, cur_type, &stat, NULL );

			job_controller_timeout( JC_T_START );
			return;

		case JC_UPDATE :
			main_data = saved_main_data;

			temp = read_and_process_plugin_output( read_fd, &progress, msg );
			current = main_data->track[ cur_track ].length * progress;

			switch ( temp )
			{
				case PLUGIN_MSG_PARSE_ERR :
					/* Nothing to do. Let's wait more */
					return;

				case PLUGIN_PROGRESS_MSG :
					// progress report, update status display
					calc_stat( main_data, &stat, current,
           					cur_track, cur_type, CALC_UPDATE );
					/* Update status widget */
					if ( msg[ 0 ] == '\0' )
						str = NULL;
					else
						str = msg;
					wm_status_frame_handler( WIDGET_UPDATE, cur_type, &stat, str );
					return;

				case PLUGIN_NO_MSG_AVAILABLE :
					// Check if the plugin has exited
					// It only happens when ripperX failed to execute the plugin
					if ( pi_pid >= 0 )
						if ( waitpid( pi_pid, NULL, WNOHANG ) == pi_pid )
						{
							err_handler( PLUGIN_NOT_PRESENT_ERR,
             							"Maybe ripperX has failed to execute the plugin" );
							pi_pid = -1;
							job_controller_timeout( ( gpointer )JC_T_STOP );
							job_controller( JC_ABORT_ALL_DELETE, main_data );
							return;
						}

					/* Check if the job is finished */
					if ( waitpid( pg_pid, NULL, WNOHANG ) != pg_pid )
						// nothing has happened. need to wait more
						return;
					else
					{
						/* FIXME: try looking here */
						/* One job finished, go for next one */

						// kill the plugin
						if ( waitpid( pi_pid, NULL, WNOHANG ) != pi_pid )
						{
							kill( pi_pid, SIGTERM );
							waitpid( pi_pid, NULL, 0 );
						}
						/* Close the fd */
						close( read_fd );
						calc_stat( main_data, &stat, current,
           						cur_track, cur_type, CALC_STOP );

						if ( cur_type == WAV )
						{
							/* Destroy wav status frame */
							wm_status_frame_handler( WIDGET_DESTROY, WAV, &stat, msg );
							/* Mark that it exists */
							main_data->track[ cur_track ].wav_exist = TRUE;
						}
						else
						{
							/* Destroy mp3 status frame */
							wm_status_frame_handler( WIDGET_DESTROY, MP3, &stat, NULL );

							main_data->track[ cur_track ].mp3_exist = TRUE;
							/* Delete WAV file if he/she doesn't want it */
							if ( main_data->track[ cur_track ].make_wav == FALSE )
							{
								if ( remove( main_data->track[ cur_track ].wav_file ) < 0 )
									err_handler( FILE_DELETE_ERR,
             									main_data->track[ cur_track ].wav_file );
								/* Mark that it has been deleted */
								main_data->track[ cur_track ].wav_exist = FALSE;
							}
						}

						if ( find_next_job( main_data, cur_track, cur_type,
                							&cur_track, &cur_type ) < 0 )
						{
							/* All finished */
							pg_pid = -1;
							pi_pid = -1;
							cur_track = -1;
							cur_type = -1;
							calc_stat( main_data, &stat, current,
           							cur_track, cur_type, CALC_STOP_SESSION );
							job_controller_timeout( ( gpointer )JC_T_STOP );
							job_finisher( main_data );
							return;
						}

						calc_stat( main_data, &stat, current,
           						cur_track, cur_type, CALC_START );

						// start ripping/encoding
						if ( start_ripping_encoding( cur_type,
                         							main_data->track[ cur_track ].begin,
                         							main_data->track[ cur_track ].length,
                         							cur_track,
                         							main_data->track[ cur_track ].wav_file,
                         							main_data->track[ cur_track ].mp3_file,
                         							&pg_pid, &pi_pid,
                         							&read_fd ) < 0 )
						{
							calc_stat( main_data, &stat, current,
           							cur_track, cur_type, CALC_STOP_SESSION );
							job_controller_timeout( ( gpointer )JC_T_STOP );
							job_finisher( main_data );
							return;
						}
						/* Create status widget */
						wm_status_frame_handler( WIDGET_CREATE, cur_type, &stat, NULL );
						return;
					}
			}
			// end of JC_UPDATE
		case JC_PAUSE :
			main_data = saved_main_data;

			if ( pg_pid >= 0 )
				if ( waitpid( pg_pid, NULL, WNOHANG ) != pg_pid )
					kill( pg_pid, SIGTSTP );
			if ( pi_pid >= 0 )
				if ( waitpid( pi_pid, NULL, WNOHANG ) != pi_pid )
					kill( pi_pid, SIGTSTP );
			calc_stat( main_data, &stat, current,
           			cur_track, cur_type, CALC_PAUSE );
			job_controller_timeout( ( gpointer )JC_T_STOP );
			return;

		case JC_CONT :
			main_data = saved_main_data;

			if ( pg_pid >= 0 )
				if ( waitpid( pg_pid, NULL, WNOHANG ) != pg_pid )
					kill( pg_pid, SIGCONT );
			if ( pi_pid >= 0 )
				if ( waitpid( pi_pid, NULL, WNOHANG ) != pi_pid )
					kill( pi_pid, SIGCONT );
			calc_stat( main_data, &stat, current,
           			cur_track, cur_type, CALC_CONT );
			job_controller_timeout( ( gpointer )JC_T_START );
			return;

		case JC_ABORT :
		case JC_ABORT_DELETE :
			main_data = saved_main_data;

			if ( pg_pid >= 0 )
				if ( waitpid( pg_pid, NULL, WNOHANG ) != pg_pid )
				{
					job_controller( JC_CONT, NULL );
					kill( pg_pid, SIGINT );
					waitpid( pg_pid, NULL, 0 );
				}
			if ( pi_pid >= 0 )
				if ( waitpid( pi_pid, NULL, WNOHANG ) != pi_pid )
				{
					kill( pi_pid, SIGINT );
					waitpid( pi_pid, NULL, 0 );
				}
			/* Clear the pipe */
			close( read_fd );
			if ( cur_type == WAV )
			{
				/* Destroy wav status widget */
				wm_status_frame_handler( WIDGET_DESTROY, WAV, &stat, msg );

				/* Delete current file if we're called with DELETE */
				if ( ops == JC_ABORT_DELETE )
				{
					if ( remove( main_data->track[ cur_track ].wav_file ) < 0 )
						err_handler( FILE_DELETE_ERR, main_data->track[ cur_track ].wav_file );
				}
				else
					/* Mark that it exists */
					main_data->track[ cur_track ].wav_exist = TRUE;
			}
			else
			{
				/* Destroy wav status widget */
				wm_status_frame_handler( WIDGET_DESTROY, MP3, &stat, NULL );

				if ( ops == JC_ABORT_DELETE )
				{
					if ( remove( main_data->track[ cur_track ].mp3_file ) < 0 )
						err_handler( FILE_DELETE_ERR, main_data->track[ cur_track ].mp3_file );
				}
				else
					main_data->track[ cur_track ].mp3_exist = TRUE;

				/* Delete WAV file if he/she doesn't want it */
				if ( main_data->track[ cur_track ].make_wav == FALSE )
				{
					if ( remove( main_data->track[ cur_track ].wav_file ) < 0 )
						err_handler( FILE_DELETE_ERR,
             						main_data->track[ cur_track ].wav_file );
					/* Mark that it has been deleted */
					main_data->track[ cur_track ].wav_exist = FALSE;
				}
			}

			calc_stat( main_data, &stat, current,
           			cur_track, cur_type, CALC_STOP );

			/* If ABORT_DELETE, go for next track */
			if ( ops == JC_ABORT_DELETE )
				cur_type = MP3;

			if ( find_next_job( main_data, cur_track, cur_type,
                				&cur_track, &cur_type ) < 0 )
			{
				/* All finished */
				pg_pid = -1;
				pi_pid = -1;
				cur_track = -1;
				cur_type = -1;
				calc_stat( main_data, &stat, current,
           				cur_track, cur_type, CALC_STOP_SESSION );
				job_controller_timeout( ( gpointer )JC_T_STOP );
				job_finisher( main_data );
				return;
			}

			calc_stat( main_data, &stat, current,
           			cur_track, cur_type, CALC_START );

			// start ripping/encoding
			if ( start_ripping_encoding( cur_type,
                         				main_data->track[ cur_track ].begin,
                         				main_data->track[ cur_track ].length,
                         				cur_track,
                         				main_data->track[ cur_track ].wav_file,
                         				main_data->track[ cur_track ].mp3_file,
                         				&pg_pid, &pi_pid,
                         				&read_fd ) < 0 )
			{
				calc_stat( main_data, &stat, current,
           				cur_track, cur_type, CALC_STOP_SESSION );
				job_controller_timeout( ( gpointer )JC_T_STOP );
				job_finisher( main_data );
				return;
			}
			/* Create status widget */
			wm_status_frame_handler( WIDGET_CREATE, cur_type, &stat, NULL );

			return;

		case JC_ABORT_ALL :
		case JC_ABORT_ALL_DELETE :
			main_data = saved_main_data;

			if ( pg_pid >= 0 )
				if ( waitpid( pg_pid, NULL, WNOHANG ) != pg_pid )
				{
					job_controller( JC_CONT, NULL );
					kill( pg_pid, SIGINT );
					waitpid( pg_pid, NULL, 0 );
				}
			if ( pi_pid >= 0 )
				if ( waitpid( pi_pid, NULL, WNOHANG ) != pi_pid )
				{
					kill( pi_pid, SIGINT );
					waitpid( pi_pid, NULL, 0 );
				}

			/* Close the pipe or pty */
			close( read_fd );
			calc_stat( main_data, &stat, current,
           			cur_track, cur_type, CALC_STOP_SESSION );

			if ( cur_type == WAV )
			{
				/* Destroy wav status widget */
				wm_status_frame_handler( WIDGET_DESTROY, WAV, &stat, msg );

				if ( ops == JC_ABORT_ALL_DELETE )
				{
					if ( remove( main_data->track[ cur_track ].wav_file ) < 0 )
						err_handler( FILE_DELETE_ERR, main_data->track[ cur_track ].wav_file );
				}
				else
					/* Mark that it exists */
					main_data->track[ cur_track ].wav_exist = TRUE;
			}
			else
			{
				/* Destroy mp3 status widget */
				wm_status_frame_handler( WIDGET_DESTROY, MP3, &stat, NULL );

				if ( ops == JC_ABORT_ALL_DELETE )
				{
					if ( remove( main_data->track[ cur_track ].mp3_file ) < 0 )
						err_handler( FILE_DELETE_ERR, main_data->track[ cur_track ].mp3_file );
				}
				else
					main_data->track[ cur_track ].mp3_exist = TRUE;

				/* Do not delete WAV file even if he/she doesn't want it */
			}

			cur_track = -1;
			cur_type = -1;
			calc_stat( main_data, &stat, current,
           			cur_track, cur_type, CALC_STOP_SESSION );
			job_controller_timeout( ( gpointer )JC_T_STOP );
			job_finisher( main_data );
			return;

			default :
			return;
	}
}
