/*
 *  Routines for sampling from Linux soundcards (/dev/dsp)
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "freq.h"

#ifdef SC_LINUX

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include "extern.h"

/* Function prototypes */
void reset_linux_sc(void);
void halt_linux_sc(void);
void cleanup_linux_sc(void);
void recordblock_linux_sc(void *buffer);
void set_mixer_linux_sc(int mix,int level);

int _fp_sound=0;

int lx_devmask,lx_stereodev,lx_devcaps,lx_exclusive_input=1;
int lx_pcmlevel=0;

void init_linux_sc(void)
{
   int parm;
   int mix;

   DOUT("Setting up soundcard pointers");
   reset_soundcard=reset_linux_sc;
   halt_soundcard=halt_linux_sc;
   cleanup_soundcard=cleanup_linux_sc;
   recordblock=recordblock_linux_sc;
   set_mixer=NULL;
   mixers=0;

   /* If the audio device is undefined, define the default */
   if(audio_device==NULL)
      audio_device="/dev/dsp";

   /* If the audio device begins with /dev/audio, flag it */
   if(strncmp(audio_device,"/dev/audio",10)==0)
   {
      printf("You selected %s for DSP input, changing this to /dev/dsp\n",audio_device);
      audio_device="/dev/dsp";
   }

   /* Open and initialize the PCM device */
   DOUT("Opening the DSP device");
   _fp_sound=open(audio_device,O_RDONLY);
   if(_fp_sound<=0)
   {
      printf("Unable to open the audio device: %s\n",audio_device);
      exit(1);
   }
   DOUT("Setting the buffer block size - 32 blocks of 256 bytes");
   parm=0x00200008;
   if(ioctl(_fp_sound,SNDCTL_DSP_SETFRAGMENT,&parm)>=0)
      { DOUT("  Successful"); }
   else
      { DOUT("  Failed"); }

   DOUT("Resetting the DSP device");
   if(ioctl(_fp_sound,SOUND_PCM_RESET,0)>=0) 
      { DOUT("  Successful"); }
   else 
      { DOUT("  Failed"); }

   DOUT("Syncing the DSP device");
   if(ioctl(_fp_sound,SOUND_PCM_SYNC,0)>=0)
      { DOUT("  Successful"); }
   else
      { DOUT("  Failed"); }

   DOUT("Setting the DSP device bits per sample to 16");
   sample_size=16;
   if(ioctl(_fp_sound,SOUND_PCM_WRITE_BITS,&sample_size)>=0)
      { DOUT("  Successful"); }
   else
      { DOUT("  Failed"); }

   DOUT("Setting the DSP device number of channels to 1");
   parm=1;
   if(ioctl(_fp_sound,SOUND_PCM_WRITE_CHANNELS,&parm)>=0)
      { DOUT("  Successful"); }
   else
      { DOUT("  Failed"); }

   /* Verify the results */
   if((sample_size!=16) && (sample_size!=8))
   {
      printf("DSP error: returned a sample size of %d bits.\n",sample_size);
      exit(1);
   }

   #ifdef LINUX_MIXER
      /* If the mixer device is undefined, set the default */
      if(mixer_device==NULL)
	 mixer_device="/dev/mixer";

      DOUT("Checking out the mixer");
      mix=open(mixer_device,O_RDWR);
      if(mix >= 0)
      {
	 DOUT("A mixer device does exist");
	 ioctl(mix,SOUND_MIXER_READ_DEVMASK,&lx_devmask);
	 ioctl(mix,SOUND_MIXER_READ_STEREODEVS,&lx_stereodev);
	 ioctl(mix,SOUND_MIXER_READ_CAPS,&lx_devcaps);
	 
	 /* Get and save the PCM level, then set it to 0 */
	 /* This prevents the clicking noises when buffers start and stop */
	 ioctl(mix,SOUND_MIXER_READ_PCM,&lx_pcmlevel);
	 parm=0;
	 ioctl(mix,SOUND_MIXER_WRITE_PCM,&parm);
	 
	 if(lx_devcaps&SOUND_CAP_EXCL_INPUT)
	 {
	    lx_exclusive_input=1;
	    DOUT("Mixer requires exclusive inputs");
	 }
	 else
	 {
	    int src=SOUND_MASK_LINE|SOUND_MASK_CD|SOUND_MASK_MIC;
	    lx_exclusive_input=0;
	    DOUT("Mixer allows multiple simultaneous inputs");
	    ioctl(mix,SOUND_MIXER_WRITE_RECSRC,&src);
	 }
	 if(lx_devmask&SOUND_MASK_LINE)
	 {
	    int level,left,right=0;
	    DOUT("Found a Line mixer");
	    ioctl(mix,SOUND_MIXER_READ_LINE,&level);
	    left=level&0x7f;
	    if(lx_stereodev&SOUND_MASK_LINE)
	    {
	       right=(level>>8)&0x7f;
	       DOUT("   in stereo");
	    }
	    if(right>left) left=right; /* Use the larger */
	    ext_level=left; 
	    mixers=1;
	 }
	 if(lx_devmask&SOUND_MASK_MIC)
	 {
	    int level,left,right=0;
	    DOUT("Found a Mic mixer");
	    ioctl(mix,SOUND_MIXER_READ_MIC,&level);
	    left=level&0x7f;
	    if(lx_stereodev&SOUND_MASK_MIC)
	    {
	       right=(level>>8)&0x7f;
	       DOUT("   in stereo");
	    }
	    if(right>left) left=right; /* Use the larger */
	    mic_level=left; 
	    mixers=1;
	 }
	 if(lx_devmask&SOUND_MASK_CD)
	 {
	    int level,left,right=0;
	    DOUT("Found a CD mixer");
	    ioctl(mix,SOUND_MIXER_READ_CD,&level);
	    left=level&0x7f;
	    if(lx_stereodev&SOUND_MASK_CD)
	    {
	       right=(level>>8)&0x7f;
	       DOUT("   in stereo");
	    }
	    if(right>left) left=right; /* Use the larger */
	    int_level=left; 
	    mixers=1;
	 }
	 DOUT("Closing the mixer device");
	 close(mix);
      }
      if(mixers)
         set_mixer=set_mixer_linux_sc;
   #endif
}

void reset_linux_sc(void)
{
   int i,sr=SampleRate;

   DOUT("Syncing the DSP device");
   ioctl(_fp_sound,SOUND_PCM_SYNC,0);

   DOUT("Setting the DSP device sampling rate");
   /* Correct the sampling rate using the fudgefactor */
   if((fudgefactor>0.5) && (fudgefactor<2))
      sr*=fudgefactor;
   ioctl(_fp_sound,SOUND_PCM_WRITE_RATE,&sr);
   if((fudgefactor>0.5) && (fudgefactor<2))
      sr/=fudgefactor;
   SampleRate=sr;
   if(SampleRate<1000)
   {
      printf("DSP error: returned a sample rate of %ld samples/second.\n",
             SampleRate);
      exit(1);
   }

   DOUT("Initializing the buffers");
   /* Reset the buffer pointers */
   queue_buffer=0;     /* Pointer to next buffer to be queued */
   record_buffer=0;    /* Pointer to next buffer to be filled */
   process_buffer=0;   /* Pointer to next buffer to be FFTed  */

   for(i=0;i<BUFFERS;i++)
      flag[i]=0;
   /*
    *  This function starts the DMA process.
    */
   recordblock_linux_sc(buffer[queue_buffer]);
   if(++queue_buffer>=BUFFERS)
      queue_buffer=0;
}


void halt_linux_sc(void)
{
   ioctl(_fp_sound,SOUND_PCM_SYNC,0);
}


void cleanup_linux_sc(void)
{
   #ifdef LINUX_MIXER
      if(mixers)
      {
	 /* Restore the PCM mixer level */
	 int mixer=open(mixer_device,O_RDWR);
	 if(mixer>=0) 
	 {
	    ioctl(mixer,SOUND_MIXER_WRITE_PCM,&lx_pcmlevel);
	    close(mixer);
	 }
      }
      ioctl(_fp_sound,SOUND_PCM_RESET,0);
   #endif
   close(_fp_sound);
}

void recordblock_linux_sc(void *buffer)
{
   if(sample_size==8)
      read(_fp_sound,buffer,fftlen);
   else
      read(_fp_sound,buffer,fftlen*2);
   /* Toggle the buffer-filled flag */
   flag[record_buffer]=1;
   if(++record_buffer>=BUFFERS)
      record_buffer=0;   
}

#ifdef LINUX_MIXER

void set_mixer_linux_sc(int mix,int level)
{
   /* Set the mixer level for the Linux mixer device */
   int mixer=open(mixer_device,O_RDWR);
   int src=SOUND_MASK_LINE|SOUND_MASK_CD|SOUND_MASK_MIC;

   DOUT("Opening the mixer device");
   if(mixer>=0)
   {
      if((mix==MIXER_EXT) && (lx_devmask&SOUND_MASK_LINE))
      {
	 DOUT("Setting the external line level");
	 if(lx_stereodev&SOUND_MASK_LINE)
	    level+=level*256;
	 /* Set the mixer level, select this as the active recording source */
	 ioctl(mixer,SOUND_MIXER_WRITE_LINE,&level);
	 if(lx_exclusive_input)
	    src=SOUND_MASK_LINE;
	 ioctl(mixer,SOUND_MIXER_WRITE_RECSRC,&src);
      }
      else if((mix==MIXER_INT) && (lx_devmask&SOUND_MASK_CD))
      {
	 DOUT("Setting the internal line (CD) level");
	 if(lx_stereodev&SOUND_MASK_CD)
	    level+=level*256;
	 /* Set the mixer level, select this as the active recording source */
	 ioctl(mixer,SOUND_MIXER_WRITE_CD,&level);
	 if(lx_exclusive_input)
	    src=SOUND_MASK_CD;
	 ioctl(mixer,SOUND_MIXER_WRITE_RECSRC,&src);
      }
      else if((mix==MIXER_MIC) && (lx_devmask&SOUND_MASK_MIC))
      {
	 DOUT("Setting the mic level");
	 if(lx_stereodev&SOUND_MASK_MIC)
	    level+=level*256;
	 /* Set the mixer level, select this as the active recording source */
	 ioctl(mixer,SOUND_MIXER_WRITE_MIC,&level);
	 if(lx_exclusive_input)
	    src=SOUND_MASK_MIC;
	 ioctl(mixer,SOUND_MIXER_WRITE_RECSRC,&src);
      }
      DOUT("Closing the mixer device");
      close(mixer);
   }
}
#endif /* LINUX_MIXER */
#endif /* SC_LINUX */



