/*
**  The JAZZ++ Midi Sequencer
**
** Copyright (C) 1994-2000 Andreas Voss and Per Sigmond, all rights reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
*/                                                                              

#include "alsaplay.h"
#include "jazz.h"
#include "trackwin.h"

#include <stdlib.h>
#include <errno.h>
#include <sys/ioctl.h>


#if USE_PURGE
// wait until this code appears in the official distribution
void tAlsaPlayer::purge_queues()
{
  snd_seq_reset_pool_t info;
  memset(&info, 0, sizeof(info));
  info.reset_input = SND_SEQ_RESET_POOL_ALL;
  info.reset_output = SND_SEQ_RESET_POOL_ALL;
  info.output_match = obcast;
  info.output_match = ibcast;
  int fd = snd_seq_file_descriptor(handle);
  if (ioctl(fd, SND_SEQ_IOCTL_RESET_POOL, &info) < 0)
    perror("reset_pool");
}
void tAlsaPlayer::clear_input_queue() { }

#else

void tAlsaPlayer::purge_queues() {}
void tAlsaPlayer::clear_input_queue() {
  snd_seq_event_t *ie;
  while (snd_seq_event_input(handle, &ie) >= 0 && ie != 0)
    snd_seq_free_event(ie);
}

#endif



#if 0
void tAlsaPlayer::set_pool_sizes()
{
  snd_seq_client_pool_t pool;
  memset(&pool, 0, sizeof(pool));
  pool.client = client;
  if (snd_seq_get_client_pool(handle, &pool) < 0)
    perror("get_pool");
  // defaults: output_pool = 500, input_pool = 200, output_room = 250
  // cout << "op " << pool.output_pool << ", ip " << pool.input_pool << ", or " << pool.output_room << endl;
  pool.output_pool = 5000;
  pool.input_pool  = 2000;
  pool.output_room = 1000;
  if (snd_seq_set_client_pool(handle, &pool) < 0)
    perror("set_pool");

}
#else
void tAlsaPlayer::set_pool_sizes() {}
#endif




tAlsaPlayer::tAlsaPlayer(tSong *song)
  : tPlayer(song)
{
  int i;
  thru = 0;
  ithru = othru = 0;

  installed = 1;
  poll_millisec = 25;
  start_clock = 0;
  recd_clock = 0;
  echo_clock = 0;

  if (snd_seq_open(&handle, SND_SEQ_OPEN) < 0) {
    perror("open sequencer");
    installed = 0;
    return;
  }

  // set myself into non blocking mode
  if (set_blocking_mode(0) < 0) {
    installed = 0;
    return;
  }
  client = snd_seq_client_id(handle);

  // create my input addrs
  memset(&iself, 0, sizeof(iself));
  iself.client  = client;
  iself.port    = create_port(handle, "Input");
  iself.queue   = snd_seq_alloc_queue(handle);
  iself.channel = 0;

  // create my output address
  memset(&oself, 0, sizeof(iself));
  oself.client  = client;
  oself.port    = create_port(handle, "Output");
  oself.queue   = snd_seq_alloc_queue(handle);
  oself.channel = 0;

  // register my name
  set_client_info(handle, "The JAZZ++ Midi Sequencer");

  // scan input addressess
  scan_clients(iaddr, SND_SEQ_PORT_CAP_READ | SND_SEQ_PORT_CAP_SUBS_READ);
  //iaddr.print("Input Devices");
  for (i = 0; i < iaddr.GetCount(); i++) {
    iaddr[i].queue   = iself.queue;
    iaddr[i].channel = 0;
  }
  subscribe_inp();

  // scan output addresses
  scan_clients(oaddr, SND_SEQ_PORT_CAP_WRITE | SND_SEQ_PORT_CAP_SUBS_WRITE);
  //oaddr.print("Output Devices");
  for (i = 0; i < oaddr.GetCount(); i++) {
    oaddr[i].queue   = oself.queue;
    oaddr[i].channel = 0;
  }
  subscribe_out();

  obcast = oself;
  obcast.client = SND_SEQ_ADDRESS_BROADCAST;
  obcast.channel = SND_SEQ_ADDRESS_BROADCAST;
  ibcast = iself;
  ibcast.client = SND_SEQ_ADDRESS_BROADCAST;
  ibcast.channel = SND_SEQ_ADDRESS_BROADCAST;

  set_pool_sizes();

  if (installed) {
    thru = new tAlsaThru();
    SetSoftThru(Config(C_SoftThru), Config(C_ThruInput), Config(C_ThruOutput));
  }
}

void tAlsaPlayer::SetSoftThru(int on, int idev, int odev) 
{
  if (idev != ithru || odev != othru) {
    ithru = idev;
    othru = odev;
    thru->Stop();
  }
  if (on && !thru->IsRunning()) {
    thru->SetSource(iaddr[ithru].client, iaddr[ithru].port);
    thru->SetDestin(oaddr[othru].client, oaddr[othru].port);
    thru->Start();
  }
  else if (!on && thru->IsRunning())
    thru->Stop();
}

// connect output addrs/queue with my client/oport
void tAlsaPlayer::subscribe_out() {
  int i;
  snd_seq_port_subscribe_t subs;
  memset(&subs, 0, sizeof(subs));
  subs.realtime = 0;
  subs.exclusive = 0;
  for (i = 0; i < oaddr.GetCount(); i++) {
    subs.sender = oself;
    subs.dest   = oaddr[i];
    if (snd_seq_subscribe_port(handle, &subs) < 0) {
      perror("subscribe output");
      installed = 0;
    }
  }
}

// connect input addrs/queue with my client/iport
void tAlsaPlayer::subscribe_inp() {
  int i;
  snd_seq_port_subscribe_t subs;
  memset(&subs, 0, sizeof(subs));
  subs.realtime = 0;
  subs.exclusive = 0;
  for (i = 0; i < iaddr.GetCount(); i++) {
    subs.dest   = iself;
    subs.sender = iaddr[i];
    if (snd_seq_subscribe_port(handle, &subs) < 0) {
      perror("subscribe input");
      installed = 0;
    }
  }
}

// set the name of this client 
void tAlsaPlayer::set_client_info(snd_seq_t *handle, const char *name) {
  snd_seq_client_info_t inf;
  memset(&inf, 0, sizeof(snd_seq_client_info_t));
  snd_seq_get_client_info(handle, &inf);
  strcpy(inf.name, name);
  if (snd_seq_set_client_info(handle, &inf) < 0) {
    perror("ioctl");
  }
}

// create a new port
int tAlsaPlayer::create_port(snd_seq_t *handle, const char *name)
{
  snd_seq_port_info_t info;

  memset(&info, 0, sizeof(info));
  info.capability = SND_SEQ_PORT_CAP_READ | SND_SEQ_PORT_CAP_SUBS_READ 
                  | SND_SEQ_PORT_CAP_WRITE | SND_SEQ_PORT_CAP_SUBS_WRITE;
  info.type =  SND_SEQ_PORT_TYPE_MIDI_GENERIC;
  info.midi_channels = 16;
  info.read_use = 1;
  info.write_use = 1;
  info.kernel = NULL;
  strcpy(info.name, name);
  
  if (snd_seq_create_port(handle, &info) < 0) {
    perror("error creating alsa port");
    return -1;
  }
  return info.port;
}


int tAlsaPlayer::Installed()
{
  return installed;
}

tAlsaPlayer::~tAlsaPlayer()
{
  snd_seq_close(handle);
  if (thru)
    delete thru;
}

// 0 = event successfully sent to driver
// 1 = try again later
int tAlsaPlayer::OutEvent(tEvent *e, int now)
{
  int rc = 0;
  snd_seq_event_t ev;
  long clock = now ? start_clock : e->Clock;
  int  device = e->GetDevice();
  switch (e->Stat)
  {
    case StatKeyOn:
      {
	tKeyOn *k = e->IsKeyOn();
        set_event_header(&ev, device, clock, k->Channel, SND_SEQ_EVENT_NOTEON);
        ev.data.note.note = k->Key;
        ev.data.note.velocity = k->Veloc;
        rc = (write(&ev, now) < 0);
      }
      break;

    case StatKeyOff:
      {
	tKeyOff *k = e->IsKeyOff();
        set_event_header(&ev, device, clock, k->Channel, SND_SEQ_EVENT_NOTEOFF);
        ev.data.note.note = k->Key;
        ev.data.note.velocity = k->OffVeloc;
        rc = (write(&ev, now) < 0);
      }
      break;

    case StatProgram:
      {
	tProgram *k = e->IsProgram();
	set_event_header(&ev, device, clock, k->Channel, SND_SEQ_EVENT_PGMCHANGE);
	ev.data.control.value = k->Program;
	rc = (write(&ev, now) < 0);
      }
      break;

    case StatKeyPressure:
      {
	tKeyPressure *k = e->IsKeyPressure();
	set_event_header(&ev, device, clock, k->Channel, SND_SEQ_EVENT_KEYPRESS);
	ev.data.control.param = k->Key;
	ev.data.control.value = k->Value;
	rc = (write(&ev, now) < 0);
      }
      break;

    case StatChnPressure:
      {
	tChnPressure *k = e->IsChnPressure();
	set_event_header(&ev, device, clock, k->Channel, SND_SEQ_EVENT_CHANPRESS);
	ev.data.control.value = k->Value;
	rc = (write(&ev, now) < 0);
      }
      break;

    case StatControl:
      {
	tControl *k = e->IsControl();
	set_event_header(&ev, device, clock, k->Channel, SND_SEQ_EVENT_CONTROLLER);
	ev.data.control.param = k->Control;
	ev.data.control.value = k->Value;
	rc = (write(&ev, now) < 0);
      }
      break;

    case StatPitch:
      {
	tPitch *k = e->IsPitch();
	set_event_header(&ev, device, clock, k->Channel, SND_SEQ_EVENT_PITCHBEND);
	ev.data.control.value = k->Value;
	rc = (write(&ev, now) < 0);
      }
      break;

    case StatSetTempo:
      {
        int bpm = e->IsSetTempo()->GetBPM();
	int us  = (int)( 60.0E6 / (double)bpm );
	set_timer_event_header(&ev, oself, clock, SND_SEQ_EVENT_TEMPO);
	ev.data.control.value = us;
	ev.dest = obcast;   // send to all clients
	rc = (write(&ev) < 0);
	set_timer_event_header(&ev, iself, clock, SND_SEQ_EVENT_TEMPO);
	ev.data.control.value = us;
	ev.dest = ibcast;   // send to all clients
	rc = (write(&ev, now) < 0);
      }
      break;

    case StatSysEx:
      {
        tSysEx *s = e->IsSysEx();
	// prepend 0xf0
	char *buf = new char[s->Length + 1];
	buf[0] = 0xF0;
	memcpy(buf + 1, s->Data, s->Length);
	set_event_header(&ev, clock, device, 0, s->Length + 1, buf);
	rc = (write(&ev, now) < 0);
	delete [] buf;
      }
      break;

    default:
      break;
  }
  if (rc)
    cout << "full" << endl;
  if (rc || now)
    snd_seq_flush_output(handle);
  return rc;
}


void tAlsaPlayer::OutBreak()
{
  snd_seq_flush_output(handle);
  // OutBreak(OutClock);
}

void tAlsaPlayer::OutBreak(long clock)
{
  snd_seq_flush_output(handle);
}

void tAlsaPlayer::StartPlay(long clock, long loopClock, int cont)
{
  recd_clock = clock;
  echo_clock = clock;
  start_clock = clock;
  purge_queues();
  clear_input_queue();
  set_blocking_mode(1);
  start_timer(clock);
  set_blocking_mode(0);
  tPlayer::StartPlay(clock, loopClock, cont);
  Notify();
  snd_seq_flush_output(handle);
}


void tAlsaPlayer::set_event_header(snd_seq_event_t *ev, int device, long clock, int chan, int type)
{
  memset(ev, 0, sizeof(*ev));
  ev->source = oself;
  ev->source.channel = chan;
  ev->dest = oaddr[device];
  ev->dest.channel = chan;
  ev->flags = SND_SEQ_TIME_STAMP_TICK | SND_SEQ_TIME_MODE_ABS | SND_SEQ_EVENT_LENGTH_FIXED;
  ev->time.tick = clock - start_clock;
  ev->type = type;
}

void tAlsaPlayer::set_event_header(snd_seq_event_t *ev, int device, long clock, int chan, int len, void *ptr)
{
  memset(ev, 0, sizeof(*ev));
  ev->source = oself;
  ev->dest = oaddr[device];
  ev->dest.channel = chan & 0xf;
  ev->flags = SND_SEQ_TIME_STAMP_TICK | SND_SEQ_TIME_MODE_ABS | SND_SEQ_EVENT_LENGTH_VARIABLE;
  ev->time.tick = clock - start_clock;
  ev->type = SND_SEQ_EVENT_SYSEX;
  ev->data.ext.len = len;
  ev->data.ext.ptr = ptr;
}


void tAlsaPlayer::set_timer_event_header(snd_seq_event_t *ev, snd_seq_addr_t &addr, long clock, int type) 
{
  ev->source = addr;
  ev->dest = addr;
  ev->dest.client = SND_SEQ_CLIENT_SYSTEM;        /* system */
  ev->dest.port = SND_SEQ_PORT_SYSTEM_TIMER;      /* timer */
  ev->time.tick = clock - start_clock;
  ev->flags = SND_SEQ_TIME_STAMP_TICK | SND_SEQ_TIME_MODE_ABS | SND_SEQ_EVENT_LENGTH_FIXED;
  ev->type = type;
}


int tAlsaPlayer::start_timer(long clock) 
{
  int time_base = Song->TicksPerQuarter;
  int cur_speed = Song->GetTrack(0)->GetCurrentSpeed(clock);
  init_queue_tempo(iself.queue, time_base, cur_speed);
  init_queue_tempo(oself.queue, time_base, cur_speed);
  start_queue_timer(iself, clock);
  start_queue_timer(oself, clock);
}

void tAlsaPlayer::init_queue_tempo(int queue, int time_base, int bpm)
{
  // set initial tempo
  snd_seq_queue_tempo_t qtempo;
  memset(&qtempo, 0, sizeof(qtempo));
  qtempo.queue = queue;
  qtempo.ppq   = time_base;
  qtempo.tempo = 60*1000000/bpm;
  if (snd_seq_set_queue_tempo(handle, queue, &qtempo) < 0)
    perror("set_queue_tempo");
}


void tAlsaPlayer::start_queue_timer(snd_seq_addr_t &addr, long clock)
{
  snd_seq_event_t ev;
  set_timer_event_header(&ev, addr, clock, SND_SEQ_EVENT_START);
  write(&ev, 1);
}

void tAlsaPlayer::stop_queue_timer(snd_seq_addr_t &addr, long clock)
{
  snd_seq_event_t ev;
  set_timer_event_header(&ev, addr, clock, SND_SEQ_EVENT_STOP);
  write(&ev, 1);
}

// write an event, return < 0 on failure
int tAlsaPlayer::write(snd_seq_event_t *ev, int now) {
  if (now) {
#ifdef USE_DIRECT
    ev->flags &= ~SND_SEQ_DEST_MASK;
    ev->flags |= SND_SEQ_DEST_DIRECT;
#endif
    ev->time.tick = 0;
  }
  return snd_seq_event_output(handle, ev);
}

void tAlsaPlayer::flush_output() {
  while (snd_seq_flush_output(handle) > 0)
    usleep(2000); // fixme
}

int tAlsaPlayer::set_blocking_mode(int enable) {
  int rc;
  if ((rc = snd_seq_block_mode(handle, enable)) < 0)
    perror("blocking mode");
  return rc;
}

void tAlsaPlayer::StopPlay()
{
  snd_seq_event_t *ep;
  tPlayer::StopPlay();
  snd_seq_drain_output(handle);
  snd_seq_drain_input(handle);
  set_blocking_mode(1);
  purge_queues();
  AllNotesOff();
  flush_output();
  stop_queue_timer(iself, start_clock);
  stop_queue_timer(oself, start_clock);
  flush_output();
  set_blocking_mode(0);
  while (snd_seq_event_input(handle, &ep) >= 0 && ep != 0)
    snd_seq_free_event(ep);
  TrackWin->NewPlayPosition(-1L);
  RecdBuffer.Keyoff2Length();
}

void tAlsaPlayer::recd_event(snd_seq_event_t *ev)
{
  tEvent *e = 0;
#if 0
  if (ev->type < 20)
    cout << ev->time.tick << ":" << (int)ev->type << endl;
  switch (ev->type) 
  {
    case SND_SEQ_EVENT_NOTEON:
      cout << "note on  clk: " << ev->time.tick << ", ch: " << (int)ev->source.channel << ", key: " << (int)ev->data.note.note << ", vel: " << (int)ev->data.note.velocity << endl;
      break;
    case SND_SEQ_EVENT_NOTEOFF:
      cout << "note off clk: " << ev->time.tick << ", ch: " << (int)ev->source.channel << ", key: " << (int)ev->data.note.note << ", vel: " << (int)ev->data.note.velocity << endl;
      break;
    case SND_SEQ_EVENT_CONTROLLER:
      cout << "ctrl: " << ev->time.tick << ", ch: " << (int)ev->source.channel << ", nr: " << ev->data.control.param << ", val: " << ev->data.control.value << endl;
      break;
  }
#endif
  switch (ev->type) 
  {

    case SND_SEQ_EVENT_NOTEON:
      if (ev->data.note.velocity > 0)
        e = new tKeyOn(0, ev->source.channel, ev->data.note.note, ev->data.note.velocity);
      else
        e = new tKeyOff(0, ev->source.channel, ev->data.note.note, 0);
      break;

    case SND_SEQ_EVENT_NOTEOFF:
      e = new tKeyOff(0, ev->source.channel, ev->data.note.note, ev->data.note.velocity);
      break;

    case SND_SEQ_EVENT_PGMCHANGE:
      e = new tProgram(0, ev->source.channel, ev->data.control.value);
      break;

    case SND_SEQ_EVENT_KEYPRESS:
      e = new tKeyPressure(0, ev->source.channel, ev->data.note.note, ev->data.note.velocity);
      break;

    case SND_SEQ_EVENT_CHANPRESS:
      e = new tChnPressure(0, ev->source.channel, ev->data.control.value);
      break;

    case SND_SEQ_EVENT_CONTROLLER:
      e = new tControl(0, ev->source.channel, ev->data.control.param, ev->data.control.value);
      break;

    case SND_SEQ_EVENT_PITCHBEND:
      e = new tPitch(0, ev->source.channel, ev->data.control.value);
      break;

    case SND_SEQ_EVENT_SYSEX:
      e = new tSysEx(0, ((unsigned char *)ev->data.ext.ptr) + 1, ev->data.ext.len - 1);
      break;

  }
  if (e) {
    e->Clock = PlayLoop->Ext2IntClock(ev->time.tick + start_clock);
    RecdBuffer.Put(e);
  }
}


long tAlsaPlayer::GetRealTimeClock()
{
  // send echo events to myself to keep track of current play position
  while (echo_clock < OutClock)
  {
    snd_seq_event_t ev;
    ev.source = iself;
    ev.dest = iself;
    ev.flags = SND_SEQ_TIME_STAMP_TICK | SND_SEQ_TIME_MODE_ABS | SND_SEQ_EVENT_LENGTH_FIXED;
    ev.type = SND_SEQ_EVENT_ECHO;
    ev.time.tick = 48 + echo_clock - start_clock;
    if (write(&ev) < 0)
      break;
    echo_clock += 48;
  }
  snd_seq_flush_output(handle);
  
  // input recorded events (including my echo events)
  snd_seq_event_t *ie;
  long old_recd_clock = recd_clock;
  while (snd_seq_event_input(handle, &ie) >= 0 && ie != 0) {
    recd_clock = ie->time.tick + start_clock;
    recd_event(ie);
    snd_seq_free_event(ie);
  }
  if (recd_clock != old_recd_clock)
    TrackWin->NewPlayPosition(PlayLoop->Ext2IntClock(recd_clock/48 * 48));
  return recd_clock;
}



void tAlsaPlayer::scan_clients(tAlsaDeviceList &list, int cap)
{
  /* stolen from pmidi */
  snd_seq_client_info_t cinfo;
  snd_seq_port_info_t pinfo;
  snd_seq_system_info_t sysinfo;
  int client;
  int port;

  list.Clear();

  snd_seq_system_info (handle, &sysinfo);
  for (client = 0; client < sysinfo.clients; client++)
  {
    snd_seq_get_any_client_info (handle, client, &cinfo);
    for (port = 0; port < sysinfo.ports; port++)
    {
      snd_seq_get_any_port_info (handle, client, port, &pinfo);
      if ((pinfo.capability & cap) == cap) {
        if (pinfo.client == this->client) // dont play to myself
	  continue;
        snd_seq_addr_t a;
	memset(&a, 0, sizeof(a));
	a.client = pinfo.client;
	a.port   = pinfo.port;
	char buf[500];
	strcpy(buf, cinfo.name);
	strcat(buf, " ");
	strcat(buf, pinfo.name);
        list.add(buf, a);
      }
    }
  }
}


void tAlsaDeviceList::print(const char *msg) {
  cout << msg << endl;
  for (int i = 0; i < count; i++) {
    snd_seq_addr_t &a = operator[](i);
    cout << GetName(i) << " = " << (int)a.client << ":" << (int)a.port << endl;
  }
}

int tAlsaDeviceList::add(const char *name, const snd_seq_addr_t &a) {
  addr[count] = a;
  names[count] = copystring(name);
  return count++;
}

snd_seq_addr_t& tAlsaDeviceList::operator[](int i) {
  if (i >= count)
    return addr[0];
  return addr[i];
}

