/*
**  The JAZZ++ Midi Sequencer
**
** Copyright (C) 1994-2000 Andreas Voss and Per Sigmond, all rights reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
*/                                                                              

#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <sys/asoundlib.h>
#include <sys/wait.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>

#include "alsathru.h"
#include "alsaplay.h"

/*
** midi thru for alsa. it creates a new process (because threads dont work 
** with wxwin) that copies from input to output. 
*/

tAlsaThru::tAlsaThru()
{
  source_client = 64;
  source_port   = 0;
  destin_client = 64;
  destin_port   = 0;
  running       = 0;
}


tAlsaThru::~tAlsaThru()
{
  if (running)
    Stop();
}


void tAlsaThru::connect(connection_t *con, int source_client, int source_port, int destin_client, int destin_port)
{
  snd_seq_port_subscribe_t subs;

  int queue = snd_seq_alloc_queue(handle);
  memset(con, 0, sizeof(*con));
  con->source.client  = source_client;
  con->source.port    = source_port;
  con->source.queue   = queue;
  con->source.channel = 0;

  con->destin.client  = destin_client;
  con->destin.port    = destin_port;
  con->destin.queue   = queue;
  con->destin.channel = 0;

  /* subscribe */
  memset(&subs, 0, sizeof(subs));
  subs.realtime = 0;
  subs.exclusive = 0;
  subs.sender = con->source;
  subs.dest = con->destin;
  if (snd_seq_subscribe_port(handle, &subs) < 0) {
    perror("subscribe");
    exit(1);
  }
}


void tAlsaThru::initialize()
{
  int client, port;

  if (snd_seq_open(&handle, SND_SEQ_OPEN) < 0) {
    perror("open");
    exit(1);
  }
  tAlsaPlayer::set_client_info(handle, "Jazz++ Midi Thru");

  if (snd_seq_block_mode(handle, 1) < 0) {
    perror("blocking mode");
    exit(1);
  }

  client  = snd_seq_client_id(handle);
  port    = tAlsaPlayer::create_port(handle, "Input/Output");
  connect(&out, client, port, destin_client, destin_port);
  connect(&inp, source_client, source_port, client, port);
}


void tAlsaThru::loop()
{
  snd_seq_event_t *ev;
  while (snd_seq_event_input(handle, &ev) >= 0 && ev != 0) {
    int channel = ev->source.channel;
    ev->flags &= ~SND_SEQ_TIME_STAMP_MASK;
    ev->flags |= SND_SEQ_TIME_STAMP_TICK;
#ifdef USE_DIRECT
    ev->flags &= ~SND_SEQ_DEST_MASK;
    ev->flags |= SND_SEQ_DEST_DIRECT;
#endif
    ev->time.tick = 0;
    ev->source = out.source;
    ev->dest   = out.destin;
    ev->dest.channel = channel;
    snd_seq_event_output(handle, ev);
    snd_seq_flush_output(handle);
    snd_seq_free_event(ev);
  }
}


#if 0

// thread version

void tAlsaThru::stopworker(int sig) 
{
  running = 0;
  snd_seq_close(handle);
  pthread_exit((void *)0);
}


void * tAlsaThru::startworker(void *p)
{
  running = 1;
  signal(SIGHUP, stopworker);
  tAlsaThru *thru = (tAlsaThru *)p;
  thru->initialize();
  thru->loop();
  return 0;
}

void tAlsaThru::Start() 
{
  if (!running)
    pthread_create(&worker, (void *)0, startworker, (void *)this);
}

void tAlsaThru::Stop()
{
  if (running)
    pthread_kill(worker, SIGHUP); 
}



#else

// fork version

static snd_seq_t *static_handle;  // ugly!!

void tAlsaThru::stopworker(int sig) 
{
  snd_seq_close(static_handle);
  exit(0);
}

void tAlsaThru::Start() 
{
  if (!running) {
    worker = fork();
    if (worker < 0) {
      perror("fork");
      exit(1);
    }
    if (worker == 0) {
      signal(SIGHUP, stopworker);
      initialize();
      loop();
      static_handle = handle;
      // never come here
      perror("loop");
      exit(1);
    }
    running = 1;
  }
}

void tAlsaThru::Stop()
{
  if (running) {
    int status = 0;
    kill(worker, SIGHUP);
    wait(&status);
    running = 0;
  }
}


#endif


