/*
**  The JAZZ++ Midi Sequencer
**
** Copyright (C) 1994-2000 Andreas Voss and Per Sigmond, all rights reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
*/                                                                              

#include "wx.h"
#pragma hdrstop

#include "config.h"
#include "dialogs.h"
#include "song.h"
#include "command.h"
#include "eventwin.h"
#include "track.h"
#include "events.h"
#include "util.h"
#include "jazz.h"
#include "player.h"
#include "pianowin.h"

// **************************************************************************
// Quantize
// *************************************************************************

Bool tQuantizeDlg::NoteStart = 1;
Bool tQuantizeDlg::NoteLength = 0;
long tQuantizeDlg::QntStep = 16;
int  tQuantizeDlg::Delay = 0;
int  tQuantizeDlg::Groove = 0;

static tNamedValue QntSteps[] =
{
  tNamedValue( "1/8",   8 ),
  tNamedValue( "1/12", 12 ),
  tNamedValue( "1/16", 16 ),
  tNamedValue( "1/24", 24 ),
  tNamedValue( "1/32", 32 ),
  tNamedValue( "1/48", 48 ),
  tNamedValue( "1/96", 96 ),
  tNamedValue(  0,      1  )
};


tQuantizeDlg::tQuantizeDlg(tEventWin *w, tFilter *f)
   : wxForm( USED_WXFORM_BUTTONS ), Steps("steps", QntSteps, &QntStep)
{
  Filter = f;
  Song = f->Song;
  EventWin = w;
}



void tQuantizeDlg::OnOk()
{
  Steps.GetValue();
  int step = Song->TicksPerQuarter * 4 / QntStep;
  tCmdQuantize qnt(Filter, step, Groove * step / 100, Delay * step / 100);
  qnt.NoteStart = NoteStart;
  qnt.NoteLength = NoteLength;
  qnt.Execute();
  EventWin->Redraw();
  if (EventWin->NextWin)
 	 EventWin->NextWin->Redraw();
  wxForm::OnOk();
}

void tQuantizeDlg::OnHelp()
{
  	if (EventWin->NextWin)
		HelpInstance->ShowTopic("Quantize");
	else
		HelpInstance->ShowTopic("Pianowin Quantize");
}

void tQuantizeDlg::EditForm(wxPanel *panel)
{
  Add(Steps.mkFormItem(100));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormBool("Note start", &NoteStart));
  Add(wxMakeFormBool("Note length", &NoteLength));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Groove ", &Groove, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(-100, 100), 0)));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Delay ", &Delay, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(-100, 100), 0)));
  AssociatePanel(panel);
}


// **************************************************************************
// Cleanup
// *************************************************************************

long tCleanupDlg::lowLimit = 48;
Bool tCleanupDlg::shortenOverlaps = 1;



tCleanupDlg::tCleanupDlg(tEventWin *w, tFilter *f)
   : wxForm( USED_WXFORM_BUTTONS ), Steps("Shortest note", limitSteps, &lowLimit)
{
  Filter = f;
  Song = f->Song;
  EventWin = w;
}



void tCleanupDlg::OnOk()
{
  Steps.GetValue();
  int limit = Song->TicksPerQuarter * 4 / lowLimit;
  tCmdCleanup cln(Filter, limit, shortenOverlaps);
  cln.Execute();
  EventWin->Redraw();
  if (EventWin->NextWin)
 	 EventWin->NextWin->Redraw();
  wxForm::OnOk();
}

void tCleanupDlg::OnHelp()
{
	HelpInstance->ShowTopic("Cleanup");
}

void tCleanupDlg::EditForm(wxPanel *panel)
{
  panel->SetLabelPosition(wxVERTICAL);
  Add(wxMakeFormMessage("Delete notes shorter than"));
  Add(wxMakeFormNewLine());
  Add(Steps.mkFormItem(100));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormBool("Shorten overlapping notes", &shortenOverlaps));
  Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}

// **************************************************************************
// SearchReplace
// *************************************************************************

long tSearchReplaceDlg::frCtrl = 1;
long tSearchReplaceDlg::toCtrl = 1;

tSearchReplaceDlg::tSearchReplaceDlg(tEventWin *w, tFilter *f)
   : wxForm( USED_WXFORM_BUTTONS ),
     frList("Search", &Config.CtrlName(0), &frCtrl),
     toList("Replace", &Config.CtrlName(0), &toCtrl)
{
  Filter = f;
  Song = f->Song;
  EventWin = w;
}

void tSearchReplaceDlg::OnOk()
{
  frList.GetValue();
  toList.GetValue();
  tCmdSearchReplace sr(Filter, frCtrl-1, toCtrl-1);
  sr.Execute();
  EventWin->Redraw();
  if (EventWin->NextWin)
    EventWin->NextWin->Redraw();
  wxForm::OnOk();
}

void tSearchReplaceDlg::OnHelp()
{
  HelpInstance->ShowTopic("Search Replace");
}

void tSearchReplaceDlg::EditForm(wxPanel *panel)
{
  //panel->SetLabelPosition(wxVERTICAL);
  Add(wxMakeFormMessage("Search and replace controller types"));
  Add(wxMakeFormNewLine());
  Add(frList.mkFormItem(200));
  Add(toList.mkFormItem(200));
  //Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}

// **************************************************************************
// Shift
// *************************************************************************

int tShiftDlg::Steps = 0;

tShiftDlg::tShiftDlg(tEventWin *w, tFilter *f, long unit)
: wxForm( USED_WXFORM_BUTTONS )
{
  Filter = f;
  Song = f->Song;
  Unit  = unit;
  EventWin = w;
}



void tShiftDlg::OnOk()
{
  tCmdShift cmd(Filter, Steps * Unit);
  cmd.Execute();
  EventWin->Redraw();
  if (EventWin->NextWin)
  	EventWin->NextWin->Redraw();
  wxForm::OnOk();
}

void tShiftDlg::OnHelp()
{
	HelpInstance->ShowTopic("Shift");
}


void tShiftDlg::EditForm(wxPanel *panel)
{
  char buf[100];
  Add(wxMakeFormMessage("Shift events left/right"));
  Add(wxMakeFormNewLine());
  sprintf(buf, "Snap is currently %ld clocks", Unit);
  Add(wxMakeFormMessage(buf));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Snaps", &Steps, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(-16.0, 16.0), 0)));
  Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}


// **************************************************************************
// Transpose
// *************************************************************************

static tNamedValue ScaleNames[] =
{
  tNamedValue("None",     ScaleChromatic),
  tNamedValue("Selected", ScaleSelected),
  tNamedValue("C",   0),
  tNamedValue("C#",  1),
  tNamedValue("D",   2),
  tNamedValue("D#",  3),
  tNamedValue("E",   4),
  tNamedValue("F",   5),
  tNamedValue("F#",  6),
  tNamedValue("G",   7),
  tNamedValue("G#",  8),
  tNamedValue("A",   9),
  tNamedValue("A#", 10),
  tNamedValue("B",  11),
  tNamedValue( 0,   ScaleChromatic)
};


int tTransposeDlg::Notes  = 0;
long tTransposeDlg::Scale  = ScaleChromatic;
Bool tTransposeDlg::FitIntoScale = 0;

tTransposeDlg::tTransposeDlg(tEventWin *w, tFilter *f)
   : wxForm( USED_WXFORM_BUTTONS ), ScaleDlg("scale", ScaleNames, &Scale)
{
  EventWin = w;
  Filter = f;
  Song   = f->Song;
}


void tTransposeDlg::OnOk()
{
  ScaleDlg.GetValue();
  tCmdTranspose trn(Filter, Notes, Scale, FitIntoScale);
  trn.Execute();
  if (EventWin->NextWin)
  	EventWin->NextWin->Redraw();
  else
  	EventWin->Redraw();
  wxForm::OnOk();
}

void tTransposeDlg::OnHelp()
{
	HelpInstance->ShowTopic("Transpose");
}


void tTransposeDlg::EditForm(wxPanel *panel)
{
  char buf[100];
  int s = tScale::Analyze(Filter);
  sprintf(buf, "selection looks like %s", ScaleNames[s+2].Name);
  Add(wxMakeFormMessage(buf));
  Add(wxMakeFormNewLine());

  Add(wxMakeFormShort("Amount", &Notes, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(-12.0, 12.0), 0)));
  Add(wxMakeFormNewLine());
  Add(ScaleDlg.mkFormItem(150));
  Add(wxMakeFormBool("Fit into Scale", &FitIntoScale));
  Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}


// **************************************************************************
// SetChannel
// *************************************************************************

int tSetChannelDlg::NewChannel = 1;

tSetChannelDlg::tSetChannelDlg(tFilter *f)
: wxForm( USED_WXFORM_BUTTONS )
{
  Filter = f;
  Song = f->Song;
}



void tSetChannelDlg::OnOk()
{
  if (NewChannel)
  {
    tCmdSetChannel exe(Filter, NewChannel - 1);
    exe.Execute();
  }
  wxForm::OnOk();
}

void tSetChannelDlg::OnHelp()
{
        HelpInstance->ShowTopic("Set MIDI Channel");
}


void tSetChannelDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormShort("new Channel", &NewChannel, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(1.0, 16.0), 0)));
  Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}


// **************************************************************************
// Velocity
// *************************************************************************

int tVelocityDlg::FromValue = 64;
int tVelocityDlg::ToValue = 0;
int tVelocityDlg::Mode = 0;
char *tVelocityDlg::mode_str = 0;

tVelocityDlg::tVelocityDlg(tFilter *f)
: wxForm( USED_WXFORM_BUTTONS )
{
  Filter = f;
  Song = f->Song;
  if (!mode_str)
    mode_str = copystring("Set");
}


void tVelocityDlg::OnOk()
{
  if (!strcmp(mode_str, "Set")) Mode = 0;
  if (!strcmp(mode_str, "Add")) Mode = 1;
  if (!strcmp(mode_str, "Sub")) Mode = 2;
  tCmdVelocity cmd(Filter, FromValue, ToValue, Mode);
  cmd.Execute();
  wxForm::OnOk();
}

void tVelocityDlg::OnHelp()
{
        HelpInstance->ShowTopic("Velocity");
}


void tVelocityDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormShort("Start", &FromValue, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(0.0, 127.0), 0)));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Stop ", &ToValue, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(0.0, 127.0), 0)));
  Add(wxMakeFormNewLine());

  Add(wxMakeFormString("Mode", &mode_str, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintStrings("Set", "Add", "Sub", 0) , 0), NULL, wxHORIZONTAL));
  AssociatePanel(panel);
}


// **************************************************************************
// Length
// *************************************************************************

int tLengthDlg::FromValue = 30;
int tLengthDlg::ToValue = 0;
int tLengthDlg::Mode = 0;
char *tLengthDlg::mode_str = 0;

tLengthDlg::tLengthDlg(tEventWin *w, tFilter *f)
: wxForm( USED_WXFORM_BUTTONS )
{
  Filter = f;
  Song = f->Song;
  EventWin = w;
  if (!mode_str)
    mode_str = copystring("Set");
}


void tLengthDlg::OnOk()
{
  if (!strcmp(mode_str, "Set")) Mode = 0;
  if (!strcmp(mode_str, "Add")) Mode = 1;
  if (!strcmp(mode_str, "Sub")) Mode = 2;
  tCmdLength cmd(Filter, FromValue, ToValue, Mode);
  cmd.Execute();
  EventWin->Redraw();
  if (EventWin->NextWin)
  	EventWin->NextWin->Redraw();
  wxForm::OnOk();
}

void tLengthDlg::OnHelp()
{
        HelpInstance->ShowTopic("Length");
}


void tLengthDlg::EditForm(wxPanel *panel)
{
  char buf[200];
  sprintf(buf, "Ticks per Quarter: %d", (int)Song->TicksPerQuarter);
  Add(wxMakeFormMessage(buf));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Start", &FromValue, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(0.1, Song->TicksPerQuarter * 4.0), 0)));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Stop ", &ToValue, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(0.0, Song->TicksPerQuarter * 4.0), 0)));
  Add(wxMakeFormNewLine());

  Add(wxMakeFormString("Mode", &mode_str, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintStrings("Set", "Add", "Sub", 0) , 0), NULL, wxHORIZONTAL));
  AssociatePanel(panel);
}



// *************************************************************************
// Delete
// *************************************************************************

Bool tDeleteDlg::LeaveSpace = 1;

tDeleteDlg::tDeleteDlg(tEventWin *w, tFilter *f)
: wxForm( USED_WXFORM_BUTTONS )
{
  Filter = f;
  EventWin = w;
}


void tDeleteDlg::OnOk()
{
  tCmdErase cmd(Filter, LeaveSpace);
  cmd.Execute();
  EventWin->Redraw();
  if (EventWin->NextWin)
  	EventWin->NextWin->Redraw();
  wxForm::OnOk();
}

void tDeleteDlg::OnHelp()
{
        HelpInstance->ShowTopic("Delete");
}

void tDeleteDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormBool("Leave Space", &LeaveSpace));
  AssociatePanel(panel);
}


// **************************************************************************
// Snap
// *************************************************************************

tSnapDlg::tSnapDlg(tPianoWin *w, long *snapptr)
   : wxForm( USED_WXFORM_BUTTONS ), Steps("Snap value", limitSteps, snapptr)
{
   win = w;
   ptr = snapptr;
}



void tSnapDlg::OnOk()
{
  Steps.GetValue();
  // toggle the tool buttons
  win->SetSnapDenom(*ptr);
  wxForm::OnOk();
}

void tSnapDlg::OnHelp()
{
  HelpInstance->ShowTopic("Snap");
}

void tSnapDlg::EditForm(wxPanel *panel)
{
  panel->SetLabelPosition(wxVERTICAL);
  Add(wxMakeFormMessage("quantize cut/paste events"));
  Add(wxMakeFormNewLine());
  Add(Steps.mkFormItem(100));
  Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}

// **************************************************************************
// MidiDevice
// *************************************************************************

tMidiDeviceDlg::tMidiDeviceDlg(tEventWin *w, tNamedValue *devs, long *devptr)
   : wxForm( USED_WXFORM_BUTTONS ), Steps("Devices", devs, devptr)
{
   win = w;
   ptr = devptr;
}



void tMidiDeviceDlg::OnOk()
{
  Steps.GetValue();
  wxForm::OnOk();
}

void tMidiDeviceDlg::OnHelp()
{
	HelpInstance->ShowTopic("MIDI device");
}

void tMidiDeviceDlg::EditForm(wxPanel *panel)
{
  panel->SetLabelPosition(wxVERTICAL);
  Add(Steps.mkFormItem(200));
  Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}

// ***********************************************************************
// Event-Dialoge
// ***********************************************************************

class tEventDlg : public wxForm
{
  public:

    tTrack    *Track;
    tClockDlg ClockDlg;
    tEventWin *Win;

    tEvent    *Event;
    tEvent    *Copy;

    tEventDlg(tEvent *e, tEventWin *w, tTrack *t);
    virtual void EditForm(wxPanel *panel);
    virtual void OnOk();
    virtual void OnHelp();
    virtual void OnCancel();
};


tEventDlg::tEventDlg(tEvent *e, tEventWin *w, tTrack *t)
  : wxForm( USED_WXFORM_BUTTONS ), ClockDlg(w->Song, "Time: ", e->Clock)
{
  Win   = w;
  Track = t;
  Event = e;
  Copy  = e->Copy();
}

void tEventDlg::EditForm(wxPanel *panel)
{
  Add(ClockDlg.mkFormItem(200));
  Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}

void tEventDlg::OnCancel()
{
  delete Copy;
  wxForm::OnCancel();
}

void tEventDlg::OnOk()
{
  Copy->Clock = ClockDlg.GetClock();
  Track->Kill(Event);
  Track->Put(Copy);
  Track->Cleanup();
  Win->Redraw();
  if (Win->NextWin)
  	Win->NextWin->Redraw();
  wxForm::OnOk();
}

void tEventDlg::OnHelp()
{
        HelpInstance->ShowTopic("Piano Window");
}

// --------------------------- ChannelEvent ----------------------------


class tChEventDlg : public tEventDlg
{
  public:

    int Channel;

    tChEventDlg(tChannelEvent *e, tEventWin *w, tTrack *t)
      : tEventDlg(e, w, t)
    {
      Channel = e->Channel + 1;		// 1..16
    }
    void EditForm(wxPanel *panel);
    void OnOk();
};

void tChEventDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormShort("Channel:", &Channel, wxFORM_DEFAULT, new wxList(wxMakeConstraintRange(1.0, 16.0), 0)));
  Add(wxMakeFormNewLine());
  tEventDlg::EditForm(panel);
}


void tChEventDlg::OnOk()
{
  ((tChannelEvent *)Copy)->Channel = Channel - 1;
  tEventDlg::OnOk();
}

// -------------------------------- Note On -------------------------------

class tKeyOnDlg : public tChEventDlg
{
 public:

  tKeyDlg PitchDlg;
  int Pitch;
  int Veloc;
  int Length;
  // SN++
  int OffVeloc;

  tKeyOnDlg(tKeyOn *e, tEventWin *w, tTrack *t);

  void EditForm(wxPanel *panel);
  void OnOk();
};


tKeyOnDlg::tKeyOnDlg(tKeyOn *e, tEventWin *w, tTrack *t)
  : tChEventDlg(e, w, t),
    PitchDlg("Pitch:", e->Key)
{
  Event = e;
  Veloc = e->Veloc;
  Pitch = e->Key;
  Length = e->Length;
  // SN++ Off veloc support
  OffVeloc = e->OffVeloc;
}


void tKeyOnDlg::OnOk()
{
  tKeyOn *k = (tKeyOn *)Copy;
  k->Key = PitchDlg.GetKey();
  k->Veloc = Veloc;
  k->Length = Length;
  // SN++ off veloc support
  k->OffVeloc = OffVeloc;
  tChEventDlg::OnOk();
}

void tKeyOnDlg::EditForm(wxPanel *panel)
{
  Add(PitchDlg.mkFormItem(100));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Veloc:", &Veloc, wxFORM_DEFAULT, new wxList(wxMakeConstraintRange(1.0, 127.0), 0)));
  Add(wxMakeFormNewLine());
  // SN++ off veloc support
  Add(wxMakeFormShort("OffVel:", &OffVeloc, wxFORM_DEFAULT, new wxList(wxMakeConstraintRange(0.0, 127.0), 0)));

  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Length:", &Length, wxFORM_DEFAULT,0,0,0,120));
  Add(wxMakeFormNewLine());
  tChEventDlg::EditForm(panel);
}


// -------------------------------- Pitch -------------------------------

class tPitchDlg : public tChEventDlg
{
 public:

  int Value;

  tPitchDlg(tPitch *e, tEventWin *w, tTrack *t);

  void EditForm(wxPanel *panel);
  void OnOk();
};


tPitchDlg::tPitchDlg(tPitch *e, tEventWin *w, tTrack *t)
  : tChEventDlg(e, w, t)
{
  Event = e;
  Value = e->Value;
}


void tPitchDlg::OnOk()
{
  ((tPitch *)Copy)->Value = Value;
  tChEventDlg::OnOk();
}

void tPitchDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormShort("Pitch:", &Value, wxFORM_DEFAULT, new wxList(wxMakeConstraintRange(-8191.0, 8191.0), 0)));
  Add(wxMakeFormNewLine());
  tChEventDlg::EditForm(panel);
}

// -------------------------------- Controller ---------------------------

class tControlDlg : public tChEventDlg
{
 public:

  int Value;
  long Control;
  tNamedChoice Choice;

  tControlDlg(tControl *e, tEventWin *w, tTrack *t);

  void EditForm(wxPanel *panel);
  void OnOk();
};


tControlDlg::tControlDlg(tControl *e, tEventWin *w, tTrack *t)
  : tChEventDlg(e, w, t),
    Choice("Controller", &Config.CtrlName(0), &Control)
{
  Event = e;
  Value = e->Value;
  Control = e->Control + 1;
}


void tControlDlg::OnOk()
{
  ((tControl *)Copy)->Value = Value;
  Choice.GetValue();
  ((tControl *)Copy)->Control = Control - 1;
  tChEventDlg::OnOk();
}

void tControlDlg::EditForm(wxPanel *panel)
{
  Add(Choice.mkFormItem(300, 300));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Value:", &Value, wxFORM_DEFAULT, new wxList(wxMakeConstraintRange(0.0, 127.0), 0)));
  Add(wxMakeFormNewLine());
  tChEventDlg::EditForm(panel);
}

// -------------------------------- Program ---------------------------

class tProgramDlg : public tChEventDlg
{
 public:

  long Program;
  tNamedChoice Choice;

  tProgramDlg(tProgram *e, tEventWin *w, tTrack *t);

  void EditForm(wxPanel *panel);
  void OnOk();
};


tProgramDlg::tProgramDlg(tProgram *e, tEventWin *w, tTrack *t)
  : tChEventDlg(e, w, t),
    Program(e->Program + 1),
    Choice("Program", &Config.VoiceName(0), &Program)
{
  Event = e;
}


void tProgramDlg::OnOk()
{
  Choice.GetValue();
  if (Program <= 0)
    Program = 1;
  ((tProgram *)Copy)->Program = Program - 1;
  tChEventDlg::OnOk();
}

void tProgramDlg::EditForm(wxPanel *panel)
{
  Add(Choice.mkFormItem(300, 300));
  Add(wxMakeFormNewLine());
  tChEventDlg::EditForm(panel);
}

// -------------------------------- Set Tempo -------------------------------

class tSetTempoDlg : public tEventDlg
{
 public:

  int Value;

  tSetTempoDlg(tSetTempo *e, tEventWin *w, tTrack *t);

  void EditForm(wxPanel *panel);
  void OnOk();
};


tSetTempoDlg::tSetTempoDlg(tSetTempo *e, tEventWin *w, tTrack *t)
  : tEventDlg(e, w, t)
{
  Event = e;
  Value = e->GetBPM();
}


void tSetTempoDlg::OnOk()
{
  ((tSetTempo *)Copy)->SetBPM( Value );
  tEventDlg::OnOk();
}

void tSetTempoDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormShort("Tempo:", &Value, wxFORM_DEFAULT, new wxList(wxMakeConstraintRange(20.0, 240.0), 0)));
  Add(wxMakeFormNewLine());
  tEventDlg::EditForm(panel);
}

// -------------------------------- Sysex Edit-------------------------------

class tSysexDlg : public tEventDlg
{
  public:

  char *str;

  tSysexDlg(tSysEx *s, tEventWin *w, tTrack *t);

  void EditForm(wxPanel *panel);
  void OnOk();
};


tSysexDlg::tSysexDlg(tSysEx *s, tEventWin *w, tTrack *t)
  : tEventDlg(s, w, t)
{
  Event = s;
  char hexbyte[10];

  str = new char[256];
  str[0] = 0;

  if (s->Length)
     strcat( str, "f0 " );

  for (int i = 0; i < s->Length; i++)
  {
     sprintf( hexbyte, "%02x ", s->Data[i] );
     strcat( str, hexbyte );
  }
}


void tSysexDlg::OnOk()
{
   int i;
   int j;
   int k;
   int len;

   uchar d[256];
   memset( d, 0, 256 );

   int jstop = strlen(str);

   for (i = 0, j = 0; j <= jstop; j += k, i++)
   {
      sscanf( str + j, "%02x %n", &d[i], &k );

      if (d[i] == 0xf7)
	 break;
   }

   int found = 0;
   for (i = 0; i < 256; i++)
   {
      if (d[i] == 0xf7)
      {
	 found = 1;
	 break;
      }
   }

   if (found)
      len = i + 1;
   else
      len = 0;

   long clk = ((tSysEx *)Copy)->Clock;
   delete Copy;
   Copy = new tSysEx( clk, d + 1, len - 1 );

   Synth->FixSysexCheckSum( Copy->IsSysEx() );

#if 0
   printf("Sysex:");
   for (i = 0; i < ((tSysEx *)Copy)->Length; i++)
   {
      printf( "%02x ", ((tSysEx *)Copy)->Data[i] );
   }
   printf("\n");

   Midi->OutNow( (tSysEx *)Copy );
#endif

   delete str;

   tEventDlg::OnOk();
}

void tSysexDlg::EditForm(wxPanel *panel)
{
   char label1[100];
   char label2[100];
   uchar *uptr;

   if (Event->IsSysEx()->Length)
   {
      sprintf( label1, "Loaded sysex: %s", SysexNames[Synth->GetSysexId(Event->IsSysEx())] );

      Add(wxMakeFormMessage(label1));
      Add(wxMakeFormNewLine());

      uptr = Synth->GetSysexValPtr(Event->IsSysEx());

      if (uptr)
      {
	 sprintf( label2, "First data byte is at offset %d, value %02x (%d decimal)", uptr - Event->IsSysEx()->Data + 1, *uptr, *uptr );
	 Add(wxMakeFormMessage(label2));
	 Add(wxMakeFormNewLine());
      }
   }
   else
   {
      Add(wxMakeFormMessage("Example input: f0 7f 7f 04 01 00 7f f7"));
      Add(wxMakeFormNewLine());
   }

   Add(wxMakeFormMessage("(any DT1/RQ1 checksums will be corrected)"));
   Add(wxMakeFormNewLine());

   Add(wxMakeFormString("SysEx (hex):", &str, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL, 300 ));
   Add(wxMakeFormNewLine());
   tEventDlg::EditForm(panel);
}

// --------------------------------------------------------------------------
// create new event
// --------------------------------------------------------------------------

static tEvent *CreateEventDialog(long Clock, int Channel, int Pitch)
{
  static char *Names[] =
  // { "Note On", "Pitch", "Controller", "Program Change", (char *)0 };
  { "Note On", "Controller", "Program Change", "Set Tempo", "SysEx", (char *)0 };
  static long Values[] =
  // { StatKeyOn, StatPitch, StatControl, StatProgram, -1 };
  { StatKeyOn, StatControl, StatProgram, StatSetTempo, StatSysEx, -1 };
  tEvent *e = 0;
  int i = wxGetSingleChoiceIndex("Select event to create", "Create Event", 5, Names);
  if (i >= 0)
  {
    switch (Values[i])
    {
      case StatKeyOn:
        e = new tKeyOn(Clock, Channel, Pitch, 64, 64);
        break;
      case StatPitch:
        e = new tPitch(Clock, Channel, 0);
        e->SetPitch(Pitch);
        break;
      case StatControl:
        e = new tControl(Clock, Channel, Pitch, 64);
        break;
      case StatProgram:
        e = new tProgram(Clock, Channel, Pitch);
        break;
     case StatSetTempo:
        e = new tSetTempo(Clock, 100);
        break;
     case StatSysEx:
	e = new tSysEx(Clock, (uchar*) "", 0);
	break;
    }
  }
  return e;
}




void EventDialog(tEvent *e, tEventWin *w, tTrack *t, long Clock, int Channel, int Pitch)
{
  if (!e)
    e = CreateEventDialog(Clock, Channel, Pitch);
  if (!e)
    return;

  tEventDlg *dlg = 0;
  char *str = 0;
  switch (e->Stat)
  {
    case StatKeyOn:
#ifdef AUDIO
      if (t->GetAudioMode()) {
        if (!Midi->IsPlaying())
	  Midi->EditSample(e->IsKeyOn()->Key);
	break;
      }
#endif
      str = "Key On";
      dlg = new tKeyOnDlg(e->IsKeyOn(), w, t);
      break;

    case StatPitch:
      str = "Pitch Wheel";
      dlg = new tPitchDlg(e->IsPitch(), w, t);
      break;

    case StatControl:
      str = "Controller";
      dlg = new tControlDlg(e->IsControl(), w, t);
      break;

    case StatProgram:
      str = "Program Change";
      dlg = new tProgramDlg(e->IsProgram(), w, t);
      break;

    case StatSetTempo:
      str = "Set Tempo (for track 0)";
      dlg = new tSetTempoDlg(e->IsSetTempo(), w, w->Song->GetTrack(0) );
      break;

    case StatSysEx:
      str = "System Exclusive";
      dlg = new tSysexDlg(e->IsSysEx(), w, t );
      break;

    default:
      break;
  }
  if (dlg)
  {
    wxDialogBox *panel = new wxDialogBox(w, str, FALSE );
    dlg->EditForm(panel);
    panel->Fit();
    panel->Show(TRUE);
  }
}
