/*
**  The JAZZ++ Midi Sequencer
**
** Copyright (C) 1994-2000 Andreas Voss and Per Sigmond, all rights reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
*/                                                                              

#include "sample.h"
#include "samplcmd.h"
#include "audio.h"
#include "signali.h"
#include "util.h"
#include <math.h>

#include <iostream.h>
#define db(a) cout << #a << " = " << a << endl


void tPaintableCommand1::Initialize()
{
  for (int i = 0; i < arr.Size(); i++)
    arr[i] = 0;
}

/**
 * adjust volume graphically. Compute in float because
 * new volume may exceed 32767
 */

void tSplVolume::Execute(long fr, long to)
{
  long n = to - fr;
  if (n <= 0)
    return; // no data

  tFloatSample fs(spl);
  float i_fact  = (float)arr.Size() / (float)n;
  for (long i = 0; i < n; i++)
  {
    float x = (float)i * i_fact;
    fs[fr+i] *= (100.0 + arr[x]) / 100.0;
  }
  fs.RescaleToShort();
  spl.Set(fs);
}


/**
 * adjust pan graphically, that is decrease volume of
 * one channel.
 */

void tSplPan::Execute(long fr, long to)
{
  long n = to - fr;
  if (n <= 0)
    return; // no data

  float i_fact  = (float)arr.Size() / (float)n;
  short *data = spl.data;
  for (long i = 0; i < n-1; i += 2)
  {
    float x = (float)i * i_fact;
    float p = arr[x];
    if (p < 0)
      data[fr+i]   = (short)((float)data[fr+i]   * (100 + p) / 100);
    else
      data[fr+i+1] = (short)((float)data[fr+i+1] * (100 - p) / 100);
  }
}


/**
 * adjust pitch graphically
 */

void tSplPitch::Execute(long fr, long to)
{
  long n = to - fr;
  if (n <= 0)
    return; // no data

  long channels = spl.GetChannels();
  float N       = spl.length / channels - 2;
  short *data   = spl.data;

  tFloatSample out(channels, spl.GetSamplingRate());
  out.SetNote(0, 0);

  float p[10];
  tMapper arr_ind(0, N, 0, arr.Size());
  //tMapper arr_val(-100, 100, 1.0 / range, range);
  tExpMapper arr_val(100, range);

  float x = 0;
  while (x < N)
  {
    float ofs = floor(x);
    float rem = x - ofs;

    long i = (long)ofs * channels;
    for (long c = 0; c < channels; c++)
    {
      tMapper map(0, 1, data[i + c], data[i + channels + c]);
      p[c] = map(rem);
    }
    out.AddOut(p);
    x += arr_val(arr[arr_ind(x)]);
  }
  out.EndNote();
  out.ClipToCurrent();
  spl.Set(out);
}


// ----------------------------------------------------------------
//                       small CMIX interface
// ----------------------------------------------------------------

tCMixCmd::tCMixCmd(float sr)
  : SR(sr)
{
  int i;
  resetval = 1000;
  for(i=0; i<SIZE; i++)
    array[i] = 1;
  lineset = 0;
}

void tCMixCmd::tableset(float dur, int size, float *tab)
{
  *tab = (long)(dur * SR  -.9999);
  *(tab+1) = size - 1;
}


float tCMixCmd::tablei(long nsample, float *array, float *tab)
{
	register int loc1,loc2;
        float frac = ((float)(nsample)/(*tab)) * *(tab+1);
	if(frac < 0) return(array[0]);
	if(frac >= *(tab+1)) return(array[(int)*(tab+1)]);
	loc1 = (int)frac;
	loc2 = loc1+1;
	frac = frac - (float)loc1;
	return(*(array+loc1) + frac * (*(array+loc2) - *(array+loc1)));
}

/* p0,2,4,5,6,8,10.. are times, p1,3,5,7,9,11.. are amps, total number of
 * arguments is n_args, result is stuffed into array array of length length
 */

void tCMixCmd::setline(const float *p, short n_args,int length,float *array)
{
	double increm;
	int i,j,k,points;

	increm = (double)(p[n_args - 2] - p[0])/(double)length;
	for(j=0,i=0; j < (n_args-2); j += 2) {
		points = (int)((double)(p[j+2] - p[j]) / increm +.5);
		if(p[j+2] != p[j]) {
			if(points <= 0) points = 1;
			for(k=0; k < points; k++) {
				array[i++] = ((float)k/(float)points)
					* (p[j+3] - p[j+1]) + p[j+1];
				if(i == length) return;
			}
		}
	}
	i--;
	while(++i < length) array[i] = array[i-1];
}


float tCMixCmd::cpspch(float pch)
{
  int oct = (int)pch;
  return (float) ((pow(2.,oct+8.333333*(pch-oct))*1.021975));
}

// ------------------------ wahwah ------------------------

tWahWah::tWahWah(tSample &s)
: tPaintableCommand1(s, 200, 0, 100)
{
  for (int i = 0; i < arr.Size(); i++)
    arr[i]  = i * 100 / arr.Size();
  filter_type = tSplFilter::BANDPASS;
  order       = 2;
  lo_freq     = 400;
  hi_freq     = 2000;
  band_width  = 0.2;
}



void tWahWah::Initialize()
{
}


void tWahWah::Execute(long fr, long to)
{
  tFloatSample *out = new tFloatSample(spl);
  for (int c = 0; c < out->GetChannels(); c++)
    Wah(c, *out);
  out->Rescale();
  spl.Set(*out);
  delete out;
}



void tWahWah::Wah(int channel, tFloatSample &out)
{
  long N = spl.GetLength();
  float SR = spl.GetSamplingRate();
  long channels = spl.GetChannels();
  tMapper xmap(0, N, 0, arr.Size());
  tMapper fmap(0, 100, lo_freq, hi_freq);

  tSplFilter flt;
  {
    float f = fmap(arr[0]);
    flt.Init(filter_type, SR, f, band_width);
  }
  int j = 0;
  short *data = spl.GetData();
  for (long i = channel; i < N; i += channels)
  {
    if (j-- == 0)
    {
      float x = (float)xmap(i);
      float f = fmap(arr[x]);
      flt.ReInit(f, band_width);
      j = 100;
    }
    out[i] = flt.Loop(data[i]);
  }
}


// ------------------------ rotater ------------------------

tShifterCmd::tShifterCmd(long sampling_rate)
  : tCMixCmd(sampling_rate)
{
}

/**
 * change the pitch of a sample. If keep_length == true, winsize should
 * be in 0..100
 */

void tShifterCmd::ShiftPitch(tSample &spl, float semis,  Bool keep_length, float winsize)
{
  if (semis == 0)
    return;

  if (keep_length)
  {
    tMapper wmap(0, 100, 0.05, 0.3);
    tFloatSample inp(spl);
    tFloatSample out(inp.GetChannels(), inp.GetSamplingRate());
    float p[8];
    p[0] = 0;
    p[1] = 0;
    p[2] = spl.Samples2Seconds(spl.GetLength());
    p[3] = 1.0;
    p[4] = semis / 100.0;
    p[5] = (float)wmap(winsize);
    p[6] = 0;
    p[7] = 0;
    rotate(p, 8, inp, out);
    if (inp.GetChannels() == 2)
    {
      p[6] = 1;
      p[7] = 1;
      rotate(p, 8, inp, out);
    }
    out.ClipToCurrent();
    spl.Set(out);
  }
  else
    spl.TransposeSemis(semis);
}

void tShifterCmd::StretchLength(tSample &spl, long newlen, Bool keep_pitch, float winsize)
{
  long oldlen = spl.GetLength();
  if (oldlen == newlen)
    return;

  double df = (double)oldlen / (double)newlen;
  // msvc cannot do this!
  // float  semis = (float)(log(df) / log( pow(2.0, 1.0/12.0) ));
  float semis = (float)( log(df) / 0.057762264 );
  spl.TransposeSemis(semis);

  if (keep_pitch)
  {
    tMapper wmap(0, 100, 0.05, 0.3);
    tFloatSample inp(spl);
    tFloatSample out(inp.GetChannels(), inp.GetSamplingRate());
    float p[8];
    p[0] = 0;
    p[1] = 0;
    p[2] = spl.Samples2Seconds(spl.GetLength());
    p[3] = 1.0;
    p[4] = -semis / 100.0;
    p[5] = (float)wmap(winsize);
    p[6] = 0;
    p[7] = 0;
    rotate(p, 8, inp, out);
    if (inp.GetChannels() == 2)
    {
      p[6] = 1;
      p[7] = 1;
      rotate(p, 8, inp, out);
    }
    out.ClipToCurrent();
    spl.Set(out);
  }
}


/*  rotate -- a pitch-shifting instrument based upon the idea
*	of old rotating tape-head pitch shifters
*
*  p0 = input skip
*  p1 = output skip
*  p2 = duration
*  p3 = amplitude multiplier (switched off - av)
*  p4 = pitch shift up or down (oct.pc)
*  p5 = window size
*  p6 = input channel number
*  p7 = stereo spread (0-1) [optional]
*  assumes function table 1 is the amplitude envelope  (switched off - av)
*  assumes function table 2 is the window envelope     (inline - av)
*	<usually a hanning window -- use "makegen(2, 25, 1000, 1)">
*
*/


double tShifterCmd::rotate(float p[], int n_args, tFloatSample &sinp, tFloatSample &sout)
{
  float samplenum1,samplenum2,x,interval;
  float val1,val2,in[2],out[2];
  int nsamps,i,j,k,off,reinit,chans,inchan;
  int octpart;
  float pcpart;
  float *wintable;
  int wlen;
  extern int resetval;

  sinp.SetNote(p[0], p[2]);
  nsamps = sout.SetNote(p[1], p[2]);

  tFloatSample hanning(1, sout.GetSamplingRate());
  hanning.HanningWindow(1000);
  wlen = 1000;
  wintable = hanning.GetData();

  octpart = (int)p[4] * 12;
  pcpart = (p[4] * 100.0) - (float)(octpart*100);
  interval =  pow(2.0, ((float)octpart + pcpart)/12.0) - 1.0;

  reinit = (int)(p[5] * SR);
  off = reinit/2;
  k = off;
  chans = sout.GetChannels();
  inchan = (int)p[6];
  j = 0;
  for(i = 0; i < nsamps; i++) {

    j = (j+1) % reinit;
    k = (k+1) % reinit;

    samplenum1 = (float)i + (float)j * interval;
    if(!sinp.GetSample(samplenum1, in)) break;
    x = wintable[(int)(((float)j/reinit) * wlen)];
    val1 = in[inchan] * x;

    samplenum2 = (float)(i) + (float)(k-off) * interval;
    if(!sinp.GetSample(samplenum2, in)) break;
    x = wintable[(int)(((float)k/reinit) * wlen)];
    val2 = in[inchan] * x;

    out[0] = (val1 + val2);
    if (chans > 1) {
      out[1] = (1.0 - p[7]) * out[0];
      out[0] *= p[7];
    }

    sout.AddOut(out);
  }

  sout.EndNote();
  return 0.0;
}


// -------------------------------------------------------------------------
//                        2-nd nogo equalizer
// -------------------------------------------------------------------------

tSplEqualizer::tSplEqualizer(tRndArray &arr, long sr)
  : array(arr),
    sampling_rate(sr)
{
  nfilters = array.Size();
  filters = new tSplFilter[nfilters];
}

void tSplEqualizer::Prepare()
{
  int i;

  tSplFilter::Type type;
  double sr = (double)sampling_rate;
  int order = 2;

  // first one is lowpass/highpass
  type = array[0] > 0 ? tSplFilter::LOWPASS : tSplFilter::HIGHPASS;
  filters[0].Init(type, sr, Index2Hertz(0), 0);
  // some band pass filters
  for (i = 1; i < nfilters-1; i++) {
    double f  = Index2Hertz(i);
    double f0 = Index2Hertz(i-1);
    double f1 = Index2Hertz(i+1);
    //double bw = (1 - f0/f1) / (1 + f0/f1);
    double bw = (f1 - f0) / f;
    type = array[i] > 0 ? tSplFilter::BANDPASS : tSplFilter::BANDSTOP;
    filters[i].Init(type, sr, f, bw);
  }
  // last one is high/low pass
  type = array[nfilters-1] > 0 ? tSplFilter::HIGHPASS : tSplFilter::LOWPASS;
  filters[nfilters-1].Init(type, sr, Index2Hertz(nfilters-1), 0);
}


tSplEqualizer::~tSplEqualizer()
{
  delete [] filters;
}


float tSplEqualizer::Index2Hertz(float index)
{
  tMapper xmap(0, nfilters, -1, 1);
  tExpMapper emap(1, 10);
  return 2000.0 * emap(xmap(index));
}



float tSplEqualizer::operator()(float x)
{
  // add all the outputs
  double y = x;
  tMapper ymap(array.Min(), array.Max(), -1, 1);
  for (int i = 0; i < nfilters; i++)
  {
    double amp = fabs(ymap(array[i]));
    y = (1 - amp) * y + amp * filters[i].Loop(y);
  }
  return (float)y / (float)nfilters;
}


