/*
**  The JAZZ++ Midi Sequencer
**
** Copyright (C) 1994-2000 Andreas Voss and Per Sigmond, all rights reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
*/                                                                              

#include "wx.h"
#pragma hdrstop

#include "config.h"
#include "trackwin.h"
#include "song.h"
#include "mstdfile.h"
#include "filter.h"
#include "pianowin.h"
#include "command.h"
#include "dialogs.h"
#include "harmony.h"
#include "rhythm.h"
#include "mapper.h"
#include "gs_dlgs.h"
#include "jazz.h"
#include "toolbar.h"
#include "shuffle.h"
#include "genmeldy.h"
#include "eventlst.h"
#include "arpeggio.h"

#ifdef wx_msw
#include "winplay.h"
#include "winaudio.h"
#else // not wx_msw
#include "player.h"
#include "audiodrv.h"
#endif

#include "about.h"

#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#ifdef wx_xt
#define wxbMessageBox wxMessageBox
#endif

tTrackWin *TrackWin = 0;
static char *defsong = 0;
static char *defpattern = 0;
char *lasts = "noname.mid";

#define MEN_LOAD 	1
#define MEN_SAVE 	2
#define MEN_QUIT 	3
#define MEN_ABOUT 	4
#define MEN_TWSETTING	5
#define MEN_FILTER	6
#define MEN_METERCH	7
#define MEN_MERGE	8
#define MEN_NEW		9
#define MEN_REPLICATE	10
#define MEN_DELETE	11
#define MEN_QUANTIZE	12
#define MEN_UNDO	13
#define MEN_DEBUG	14
#define MEN_SETCHAN	15
#define MEN_TRANSP	16
#define MEN_VELOC	17
#define MEN_LENGTH	18
#define MEN_RESET       19
#define MEN_SONG	20
#define MEN_TMERGE	21
#define MEN_TSPLIT	22
#define MEN_MIXER	23
#define MEN_SOUND	24
#define MEN_VIBRATO	25
#define MEN_ENVELOPE	26
#define MEN_BEND_BASIC	27
#define MEN_BEND_LFO1	28
#define MEN_BEND_LFO2	29
#define MEN_EFFECTS	30
#define MEN_TIMING	31
#define MEN_MOD_BASIC	32
#define MEN_MOD_LFO1	33
#define MEN_MOD_LFO2	34
#define MEN_CAF_BASIC	35
#define MEN_CAF_LFO1	36
#define MEN_CAF_LFO2	37
#define MEN_PAF_BASIC	38
#define MEN_PAF_LFO1	39
#define MEN_PAF_LFO2	40
#define MEN_CC1_BASIC	41
#define MEN_CC1_LFO1	42
#define MEN_CC1_LFO2	43
#define MEN_CC2_BASIC	44
#define MEN_CC2_LFO1	45
#define MEN_CC2_LFO2	46
#define MEN_PART_RSRV	47
#define MEN_PART_MODE	48
#define MEN_HARMONY     49
#define MEN_RHYTHM      50
#define MEN_MASTER	51
#define MEN_SHIFT	52
#define MEN_HELP_MOUSE  53
#define MEN_DRUM_PARAM  54
#define MEN_HELP_JAZZ   55
#define MEN_HELP_TWIN   56
#define MEN_SUB_BENDER  57
#define MEN_SUB_MODUL   58
#define MEN_SUB_CAF     59
#define MEN_SUB_PAF     60
#define MEN_SUB_CC1     61
#define MEN_SUB_CC2     62
#define MEN_CLEANUP     63
#define MEN_DEVICE	64
#define MEN_SAVE_SET    65
#define MEN_MIDI_THRU   66
#define MEN_COPYRIGHT   67
#define MEN_SEARCHREP   68
#define MEN_LOADPATTERN 69
#define MEN_SAVEPATTERN 70
#define MEN_SAVEAS	71
#define MEN_SAVE_THRU   72
#define MEN_SAVE_TIM    73
#define MEN_SAVE_EFF    74
#define MEN_SAVE_GEO    75
#define MEN_SAVE_ALL    76
#define MEN_MAPPER      77
#define MEN_SAVE_SYNTH 	79
#define MEN_SYNTH_SETTINGS 	80
#define MEN_PIANOWIN    81
#define MEN_METRONOME   82
#define MEN_PLAY   	83
#define MEN_SAVE_METRO	84
#define MEN_METRO_ON	85
#define MEN_PLAYLOOP    86
#define MEN_RECORD      87
#define MEN_REDO        88
#define MEN_ZOOMIN      91
#define MEN_ZOOMOUT     92
#define MEN_LOAD_TMPL   93

#define MEN_CLP_CUT     94
#define MEN_CLP_COPY    95
#define MEN_CLP_PASTE   96
#define MEN_SHUFFLE     97
#define MEN_GENMELDY    98
#define MEN_EVENTLIST   99
#define MEN_ARPEGGIO   100

#ifdef wx_x
#include "../bitmaps/open.xpm"
#include "../bitmaps/save.xpm"
#include "../bitmaps/new.xpm"
#include "../bitmaps/repl.xpm"
#include "../bitmaps/delete.xpm"
#include "../bitmaps/quantize.xpm"
#include "../bitmaps/mixer.xpm"
#include "../bitmaps/play.xpm"
#include "../bitmaps/undo.xpm"
#include "../bitmaps/redo.xpm"
#include "../bitmaps/zoomin.xpm"
#include "../bitmaps/zoomout.xpm"
#include "../bitmaps/panic.xpm"
#include "../bitmaps/help.xpm"
#include "../bitmaps/pianowin.xpm"
#include "../bitmaps/metro.xpm"
#include "../bitmaps/playloop.xpm"
#include "../bitmaps/record.xpm"
static tToolDef tdefs[] = {
  { MEN_LOAD,  		FALSE, 0, tb_open },
  { MEN_SAVE,  		FALSE, 0, tb_save },
  { MEN_NEW,   		FALSE, 1, tb_new  },

  { MEN_REPLICATE,  	FALSE, 0, tb_repl },
  { MEN_DELETE, 	FALSE, 0, tb_delete  },
  { MEN_QUANTIZE, 	FALSE, 0, tb_quantize  },
  { MEN_MIXER, 		FALSE, 0,  tb_mixer  },
  { MEN_PIANOWIN,	FALSE, 1, tb_pianowin  },

  { MEN_PLAY, 		FALSE, 0, tb_play },
  { MEN_PLAYLOOP, 	FALSE, 0, tb_playloop },
  { MEN_RECORD, 	FALSE, 0, tb_record },
  { MEN_METRO_ON,	TRUE,  1, tb_metro },

  { MEN_ZOOMIN, 	FALSE, 0, tb_zoomin },
  { MEN_ZOOMOUT,	FALSE, 0, tb_zoomout },
  { MEN_UNDO, 		FALSE, 0, tb_undo },
  { MEN_REDO, 		FALSE, 0, tb_redo },
  { MEN_RESET, 		FALSE, 0, tb_panic },
  { MEN_HELP_JAZZ, 	FALSE, 0, tb_help }

};

#else

static tToolDef tdefs[] = {
  { MEN_LOAD,  		FALSE, 0, "tb_open", "load song" },
  { MEN_SAVE,  		FALSE, 0, "tb_save", "save song" },
  { MEN_NEW,   		FALSE, 1, "tb_new" , "new song" },

  { MEN_REPLICATE,  	FALSE, 0, "tb_repl", "replicate selection" },
  { MEN_DELETE, 	FALSE, 0, "tb_delete", "delete selection"  },
  { MEN_QUANTIZE, 	FALSE, 0, "tb_quantize", "quantize selection"  },
  { MEN_MIXER, 		FALSE, 0, "tb_mixer", "mixer"  },
  { MEN_PIANOWIN,	FALSE, 1, "tb_pianowin", "show piano window"  },

  { MEN_PLAY, 		FALSE, 0, "tb_play", "start play" },
  { MEN_PLAYLOOP, 	FALSE, 0, "tb_playloop", "loop play" },
  { MEN_RECORD, 	FALSE, 0, "tb_record", "record" },
  { MEN_METRO_ON, 	TRUE,  1, "tb_metro", "metronome" },

  { MEN_ZOOMIN, 	FALSE, 0, "tb_zoomin", "zoom in" },
  { MEN_ZOOMOUT,	FALSE, 0, "tb_zoomout", "zoom out" },
  { MEN_UNDO, 		FALSE, 0, "tb_undo", "undo" },
  { MEN_REDO, 		FALSE, 0, "tb_redo", "redo" },
  { MEN_RESET, 		FALSE, 0, "tb_panic", "all notes off" },
  { MEN_HELP_JAZZ, 	FALSE, 0, "tb_help", "help" }

};
#endif

tTrackWin::tTrackWin(wxFrame *frame, char *title, tSong *song, int x, int y, int width, int height)
  : tEventWin(frame, title, song, x, y, width, height)
{
  rhythm_win  = 0;
  mapper_win  = 0;
  meldy_win   = 0;
  eventlst_win = 0;
  arpeggio_win = 0;
  prev_clock  = 0;
  prev_loop   = 0;
  prev_muted  = 0;
  prev_record = 0;
  paste_buffer = 0;

  defsong = copystring(lasts);
  defpattern = copystring("noname.mid");

  tool_bar = new tToolBar(this, tdefs, 18);

  int i;
  int opt;

  opt = GetArgOpt( "-pianowin" ) + 1;
  for (i = 0; i < 4; i++, opt++) {
  if ((wxTheApp->argc > opt) && isdigit( wxTheApp->argv[opt][0] ))
	Config(i+C_PianoWinXpos) = atoi( wxTheApp->argv[opt] );
  else
	break;
  }

  NextWin = new tPianoWin(frame, "Piano Roll", Song, Config(C_PianoWinXpos), Config(C_PianoWinYpos), Config(C_PianoWinWidth), Config(C_PianoWinHeight) );
  NextWin->Create();

  nBars = 0;
  RecInfo.Track = 0;
  RecInfo.Muted = 0;

  CounterMode = CmProgram;
  NumberMode = NmMidiChannel;

  MetronomeInfo.IsAccented = Config(C_MetroIsAccented);
  MetronomeInfo.Veloc = Config(C_MetroVelocity);
  MetronomeInfo.KeyNorm = Config(C_MetroNormalClick);
  MetronomeInfo.KeyAcc = Config(C_MetroAccentedClick);
}

tTrackWin::~tTrackWin()
{
  delete rhythm_win;
  delete mapper_win;
  delete meldy_win;
  delete [] defsong;
  delete [] defpattern;
  delete paste_buffer;
  delete arpeggio_win;
  delete eventlst_win;
}


Bool tTrackWin::OnClose()
{
  if (tTrack::changed) {
    if (wxMessageBox("Song has changed. Quit anyway?", "Quit ?", wxYES_NO) == wxNO)
      return FALSE;
  }

  if (Midi->Playing)
  {
    Midi->StopPlay();
#ifndef wx_msw
    sleep(1);
#endif
  }
  delete the_harmony_browser;
  delete Midi;
  delete NextWin;
  return TRUE;
}



#if 0
static void debug(tTrackWin *tw)
{
#ifndef wx_msw
  // debug: place a Reset Message into the song
  if (0) {
    tEvent *e = Synth->Reset();
    tTrack *t = tw->Song->GetTrack(0);
    e = e->Copy();
    e->Clock = tw->Song->TicksPerQuarter * 18;
    t->Put(e);
    t->Cleanup();
  }
  /*
  for (int i = 16; i < 20; i++)
  {
     Midi->OutNow( Synth->ReverbMacroSX( 0, i, 0 ) );
     printf("%d\n", i );
     sleep(1);
  }
  */
  for (int i = 64; i < 73; i++)
  {
     Midi->OutNow( Synth->ChorusMacroSX( 0, i, 0 ) );
     printf("%d\n", i );
     sleep(1);
  }
#endif
}
#endif


void tTrackWin::Setup()
{
  float x, y;

  tEventWin::Setup();

  dc->GetTextExtent("H", &x, &y);
  LittleBit = (int)(x/2);

  dc->GetTextExtent("HXWjgi", &x, &y);
  hLine = (int)y + LittleBit;
  hTop = hFixedFont + 2 * LittleBit;

  dc->GetTextExtent("99", &x, &y);
  wNumber = (int)x + LittleBit;

  dc->GetTextExtent("Normal Trackname", &x, &y);
  wName = (int)x + LittleBit;

  dc->GetTextExtent("m", &x, &y);
  wState = (int)x + LittleBit;

  dc->GetTextExtent("999", &x, &y);
  wPatch = (int)x + 2 * LittleBit;

  wLeft = wNumber + wName + wState + wPatch + 1;

  UnMark();
}



// ************************************************************************
// Menubar
// ************************************************************************

static wxMenu *file_menu;
static wxMenu *edit_menu;
static wxMenu *parts_menu;
static wxMenu *bender_menu;
static wxMenu *modulation_menu;
static wxMenu *caf_menu;
static wxMenu *paf_menu;
static wxMenu *cc1_menu;
static wxMenu *cc2_menu;
static wxMenu *setting_menu;
static wxMenu *save_settings_menu;
static wxMenu *misc_menu;
static wxMenu *help_menu;
#ifdef AUDIO
static wxMenu *audio_menu;
#endif

static void EnableDisableMenus();

void tTrackWin::CreateMenu()
{
  wxMenuBar *menu_bar = NULL;
  file_menu = new wxMenu;
  file_menu->Append(MEN_LOAD,          "&Load ...");
  file_menu->Append(MEN_SAVE,          "&Save");
  file_menu->Append(MEN_SAVEAS,        "Save &as...");
  file_menu->Append(MEN_NEW,           "&New ...");
  file_menu->Append(MEN_LOAD_TMPL,     "Load &Template...");
  file_menu->Append(MEN_LOADPATTERN,   "Load Pattern...");
  file_menu->Append(MEN_SAVEPATTERN,   "Save Pattern...");
  file_menu->AppendSeparator();
  file_menu->Append(MEN_QUIT,          "&Quit");

  edit_menu = new wxMenu;

  edit_menu->Append(MEN_CLP_CUT,       "&Cut");
  edit_menu->Append(MEN_CLP_COPY,      "C&opy");
  edit_menu->Append(MEN_CLP_PASTE,     "&Paste");
  edit_menu->AppendSeparator();

  edit_menu->Append(MEN_REPLICATE,     "&Replicate ...");
  edit_menu->Append(MEN_DELETE,        "&Delete ...");
  edit_menu->Append(MEN_QUANTIZE,      "&Quantize ...");
  edit_menu->Append(MEN_SETCHAN,       "&Set MIDI Channel ...");
  edit_menu->Append(MEN_TRANSP,        "&Transpose ...");
  edit_menu->Append(MEN_VELOC,         "&Velocity ...");
  edit_menu->Append(MEN_LENGTH,        "&Length ...");
  edit_menu->Append(MEN_SHIFT,         "Shi&ft ...");
  edit_menu->Append(MEN_CLEANUP,       "C&leanup ...");
  edit_menu->Append(MEN_SEARCHREP,     "Search Re&place ...");

  parts_menu = new wxMenu;
  parts_menu->Append(MEN_MIXER,    "&Mixer ...");
  parts_menu->Append(MEN_MASTER,   "Mas&ter ...");
  parts_menu->Append(MEN_SOUND,    "&Sound ...");
  parts_menu->Append(MEN_VIBRATO,  "&Vibrato ...");
  parts_menu->Append(MEN_ENVELOPE, "&Envelope ...");

  bender_menu = new wxMenu;
  bender_menu->Append(MEN_BEND_BASIC,    "&Bender Basic...");
  bender_menu->Append(MEN_BEND_LFO1,    "&Bender LFO1...");
  bender_menu->Append(MEN_BEND_LFO2,    "&Bender LFO2...");
  parts_menu->Append(MEN_SUB_BENDER, "&Bender...", bender_menu );

  modulation_menu = new wxMenu;
  modulation_menu->Append(MEN_MOD_BASIC,    "&Modulation Basic...");
  modulation_menu->Append(MEN_MOD_LFO1,    "&Modulation LFO1...");
  modulation_menu->Append(MEN_MOD_LFO2,    "&Modulation LFO2...");
  parts_menu->Append(MEN_SUB_MODUL, "&Modulation...", modulation_menu );

  caf_menu = new wxMenu;
  caf_menu->Append(MEN_CAF_BASIC,    "&CAf Basic...");
  caf_menu->Append(MEN_CAF_LFO1,    "&CAf LFO1...");
  caf_menu->Append(MEN_CAF_LFO2,    "&CAf LFO2...");
  parts_menu->Append(MEN_SUB_CAF, "&CAf...", caf_menu );

  paf_menu = new wxMenu;
  paf_menu->Append(MEN_PAF_BASIC,    "&PAf Basic...");
  paf_menu->Append(MEN_PAF_LFO1,    "&PAf LFO1...");
  paf_menu->Append(MEN_PAF_LFO2,    "&PAf LFO2...");
  parts_menu->Append(MEN_SUB_PAF, "&PAf...", paf_menu );

  cc1_menu = new wxMenu;
  cc1_menu->Append(MEN_CC1_BASIC,    "&CC1 Basic...");
  cc1_menu->Append(MEN_CC1_LFO1,    "&CC1 LFO1...");
  cc1_menu->Append(MEN_CC1_LFO2,    "&CC1 LFO2...");
  parts_menu->Append(MEN_SUB_CC1, "&CC1...", cc1_menu );

  cc2_menu = new wxMenu;
  cc2_menu->Append(MEN_CC2_BASIC,    "&CC2 Basic...");
  cc2_menu->Append(MEN_CC2_LFO1,    "&CC2 LFO1...");
  cc2_menu->Append(MEN_CC2_LFO2,    "&CC2 LFO2...");
  parts_menu->Append(MEN_SUB_CC2, "&CC2...", cc2_menu );

  parts_menu->Append(MEN_DRUM_PARAM,    "&Drum Parameters...");
  parts_menu->Append(MEN_PART_RSRV,    "&Partial Reserve...");
  parts_menu->Append(MEN_PART_MODE,    "&Part Mode...");

  setting_menu = new wxMenu;
  setting_menu->Append(MEN_FILTER,    "&Filter ...");
  setting_menu->Append(MEN_TWSETTING, "&Window ...");
  setting_menu->Append(MEN_SONG,      "&Song ...");
  setting_menu->Append(MEN_METRONOME, "&Metronome ...");
  setting_menu->Append(MEN_EFFECTS,   "&Effects ...");
  setting_menu->Append(MEN_TIMING,    "&Timing ...");
  setting_menu->Append(MEN_MIDI_THRU, "&Midi Thru ...");
  setting_menu->Append(MEN_SYNTH_SETTINGS, "&Synth Type ...");

#ifdef wx_msw
  setting_menu->Append(MEN_DEVICE,    "&Midi Device...");
#else
  if (Config(C_MidiDriver) == C_DRV_OSS)
    setting_menu->Append(MEN_DEVICE,    "&Midi Device...");
#endif
  save_settings_menu = new wxMenu;
  save_settings_menu->Append( MEN_SAVE_THRU, "&Midi Thru" );
  save_settings_menu->Append( MEN_SAVE_TIM, "&Timing" );
  save_settings_menu->Append( MEN_SAVE_EFF, "&Effect Macros" );
  save_settings_menu->Append( MEN_SAVE_GEO, "&Window Geometry" );
  save_settings_menu->Append( MEN_SAVE_METRO, "&Metronome" );
  // save_settings_menu->Append( MEN_SAVE_SYNTH, "&Synth Type" );
  save_settings_menu->Append( MEN_SAVE_ALL, "&Save All" );
  setting_menu->Append(MEN_SAVE_SET, "&Save settings", save_settings_menu );

  misc_menu = new wxMenu;
  misc_menu->Append(MEN_UNDO,     "&Undo ...");
  misc_menu->Append(MEN_REDO,     "&Redo ...");
  misc_menu->Append(MEN_TMERGE,   "Mer&ge Tracks ...");
  misc_menu->Append(MEN_TSPLIT,   "&Split Tracks ...");
  misc_menu->Append(MEN_METERCH,  "&Meterchange ...");
  misc_menu->Append(MEN_RESET,    "&Reset Midi");
  misc_menu->Append(MEN_HARMONY,  "&Harmony Browser...");
  misc_menu->Append(MEN_RHYTHM,   "Random R&hythm...");
  misc_menu->Append(MEN_SHUFFLE,  "Random Sh&uffle...");
  misc_menu->Append(MEN_GENMELDY, "Random Melod&y...");
  misc_menu->Append(MEN_ARPEGGIO, "Random Arpeggio...");
  misc_menu->Append(MEN_MAPPER,   "Ma&pper...");
  misc_menu->Append(MEN_EVENTLIST, "Event &List...");

  misc_menu->Append(MEN_COPYRIGHT,"&Set Music Copyright ...");

  help_menu = new wxMenu;
  help_menu->Append(MEN_HELP_JAZZ, "&Jazz");
  help_menu->Append(MEN_HELP_TWIN, "&Trackwin");
  help_menu->Append(MEN_HELP_MOUSE, "&Mouse");
  help_menu->Append(MEN_ABOUT, "&About");

  menu_bar = new wxMenuBar;
  menu_bar->Append(file_menu,    "&File");
  menu_bar->Append(edit_menu,    "&Edit");
  menu_bar->Append(parts_menu,   "&Parts");
  menu_bar->Append(setting_menu, "&Settings");
  menu_bar->Append(misc_menu,    "&Misc");

#ifdef AUDIO
    audio_menu = new wxMenu();
    audio_menu->Append(MEN_AUDIO_GLOBAL,        "&Global Settings ...");
    audio_menu->Append(MEN_AUDIO_SAMPLES,       "Sample Se&ttings ... ");
    audio_menu->Append(MEN_AUDIO_LOAD,          "&Load Set ...");
    audio_menu->Append(MEN_AUDIO_SAVE,          "&Save Set");
    audio_menu->Append(MEN_AUDIO_SAVE_AS,       "Save Set &As");
    audio_menu->Append(MEN_AUDIO_NEW,           "&New Set");
    menu_bar->Append(audio_menu, "&Audio");
#endif

  menu_bar->Append(help_menu,    "&Help");

  SetMenuBar(menu_bar);

  EnableDisableMenus();
}

static void EnableDisableMenus()
{
   bender_menu->Enable(MEN_BEND_LFO1, TRUE);
   bender_menu->Enable(MEN_BEND_LFO2, TRUE);
   parts_menu->Enable(MEN_SUB_MODUL, TRUE);
   parts_menu->Enable(MEN_SUB_CAF, TRUE);
   parts_menu->Enable(MEN_SUB_PAF, TRUE);
   parts_menu->Enable(MEN_SUB_CC1, TRUE);
   parts_menu->Enable(MEN_SUB_CC2, TRUE);

   bender_menu->Enable(MEN_BEND_LFO2, TRUE);
   modulation_menu->Enable(MEN_MOD_LFO2, TRUE);
   caf_menu->Enable(MEN_CAF_LFO2, TRUE);
   paf_menu->Enable(MEN_PAF_LFO2, TRUE);
   cc1_menu->Enable(MEN_CC1_LFO2, TRUE);
   cc2_menu->Enable(MEN_CC2_LFO2, TRUE);
   parts_menu->Enable(MEN_PART_RSRV, TRUE);
   setting_menu->Enable(MEN_EFFECTS, TRUE);

   if (Synth->IsGM())
   {
      bender_menu->Enable(MEN_BEND_LFO1, FALSE);
      bender_menu->Enable(MEN_BEND_LFO2, FALSE);
      parts_menu->Enable(MEN_SUB_MODUL, FALSE);
      parts_menu->Enable(MEN_SUB_CAF, FALSE);
      parts_menu->Enable(MEN_SUB_PAF, FALSE);
      parts_menu->Enable(MEN_SUB_CC1, FALSE);
      parts_menu->Enable(MEN_SUB_CC2, FALSE);
      setting_menu->Enable(MEN_EFFECTS, FALSE);
   }
   else if (Synth->IsXG())
   {
      bender_menu->Enable(MEN_BEND_LFO2, FALSE);
      modulation_menu->Enable(MEN_MOD_LFO2, FALSE);
      caf_menu->Enable(MEN_CAF_LFO2, FALSE);
      paf_menu->Enable(MEN_PAF_LFO2, FALSE);
      cc1_menu->Enable(MEN_CC1_LFO2, FALSE);
      cc2_menu->Enable(MEN_CC2_LFO2, FALSE);
      parts_menu->Enable(MEN_PART_RSRV, FALSE);
   }
}



#ifndef wxRELEASE_NUMBER

#define _MAXPATHLEN 500

// Return just the filename, not the path
// (basename)
char *
wxFileNameFromPath (char *path)
{
  if (path)
    {
      register char *tcp;

      tcp = path + strlen (path);
      while (--tcp >= path)
	{
	  if (*tcp == '/' || *tcp == '\\'
#ifdef VMS
     || *tcp == ':' || *tcp == ']')
#else
     )
#endif
	    return tcp + 1;
	}                       /* while */
#ifdef wx_msw
      if (isalpha (*path) && *(path + 1) == ':')
	return path + 2;
#endif
    }
  return path;
}

// Return just the directory, or NULL if no directory
char *
wxPathOnly (char *path)
{
  if (path && *path)
    {
      static char buf[_MAXPATHLEN];

      // Local copy
      strcpy (buf, path);

      int l = strlen(path);
      Bool done = FALSE;

      int i = l - 1;

      // Search backward for a backward or forward slash
      while (!done && i > -1)
      {
	if (path[i] == '/' || path[i] == '\\')
	{
	  done = TRUE;
	  buf[i] = 0;
	  return buf;
	}
	else i --;
      }

/* there's a bug here somewhere, so replaced with my original code.
      char *tcp;
      // scan back
      for (tcp = &buf[strlen (buf) - 1]; tcp >= buf; tcp--)
	{
	  // Search for Unix or Dos path sep {'\\', '/'}
	  if (*tcp == '\\' || *tcp == '/')
	    {
	      *tcp = '\0';
	      return buf;
	    }
	}                       // for()
*/
#ifdef wx_msw
      // Try Drive specifier
      if (isalpha (buf[0]) && buf[1] == ':')
	{
	  // A:junk --> A:. (since A:.\junk Not A:\junk)
	  buf[2] = '.';
	  buf[3] = '\0';
	  return buf;
	}
#endif
    }

  return NULL;
}
#endif


void tTrackWin::OnMenuCommand(int id)
{
  char *s;

  if (Midi->OnMenuCommand(id))
    return;

  switch (id)
  {

#ifdef wx_msw
  case MEN_DEVICE:
    {
      long idev = Config(C_WinInputDevice);
      long odev = Config(C_WinOutputDevice);
      tWinPlayer::SettingsDlg(idev, odev);
      wxMessageBox("Restart jazz to activate changes in device settings", "Info", wxOK);
    }
  break;
#else
  case MEN_DEVICE:
    if (Config(C_MidiDriver) == C_DRV_OSS) {
      int dev = Midi->FindMidiDevice();
      if (dev >= 0)
      {
	SaveMidiDeviceSettings( dev );
	wxMessageBox("Restart jazz to activate changes in device settings", "Info", wxOK);
      }
      else
	wxMessageBox("No midi device found", "Info", wxOK);
    }
    break;
#endif

    case MEN_PIANOWIN:
      NextWin->Show(TRUE);
      break;

    case MEN_ZOOMIN:
      if (ClocksPerPixel>12)
	ZoomIn();
      break;

    case MEN_ZOOMOUT:
      if (ClocksPerPixel<120)
	ZoomOut();
      break;

    case MEN_PLAY:
      MousePlay(0, PlayButton);
      break;

    case MEN_PLAYLOOP:
      MousePlay(0, PlayLoopButton);
      break;

    case MEN_RECORD:
      MousePlay(0, RecordButton);
      break;

    case MEN_METRO_ON:
      MetronomeInfo.IsOn = !MetronomeInfo.IsOn;
      break;

    case MEN_SAVE_THRU:
      SaveThruSettings();
      break;

    case MEN_SAVE_TIM:
      SaveTimingSettings();
      break;

    case MEN_SAVE_EFF:
      SaveEffectSettings();
      break;

    case MEN_SAVE_GEO:
      SaveGeoSettings();
      break;

    case MEN_SAVE_METRO:
      SaveMetronomeSettings();
      break;

    case MEN_SAVE_ALL:
      SaveThruSettings();
      SaveTimingSettings();
      SaveEffectSettings();
      SaveGeoSettings();
      SaveMetronomeSettings();
      break;

    case MEN_RHYTHM:
      if (!rhythm_win)
	rhythm_win = new tRhythmWin(this, Song);
      rhythm_win->Show(TRUE);
      break;

    case MEN_MAPPER:
      if (!mapper_win)
	mapper_win = new tMapperWin(this, Song);
      mapper_win->Show(TRUE);
      break;

    case MEN_GENMELDY:
      if (!meldy_win)
        meldy_win = new tGenMelody(this, (wxFrame **)&meldy_win);
      meldy_win->Show(TRUE);
      break;

    case MEN_EVENTLIST:
      if (!EventsSelected())
	return;
      if (Filter->FromTrack != Filter->ToTrack) {
        wxMessageBox("you must select exacty 1 track", "Error", wxOK);
        return;
      }

      if (!eventlst_win)
        eventlst_win = new tEventList(this, (wxFrame **)&eventlst_win);
      eventlst_win->Scan(Filter);
      eventlst_win->Show(TRUE);
      break;

    case MEN_ARPEGGIO:
      if (!EventsSelected())
	return;
      if (!arpeggio_win)
        arpeggio_win = new tArpeggioWin(this, (wxFrame **)&arpeggio_win);
      arpeggio_win->Show(TRUE);
      break;

    case MEN_HARMONY:
      harmony_browser(this);
      break;

    case MEN_NEW:
      if (MixerForm) {
	wxMessageBox("Quit parts dialog first.", "Info", wxOK);
	break;
      }
      if (wxMessageBox("Clear Song?", "Sure?", wxOK | wxCANCEL) == wxOK)
      {
	Song->Clear();
	Redraw();
	delete [] defsong;
	defsong = copystring("noname.mid");
	SetTitle(defsong);
	NextWin->NewPosition(1, 0);
      }
      break;

    case MEN_LOADPATTERN:
      MenLoadPattern();
      break;

    case MEN_SAVEPATTERN:
      MenSavePattern();
      break;

    case MEN_CLP_COPY:
      MenClpCopy(FALSE);
      break;

    case MEN_CLP_CUT:
      MenClpCopy(TRUE);
      break;

    case MEN_CLP_PASTE:
      MenClpPaste();
      break;

    case MEN_LOAD:
      if (MixerForm) {
	wxMessageBox("Quit parts dialog first.", "Info", wxOK);
	break;
      }
      s = file_selector(defsong, "Load File", 0, tTrack::changed, "*.mid");
      if (s)
      {
	tStdRead io;
	Song->Clear();
	Song->Read(io, s);
	SetTitle(s);
	NextWin->NewPosition(1, 0);
	Canvas->SetScrollRanges();
	NextWin->Canvas->SetScrollRanges();
	Redraw();
	tTrack::changed = 0;
      }
      break;

    case MEN_LOAD_TMPL:
      if (MixerForm) {
	wxMessageBox("Quit parts dialog first.", "Info", wxOK);
	break;
      }
      s = file_selector(defsong, "Load Template", 0, tTrack::changed, "*.mid");
      if (s)
      {
	tStdRead io;
	Song->Clear();
	Song->Read(io, s);
	delete [] defsong;
	defsong = copystring("noname.mid");
	SetTitle(defsong);
	NextWin->NewPosition(1, 0);
	Canvas->SetScrollRanges();
	NextWin->Canvas->SetScrollRanges();
	Redraw();
	tTrack::changed = 0;
      }
      break;

    case MEN_SAVEAS:
      {
	 char *s = file_selector(defsong, "Save File", 1, 0, "*.mid");
	 if (s)
	 {
	    tStdWrite io;
	    Song->Write(io, s);
	    SetTitle(s);
	    tTrack::changed = 0;
	    Config.Put(C_StartUpSong, s);
	 }
      }
      break;

    case MEN_SAVE:
      if (strcmp(defsong, "noname.mid") == 0)
        OnMenuCommand(MEN_SAVEAS);
      else
      {
	 tStdWrite io;
	 Song->Write(io, defsong);
	 tTrack::changed = 0;
	 Config.Put(C_StartUpSong, defsong);
      }
      break;


    case MEN_QUIT:
      if ( OnClose() == FALSE)
        return;
      delete this;
      break;

    case MEN_TWSETTING: SettingsDialog(0); break;
    case MEN_REPLICATE: MenCopy(); break;
    case MEN_QUANTIZE:  MenQuantize(); break;
    case MEN_DELETE:    MenDelete(); break;
    case MEN_SETCHAN:   MenSetChannel(); break;
    case MEN_TRANSP:    MenTranspose(); break;
    case MEN_VELOC:     MenVelocity(); break;
    case MEN_LENGTH:    MenLength(); break;
    case MEN_SHIFT:     MenShift(GetPianoWin()->SnapClocks()); break;
    case MEN_CLEANUP:   MenCleanup(); break;
    case MEN_SEARCHREP: MenSearchReplace(); break;
    case MEN_METERCH:   MenMeterChange(); break;
    case MEN_RESET:     Midi->AllNotesOff(1); break;

    case MEN_SONG:      MenSongSettings(); break;
    case MEN_METRONOME: MenMetronomeSettings(); break;
    case MEN_TSPLIT:
      if (MixerForm) {
	wxMessageBox("Quit parts dialog first.", "Info", wxOK);
	break;
      }
      MenSplitTracks();
      break;
    case MEN_TMERGE:
      if (MixerForm) {
	wxMessageBox("Quit parts dialog first.", "Info", wxOK);
	break;
      }
      MenMergeTracks();
      break;

    case MEN_UNDO:
      Song->Undo();
      Redraw();
      NextWin->Redraw();
      break;

    case MEN_REDO:
      Song->Redo();
      Redraw();
      NextWin->Redraw();
      break;

    case MEN_SHUFFLE:
      {
	if (!EventsSelected())
	  return;
	wxDialogBox *panel = new wxDialogBox(this, "Random Shuffle", FALSE );
	tShuffleDlg * dlg = new tShuffleDlg(this, Filter);
	dlg->EditForm(panel);
	panel->Fit();
	panel->Show(TRUE);
      }
      break;

    case MEN_DEBUG:
      //debug(this);
      break;

    case MEN_MIXER:
      MenMixer();
      break;

    case MEN_MASTER:
      MenMaster();
      break;

    case MEN_VIBRATO:
      MenVibrato();
      break;

    case MEN_SOUND:
      MenSound();
      break;

    case MEN_ENVELOPE:
      MenEnvelope();
      break;

    case MEN_BEND_BASIC:
      MenBendBasic();
      break;

    case MEN_BEND_LFO1:
      MenBendLfo1();
      break;

    case MEN_BEND_LFO2:
      MenBendLfo2();
      break;

    case MEN_MOD_BASIC:
      MenModBasic();
      break;

    case MEN_MOD_LFO1:
      MenModLfo1();
      break;

    case MEN_MOD_LFO2:
      MenModLfo2();
      break;

    case MEN_CAF_BASIC:
      MenCAfBasic();
      break;

    case MEN_CAF_LFO1:
      MenCAfLfo1();
      break;

    case MEN_CAF_LFO2:
      MenCAfLfo2();
      break;

    case MEN_PAF_BASIC:
      MenPAfBasic();
      break;

    case MEN_PAF_LFO1:
      MenPAfLfo1();
      break;

    case MEN_PAF_LFO2:
      MenPAfLfo2();
      break;

    case MEN_CC1_BASIC:
      MenCC1Basic();
      break;

    case MEN_CC1_LFO1:
      MenCC1Lfo1();
      break;

    case MEN_CC1_LFO2:
      MenCC1Lfo2();
      break;

    case MEN_CC2_BASIC:
      MenCC2Basic();
      break;

    case MEN_CC2_LFO1:
      MenCC2Lfo1();
      break;

    case MEN_CC2_LFO2:
      MenCC2Lfo2();
      break;

    case MEN_DRUM_PARAM:
      MenDrumParam();
      break;

    case MEN_PART_RSRV:
      MenPartRsrv();
      break;

    case MEN_PART_MODE:
      MenPartMode();
      break;

    case MEN_FILTER:
      Filter->Dialog(0);
      break;

    case MEN_EFFECTS:
      MenEffects();
      break;

    case MEN_TIMING:
      MenTiming();
      break;

    case MEN_MIDI_THRU:
      MenMidiThru();
      break;

    case MEN_SYNTH_SETTINGS:
      MenSynthSettings();
      break;

    case MEN_COPYRIGHT:
      MenCopyright();
      break;

    case MEN_ABOUT:
      extern const char *about_text;
      wxMessageBox( (char *) about_text, "About", wxOK);
      break;

    case MEN_HELP_MOUSE:
      wxbMessageBox( (char *)
	"topline:\n"
	"  left: start/stop record/play\n"
	"    +shift: start/stop cycle record/play\n"
	"  middle: same as left+shift\n"
	"  right: start/stop record/play, mute selected track\n"
	"\n"
	"events:\n"
	"  left: select events\n"
	"    +shift: continue selection\n"
	"  right: open and position pianowin\n", "Help", wxOK);
      break;
    case MEN_HELP_JAZZ:
      HelpInstance->DisplayContents();
      break;
    case MEN_HELP_TWIN:
      HelpInstance->ShowTopic("Track Window");
      break;

  }
}


// ************************************************************************
// Painting
// ************************************************************************


const char *tTrackWin::CounterStr()
{
  const char *str;
  switch (CounterMode)
  {
    case CmProgram: str = "Prg"; break;
    case CmBank   : str = "Bnk"; break;
    case CmVolume : str = "Vol"; break;
    case CmPan    : str = "Pan"; break;
    case CmReverb : str = "Rev"; break;
    case CmChorus : str = "Cho"; break;
    default       : str = "???"; break;
  }
  return str;
}


void tTrackWin::DrawCounters()
{
  int i;
  const char *str = CounterStr();
  LineText(xPatch, yy-1, wPatch, str, hTop);

  dc->SetClippingRegion(xPatch, yEvents, xPatch + wPatch, yEvents + hEvents);
  for (i = FromLine; i < ToLine; i++)
  {
    tTrack *t = Song->GetTrack(i);
    if (t)
    {
      char buf[20];
      int  val;
      switch (CounterMode)
      {
	case CmProgram: val = t->GetPatch(); break;
	case CmBank   : val = t->GetBank(); break;
	case CmVolume : val = t->GetVolume(); break;
	case CmPan    : val = t->GetPan(); break;
	case CmReverb : val = t->GetReverb(); break;
	case CmChorus : val = t->GetChorus(); break;
	default : val = 0; break;
      }
      sprintf(buf, "%3d", val);
      LineText(xPatch, Line2y(i), wPatch, buf);
    }
    else
      LineText(xPatch, Line2y(i), wPatch, "?");
  }
  dc->DestroyClippingRegion();
}


const char* tTrackWin::NumberStr()
{
  const char *str;
  switch (NumberMode)
  {
    case NmTrackNr     : str = "T"; break;
    case NmMidiChannel : str = "M"; break;
    default            : str = "?"; break;
  }
  return str;
}


void tTrackWin::DrawNumbers()
{
  const char *str = NumberStr();
  int i;
  LineText(xNumber, yy-1, wNumber, str, hTop);

  dc->SetClippingRegion(xNumber, yEvents, xNumber + wNumber, yEvents + hEvents);
  for (i = FromLine; i < ToLine; i++)
  {
    tTrack *t = Song->GetTrack(i);
    if (t != 0)
    {
#ifdef AUDIO
      if (t->GetAudioMode())
	LineText(xNumber, Line2y(i), wNumber, "Au");
      else
#endif
      {
	char buf[20];
	int  val;
	switch (NumberMode)
	{
	  case NmTrackNr: val = i; break;
	  case NmMidiChannel : val = t->Channel; break;
	  default : val = 0; break;
	}
	sprintf(buf, "%02d", val);
	LineText(xNumber, Line2y(i), wNumber, buf);
      }
    }
  }
  dc->DestroyClippingRegion();
}



void tTrackWin::DrawSpeed(int Value, Bool down)
{
  char buf[50];
  if (Value < 0)
    Value = Song->GetTrack(0)->GetDefaultSpeed();
  sprintf(buf, "speed: %3d", Value);
  LineText(xName, yy-1, wName, buf, hTop, down);
}


void tTrackWin::OnPaint(long x, long y)
{
  tEventWin::OnPaint(x, y);

  xNumber  = xx;
  xName    = xNumber  + wNumber;
  xState   = xName    + wName;
  xPatch   = xState   + wState;

  long StopClk;
  tBarInfo BarInfo(Song);
  char buf[20];

  dc->BeginDrawing();
  dc->DestroyClippingRegion();
  DrawPlayPosition();
  SnapSel->Draw(xEvents, yEvents, wEvents, hEvents);
  dc->SetBackground(wxWHITE_BRUSH);
  dc->Clear();

  // clear playposition and selection

  #define VLine(x) DrawLine(x,  yy, x, yEvents+hEvents)
  #define HLine(y) DrawLine(xx, y, xx + ww, y)

  dc->SetPen(wxBLACK_PEN);

  // vertikale Linien
  dc->VLine(xNumber);
  dc->VLine(xName);
  dc->VLine(xState);
  dc->VLine(xPatch);
  // SN+ dc->VLine(xEvents);
  dc->VLine(xEvents-1);
  dc->HLine(yEvents);
  dc->HLine(yEvents-1);


  // Taktstriche und -nummern

  BarInfo.SetClock(FromClock);
  StopClk = x2Clock(xx + ww);
  nBars = 0;
  int intro = Song->GetIntroLength();
  dc->SetPen(wxGREY_PEN);
  while (1)
  {
    x = Clock2x(BarInfo.Clock);
    if (x > xx + ww)
      break;
    if (x >= xEvents)   // so ne Art clipping
    {
      // SN+-      if ((BarInfo.BarNr % 4) == 0)
      int c;
      if (ClocksPerPixel > 48) c = 8; else c = 4;
      if (((BarInfo.BarNr - intro + 96) % c) == 0)
      {
	dc->SetPen(wxBLACK_PEN);
	sprintf(buf, "%d", BarInfo.BarNr + 1 - intro);
	dc->DrawText(buf, x + LittleBit, yEvents - hLine);
	dc->SetPen(wxGREY_PEN);
	dc->DrawLine(x, yEvents + 1 - hLine, x, yEvents + hEvents);
      }
      else
      {
	dc->SetPen(wxLIGHT_GREY_PEN);
	dc->DrawLine(x, yEvents+1, x, yEvents+hEvents);
      }

      if (nBars < MaxBars)      // x-Koordinate fuer MouseAction->Snap()
	xBars[nBars++] = x;

    }
    BarInfo.Next();
  }
  dc->SetPen(wxBLACK_PEN);

  // for each track show num, name, state, prg

  dc->SetClippingRegion(xx, yEvents, xx+ww, yEvents + hEvents);
  int TrackNr = FromLine;
  for (y = Line2y(TrackNr); y < yEvents + hEvents; y += hLine)
  {
// SN+    dc->HLine(y);
    dc->SetPen(wxGREY_PEN);
    dc->DrawLine(xEvents+1,y,xx + ww, y);
    dc->SetPen(wxBLACK_PEN);
    dc->DrawLine(xx, y, xEvents,y);
//
    tTrack *Track = Song->GetTrack(TrackNr);
    if (Track)
    {
      // TrackName, show the button pressed when dialog is open
      //dc->DrawText(Track->GetName(), xName + LittleBit, y + LittleBit);
      if (Track->DialogBox)
	LineText(xName, y, wName, Track->GetName(), -1, TRUE);
      else
	LineText(xName, y, wName, Track->GetName(), -1, FALSE);

      // TrackStatus
      //dc->DrawText(Track->GetStateChar(), xState + LittleBit, y + LittleBit);
      LineText(xState, y, wState, Track->GetStateChar());
    }
    ++ TrackNr;
  }
  dc->DestroyClippingRegion();

  DrawNumbers();
  DrawSpeed();
  DrawCounters();
  LineText(xState, yy-1, wState, "", hTop);

  // draw events
  dc->SetClippingRegion(xEvents, yEvents, wEvents, hEvents);
  TrackNr = FromLine;
  for (y = Line2y(TrackNr); y < yEvents + hEvents; y += hLine)
  {
    tTrack *Track = Song->GetTrack(TrackNr);
    if (Track)
    {
      tEventIterator Iterator(Track);
      tEvent *e = Iterator.Range(FromClock, StopClk);
      float y0 = y + LittleBit;
      float y1 = y + hLine - LittleBit;

      if (UseColors)
      {
#if 0
	while (e)       // slow!
	{
	  float x = Clock2x(e->Clock);
	  dc->SetPen(e->GetPen());
	  dc->DrawLine(x, y0, x, y1);
	  e = Iterator.Next();
	}
#else
	float xdone = -1;
	float h = y1 - y0;
	while (e)       // very slow!
	{
	  float x1 = Clock2x(e->Clock + e->GetLength());
	  if (x1 > xdone) {
	    float x0 = Clock2x(e->Clock);
	    if (x0 < xdone)
	      x0 = xdone;
	    float w = x1 - x0;
	    if (w < 2)
	      w = 2;
	    xdone = x0 + w;
	    dc->SetPen(e->GetPen());
	    dc->SetBrush(e->GetBrush());
	    dc->DrawRectangle(x0, y0, w, h);
	  }
	  e = Iterator.Next();
	}
#endif
	dc->SetPen(wxBLACK_PEN);
      }
      else
      {
	float xblack = -1.0;
	while (e)
	{
	  float x = Clock2x(e->Clock);

	  // Avoid painting events ON the bar
	  if ( !(e->Clock % BarInfo.TicksPerBar) ) x = x + 1;

	  if (x > xblack)
	  {
	    dc->DrawLine(x, y0, x, y1);
#ifndef SLOW_MACHINE
	    xblack = x;
#else
	    xblack = x + 4;
#endif
	  }
	  e = Iterator.Next();
	}
      }
    }

    ++ TrackNr;
  }

  // strange bug? after SetPen(GREY_PEN) XOR-Painting doesnt work anymore
  // unless SetBackground(WHITE_BRUSH) is called
  dc->SetBackground(wxWHITE_BRUSH);

  SnapSel->Draw(xEvents, yEvents, wEvents, hEvents);

  if (Marked.x > 0)
    LineText((long)Marked.x, (long)Marked.y, (long)Marked.w, ">");

  dc->DestroyClippingRegion();
  DrawPlayPosition();
  dc->EndDrawing();
}

// ************************************************************************
// Utilities
// ************************************************************************

long tTrackWin::x2xBar(long x)
{
  for (int i = 1; i < nBars; i++)
    if (x < xBars[i])
      return xBars[i - 1];
  return -1;
}


long tTrackWin::x2wBar(long x)
{
  for (int i = 1; i < nBars; i++)
    if (x < xBars[i])
      return xBars[i] - xBars[i - 1];
  return 0;
}


tTrack *tTrackWin::y2Track(long y)
{
  return Song->GetTrack(y2Line(y));
}

void tTrackWin::Mark(long x, long y)
{
  Marked.x = x2xBar(x);
  Marked.y = y2yLine(y);
  Marked.w = x2wBar(x);
  Marked.h = hLine;
  LineText((int)Marked.x, (int)Marked.y, (int)Marked.w, ">");
}

void tTrackWin::UnMark()
{
  Marked.x = -1;
}


// ********************************************************************
// Snapper
// ********************************************************************

void tTrackWin::SnapSelStart(wxMouseEvent &e)
{
  SnapSel->SetXSnap(nBars, xBars);
  SnapSel->SetYSnap(Line2y(FromLine), yEvents + hEvents, hLine);
}


void tTrackWin::SnapSelStop(wxMouseEvent &e)
{
  if (SnapSel->Selected)
  {
    Filter->FromTrack = y2Line((long)SnapSel->r.y);
    Filter->ToTrack   = y2Line((long)(SnapSel->r.y + SnapSel->r.h - 1));
    Filter->FromClock = x2BarClock((long)SnapSel->r.x + 1);
    Filter->ToClock   = x2BarClock((long)(SnapSel->r.x + SnapSel->r.w + 1));
    NextWin->NewPosition(Filter->FromTrack, Filter->FromClock);
  }
}



// --------------------------------------------------------------------
// Tracknummer
// --------------------------------------------------------------------

void tTrackWin::MouseNumber(wxMouseEvent &e)
{
  if (e.LeftDown())
  {
    float x, y;
    e.Position(&x, &y);
    tTrack *t = y2Track((long)y);
    if (t != 0)
    {
      tRect r;
      r.x = 0;
      r.y = y2yLine((long)y);
      r.w = Clock2x( Song->MaxQuarters * Song->TicksPerQuarter );
      r.h = hLine;
      SnapSel->Select(r, xEvents, yEvents, wEvents, hEvents);
      SnapSelStop(e);
    }
  }
}


// --------------------------------------------------------------------
// PatchCounter
// --------------------------------------------------------------------

class tPatchCounter : public tMouseCounter
{
    tTrackWin *tw;
  public:
    int Event(wxMouseEvent &e);
    tPatchCounter(tTrackWin *t, tRect *r, int v, int min, int max)
      : tMouseCounter(t, r, v, min, max)
    {
      tw = t;
    }
};


int tPatchCounter::Event(wxMouseEvent &e)
{
  if (tMouseCounter::Event(e))
  {
    tTrack *t = tw->y2Track((long)r.y);
    if (t)
    {
      switch (tw->CounterMode)
      {
	case CmProgram : t->SetBank( t->GetBank() );
                         t->SetPatch(Value);
                         tw->DrawCounters();
                         break;
	case CmBank    : t->SetBank(Value);
                         t->SetPatch( t->GetPatch() );
                         tw->DrawCounters();
                         break;
	case CmVolume  : t->SetVolume(Value);
			 tMixerDlg::SetSliderVal( t->Channel - 1, MxVol, Value );
			 break;
	case CmPan     : t->SetPan(Value);
			 tMixerDlg::SetSliderVal( t->Channel - 1, MxPan, Value );
			 break;
	case CmReverb  : t->SetReverb(Value);
			 tMixerDlg::SetSliderVal( t->Channel - 1, MxRev, Value );
			 break;
	case CmChorus  : t->SetChorus(Value);
			 tMixerDlg::SetSliderVal( t->Channel - 1, MxCho, Value );
			 break;
	default: break;
      }
    }
    tw->MouseAction = 0;
    delete this;
  }
  return 0;
}


void tTrackWin::MousePatch(wxMouseEvent &e)
{
  float x, y;

  if (!e.LeftDown() && !e.RightDown())
    return;

  e.Position(&x, &y);
  tTrack *t = y2Track((long)y);
  if (t)
  {
    tRect r;
    int Value;
    switch (CounterMode)
    {
      case CmProgram: Value = t->GetPatch(); break;
      case CmBank   : Value = t->GetBank(); break;
      case CmVolume : Value = t->GetVolume(); break;
      case CmPan    : Value = t->GetPan(); break;
      case CmReverb : Value = t->GetReverb(); break;
      case CmChorus : Value = t->GetChorus(); break;
      default       : Value = 0; break;
    }
    r.x = xPatch;
    r.y = y2yLine((long)y);
    r.w = wPatch;
    r.h = 0;

    tPatchCounter *PatchCounter;

    if ((CounterMode == CmBank) && Config(C_UseTwoCommandBankSelect))
    {
       PatchCounter = new tPatchCounter(this, &r, Value, 0, Config(C_MaxBankTableEntries) - 1);
    }
    else
    {
       PatchCounter = new tPatchCounter(this, &r, Value, 0, 128);
    }

    PatchCounter->Event(e);
    MouseAction = PatchCounter;
  }
}

// --------------------------------------------------------------------
// SpeedCounter
// --------------------------------------------------------------------

class tSpeedCounter : public tMouseCounter
{
    tTrackWin *tw;
  public:
    int Event(wxMouseEvent &e);
    tSpeedCounter(tTrackWin *t, tRect *r, int v, int min, int max)
      : tMouseCounter(t, r, v, min, max)
    {
      tw = t;
    }
    virtual void ShowValue(Bool down);
};


int tSpeedCounter::Event(wxMouseEvent &e)
{
  if (tMouseCounter::Event(e))
  {
    tTrack *t = tw->Song->GetTrack(0);
    t->SetDefaultSpeed(Value);
#ifdef AUDIO
    for (int i = 0; i < tw->Song->nTracks; i++)
      Midi->AdjustAudioLength(tw->Song->GetTrack(i));
    tw->NextWin->Redraw();
#endif
    tw->MouseAction = 0;
    delete this;
  }
  return 0;
}

void tSpeedCounter::ShowValue(Bool down)
{
  tw->DrawSpeed(Value, down);
}

void tTrackWin::MouseSpeed(wxMouseEvent &e)
{
  tRect r;
  int Value;

  if (!e.LeftDown() && !e.RightDown())
    return;

#ifdef AUDIO
  if (Midi->GetAudioEnabled() && Midi->Playing)
    return;
#endif

  Value = Song->GetTrack(0)->GetDefaultSpeed();
  tSpeedCounter *SpeedCounter = new tSpeedCounter(this, &r, Value, 20, 250);
  SpeedCounter->Event(e);
  MouseAction = SpeedCounter;
}

// -----------------------------------------------------------------------
// Track-Status
// -----------------------------------------------------------------------


void tTrackWin::MouseState(wxMouseEvent &e)
{
  if (e.LeftDown() || e.RightDown())
  {
    float x, y;
    e.Position(&x, &y);
    tTrack *t = y2Track((long)y);
    if (t)
    {
      const char *down = t->GetStateChar();
      t->ToggleState(e.LeftDown() ? 1 : -1); // toggle
      const char *up   = t->GetStateChar();
      //LineText(xState, (long)y, wState, t->GetStateChar());
      tRect r;
      r.x = xState;
      r.y = y2yLine((long)y);
      r.w = wState;
      r.h = 0;
      MouseAction = new tMouseButton(this, &r, up, up);
    }
  }
}


// --------------------------------------------------------------------------
// Name
// --------------------------------------------------------------------------

class tTrackNameButton : public tMouseButton
{
  public:
    tTrackNameButton(tTrackWin *win, tTrack *trk, tRect *r, const char *name)
    : tMouseButton(win, r, name, name), tw(win), track(trk) {}
    virtual void Action() {
      track->Dialog(tw);
    }
  private:
    tTrackWin *tw;
    tTrack    *track;

};

void tTrackWin::MouseName(wxMouseEvent &e)
{
  static int from=0,to=0;
  static int  state = 0;
  static long last  = 0;
  long i;
  float x,y;

  if (e.LeftDown())
  {
    e.Position(&x, &y);
    tTrack *t = y2Track((long)y);
    if (t && !MixerForm) {
      //t->Dialog(this);
      tRect r;
      r.x = xName;
      r.y = y2yLine((long)y);
      r.w = wName;
      r.h = 0;
      MouseAction = new tTrackNameButton(this, t, &r, t->GetName());
    }
    else {
      wxMessageBox("Quit parts dialog first.", "Info", wxOK);
    }
  }

  // SN++
  if (e.RightDown() && !state) {

      e.Position(&x, &y);
      tTrack *t = y2Track((long)y);
      if (t && !MixerForm) {
        dc->SetBrush(wxTRANSPARENT_BRUSH);
	    dc->DrawRectangle(xName+1,y2yLine((long)y),wName-2,hLine-1);
	    dc->SetBrush(wxWHITE_BRUSH);
	    if(!state) {
               from = y2Line((long)y);
	       state=1;
	       last = y2yLine((long)y);
	    }
      }
   }
   if (e.RightUp() && state) {
         state = 0;
         e.Position(&x, &y);
         tTrack *t = y2Track((long)y);
         if (t && !MixerForm) {
           to = y2Line((long)y);
           Song->moveTrack(from,to);
	 }
	 Redraw();
   }
   if (e.Dragging() && state) {
      e.Position(&x, &y);
      i = y2yLine((long)y);
      if (i!=last) {
        dc->SetBrush(wxTRANSPARENT_BRUSH);
        dc->SetLogicalFunction(wxXOR);
        dc->SetPen(wxGREY_PEN);
        dc->DrawRectangle(xName, last  ,wName-1, hLine-1);
        dc->DrawRectangle(xName, last+1,wName-2, hLine-3);
        dc->DrawRectangle(xName, i     ,wName-1, hLine-1);
        dc->DrawRectangle(xName, i+1   ,wName-2, hLine-3);
        dc->SetLogicalFunction(wxCOPY);
	dc->SetPen(wxBLACK_PEN);
        dc->SetBrush(wxWHITE_BRUSH);
        last = i;
      }
   }
   //
}

// --------------------------------------------------------------------------
// Event-Bereich
// --------------------------------------------------------------------------

void tTrackWin::MouseEvents(wxMouseEvent &e)
{
  if (e.RightDown())
  {
    float x, y;
    e.Position(&x, &y);
    int TrackNr = y2Line((long)y);
    long Clock  = x2BarClock((long)x);
    NextWin->NewPosition(TrackNr, Clock);
    NextWin->Show(TRUE);
  }
  else
    tEventWin::OnMouseEvent(e);
}

// --------------------------------------------------------------------------
// Playbar
// --------------------------------------------------------------------------

/*
 * not playing:
 *  events selected:
 *    left : start rec/play
 *    right: mute + start rec/play
 *  no events selected:
 *    left+right: start play
 * playing:
 *  left+right: stop
 */

void tTrackWin::MousePlay(wxMouseEvent *e, MousePlayMode mode)
{
  if (mode == Mouse && !e->ButtonDown())
    return;

  if (!Midi->Playing)
  {
    switch (mode) {
      case Mouse:
	{
	  float x, y;
	  e->Position(&x, &y);
	  prev_clock = x2BarClock((long)x);
	  prev_muted = (e->RightDown() != 0);
	  if (SnapSel->Selected && (e->ShiftDown() || e->MiddleDown()))
	    prev_loop = TRUE;
	  else
	    prev_loop = FALSE;
          prev_record = SnapSel->Selected;
	}
	break;

      case SpaceBar:
        // do it again, sam
        break;

      case PlayButton:
        prev_loop = FALSE;
	prev_record = FALSE;
        break;

      case PlayLoopButton:
        if (!EventsSelected("please select loop range first"))
	  return;
        prev_loop   = TRUE;
	prev_record = FALSE;
	break;

      case RecordButton:
        {
	  if (!EventsSelected("please select record track/bar first"))
	    return;
	  tBarInfo bi(Song);
	  bi.SetClock(Filter->FromClock);
	  if (bi.BarNr > 0)
	    bi.SetBar(bi.BarNr - 1);
	  prev_clock  = bi.Clock;
	  prev_record = TRUE;
	  prev_loop   = FALSE;
	}
        break;
    }

    Bool loop   = prev_loop;
    Bool muted  = prev_muted;
    Bool record = prev_record;

    // possible to record?
    if (record && SnapSel->Selected)
    {
      RecInfo.TrackNr   = Filter->FromTrack;
      RecInfo.Track     = Song->GetTrack(RecInfo.TrackNr);
      RecInfo.FromClock = Filter->FromClock;
      RecInfo.ToClock   = Filter->ToClock;
      if (muted)
      {
	RecInfo.Muted = 1;
	RecInfo.Track->SetState(tsMute);
	LineText(xState, Line2y(RecInfo.TrackNr), wState, RecInfo.Track->GetStateChar());
      }
      else
	RecInfo.Muted = 0;
    }
    else
      RecInfo.Track = 0;

    // possible to loop?
    long loop_clock = 0;
    if (loop && SnapSel->Selected)
    {
      prev_clock = Filter->FromClock;
      loop_clock = Filter->ToClock;
    }

    // GO!
    if (RecInfo.Track)  // recording?
      Midi->SetRecordInfo(&RecInfo);
    else
      Midi->SetRecordInfo(0);
    Midi->StartPlay(prev_clock, loop_clock);

  }
  else
  {
    Midi->StopPlay();
    if (RecInfo.Track)
    {
      if (RecInfo.Muted)
      {
	RecInfo.Track->SetState(tsPlay);
	LineText(xState, Line2y(RecInfo.TrackNr), wState, RecInfo.Track->GetStateChar());
      }
#ifdef AUDIO
      if (RecInfo.Track->GetAudioMode())
	;
      else
#endif
      if (!Midi->RecdBuffer.IsEmpty())
      {
	//int choice = wxMessageBox("Keep recorded events?", "You played", wxOK | wxCANCEL);
	//if (choice == wxOK)
	{
	  wxBeginBusyCursor();
	  Song->NewUndoBuffer();
	  RecInfo.Track->MergeRange(&Midi->RecdBuffer, RecInfo.FromClock, RecInfo.ToClock, RecInfo.Muted);
	  wxEndBusyCursor();
	  Redraw();
	  NextWin->Redraw();
	}
      }
    }
  }
}


int tTrackWin::OnKeyEvent(wxKeyEvent &e)
{

  if (e.ControlDown()) {
    switch (e.KeyCode()) {
      case 'Z':
        OnMenuCommand(MEN_UNDO);
        return 1;
      case 'Y':
        OnMenuCommand(MEN_REDO);
        return 1;
      case 'X':
        OnMenuCommand(MEN_CLP_CUT);
        return 1;
      case 'V':
        OnMenuCommand(MEN_CLP_PASTE);
        return 1;
      case 'C':
      case WXK_INSERT:
        OnMenuCommand(MEN_CLP_COPY);
        return 1;
    }
  }

  else if (e.ShiftDown()) {
    switch (e.KeyCode()) {
      case WXK_INSERT:
        OnMenuCommand(MEN_CLP_PASTE);
        return 1;
      case WXK_DELETE:
        OnMenuCommand(MEN_CLP_CUT);
        return 1;
    }
  }

  else {
    switch (e.KeyCode()) {
      case ' ':
        MousePlay(0, SpaceBar);
        return 1;
      case WXK_DELETE:
        OnMenuCommand(MEN_DELETE);
        return 1;
    }
  }

  return 0;
}



// -----------------------------------------------------------------------
// event dispatcher
// -----------------------------------------------------------------------


int tTrackWin::OnMouseEvent(wxMouseEvent &e)
{

  if (!MouseAction)
  {
    float x, y;
    e.Position(&x, &y);
    if (y > yEvents)
    {
      // Events
      if (xNumber < x && x < xNumber + wNumber)
	MouseNumber(e);
      else if (xName < x && x < xName + wName)
	MouseName(e);
      else if(xState < x && x < xState + wState)
	MouseState(e);
      else if (xPatch < x && x < xPatch + wPatch)
	MousePatch(e);
      else if (xEvents < x && x < xEvents + wEvents)
	MouseEvents(e);
      else
	tEventWin::OnMouseEvent(e);
    }
    else
    {
      // Playbar

      if (xNumber < x && x < xNumber + wNumber)
      {
	if (e.LeftDown())
	{
 	  NumberMode = (tNumberModes)(((int)NumberMode + 1) % (int)NmModes);
	  DrawNumbers();
	  tRect r;
	  r.x = xNumber;
	  r.y = yy-1;
	  r.w = wNumber;
	  r.h = hTop;
	  MouseAction = new tMouseButton(this, &r, NumberStr());
	}
      }
      else if (xName < x && x < xName + wName)
	MouseSpeed(e);
      else if(xState < x && x < xState + wState)
	;
      else if (xPatch < x && x < xPatch + wPatch)
      {
        if (e.ButtonDown())
	{
	  if (e.LeftDown())
	    CounterMode = (tCounterModes)(((int)CounterMode + 1) % (int)CmModes);
	  else if (e.RightDown())
	    CounterMode = (tCounterModes)(((int)CounterMode + (int)CmModes - 1) % (int)CmModes);
	  DrawCounters();
	  tRect r;
	  r.x = xPatch;
	  r.y = yy-1;
	  r.w = wPatch;
	  r.h = hTop;
	  MouseAction = new tMouseButton(this, &r, CounterStr());
	}
      }
      else if (xEvents < x && x < xEvents + wEvents)
	MousePlay(&e, Mouse);
      else
	tEventWin::OnMouseEvent(e);
    }
  }

  else
    tEventWin::OnMouseEvent(e);

  return 0;
}



// **************************************************************************
// Copy
// **************************************************************************


class tCopyDlg;

class tCopyCommand : public wxObject, public tMouseAction
{
    tTrackWin *tw;
    int MarkRepeat;
    float StartX, StartY, StopX, StopY;

    tMarkDestin *Mouse;
    tCopyDlg *CopyDlg;

  public:
    static Bool RepeatCopy;
    static Bool EraseSource;
    static Bool EraseDestin;
    static Bool InsertSpace;

    int Event(wxMouseEvent &e);
    tCopyCommand(tTrackWin *t);
    void EditForm(wxPanel *panel);
    void OnOk();
    void OnCancel();
    void Execute(int doit);
};

Bool tCopyCommand::EraseDestin = 1;
Bool tCopyCommand::RepeatCopy = 0;
Bool tCopyCommand::EraseSource = 0;
Bool tCopyCommand::InsertSpace = 0;


class tCopyDlg : public wxForm
{
  tCopyCommand *cp;

 public:
  tCopyDlg(tCopyCommand *c)
	: wxForm( USED_WXFORM_BUTTONS ) { cp = c; }
  void OnOk()                           { cp->OnOk(); wxForm::OnOk(); }
  void OnCancel()                       { cp->OnCancel(); wxForm::OnCancel(); }
  void OnHelp();
  void EditForm(wxPanel *panel);
};

void tCopyDlg::OnHelp()
{
	HelpInstance->ShowTopic("Replicate");
}


void tCopyDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormBool("Erase Destin", &cp->EraseDestin));
  Add(wxMakeFormBool("Repeat Copy", &cp->RepeatCopy));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormBool("Erase Source", &cp->EraseSource));
  Add(wxMakeFormBool("Insert Space", &cp->InsertSpace));
  AssociatePanel(panel);
}



tCopyCommand::tCopyCommand(tTrackWin *t)
{
  tw = t;
  MarkRepeat = 0;
  Mouse = new tMarkDestin(tw->Canvas, tw, 0);
  CopyDlg = 0;
}


int tCopyCommand::Event(wxMouseEvent &e)
{
  if (!CopyDlg && Mouse && Mouse->Event(e))
  {
    if (Mouse->Aborted)
    {
      if (CopyDlg)
	CopyDlg->OnCancel();
      else
	Execute(0);
    }
    else if (!MarkRepeat)
    {
      e.Position(&StartX, &StartY);
      tw->Mark((long)StartX, (long)StartY);
      wxDialogBox *panel = new wxDialogBox(tw, "Replicate", FALSE );
      CopyDlg = new tCopyDlg(this);
      CopyDlg->EditForm(panel);
      panel->Fit();
      panel->Show(TRUE);
    }
    else
    {
      e.Position(&StopX, &StopY);
      Execute(1);
    }
  }
  return 0;
}


void tCopyCommand::OnOk()
{
  CopyDlg = 0;
  if (RepeatCopy)
  {
    delete Mouse;
    Mouse = new tMarkDestin(tw->Canvas, tw, 1);
    MarkRepeat = 1;
  }
  else
    Execute(1);
}


void tCopyCommand::OnCancel()
{
  CopyDlg = 0;
  Execute(0);
}


void tCopyCommand::Execute(int doit)
{

  if (doit)
  {
    long DestTrack = tw->y2Line((long)StartY);
    long DestClock = tw->x2BarClock((long)StartX);
    tCmdCopy cpy(tw->Filter, DestTrack, DestClock);

    if (RepeatCopy)
      cpy.RepeatClock = tw->x2BarClock((long)StopX, 1);
    cpy.EraseSource = EraseSource;
    cpy.EraseDestin = EraseDestin;
    cpy.InsertSpace = InsertSpace;
    cpy.Execute();
  }

  tw->UnMark();
  tw->MouseAction = 0;
  tw->Redraw();
  tw->NextWin->Redraw();
  delete Mouse;
  delete this;
}


void tTrackWin::MenCopy()
{
  if (!EventsSelected())
    return;
  MouseAction = new tCopyCommand(this);
}

// ******************************************************************
// Song-Settings Dialog
// ******************************************************************



class tSongSettingsDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  int TicksPerQuarter;
  int SongLength;
  int IntroLength;
  tSongSettingsDlg(tEventWin *w);
  void EditForm(wxPanel *panel);
  virtual void OnOk();
  virtual void OnCancel();
  virtual void OnHelp();
};



tSongSettingsDlg::tSongSettingsDlg(tEventWin *w)
: wxForm( USED_WXFORM_BUTTONS )
{
  EventWin = w;
  TicksPerQuarter = EventWin->Song->TicksPerQuarter;
  SongLength      = EventWin->Song->MaxQuarters / 4;
  IntroLength     = EventWin->Song->GetIntroLength();
}


void tSongSettingsDlg::OnHelp()
{
	HelpInstance->ShowTopic("Song");
}


void tSongSettingsDlg::OnCancel()
{
  EventWin->DialogBox = 0;
  wxForm::OnCancel();
}


void tSongSettingsDlg::OnOk()
{
  EventWin->Song->SetTicksPerQuarter(TicksPerQuarter);
  EventWin->Song->MaxQuarters = SongLength * 4;
  EventWin->Canvas->SetScrollRanges();
  if (EventWin->NextWin)
    EventWin->NextWin->Canvas->SetScrollRanges();
  EventWin->Redraw();
  EventWin->DialogBox = 0;
  EventWin->Song->SetIntroLength(IntroLength);
  wxForm::OnOk();
}


void tSongSettingsDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormMessage("Supported ticks/quarter: 48, 72, 96, 120, 144, 168, 192"));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort(" ", &TicksPerQuarter, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(48.0, 192.0), 0)));
  Add(wxMakeFormMessage("Ticks per Quarter Note"));

  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort(" ", &SongLength, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(50.0, 500.0), 0)));
  Add(wxMakeFormMessage("Song length in Bars"));

  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort(" ", &IntroLength, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(0.0, 8.0), 0)));
  Add(wxMakeFormMessage("Intro length in Bars"));
  AssociatePanel(panel);
}


void tTrackWin::MenSongSettings()
{
  tSongSettingsDlg *dlg;
  if (DialogBox)
  {
    DialogBox->Show(TRUE);
    return;
  }
  DialogBox = new wxDialogBox(this, "Song Settings", FALSE );
  dlg = new tSongSettingsDlg(this);
  dlg->EditForm(DialogBox);
  DialogBox->Fit();
  DialogBox->Show(TRUE);
}

// ******************************************************************
// Metronome-Settings Dialog
// ******************************************************************



class tMetronomeSettingsDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  int IsAccented;
  int KeyAcc;
  int KeyNorm;
  int Veloc;

  char *index2Name[130];
  int index2Pitch[130];
  int pitch2Index[130];
  int numNames;
  char *KeyAccName;
  char *KeyNormName;

  tMetronomeSettingsDlg(tEventWin *w);
  void EditForm(wxPanel *panel);
  virtual void OnOk();
  virtual void OnCancel();
  virtual void OnHelp();
};



tMetronomeSettingsDlg::tMetronomeSettingsDlg(tEventWin *w)
: wxForm( USED_WXFORM_BUTTONS )
{
  EventWin = w;
  IsAccented = ((tTrackWin*)EventWin)->MetronomeInfo.IsAccented;
  KeyAcc = ((tTrackWin*)EventWin)->MetronomeInfo.KeyAcc;
  KeyNorm = ((tTrackWin*)EventWin)->MetronomeInfo.KeyNorm;
  Veloc = ((tTrackWin*)EventWin)->MetronomeInfo.Veloc;

  numNames = 0;
  for (int i = 0; Config.DrumName(i).Name; i++)
  {
    if (Config.DrumName(i).Name[0])
    {
      index2Pitch[numNames] = Config.DrumName(i).Value - 1;
      pitch2Index[Config.DrumName(i).Value - 1] = numNames;
      index2Name[numNames++] = Config.DrumName(i).Name;
    }
  }
  KeyAccName = copystring( index2Name[ pitch2Index[ KeyAcc ] ] );
  KeyNormName = copystring( index2Name[ pitch2Index[ KeyNorm ] ] );
}


void tMetronomeSettingsDlg::OnHelp()
{
	HelpInstance->ShowTopic("Metronome Settings");
}


void tMetronomeSettingsDlg::OnCancel()
{
  EventWin->DialogBox = 0;
  wxForm::OnCancel();
}


void tMetronomeSettingsDlg::OnOk()
{
  ((tTrackWin*)EventWin)->MetronomeInfo.IsAccented = IsAccented;
  int i;
  for (i = 0; i < numNames; i++)
  {
    if ( !strcmp(index2Name[i], KeyNormName) )
      break;
  }
  if (i < numNames)
    ((tTrackWin*)EventWin)->MetronomeInfo.KeyNorm = index2Pitch[i];
  for (i = 0; i < numNames; i++)
  {
    if (!strcmp( index2Name[i], KeyAccName ) )
      break;
  }
  if (i < numNames)
    ((tTrackWin*)EventWin)->MetronomeInfo.KeyAcc = index2Pitch[i];
  ((tTrackWin*)EventWin)->MetronomeInfo.Veloc = Veloc;
  EventWin->Redraw();
  EventWin->DialogBox = 0;
  wxForm::OnOk();
}



void tMetronomeSettingsDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormBool( "Accented", &IsAccented ));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort(	" Velocity:", &Veloc, wxFORM_DEFAULT,
                       	new wxList(wxMakeConstraintRange(0.0, 127.0), 0)));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormString(	"Normal click:",
			&KeyNormName,
			wxFORM_DEFAULT,
			new wxList(wxMakeConstraintStrings(
				index2Name[ pitch2Index[ 37 ] ],
				index2Name[ pitch2Index[ 42 ] ],
				index2Name[ pitch2Index[ 56 ] ],
				0 ), 0),
			NULL,
			wxVERTICAL
			)
  );
  Add(wxMakeFormNewLine());
  Add(wxMakeFormString(	"Accented click:",
			&KeyAccName,
			wxFORM_DEFAULT,
			new wxList(wxMakeConstraintStrings(
				index2Name[ pitch2Index[ 36 ] ],
				index2Name[ pitch2Index[ 38 ] ],
				index2Name[ pitch2Index[ 54 ] ],
				0 ), 0),
			NULL,
			wxVERTICAL
			)
  );
  AssociatePanel(panel);
}


void tTrackWin::MenMetronomeSettings()
{
  tMetronomeSettingsDlg *dlg;
  if (DialogBox)
  {
    DialogBox->Show(TRUE);
    return;
  }
  DialogBox = new wxDialogBox(this, "Metronome Settings", FALSE );
  dlg = new tMetronomeSettingsDlg(this);
  dlg->EditForm(DialogBox);
  DialogBox->Fit();
  DialogBox->Show(TRUE);
}


// *************************************************************************************
// Split / Merge Tracks
// *************************************************************************************

void tTrackWin::MenMergeTracks()
{
  int choice = wxMessageBox("Merge all Tracks to Track 0?", "Sure?", wxOK | wxCANCEL);
  if (choice == wxOK)
  {
    int tn;
    Song->NewUndoBuffer();
    tTrack *dst = Song->GetTrack(0);
    for (tn = 1; tn < Song->nTracks; tn++)
    {
      tTrack *src = Song->GetTrack(tn);
      tEventIterator Iterator(src);
      tEvent *e = Iterator.First();
      while (e)
      {
	tEvent *c = e->Copy();
	src->Kill(e);
	dst->Put(c);
	e = Iterator.Next();
      }
      src->Cleanup();
    }
    dst->Cleanup();
    Redraw();
  }
}


void tTrackWin::MenSplitTracks()
{
  int choice = wxMessageBox("Split Track 0 by Midi-Channel to Track 1..16?", "Sure?", wxOK | wxCANCEL);
  if (choice == wxOK)
  {
    int ch;
    Song->NewUndoBuffer();
    tTrack *src = Song->GetTrack(0);
    tEventIterator Iterator(src);
    tEvent *e = Iterator.First();
    while (e)
    {
      tChannelEvent *ce = e->IsChannelEvent();
      if (ce)
      {
	int cn = ce->Channel;
	tTrack *dst = Song->GetTrack(cn + 1);
	if (dst)
	{
	  tEvent *cp = ce->Copy();
	  src->Kill(ce);
	  dst->Put(cp);
	  dst->Channel = cn + 1;
	}
      }
      e = Iterator.Next();
    }

    for (ch = 0; ch <= 16; ch++)
    {
      src = Song->GetTrack(ch);
      if (src)
	src->Cleanup();
    }
    Redraw();
  }
}

// ******************************************************************
// Midi Timing Dialog
// ******************************************************************

class tMidiButton;

class tTimingDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  tTimingDlg(tEventWin *w);
  void EditForm(wxPanel *panel);
  virtual void OnOk();
  virtual void CloseWindow();
  virtual void OnHelp();
  char *ClkSrcArray[4];
  char *MtcTypeArray[4];
  wxListBox *ClkSrcListBox;
  wxListBox *MtcTypeListBox;
  wxCheckBox *RealTimeCheckBox;
  wxText *MtcOffsetEntry;
  void MtcInitRec();
  void MtcFreezeRec();
  static void MtcRecFunc( tMidiButton& button, wxCommandEvent& event );
  static void OkFunc( tMidiButton& button, wxCommandEvent& event );
  static void CancelFunc( tMidiButton& button, wxCommandEvent& event );
  static void HelpFunc( tMidiButton& button, wxCommandEvent& event );
};

class tMidiButton : public wxButton {
  public:
    tMidiButton( tTimingDlg *dlg,
		 wxPanel *panel, wxFunction func, char *label, int x = -1, int y = -1,
		 int width = -1, int height = -1, long style = 0, char *name = "button")
    : wxButton( panel, func, label, x, y, width, height, style, name )
    {
      midiDlg = dlg;
    }

    void OnOk()
    {
      midiDlg->OnOk();
    }

    void OnCancel()
    {
      midiDlg->CloseWindow();
    }

    void OnHelp()
    {
      midiDlg->OnHelp();
    }

    void OnInitRec()
    {
      midiDlg->MtcInitRec();
    }

    void OnFreezeRec()
    {
      midiDlg->MtcFreezeRec();
    }
  private:
    tTimingDlg *midiDlg;
};

void tTimingDlg::OkFunc( tMidiButton& button, wxCommandEvent& event )
{
  button.OnOk();
}

void tTimingDlg::CancelFunc( tMidiButton& button, wxCommandEvent& event )
{
  button.OnCancel();
}

void tTimingDlg::HelpFunc( tMidiButton& button, wxCommandEvent& event )
{
  button.OnHelp();
}

void tTimingDlg::MtcRecFunc( tMidiButton& button, wxCommandEvent& event )
{
  if ( !strcmp( button.GetLabel(), "Start" ) )
  {
    button.OnInitRec();
    button.SetLabel( "Freeze" );
  }
  else
  {
    button.OnFreezeRec();
    button.SetLabel( "Start" );
  }
}

void tTimingDlg::MtcInitRec()
{
#ifdef wx_msw
  if (Config(C_ClockSource) != CsMtc)
  {
    delete Midi;
    Config(C_ClockSource) = CsMtc;
    ClkSrcListBox->SetStringSelection( ClkSrcArray[ Config(C_ClockSource) ] );
    Midi = new tWinMtcPlayer(EventWin->Song);
    if (!Midi->Installed())
    {
      wxMessageBox("no midi driver installed", "Error", wxOK);
      Midi = new tNullPlayer(EventWin->Song);
    }
  }
#endif
  Midi->InitMtcRec();
}

void tTimingDlg::MtcFreezeRec()
{
  tMtcTime *offs = Midi->FreezeMtcRec();
  if (offs)
  {
    char str[80];
    offs->ToString( str );
    MtcTypeListBox->SetStringSelection( MtcTypeArray[ offs->type ] );
    delete offs;
    MtcOffsetEntry->SetValue( str );
  }
}

tTimingDlg::tTimingDlg(tEventWin *w)
: wxForm( USED_WXFORM_BUTTONS )
{
  EventWin = w;
  ClkSrcArray[CsInt] = "INTERNAL";
  ClkSrcArray[CsFsk] = "FSK";
  ClkSrcArray[CsMidi] = "SONG PTR";
  ClkSrcArray[CsMtc] = "MTC";
  MtcTypeArray[Mtc24] = "24 fm/sec";
  MtcTypeArray[Mtc25] = "25 fm/sec";
  MtcTypeArray[Mtc30Df] = "30 drop";
  MtcTypeArray[Mtc30Ndf] = "30 non-drop";
  ClkSrcListBox = 0;
  MtcTypeListBox = 0;
  RealTimeCheckBox = 0;
  MtcOffsetEntry = 0;
}

void tTimingDlg::CloseWindow()
{
  EventWin->DialogBox->Show( FALSE );
  delete EventWin->DialogBox;
  EventWin->DialogBox = 0;
  DELETE_THIS();
}


void tTimingDlg::OnOk()
{
  int i;
  char *str = copystring( ClkSrcListBox->GetStringSelection() );
  for (i = 0; i < 4; i++) {
	if (!strcmp(str,ClkSrcArray[i])) {
		break;
	}
  }
  delete str;
  if (i > 3)
    i = CsInt;

  if (i != Config(C_ClockSource))
  {
    Config(C_ClockSource) = (tClockSource) i;
#ifdef wx_msw
    // Re-install the midi device
    delete Midi;

    // create new player
    switch (Config(C_ClockSource))
    {
      case CsMidi:
        Midi = new tWinMidiPlayer(EventWin->Song);
        break;
      case CsMtc:
        Midi = new tWinMtcPlayer(EventWin->Song);
        break;
      case CsFsk:
      case CsInt:
      default:
#ifdef AUDIO
	Midi = new tWinAudioPlayer(EventWin->Song);
#else
	Midi = new tWinIntPlayer(EventWin->Song);
#endif
        break;
    }
    if (!Midi->Installed())
    {
      wxMessageBox("no midi driver installed", "Error", wxOK);
      Midi = new tNullPlayer(EventWin->Song);
    }
#endif
  }

  tMtcType MtcType;
  str = copystring( MtcTypeListBox->GetStringSelection() );
  for (i = 0; i < 4; i++) {
	if (!strcmp(str,MtcTypeArray[i])) {
		MtcType = (tMtcType) i;
		break;
	}
  }
  delete str;
  if (i > 3)
    MtcType = Mtc30Ndf;
  tMtcTime *offs = new tMtcTime( MtcOffsetEntry->GetValue(), MtcType );
  EventWin->Song->GetTrack(0)->SetMtcOffset( offs );
  delete offs;

  Config(C_RealTimeOut) = RealTimeCheckBox->GetValue();
  EventWin->Redraw();
  CloseWindow();
}

void tTimingDlg::OnHelp()
{
	HelpInstance->ShowTopic("Timing");
}



void tTimingDlg::EditForm(wxPanel *panel)
{

  (void) new tMidiButton( this, panel, (wxFunction) OkFunc, "Ok" );
  (void) new tMidiButton( this, panel, (wxFunction) CancelFunc, "Cancel" );
  (void) new tMidiButton( this, panel, (wxFunction) HelpFunc, "Help" );
  panel->NewLine();

  panel->SetLabelPosition(wxVERTICAL);
  ClkSrcListBox = new wxListBox( panel,
				 NULL,
				 "Clock Source",
				 wxSINGLE|wxALWAYS_SB,
				 -1, -1, -1, -1 );

#ifdef wx_msw
  ClkSrcListBox->Append( ClkSrcArray[CsInt] );
  ClkSrcListBox->Append( ClkSrcArray[CsMidi] );
  ClkSrcListBox->Append( ClkSrcArray[CsMtc] );
#else
  int driver = Config(C_MidiDriver);
  if (driver == C_DRV_OSS || driver == C_DRV_ALSA)
  {
    ClkSrcListBox->Append( ClkSrcArray[CsInt] );
  }
  else if (driver == C_DRV_JAZZ)
  {
    ClkSrcListBox->Append( ClkSrcArray[CsInt] );
    ClkSrcListBox->Append( ClkSrcArray[CsMidi] );
    ClkSrcListBox->Append( ClkSrcArray[CsFsk] );
  }
#endif
  ClkSrcListBox->SetStringSelection( ClkSrcArray[ Config(C_ClockSource) ] );
  panel->NewLine();

  tTrack *t = EventWin->Song->GetTrack(0);
  char str[80];
  tMtcTime *offs = t->GetMtcOffset();
  offs->ToString( str );
  MtcOffsetEntry = new wxText( panel,
			       NULL,
			       "MTC offset",
			       str,
                -1,
                -1,
                100 );

  panel->NewLine();

#ifdef wx_msw
  (void) new wxMessage( panel, "Record MTC offset: " );
  (void) new tMidiButton( this, panel, (wxFunction) MtcRecFunc, "Start" );
  panel->NewLine();
#endif

  MtcTypeListBox = new wxListBox( panel,
				  NULL,
				  "MTC Type",
				  wxSINGLE|wxALWAYS_SB,
				  -1, -1, -1, -1 );
  MtcTypeListBox->Append( MtcTypeArray[0] );
  MtcTypeListBox->Append( MtcTypeArray[1] );
  MtcTypeListBox->Append( MtcTypeArray[2] );
  MtcTypeListBox->Append( MtcTypeArray[3] );
  MtcTypeListBox->SetStringSelection( MtcTypeArray[ offs->type ] );
  delete offs;
  panel->NewLine();

  RealTimeCheckBox = new wxCheckBox( panel,
				     NULL,
				     "Realtime to MIDI Out" );
  RealTimeCheckBox->SetValue( Config(C_RealTimeOut) );
  panel->NewLine();
}


void tTrackWin::MenTiming()
{
  tTimingDlg *dlg;
  if (DialogBox)
  {
    DialogBox->Show(TRUE);
    return;
  }
  DialogBox = new wxDialogBox(this, "MIDI Timing Settings", FALSE );
  dlg = new tTimingDlg(this);
  dlg->EditForm(DialogBox);
  DialogBox->Fit();
  DialogBox->Show(TRUE);
}

// ******************************************************************
// Midi thru dialog
// ******************************************************************

class tMidiThruDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  tMidiThruDlg(tEventWin *w);
  void EditForm(wxPanel *panel);
  virtual void OnOk();
  virtual void OnCancel();
  virtual void OnHelp();

  tNamedChoice InputDeviceChoice;
  tNamedChoice OutputDeviceChoice;
  long InputDevice;
  long OutputDevice;
};



tMidiThruDlg::tMidiThruDlg(tEventWin *w)
: wxForm( USED_WXFORM_BUTTONS ),
  InputDeviceChoice("Input Device", Midi->GetInputDevices().AsNamedValue(), &InputDevice),
  OutputDeviceChoice("Output Device", Midi->GetOutputDevices().AsNamedValue(), &OutputDevice)
{
  EventWin = w;
  InputDevice = Midi->GetThruInputDevice();
  OutputDevice = Midi->GetThruOutputDevice();
}


void tMidiThruDlg::OnHelp()
{
	HelpInstance->ShowTopic("Midi Thru");
}


void tMidiThruDlg::OnCancel()
{
  EventWin->DialogBox = 0;
  wxForm::OnCancel();
}


void tMidiThruDlg::OnOk()
{
  InputDeviceChoice.GetValue();
  OutputDeviceChoice.GetValue();
  Config(C_ThruInput) = InputDevice;
  Config(C_ThruOutput) = OutputDevice;
  Midi->SetSoftThru(Config(C_SoftThru), InputDevice, OutputDevice);
  Midi->SetHardThru(Config(C_HardThru), InputDevice, OutputDevice);
  EventWin->Redraw();
  EventWin->DialogBox = 0;
  wxForm::OnOk();
}



void tMidiThruDlg::EditForm(wxPanel *panel)
{
  if (Midi->SupportsMultipleDevices()) {
    Add(InputDeviceChoice.mkFormItem(300, 50));
    Add(wxMakeFormNewLine());
    Add(OutputDeviceChoice.mkFormItem(300, 50));
    Add(wxMakeFormNewLine());
  }
#ifdef wx_msw
  Add(wxMakeFormBool( "Software MIDI thru", &Config(C_SoftThru) ));
  Add(wxMakeFormNewLine());
#else
  int driver = Config(C_MidiDriver);
  if (driver == C_DRV_OSS || driver == C_DRV_ALSA)
  {
    Add(wxMakeFormBool( "Software MIDI thru", &Config(C_SoftThru) ));
    Add(wxMakeFormNewLine());
  }
  else
  {
    Add(wxMakeFormBool( "Hardware MIDI thru", &Config(C_HardThru) ));
    Add(wxMakeFormNewLine());
  }
#endif
  AssociatePanel(panel);
}

void tTrackWin::MenMidiThru()
{
  tMidiThruDlg *dlg;
  if (DialogBox)
  {
    DialogBox->Show(TRUE);
    return;
  }
  DialogBox = new wxDialogBox(this, "Midi Thru Settings", FALSE );
  dlg = new tMidiThruDlg(this);
  dlg->EditForm(DialogBox);
  DialogBox->Fit();
  DialogBox->Show(TRUE);
}

// ******************************************************************
// Synth settings dialog
// ******************************************************************

#define NumResetStrings 3

class tSynthSettingsDlg : public wxForm
{
   public:
      tEventWin *EventWin;
      char *SynthTypeName;
      char *OldSynthTypeName;
      char *ResetVal;
      char *ResetString[NumResetStrings];
      tSynthSettingsDlg(tEventWin *w);
      void EditForm(wxPanel *panel);
      virtual void OnOk();
      virtual void OnCancel();
      virtual void OnHelp();
};



tSynthSettingsDlg::tSynthSettingsDlg(tEventWin *w)
: wxForm( USED_WXFORM_BUTTONS )
{
  EventWin = w;
  SynthTypeName = copystring( Config.StrValue(C_SynthType) );
  OldSynthTypeName = copystring( Config.StrValue(C_SynthType) );

  ResetString[0] = "Never";
  ResetString[1] = "Song start";
  ResetString[2] = "Start play";

  ResetVal = copystring( ResetString[Config( C_SendSynthReset )] );
}


void tSynthSettingsDlg::OnHelp()
{
        HelpInstance->ShowTopic("Synthesizer Type Settings");
}


void tSynthSettingsDlg::OnCancel()
{
  EventWin->DialogBox = 0;
  wxForm::OnCancel();
}


void tSynthSettingsDlg::OnOk()
{
   int i;
   char* str = copystring( SynthTypeName );

   for (i = 0; i < NumSynthTypes; i++)
   {
      if (!strcmp( str, SynthTypes[i].Name))
         break;
   }

   Config.Put( C_SynthConfig, SynthTypeFiles[i].Name );
   Config.StrValue(C_SynthType) = copystring(SynthTypes[i].Name);

   delete str;
   str = copystring( ResetVal );

   for (i = 0; i < NumResetStrings; i++)
   {
      if (!strcmp( str, ResetString[i]))
         break;
   }

   Config( C_SendSynthReset ) = i;
   Config.Put( C_SendSynthReset );

   if ( Config( C_SynthDialog ) ) {
      Config( C_SynthDialog ) = 0;
      Config.Put( C_SynthDialog );
   }

   if (strcmp( SynthTypeName, OldSynthTypeName ))
      wxMessageBox("Restart jazz for the changes to take effect", "Info", wxOK);

   delete str;
   delete SynthTypeName;
   delete OldSynthTypeName;
   delete ResetVal;
   EventWin->Redraw();
   EventWin->DialogBox = 0;
   wxForm::OnOk();
}



void tSynthSettingsDlg::EditForm(wxPanel *panel)
{
  int i;
   // following adapted from wxwin/src/base/wb_form.cc
   wxList *list = new wxList;
   for (i = 0; i < NumSynthTypes; i++)
      if (*SynthTypes[i].Name)	// omit empty entries
	 list->Append((wxObject *)copystring(SynthTypes[i].Name));

   wxFormItemConstraint *constraint = wxMakeConstraintStrings(list);

   Add(wxMakeFormNewLine());
   Add(wxMakeFormString( "Synthesizer type:",
			 &SynthTypeName,
			 //wxFORM_RADIOBOX,
			 wxFORM_DEFAULT,
			 new wxList(constraint, 0),
			 NULL,
			 wxVERTICAL
      )
      );

   // following adapted from wxwin/src/base/wb_form.cc
   wxList *list2 = new wxList;
   for (i = 0; i < NumResetStrings; i++)
      list2->Append((wxObject *)copystring( ResetString[i] ));

   wxFormItemConstraint *constraint2 = wxMakeConstraintStrings(list2);

   Add(wxMakeFormNewLine());
   Add(wxMakeFormString( "Auto send MIDI reset:",
                         &ResetVal,
                         //wxFORM_RADIOBOX,
                         wxFORM_DEFAULT,
                         new wxList(constraint2, 0),
                         NULL,
                         wxVERTICAL
   )
   );

   Add(wxMakeFormNewLine());

   AssociatePanel(panel);
}

void tTrackWin::MenSynthSettings()
{
  tSynthSettingsDlg *dlg;
  if (DialogBox)
  {
    DialogBox->Show(TRUE);
    return;
  }
  DialogBox = new wxDialogBox(this, "Synth Type", FALSE );
  dlg = new tSynthSettingsDlg(this);
  dlg->EditForm(DialogBox);
  DialogBox->Fit();
  DialogBox->Show(TRUE);
}

void tTrackWin::SaveMidiDeviceSettings( int dev )
{
  Config.Put( C_Seq2Device, dev );
}

void tTrackWin::SaveThruSettings()
{
  Config.Put( C_SoftThru );
  Config.Put( C_HardThru );
  Config.Put( C_ThruInput );
  Config.Put( C_ThruOutput );
}

void tTrackWin::SaveTimingSettings()
{
  Config.Put( C_RealTimeOut );
  Config.Put( C_ClockSource );
}

void tTrackWin::SaveEffectSettings()
{
  Config.Put( C_UseReverbMacro );
  Config.Put( C_UseChorusMacro );
}

void tTrackWin::SaveMetronomeSettings()
{
  Config.Put( C_MetroIsAccented, MetronomeInfo.IsAccented );
  Config.Put( C_MetroVelocity, MetronomeInfo.Veloc );
  Config.Put( C_MetroNormalClick, MetronomeInfo.KeyNorm );
  Config.Put( C_MetroAccentedClick, MetronomeInfo.KeyAcc );
}

void tTrackWin::SaveGeoSettings()
{
  GetPosition( &Config(C_TrackWinXpos), &Config(C_TrackWinYpos) );
  GetSize( &Config(C_TrackWinWidth), &Config(C_TrackWinHeight) );
  Config.Put( C_TrackWinXpos );
  Config.Put( C_TrackWinYpos );
  Config.Put( C_TrackWinWidth );
  Config.Put( C_TrackWinHeight );

  if (NextWin)
  {
    NextWin->GetPosition( &Config(C_PianoWinXpos), &Config(C_PianoWinYpos) );
    NextWin->GetSize( &Config(C_PianoWinWidth), &Config(C_PianoWinHeight) );
    Config.Put( C_PianoWinXpos );
    Config.Put( C_PianoWinYpos );
    Config.Put( C_PianoWinWidth );
    Config.Put( C_PianoWinHeight );
  }

  if (MixerForm)
  {
    MixerForm->GetPosition( &Config(C_PartsDlgXpos), &Config(C_PartsDlgYpos) );
  }
  Config.Put( C_PartsDlgXpos );
  Config.Put( C_PartsDlgYpos );
  Config.Put( C_TrackDlgXpos );
  Config.Put( C_TrackDlgYpos );

  if (rhythm_win)
  {
    rhythm_win->GetPosition( &Config(C_RhythmXpos), &Config(C_RhythmYpos) );
  }
  Config.Put( C_RhythmXpos );
  Config.Put( C_RhythmYpos );

  if (the_harmony_browser)
  {
    ((HBFrame *)the_harmony_browser)->GetPosition( &Config(C_HarmonyXpos), &Config(C_HarmonyYpos) );
  }
  Config.Put( C_HarmonyXpos );
  Config.Put( C_HarmonyYpos );
}

// ******************************************************************
// Copyright dialog
// ******************************************************************



class tCopyrightDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  char *String;
  tCopyrightDlg(tEventWin *w);
  ~tCopyrightDlg() { delete String; }
  void EditForm(wxPanel *panel);
  virtual void OnOk();
  virtual void OnCancel();
  virtual void OnHelp();
};



tCopyrightDlg::tCopyrightDlg(tEventWin *w)
: wxForm( USED_WXFORM_BUTTONS )
{
  EventWin = w;
  char *cs = EventWin->Song->GetTrack(0)->GetCopyright();
  if ( cs && strlen(cs) )
  {
    String = copystring( cs );
  }
  else
  {
    String = copystring( "Copyright (C) <year> <owner>" );
  }
}


void tCopyrightDlg::OnHelp()
{
	HelpInstance->ShowTopic("Set Music Copyright");
}


void tCopyrightDlg::OnCancel()
{
  EventWin->DialogBox = 0;
  wxForm::OnCancel();
}


void tCopyrightDlg::OnOk()
{
  EventWin->Song->GetTrack(0)->SetCopyright( String );
  EventWin->Redraw();
  EventWin->DialogBox = 0;
  wxForm::OnOk();
}



void tCopyrightDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormString("Copyright notice:",
                       &String,
                       wxFORM_DEFAULT,
                       NULL, NULL, wxVERTICAL, 400 ));
  AssociatePanel(panel);
}


void tTrackWin::MenCopyright()
{
  tCopyrightDlg *dlg;
  if (DialogBox)
  {
    DialogBox->Show(TRUE);
    return;
  }
  DialogBox = new wxDialogBox(this, "Music copyright", FALSE );
  dlg = new tCopyrightDlg(this);
  dlg->EditForm(DialogBox);
  DialogBox->Fit();
  DialogBox->Show(TRUE);
}


class tLoadPattern : public tMarkDestin {
  public:
    tLoadPattern(tTrackWin *t, char *fn)
      : tMarkDestin(t->Canvas, t, 0), fname(fn), tw(t) {}

    int LeftDown(wxMouseEvent &e);
  private:
    char *fname;
    tTrackWin *tw;
};

int tLoadPattern::LeftDown(wxMouseEvent &e)
{
  tMarkDestin::LeftDown(e);
  tSong *sng = new tSong;
  tStdRead io;
  sng->Read(io, fname);

  sng->SetTicksPerQuarter(tw->Song->TicksPerQuarter);

  int dst_track = tw->y2Line((long)y);
  int num_tracks = sng->NumUsedTracks();
  long delta = tw->x2BarClock((long)x);
  tw->Song->NewUndoBuffer();
  for (int i = 0; i < num_tracks; i++) {
    tTrack *src = sng->GetTrack(i);
    tTrack *dst = tw->Song->GetTrack(dst_track++);
    tEventIterator iter(src);
    tEvent *e = iter.First();
    while (e) {
      e = e->Copy();
      e->Clock += delta;
      dst->Put(e);
      e = iter.Next();
    }
    dst->Cleanup();
  }
  delete sng;

  tw->Redraw();
  tw->MouseAction = 0;
  delete this;
  return 1;
}

void tTrackWin::MenLoadPattern()
{
  if (MouseAction)
    return;
  char * fname = file_selector(defpattern, "Load Pattern", 0, 0, "*.mid");
  if (fname)
    MouseAction = new tLoadPattern(this, fname);
}


void tTrackWin::MenSavePattern()
{
  if (!EventsSelected("please select events to be saved"))
    return;
  char * fname = file_selector(defpattern, "Save Selected Range", 1, 0, "*.mid");
  if (fname) {
    int tracknr = 0;
    tSong *sng = new tSong;
    delete sng;
    sng = new tSong;
    tTrackIterator Tracks(Filter);
    tTrack *src = Tracks.First();
    while (src != 0)
    {
      tTrack *dst = sng->GetTrack(tracknr++);
      tEventIterator Events(src);
      long delta = Filter->FromClock;
      tEvent *e = Events.Range(Filter->FromClock, Filter->ToClock);
      while (e) {
        if (Filter->IsSelected(e)) {
          e = e->Copy();
          e->Clock -= delta;
          dst->Put(e);
        }
        e = Events.Next();
      }
      dst->Cleanup();
      src = Tracks.Next();
    }
    tStdWrite io;
    sng->Write(io, fname);
    delete sng;
    Config.Put(C_StartUpSong, fname);
  }
}


// ----------------------------------------------------------------------------
//                              clipboard copy/paste
// ----------------------------------------------------------------------------

void tTrackWin::MenClpCopy(Bool erase)
{
  if (!EventsSelected("please select events to be copied"))
    return;
  Song->NewUndoBuffer();

  int tracknr = 0;

  delete paste_buffer;
  paste_buffer = new tSong;
  tTrackIterator Tracks(Filter);
  tTrack *src = Tracks.First();
  while (src != 0)
  {
    tTrack *dst = paste_buffer->GetTrack(tracknr++);
    tEventIterator Events(src);
    long delta = Filter->FromClock;
    tEvent *e = Events.Range(Filter->FromClock, Filter->ToClock);
    while (e) {
      if (Filter->IsSelected(e)) {
	tEvent *c = e->Copy();
	c->Clock -= delta;
	dst->Put(c);
	if (erase)
	  src->Kill(e);
      }
      e = Events.Next();
    }
    dst->Cleanup();
    if (erase)
      src->Cleanup();
    src = Tracks.Next();
  }
  paste_buffer->TicksPerQuarter = Song->TicksPerQuarter;
  if (erase)
    Redraw();
}


class tClpPaste : public tMarkDestin {
  public:
    tClpPaste(tTrackWin *t, tSong *clp)
      : tMarkDestin(t->Canvas, t, 0), sng(clp), tw(t) {}

    int LeftDown(wxMouseEvent &e);
  private:
    tSong *sng;
    tTrackWin *tw;
};

int tClpPaste::LeftDown(wxMouseEvent &e)
{
  tMarkDestin::LeftDown(e);

  sng->SetTicksPerQuarter(tw->Song->TicksPerQuarter);

  int dst_track = tw->y2Line((long)y);
  int num_tracks = sng->NumUsedTracks();
  long delta = tw->x2BarClock((long)x);
  tw->Song->NewUndoBuffer();
  for (int i = 0; i < num_tracks; i++) {
    tTrack *src = sng->GetTrack(i);
    tTrack *dst = tw->Song->GetTrack(dst_track++);
    tEventIterator iter(src);
    tEvent *e = iter.First();
    while (e) {
      e = e->Copy();
      e->Clock += delta;
      dst->Put(e);
      e = iter.Next();
    }
    dst->Cleanup();
  }

  tw->Redraw();
  tw->MouseAction = 0;
  delete this;
  return 1;
}

void tTrackWin::MenClpPaste() {
  if (MouseAction || paste_buffer == NULL)
    return;
  MouseAction = new tClpPaste(this, paste_buffer);
}

