/*
**  The JAZZ++ Midi Sequencer
**
** Copyright (C) 1994-2000 Andreas Voss and Per Sigmond, all rights reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
*/                                                                              

#include "wx.h"
#pragma hdrstop

#include "util.h"
#include "song.h"
#include "jazz.h"

#include <ctype.h>


// *************************************************************************
// tNamedChoice
// *************************************************************************

tNamedValue limitSteps[] =
{
  tNamedValue( "1/8",   8 ),
  tNamedValue( "1/12", 12 ),
  tNamedValue( "1/16", 16 ),
  tNamedValue( "1/24", 24 ),
  tNamedValue( "1/32", 32 ),
  tNamedValue( "1/48", 48 ),
  tNamedValue( "1/96", 96 ),
  tNamedValue( "1/192", 192 ),
  tNamedValue(   0,      1  )
};


tNamedChoice::tNamedChoice(char *t, tNamedValue *v, long *r)
{
  Values = v;
  Selection = 0;
  Title = t;
  Result = r;
}



wxFormItem *tNamedChoice::mkFormItem(int w, int h)
{
  SetValue();

  // following adapted from wxwin/src/base/wb_form.cc
  wxList *list = new wxList;
  for (int i = 0; Values[i].Name; i++)
    if (*Values[i].Name)	// omit empty entries
      list->Append((wxObject *)copystring(Values[i].Name));

  wxFormItemConstraint *constraint = wxMakeConstraintStrings(list);
  return wxMakeFormString(Title, &Selection, wxFORM_SINGLE_LIST, new wxList(constraint, 0),
    0, 0, w, h);
}


void tNamedChoice::GetValue()
{
  int i;

  if (Selection)
  {
    for (i = 0; Values[i].Name; i++)
    {
      if (!strcmp(Selection, Values[i].Name))
      {
	*Result = Values[i].Value;
	break;
      }
    }
  }
}


void tNamedChoice::SetValue()
{
  int i;

  for (i = 0; Values[i].Name; i++)
  {
    if (*Result == Values[i].Value)
    {
      delete Selection;
      Selection = copystring(Values[i].Name);
      break;
    }
  }
}

tNamedChoice::~tNamedChoice()
{
  delete Selection;
}


// **********************************************************************
// tNamedValuesChoice
// **********************************************************************

tNamedValueChoice::tNamedValueChoice(wxPanel *parent, wxFunction func, char *label, tNamedValue* nvals)
 : wxChoice(parent, func, label), values(nvals)
{
  tNamedValue *v = values;
  while (v->Name) {
    Append(v->Name);
    ++v;
  }
}

long tNamedValueChoice::GetValue()
{
  int i = GetSelection();
  if (i >= 0)
    return values[i].Value;
  return 16;
}


void tNamedValueChoice::SetValue(long measure)
{
  int i;
  for (i = 0; values[i].Name; i++) {
    if (values[i].Value == measure) {
      SetSelection(i);
      break;
    }
  }
}


long tMeasureChoice::GetTicks(tSong *s) {
  long m = GetMeasure();
  if (m < 0)
    m = 16;
  m = s->TicksPerQuarter * 4 / m;
  return (m > 0) ? m : 1;
}

// **********************************************************************
// tRect
// **********************************************************************


tRect::tRect(float xx, float yy, float ww, float hh)
{
  x = xx; y = yy; w = ww; h = hh;
}

void tRect::SetUnion(tRect &r)
{
  if (r.w > w)
    w = r.w;
  if (r.h > h)
    h = r.h;
}

void tRect::SetNormal()
{
  if (w < 0)
  {
    w = -w;
    x -= w;
  }
  if (h < 0)
  {
    h = -h;
    y -= h;
  }
  if (w == 0)
    w = 1;
  if (h == 0)
    h = 1;
}

int tRect::IsInside(float xx, float yy)
{
  if (xx < x || xx > x + w)
    return 0;
  if (yy < y || yy > y + h)
    return 0;
  return 1;
}

// **************************************************************************
// tClockDlg
// **************************************************************************

tClockDlg::tClockDlg(tSong *s, char *t, long c)
{
  char buf[500];
  s->Clock2String(c, buf);
  String = copystring(buf);
  Title = t;
  Song  = s;
}


tClockDlg::~tClockDlg()
{
  delete String;
}


wxFormItem *tClockDlg::mkFormItem(int w)
{
  return wxMakeFormString(Title, &String, wxFORM_DEFAULT,0,0,0,w);
}



long tClockDlg::GetClock()
{
  return Song->String2Clock(String);
}

// ***********************************************************************
// KeyStr
// ***********************************************************************


void Key2Str(int key, char *str)
{
  static char *Names[] = { "C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B" };
  strcpy(str, Names[key % 12]);
  sprintf(str + strlen(str), "%d", key / 12);
}


int Str2Key(const char *str)
{
  static char sKey[] = "cCdDeEfFgGaAbB";
  static int  nKey[]={0,0,2,2,4,4,5,5,7,7,9,9,11,11};
  int i, key = 0;

  while (*str)
  {
    if (*str == '#')
    {
      key += 1;
      str++;
    }
    else if (isdigit(*str))
    {
      int n = 0;
      while (isdigit(*str))
        n = 10 * n + *str++ - '0';
      key += 12 * n;
    }
    else
    {
      for (i = 0; sKey[i]; i++)
      {
	if (str[0] == sKey[i])
	{
	  key += nKey[i];
	  str++;
	  break;
	}
      }
      if (!sKey[i])	// error
        ++ str;
    }
  }
  return key;
}

// **************************************************************************
// tNoteDlg
// **************************************************************************

tKeyDlg::tKeyDlg(char *t, int Key)
{
  char buf[50];
  Key2Str(Key, buf);
  String = copystring(buf);
  Title = t;
}


tKeyDlg::~tKeyDlg()
{
  delete String;
}


wxFormItem *tKeyDlg::mkFormItem(int w)
{
  return wxMakeFormString(Title, &String, wxFORM_DEFAULT,0,0,0,w);
}



int tKeyDlg::GetKey()
{
  return Str2Key(String);
}

int GetArgOpt( char *opt ) {

	int arg = 0;

	for (int i = 1; i < wxTheApp->argc; i++) {
		if (!strcmp( wxTheApp->argv[i], opt )) {
				arg = i;
		}
	}
	return( arg );
}


int SelectControllerDlg()
{
  int i, n = 0;
  char *names[130];
  int   ctrls[130];
  for (i = 0; Config.CtrlName(i).Name; i++)
  {
    if (Config.CtrlName(i).Name[0])
    {
      ctrls[n]   = Config.CtrlName(i).Value;
      names[n++] = Config.CtrlName(i).Name;
    }
  }
  i = wxGetSingleChoiceIndex("Controller", "Select a controller", n, names);
  if (i >= 0)
    return ctrls[i];
  return -1;
}

#if 0
// select multiple drum instruments
// returns:
//  -1 : user pressed escape
//   0 : user pressed ok, but no instrument was selected
//   n : user pressed ok, n = number of selected instruments. The keys
//       are returned in keys parameter

int SelectMultipleDrumsDlg(int *keys)
{
  int i, n = 0;
  char *names[150];
  int  selection[150];
  int  n_sel;

  for (i = 0; DrumNames[i].Name; i++)
    if (DrumNames[i].Name[0])
      names[n++] = DrumNames[i].Name;
  n_sel = wxGetMultipleChoice("Select some instruments", "Drums", n, names, 0, selection);
  for (i = 0; i < n_sel; i++)
    keys[i] = DrumNames[selection[i]].Value - 1;
  return n_sel;
}
#endif


// read/write strings from stream

istream & ReadString(istream &is, char *buf, int maxlen)
{
  // read a string (I hate C++)
  maxlen--;	// space for \0
  int c, i;
  do
    c = is.get();	// ignore \"
  while (c != '"' && c != EOF);
  for (i = 0; i < maxlen; i++) {
    c = is.get();
    if (c == '"' || c == EOF)
      break;
    buf[i] = c;
  }
  buf[i] = 0;
  return is;
}


ostream & WriteString(ostream &os, const char *str)
{
  os << '"' << str << '"';
  return os;
}



static const char * add_default_ext(const char *fn, const char *ext)
// fn must be dynamically allocated and the result will be too!
{
  // is any extension already there?
  {
    const char *x = wxFileNameFromPath((char *)fn);
    if (strchr(x, '.') != 0)
      return fn;
  }

  const int elen = strlen(ext);
  const int flen = strlen(fn);
  char *fn2 = new char[strlen(fn) + elen + 1];
  strcpy(fn2, fn);
  strcat(fn2, ext);
  delete [] (void *)fn;
  return fn2;
}


char *file_selector(char * &deffile, const char *title, Bool save, Bool changed, const char *ext)
{
  char *s = 0;

  {
    char *file = 0, *path = 0;
    if (save)
    {
      file = wxFileNameFromPath(deffile);
      if (file)
	file = copystring(file);
    }
    path = wxPathOnly(deffile);
    if (path)
      path = copystring(path);

    int flags = save ? wxSAVE : wxOPEN;
    s = wxFileSelector((char *)title, (char *)path, file, 0, (char *)ext, flags);
    delete [] file;
    delete [] path;
    // wxFileSelector returns a static temporary
    if (s)
      s = copystring(s);
  }

  // add extension if missing
  if (s && ext) {
    s = (char *)add_default_ext(s, ext + 1);
  }

  // warn if overwriting existent file
  if (s && save)
  {
    if (wxFileExists(s))
    {
      char buf[500];
      sprintf(buf, "overwrite %s?", s);
      if (wxMessageBox(buf, "Save ?", wxYES_NO) == wxNO) {
        delete [] s;
        s = 0;
      }
    }
  }

  if (s && !save && changed)
  {
    char buf[500];
    strcpy(buf, deffile);
    strcat(buf, " has changed. Load anyway?");
    if (wxMessageBox(buf, "Load ?", wxYES_NO) == wxNO) {
      delete [] s;
      s = 0;
    }
  }

  if (s && !save)
  {
    if (!wxFileExists(s))
    {
      char buf[500];
      strcpy(buf, "Cannot find file ");
      strcat(buf, s);
      wxMessageBox( buf, "Error", wxOK );
      delete [] s;
      s = 0;
    }
  }

  if (s)
  {
    delete [] deffile;
    deffile = s;
    return deffile;
  }

  return 0;
}


