/////////////////////////////////////////////////////////////////////////////
//
// SGI audio
//
// Time-stamp: <97/02/27 15:08:32 vels>
// Copyright (c) Vladimir Lobak
//
/////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <errno.h>

#include "machine_audio.h"

////////////////////////////////////////////////////////////////////////////
// 
// Function:     AudioDevice(int samplesPerSec, int channels)
// Arguments:
//    sampleSize    - size of one sample (8 or 16 bit)
//    samplesPerSec - input/output rate
//    channels      - number of channels to use (1-mono, 2-stereo)
//
// Return value: none
// Description:  Constructor
//
////////////////////////////////////////////////////////////////////////////
AudioDevice::AudioDevice(int sampleSize, int samplesPerSec, int channels)
{
  m_port          = (ALport)0;
  m_config        = (ALconfig)0;
  m_sampleSize    = sampleSize;
  m_samplesPerSec = samplesPerSec;
  m_channels      = channels;
  m_isReadable    = FALSE;
  m_isWritable    = FALSE;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     ~AudioDevice()
// Arguments:
// Return value: none
// Description:  Destructor
//
////////////////////////////////////////////////////////////////////////////
AudioDevice::~AudioDevice()
{
  close();
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     open(int mode)
// Arguments:
//    mode       - mode to open with (AUDIO_READ, AUDIO_WRITE or
//                 AUDIO_READWRITE)
//
// Return value: TRUE on success, FALSE on any error
// Description:  Open and initialize audio device
//               Currently ONLY for write !
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::open()
{
  if (m_port)
  {
	fprintf(stderr, "AudioDevice::open: won't initialize twice !\n");
	return FALSE;
  }

  // Create config
  m_config = ALnewconfig();
  if (m_config == (ALconfig)0)
  {
	fprintf(stderr, "AudioDevice::open: ALnewconfig() failed\n");
	return FALSE;
  }
	
  // 16 bits per sample
  ALsetsampfmt(m_config, AL_SAMPFMT_TWOSCOMP);
  //
  // TODO:
  //
  // When queue size is exactly SamplesPerSec m_pBuffer is not played 
  // till the end, because Write is not called at the end of the movie.
  // (Works but dirty :)
  //
  ALsetqueuesize(m_config, samplesPerSec);
  ALsetchannels(m_config,  channels);
  
  m_isWritable = TRUE;

  // Open audio device
  m_port = ALopenport("AudioDevice", "w", m_config);
  if (m_port == (ALport)0)
  {
	switch (oserror())
	{
	case AL_BAD_NO_PORTS:
	  fprintf(stderr, "AudioDevice::open: Out of audio ports\n");
	  break;
	case AL_BAD_DEVICE_ACCESS:
	  fprintf(stderr, "AudioDevice::open: Can't access audio device\n");
	  break;
	case AL_BAD_OUT_OF_MEM:
	  fprintf(stderr, "AudioDevice::open: Out of memory\n");
	  break;
	}
	ALfreeconfig(m_config);
  }

  // Check if output rate is ok for us
  long in_buf[4] = { AL_OUTPUT_RATE, 0, 0, 0 };
  ALgetparams(AL_DEFAULT_DEVICE, in_buf, 2);
  
  // If current output rate is other that we need - check if
  // there is someone else using audio, if so - don't modify anything.
  if (in_buf[1] != samplesPerSec)
  {
	in_buf[0] = AL_OUTPUT_COUNT;
	in_buf[0] = AL_MONITOR_CTL;
	if (ALgetparams(AL_DEFAULT_DEVICE, in_buf, 4) < 0)
	{
	  fprintf(stderr, "AudioDevice::open: ALgetparams failed\n");
	  close();
	  return FALSE;
	}

	if (in_buf[1] == 0) //&& in_buf[3] == AL_MONITOR_OFF)
	{
	  // Set output rate
	  long out_buf[2] = { AL_OUTPUT_RATE, samplesPerSec };
	  ALsetparams(AL_DEFAULT_DEVICE, out_buf, 2);
	}
	else
	{
	  fprintf(stderr, "AudioDevice::open: Audio device is in use\n");
	  close();
	  return FALSE;
	}
  }
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     close()
// Arguments:
// Return value: none
// Description:  Close audio device and free all resources
//
////////////////////////////////////////////////////////////////////////////
void AudioDevice::close()
{
  if (m_config)
  {
    ALfreeconfig(m_config);
    m_config = (ALconfig)0;
  }

  if (m_port)
  {
    ALcloseport(m_port);
    m_port = (ALport)0;
  }
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     sync()
// Arguments:
// Return value: none
// Description:  Complete all data transfers that are in progress
//
////////////////////////////////////////////////////////////////////////////
void AudioDevice::sync()
{
  // IMPLEMENT
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     flush()
// Arguments:
// Return value: none
// Description:  Flush data in audio buffer
//
////////////////////////////////////////////////////////////////////////////
void AudioDevice::flush()
{
  // No idea how to do this on IRIX
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     speakerGain()
// Arguments:
// Return value: Audio volume
// Description:  Get output audio volume
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::speakerGain()
{
  // Get volume (right speaker only !)
  long in_buf[2] =
  { 
	AL_RIGHT_SPEAKER_GAIN, 0,
  };
  ALgetparams(AL_DEFAULT_DEVICE, in_buf, 2);
  return in_buf[1];
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     setSpeakerGain(int value)
// Arguments:
//    value      - new volume
//
// Return value: none
// Description:  Set output audio volume
//
////////////////////////////////////////////////////////////////////////////
void AudioDevice::SetSpeakerGain(int value)
{
  // Set volume (right&left)
  long out_buf[4] =
  { 
	AL_RIGHT_SPEAKER_GAIN, value,
	AL_LEFT_SPEAKER_GAIN, value
  };
  ALsetparams(AL_DEFAULT_DEVICE, out_buf, 4);
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     fillable()
// Arguments:
// Return value: Space in audio buffer
// Description:  Get free space in audio buffer
//               (Maximum number of bytes we can write)
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::fillable()
{
  if (m_port)
    return ALgetfillable(m_port);
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     available()
// Arguments:
// Return value: Space in audio buffer
// Description:  Get number of bytes available for read in audio device
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::available()
{
  // IMPLEMENT
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     write(char *buf, int numSamples)
// Arguments:
//    buf        - buffer
//    numSamples - number of samples in buffer
//
// Return value: ?
// Description:  Write samples to audio device
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::write(char *buf, int numSamples)
{
  if (m_port)
    return ALwritesamps(m_port, buf, numSamples);
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     read(char *buf, int maxSamples)
// Arguments:
//    buf        - buffer
//    maxSamples - maximum number of samples to put in buffer
//
// Return value: ?
// Description:  Read samples from audio device
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::read(char *buf, int maxSamples)
{
}
