/////////////////////////////////////////////////////////////////////////////
//
// Linux audio
//
// Time-stamp: <97/02/27 16:00:27 vels>
// Copyright (c) Vladimir Lobak
//
/////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "machine_audio.h"

////////////////////////////////////////////////////////////////////////////
// 
// Function:     AudioDevice(int samplesPerSec, int channels)
// Arguments:
//    sampleSize    - size of one sample (8 or 16 bit)
//    samplesPerSec - input/output rate
//    channels      - number of channels to use (1-mono, 2-stereo)
//
// Return value: none
// Description:  Constructor
//
////////////////////////////////////////////////////////////////////////////
AudioDevice::AudioDevice(int sampleSize, int samplesPerSec, int channels)
{
  m_audioFd       = -1;
  m_sampleSize    = sampleSize;
  m_samplesPerSec = samplesPerSec;
  m_channels      = channels;
  m_isReadable    = FALSE;
  m_isWritable    = FALSE;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     ~AudioDevice()
// Arguments:
// Return value: none
// Description:  Destructor
//
////////////////////////////////////////////////////////////////////////////
AudioDevice::~AudioDevice()
{
  close();
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     open(int mode)
// Arguments:
//    mode       - mode to open with (AUDIO_READ, AUDIO_WRITE or
//                 AUDIO_READWRITE)
//
// Return value: TRUE on success, FALSE on any error
// Description:  Open and initialize audio device
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::open(int mode)
{
  if (m_audioFd >= 0)
  {
	fprintf(stderr, "AudioDevice::open: won't initialize twice !\n");
	return FALSE;
  }

  // Remember mode
  if (mode & AUDIO_READ || mode & AUDIO_READWRITE)
    m_isReadable = TRUE;
  if (mode & AUDIO_WRITE || mode & AUDIO_READWRITE)
    m_isWritable = TRUE;

  m_s2b         = m_sampleSize/8;
  m_bytesPerSec = m_samplesPerSec*m_s2b;  

  // Throw data to audio device immediately
  mode |= O_SYNC;

  // Open audio device
  if ((m_audioFd = ::open(AUDIODEV, mode)) < 0)
  {
	if ((errno == EINTR) || (errno == EBUSY))
	  fprintf(stderr, "AudioDevice::open: device in use\n");
    else
	  perror("AudioDevice::open:");
	
	return FALSE;
  }

  int arg;

  arg = m_sampleSize;
  if (ioctl(m_audioFd, SOUND_PCM_WRITE_BITS, &arg) < 0)
  {
    perror("AudioDevice::ioctl(bits):");
    return FALSE;
  }
  //printf("bits: %d, ", arg);

  arg = m_channels;
  if (ioctl(m_audioFd, SOUND_PCM_WRITE_CHANNELS, &arg) < 0)
  {
    perror("AudioDevice::ioctl(channels):");
    return FALSE;
  }
  //printf("channels: %d, ", arg);

  arg = m_samplesPerSec;
  if (ioctl(m_audioFd, SOUND_PCM_WRITE_RATE, &arg) < 0)
  {
    perror("AudioDevice::ioctl(rate):");
    return FALSE;
  }
  //printf("rate: %d\n", arg);

  return TRUE;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     close()
// Arguments:
// Return value: none
// Description:  Close audio device
//
////////////////////////////////////////////////////////////////////////////
void AudioDevice::close()
{
  if (m_audioFd < 0)
    return;
  
  ::close(m_audioFd);
  m_audioFd = -1;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     sync()
// Arguments:
// Return value: none
// Description:  Complete all data transfers that are in progress
//
////////////////////////////////////////////////////////////////////////////
void AudioDevice::sync()
{
  if (m_audioFd == -1)
    return;
  
  if (ioctl(m_audioFd, SOUND_PCM_SYNC, 0) < 0)
    perror("AudioDevice::ioctl(sync)");
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     flush()
// Arguments:
// Return value: none
// Description:  Flush data in audio buffer
//
////////////////////////////////////////////////////////////////////////////
void AudioDevice::flush()
{
  if (m_audioFd == -1)
    return;
  
  if (ioctl(m_audioFd, SOUND_PCM_RESET, 0) < 0)
    perror("AudioDevice::ioctl(reset)");
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     speakerGain()
// Arguments:
// Return value: Audio volume
// Description:  Get output audio volume
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::speakerGain()
{
  if (m_audioFd < 0)
	return 0;

  // IMPLEMENT
  perror("AudioDevice::speakerGain:ioctl");
  return 0;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     setSpeakerGain(int value)
// Arguments:
//    value      - new volume
//
// Return value: none
// Description:  Set output audio volume
//
////////////////////////////////////////////////////////////////////////////
void AudioDevice::setSpeakerGain(int value)
{
  if (m_audioFd < 0)
	return;

  // IMPLEMENT
  perror("AudioDevice::setSpeakerGain:ioctl");
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     fillable()
// Arguments:
// Return value: Space in audio buffer
// Description:  Get free space in audio buffer
//               (Maximum number of bytes we can write)
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::fillable()
{
  // Not true, but i've not found any other way...
  return m_bytesPerSec;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     available()
// Arguments:
// Return value: Space in audio buffer
// Description:  Get number of bytes available for read in audio device
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::available()
{
  if (m_audioFd < 0)
	return 0;

  int size;
  if (ioctl(m_audioFd, FIONREAD, &size) < 0)
  {
	perror("AudioDevice::available:ioctl");
    return 0;
  }
  else
  {
    return size;
  }
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     write(char *buf, int numSamples)
// Arguments:
//    buf        - buffer
//    numSamples - number of samples in buffer
//
// Return value: Number of samples actually writen, -1 on error
// Description:  Write samples to audio device
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::write(char *buf, int numSamples)
{
  int ret;
  int size = numSamples*m_s2b;
  if ((ret = ::write(m_audioFd, buf, size)) != size)
  {
    if (ret < 0)
      perror("AudioDevice::write");
    else
      fprintf(stderr, "AudioDevice::write: wrote only %d bytes", ret);
  }

  return ret/m_s2b;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     read(char *buf, int maxSamples)
// Arguments:
//    buf        - buffer
//    maxSamples - maximum number of samples to put in buffer
//
// Return value: Number of samples actually read, -1 on error
// Description:  Read samples from audio device
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::read(char *buf, int maxSamples)
{
  int ret;
  int size = maxSamples*m_s2b;
  if ((ret = ::read(m_audioFd, buf, size)) != size)
  {
    if (ret < 0)
      perror("AudioDevice::read");
    else
      fprintf(stderr, "AudioDevice::read: read only %d bytes", ret);
  }
  
  return ret/m_s2b;
}
