/////////////////////////////////////////////////////////////////////////////
//
// Sun audio
//
// Time-stamp: <97/02/27 15:08:16 vels>
// Copyright (c) Vladimir Lobak
//
/////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>

#include "machine_audio.h"

////////////////////////////////////////////////////////////////////////////
// 
// Function:     AudioDevice(int samplesPerSec, int channels)
// Arguments:
//    sampleSize    - size of one sample (8 or 16 bit)
//    samplesPerSec - input/output rate
//    channels      - number of channels to use (1-mono, 2-stereo)
//
// Return value: none
// Description:  Constructor
//
////////////////////////////////////////////////////////////////////////////
AudioDevice::AudioDevice(int sampleSize, int samplesPerSec, int channels)
{
  m_audioFd       = -1;
  m_sampleSize    = sampleSize;
  m_samplesPerSec = samplesPerSec;
  m_channels      = channels;
  m_isReadable    = FALSE;
  m_isWritable    = FALSE;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     ~AudioDevice()
// Arguments:
// Return value: none
// Description:  Destructor
//
////////////////////////////////////////////////////////////////////////////
AudioDevice::~AudioDevice()
{
  close();
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     open(int mode)
// Arguments:
//    mode       - mode to open with (AUDIO_READ, AUDIO_WRITE or
//                 AUDIO_READWRITE)
//
// Return value: TRUE on success, FALSE on any error
// Description:  Open and initialize audio device
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::open(int mode)
{
  char audiodev[MAX_AUDIO_DEV_LEN+1], *t;
  char error_str[MAX_AUDIO_DEV_LEN+64];

  if (m_audioFd >= 0)
  {
	fprintf(stderr, "AudioDevice::open: won't initialize twice !\n");
	return FALSE;
  }

  // Obtain device name
  if ((t=getenv("AUDIODEV")) == NULL)
	strcpy(audiodev, DEFAULT_AUDIODEV);
  else
	strcpy(audiodev, t);

  // Remember mode
  if (mode & AUDIO_READ || mode & AUDIO_READWRITE)
    m_isWritable = TRUE;
  if (mode & AUDIO_WRITE || mode & AUDIO_READWRITE)
    m_isReadable = TRUE;

  // No delay on syscalls
  mode |= O_NDELAY;

  // Open audio device
  if ((m_audioFd = ::open(audiodev, mode)) < 0)
  {
	if ((errno == EINTR) || (errno == EBUSY))
    {
	  fprintf(stderr, "AudioDevice::open: device in use\n");
    }
	else
	{
	  strcpy(error_str, "AudioDevice::open: ");
	  strcat(error_str, audiodev);
	  perror(error_str);
	}
	
	return FALSE;
  }

  m_s2b         = m_sampleSize/8;
  m_bytesPerSec = m_samplesPerSec*m_s2b;  

  // Set stream info
  audio_info_t audioinfo;
  AUDIO_INITINFO(&audioinfo);
  audioinfo.play.precision   = m_sampleSize;
  audioinfo.play.sample_rate = m_samplesPerSec;
  audioinfo.play.channels    = m_channels;
  audioinfo.play.buffer_size = m_bytesPerSec;
  audioinfo.play.encoding    = AUDIO_ENCODING_LINEAR;
  if (ioctl(m_audioFd, AUDIO_SETINFO, &audioinfo) < 0)
  {	
	strcpy(error_str, "AudioDevice::open:ioctl");
	strcat(error_str, audiodev);
	perror(error_str);

	close();
	return FALSE;
  }
  
  return TRUE;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     close()
// Arguments:
// Return value: none
// Description:  Close audio device
//
////////////////////////////////////////////////////////////////////////////
void AudioDevice::close()
{
  if (m_audioFd < 0)
    return;
  
  flush();
  ::close(m_audioFd);
  m_audioFd = -1;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     sync()
// Arguments:
// Return value: none
// Description:  Complete all data transfers that are in progress
//
////////////////////////////////////////////////////////////////////////////
void AudioDevice::sync()
{
  // IMPLEMENT
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     flush()
// Arguments:
// Return value: none
// Description:  Flush data in audio buffer
//
////////////////////////////////////////////////////////////////////////////
void AudioDevice::flush()
{
  if (m_audioFd < 0)
    return;
  
  if (m_isWritable)
  {
    if (ioctl(m_audioFd, I_FLUSH, FLUSHW) < 0)
      perror("AudioDevice::flushw:ioctl");
  }

  if (m_isReadable)
  {
    if (ioctl(m_audioFd, I_FLUSH, FLUSHR) < 0)
      perror("AudioDevice::flushr:ioctl");
  }
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     speakerGain()
// Arguments:
// Return value: Audio volume
// Description:  Get output audio volume
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::speakerGain()
{
  if (m_audioFd < 0)
	return 0;

  audio_info_t audioinfo;
  if (ioctl(m_audioFd, AUDIO_GETINFO, &audioinfo) < 0)
	perror("AudioDevice::speakerGain:ioctl");

  return audioinfo.play.gain;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     setSpeakerGain(int value)
// Arguments:
//    value      - new volume
//
// Return value: none
// Description:  Set output audio volume
//
////////////////////////////////////////////////////////////////////////////
void AudioDevice::setSpeakerGain(int value)
{
  if (m_audioFd < 0)
	return;

  audio_info_t audioinfo;
  AUDIO_INITINFO(&audioinfo);
  audioinfo.play.gain = value;

  if (ioctl(m_audioFd, AUDIO_SETINFO, &audioinfo) < 0)
	perror("AudioDevice::setSpeakerGain:ioctl");
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     fillable()
// Arguments:
// Return value: Space in audio buffer
// Description:  Get free space in audio buffer
//               (Maximum number of bytes we can write)
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::fillable()
{
  // Not true, but i've not found any other way...
  return m_bytesPerSec;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     available()
// Arguments:
// Return value: Space in audio buffer
// Description:  Get number of bytes available for read in audio device
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::available()
{
  if (m_audioFd < 0)
	return 0;

  int size;
  if (ioctl(m_audioFd, FIONREAD, &size) < 0)
  {
	perror("AudioDevice::available:ioctl");
    return 0;
  }
  else
  {
    return size;
  }
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     write(char *buf, int numSamples)
// Arguments:
//    buf        - buffer
//    numSamples - number of samples in buffer
//
// Return value: Number of *bytes* actually writen, -1 on error
// Description:  Write samples to audio device
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::write(char *buf, int numSamples)
{
  int ret;
  int size = numSamples*m_s2b;
  if ((ret = ::write(m_audioFd, buf, size)) != size)
  {
    if (ret < 0)
      perror("AudioDevice::write");
    else
      fprintf(stderr, "AudioDevice::write: wrote only %d bytes", ret);
  }

  return ret;
}

////////////////////////////////////////////////////////////////////////////
// 
// Function:     read(char *buf, int maxSamples)
// Arguments:
//    buf        - buffer
//    maxSamples - maximum number of samples to put in buffer
//
// Return value: Number of *bytes* actually read, -1 on error
// Description:  Read samples from audio device
//
////////////////////////////////////////////////////////////////////////////
int AudioDevice::read(char *buf, int maxSamples)
{
  int ret;
  if ((ret = ::read(m_audioFd, buf, maxSamples*m_s2b)) < 0)
	perror("AudioDevice::read");
  
  return ret;
}
