#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>
#include <malloc.h>
#include <qapp.h>
#include <qmsgbox.h>
#include <qtimer.h>
#include <qfile.h>

#include "qMainDlg.h"
#include "MainDlgData.h"
#include "qSetupDlg.h"
#include "qRecDlg.h"
#include "qPlayDlg.h"
#include "../server/server.h"
#include "../server/util.h"


#define STATUS(m)  m_statusLbl->setText(m); qApp->flushX(); //debug(m);
#define SEND(str)  if (SendMsg(m_sock, str, strlen(str)+1) < 0) \
                   { \
                     fprintf(stderr, "Exiting...\n"); \
                     close(); \
                   }
                   

MainDlg::MainDlg(QWidget* parent, const char* name)	:
  QDialog(parent, name, TRUE),
  MainDlgData(this)
{
  setCaption("AU Mail");

  connect(m_quitBtn, SIGNAL(clicked()), this, SLOT(reject()));
  connect(m_setupBtn, SIGNAL(clicked()), this, SLOT(OnSetup()));
  connect(m_retreiveBtn, SIGNAL(clicked()), this, SLOT(OnRetreive()));
  connect(m_sendBtn, SIGNAL(clicked()), this, SLOT(OnSend()));
  connect(m_msgList, SIGNAL(selected(int)), this, SLOT(OnItemSelect(int)));
  
  // Set initial values
  m_serverName = "localhost";
  m_userName   = getpwuid(getuid())->pw_name;
  m_port       = LISTEN_PORT;
  m_sock       = -1;

  // Welcome user
  STATUS("Welcome to audio mail v"VERSION);
}

void MainDlg::OnItemSelect(int index)
{
  // Retreive audio mail from server
  char str[MAX_STRLEN];
  sprintf(str, "read %d", index);
  SEND(str);
    
  STATUS("Requesting audio mail...");
  m_waitFor = WF_MAILDATA;
}

// -- HACK --
void MainDlg::OnServerErrorMsg()
{
  QMessageBox::message("Error", m_lastServerErrorMsg);
  free(m_lastServerErrorMsg);
}

void MainDlg::OnMailData()
{
  PlayDlg dlg(m_receivedMail, m_receivedMailSize, this, "playdlg");
  dlg.setCaption("Play message");
    
  // Accepted means - delete message from server
  if (dlg.exec() == Accepted)
  {
    char str[MAX_STRLEN];
    sprintf(str, "del %d", m_msgList->currentItem());
    SEND(str);
    m_waitFor = WF_DELOK;
  }
}
// -- HACK --


void MainDlg::OnSetup()
{
  SetupDlg dlg(this, "setupdlg");
  dlg.setCaption("Change settings");
  dlg.setServerName(m_serverName);
  dlg.setUserName(m_userName);
  dlg.setPort(m_port);
  
  if (dlg.exec() == Accepted)
  {
    m_serverName = dlg.serverName();
    m_userName   = dlg.userName();
    m_port       = dlg.port();
  }
}

void MainDlg::OnSend()
{
  RecDlg dlg(this, "recdlg");
  dlg.setCaption("Record/Send message");
  
  if (dlg.exec() == Accepted)
  {
    m_state = SEND;

    // Create sendMsg from rcpt and subject
    // [used in initiateSend()]
    m_sendMsg = QString("send ") +
      QString(dlg.rcpt()) + QString(" ") + QString(dlg.subject());

    // Remember temporary audio filename
    strcpy(m_tempFile, dlg.tempFile());

    // Connect to server if not connected yet
    // (We may have been connected by OnRetreive)
    if (m_sock < 0)
    {
      if (!ConnectToServer())
      {
        // Erase temporary audio file
        if (unlink(m_tempFile) < 0)
          QMessageBox::message("Error", "Unable to remove temporary file.");
        return;
      }
      m_connectedBy = SEND;
    }
    else // We can initiate send now
    {
      InitiateSend();
    }
  }
  else
  {
    // Erase temporary audio file
    if (!dlg.tempFile() || unlink(dlg.tempFile()) < 0)
      QMessageBox::message("Error", "Unable to remove temporary file.");
  }
}

void MainDlg::OnRetreive()
{
  // If already connected - should disconnect
  if (m_sock >= 0)
  {
    DisconnectFromServer();
    m_retreiveBtn->setText("Retreive mail");
    STATUS("Disconnected.");
  }
  else // Connect and retreive mail headers
  {
    if (!ConnectToServer())
      return;

    m_state       = RETREIVE;
    m_connectedBy = RETREIVE;

    // Change button label
    m_retreiveBtn->setText("Disconnect");
  }
}

int MainDlg::ConnectToServer()
{
  sockaddr_in addr;
  int addrLen;

  STATUS("Resolving...");

  // Resolve server hostname
  if (!FillAddr((char *)((const char *)m_serverName), m_port, &addr))
  {
    QMessageBox::message("Error", "Unable to resolve server hostname.\n"
                         "Please check the name You typed.");
    STATUS("Could not resolve server hostname.");
    return FALSE;
  }

  // Create tcp socket
  if ((m_sock = socket(PF_INET, SOCK_STREAM, 0)) < 0)
  {
    QMessageBox::message("Error", "Unable to create TCP socket.");
    STATUS("Could not create socket.");
    return FALSE;
  }

  STATUS("Connecting...");

  // Connect
  if (::connect(m_sock, (sockaddr *)&addr, sizeof(sockaddr_in)) < 0)
  {
    QMessageBox::message("Error", "Unable to connect to server.\n"
                         "Probably host is down or no server is running.");
    m_sock = -1;
    STATUS("Could not establish connection.");
    return FALSE;
  }

  // Wait for welcome message
  m_waitFor = WF_WELCOME;
  STATUS("Waiting for welcome message...");

  m_sockNotifier = new QSocketNotifier(m_sock, QSocketNotifier::Read, this);
  connect(m_sockNotifier, SIGNAL(activated(int)), 
          this, SLOT(OnSockRead(int)));

  return TRUE;
}

void MainDlg::DisconnectFromServer()
{
  delete m_sockNotifier;
  ::close(m_sock);
  m_sock = -1;

  // Remove all items from list
  m_msgList->clear();
}

void MainDlg::InitiateSend()
{
  STATUS("Initiating send...");

  // Send 'send' command with recepient name
  SEND((char *)((const char *)m_sendMsg));
  m_waitFor = WF_RCPTOK;
}

void MainDlg::OnSockRead(int fd)
{
  int  size;
  char *buf;

  // Read message from socket
  if ((buf=RecvMsg(m_sock, &size)) == NULL)
  {
    // Will cause disconnect
    OnRetreive();

    // Show message box
    m_lastServerErrorMsg = (char *)malloc(MAX_STRLEN);
    sprintf(m_lastServerErrorMsg, "Server has disconnected !\n"
            "This usually indicates that something is wrong\n"
            "with the server or client. To be more specific -\n"
            "     You have found a bug ! :)");
    QTimer::singleShot(10, this, SLOT(OnServerErrorMsg()));
    return;
  }

  // If response is NACK - show it to user
  if (*buf == '-')
  {
    // If NACK was received during send - disconnect
    if (m_state == SEND)
      DisconnectFromServer();

    *(buf+strlen(buf)-1) = '\0';
    //QMessageBox::message("Error", buf); -- BUG  ??

    m_lastServerErrorMsg = (char *)memmove(buf, buf+1, strlen(buf));
    QTimer::singleShot(10, this, SLOT(OnServerErrorMsg()));
    //STATUS(buf+1);
    return;
  }

  // Parse message
  switch(m_waitFor)
  {
  case WF_WELCOME:
    {
      // Got welcome message - send user name
      char user[256];
      sprintf(user, "user %s", (char *)((const char *)m_userName));
      SEND(user);
      m_waitFor = WF_USEROK;
      STATUS("Sending user name...");
    }
  break;

  case WF_USEROK:
    // Got userok message
    if (m_state == RETREIVE)
    {
      // Send 'list mails' message
      SEND("list");
      m_waitFor = WF_LIST;
      STATUS("Waiting for mail list...");
    }
    else // m_state == SEND
    {
      InitiateSend();
    }
    break;

  case WF_RCPTOK:
    {
      // Got recipient OK - we're allowed to send mail data
      
      // Send recordered mail
      STATUS("Sending mail...");

      QFile *qf = new QFile(m_tempFile);
      if (!qf->open(IO_Raw | IO_ReadOnly))
      {
        char str[MAX_STRLEN];
        sprintf(str, "Temporary audio file %s lost.", m_tempFile);
        STATUS(str);
      }
      else
      {
        // Send the whole file as one message.
        // TCP is smart enough to handle it :)
        char *fileBuf = (char *)malloc(qf->size());
        qf->readBlock(fileBuf, qf->size());

        if (SendMsg(m_sock, fileBuf, qf->size()) < 0)
        {
          fprintf(stderr, "Low-level error. Exiting...\n");
          close();
        }
        free(fileBuf);
        unlink(m_tempFile);

        m_waitFor = WF_SENDOK;
      }

      break;
    }
  
  case WF_SENDOK:
    // Mail was sent successfully
    STATUS("Mail sent ok.");
    if (m_connectedBy == SEND)
      DisconnectFromServer();
    break;

  case WF_LIST:
    // Got list message - show to user

    // Skip first line
    strtok(buf, "\n");

    // Insert each line into list
    char *p;
    m_msgList->clear();
    while ((p=strtok(NULL, "\n")) != NULL)
    {
      m_msgList->insertItem(p);
    }

    STATUS("Double click on message to listen.");
    break;

  case WF_MAILDATA:
    {
      // Audio data received - play it
      m_receivedIndx = m_msgList->currentItem();
      m_receivedMail = buf;
      m_receivedMailSize = size;
      QTimer::singleShot(100, this, SLOT(OnMailData()));
      return;
    }

  case WF_DELOK:
    // Server deleted specified audio mail - remove it from our list
    m_msgList->removeItem(m_msgList->currentItem());
    break;

  case WF_NONE:
    // Unexpected message 
    QMessageBox::message("Error", 
                         "Received unexpected message from server.");
    break;

  default:
    fprintf(stderr, "WTF am i doing here ?!\n");
    break;
  }

  free(buf);
}
