#include <stdio.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <qmsgbox.h>

#include "qRecDlg.h"
#include "RecDlgData.h"

#define SIZE 160

RecDlg::RecDlg(QWidget* parent,	const char* name) :
  QDialog(parent, name, TRUE),
  RecDlgData(this)
{
  // Initialize state
  m_isPlaying    = FALSE;
  m_isRecording  = FALSE;
  m_isRecordered = FALSE;

  m_gsmHandle    = NULL;
  m_sockNotifier = NULL;
  m_timer        = NULL;
  m_audio        = NULL;
  m_tempFile     = -1;

  m_playBtn->setEnabled(FALSE);

  // GUI logic
  connect(m_recordBtn, SIGNAL(clicked()), this, SLOT(OnRecord()));
  connect(m_playBtn, SIGNAL(clicked()), this, SLOT(OnPlay()));
  connect(m_sendBtn, SIGNAL(clicked()), this, SLOT(OnSend()));
  connect(m_cancelBtn, SIGNAL(clicked()), this, SLOT(reject()));

  // Create name for temporary audio file
  QString fnameStr;
  fnameStr.setNum(getpid());
  m_tempFileName = QString("/tmp/aum")+fnameStr+QString(".gsm");
}

RecDlg::~RecDlg()
{
  cleanup();
}

void RecDlg::cleanup()
{
#ifdef USE_GSM
  // Close GSM handle
  if (m_gsmHandle)
  {
    gsm_destroy(m_gsmHandle);
    m_gsmHandle = NULL;
  }
#endif

  // Stop reading from audio device
  if (m_sockNotifier)
  {
    delete m_sockNotifier;
    m_sockNotifier = NULL;
  }

  // Stop reading from temporary file
  if (m_timer)
  {
    m_timer->stop();
    delete m_timer;
    m_timer = NULL;
  }

  if (m_audio)
  {
    delete m_audio;
    m_audio = NULL;
  }

  if (m_tempFile > 0)
  {
    ::close(m_tempFile);
    m_tempFile = -1;
  }
}


void RecDlg::OnPlay()
{
  // Stop playing
  if (m_isPlaying)
  {
    cleanup();
    
    // Enable buttons
    m_playBtn->setText("Listen");
    m_recordBtn->setEnabled(TRUE);
    m_sendBtn->setEnabled(TRUE);
    m_cancelBtn->setEnabled(TRUE);

    m_isPlaying = FALSE;
  }
  else // Start playing
  {
    // Open audio device and create GSM handle
    m_audio = new AudioDevice(16, 8000, 1);
    if (m_audio->open(AudioDevice::AUDIO_WRITE)
#ifdef USE_GSM
        && (m_gsmHandle = gsm_create()) != NULL
#endif
        )
    {
      // Disable buttons
      m_playBtn->setText("Stop");
      m_recordBtn->setEnabled(FALSE);
      m_sendBtn->setEnabled(FALSE);
      m_cancelBtn->setEnabled(FALSE);
      
      // Initialize counters
      m_isPlaying = TRUE;
      m_time.start();

      // Open temporary audio file
      if ((m_tempFile=open(m_tempFileName, O_RDONLY)) < 0)
      {
        QMessageBox::message("Error", "Unable to open temporary file.");
      }
      else
      {
        // Start reading from temporary file
        m_timer = new QTimer(this);
        connect(m_timer, SIGNAL(timeout()), this, SLOT(OnTempFileRead()));
        m_timer->start(500 / (m_audio->bytesPerSec()/SIZE) );
      }
    }
    else
    {
      QMessageBox::message("Error", "Unable to open audio device.\n"
                           "Or initialize GSM compression.");
    }
  }
}


void RecDlg::OnRecord()
{
  // Stop recording
  if (m_isRecording)
  {
    cleanup();
    
    // Enable buttons
    m_recordBtn->setText("Record");
    m_sendBtn->setEnabled(TRUE);
    m_cancelBtn->setEnabled(TRUE);
    m_playBtn->setEnabled(TRUE);

    m_isRecording  = FALSE;
    m_isRecordered = TRUE;
  }
  else // Start recording
  {
    // Open audio device and create GSM handle
    m_audio = new AudioDevice(16, 8000, 1);
    if (m_audio->open(AudioDevice::AUDIO_READ)
#ifdef USE_GSM
        && (m_gsmHandle = gsm_create()) != NULL
#endif
        )
    {
      // Disable buttons
      m_recordBtn->setText("Stop");
      m_sendBtn->setEnabled(FALSE);
      m_cancelBtn->setEnabled(FALSE);
      m_playBtn->setEnabled(FALSE);
    
      // Initialize counters
      m_isRecording     = TRUE;
      m_bytesRecordered = 0;
      m_time.start();

      // Create and open temporary audio file
      if ((m_tempFile=creat(m_tempFileName, S_IWUSR | S_IRUSR)) < 0)
      {
        QMessageBox::message("Error", "Unable to create temporary file.");
      }
      else
      {
        // Start reading from audio device
        // We should initiate first read so that socket notifier will work.
        OnAudioRead(0);
        m_sockNotifier = new QSocketNotifier(m_audio->audioFd(),
                                             QSocketNotifier::Read, this);
        connect(m_sockNotifier, SIGNAL(activated(int)),
                this, SLOT(OnAudioRead(int)));
      }
    }
    else
    {
      QMessageBox::message("Error", "Unable to open audio device.\n"
                           "Or initialize GSM compression.");
    }
  }
}


void RecDlg::OnSend()
{
  if (*m_recipientLine->text())
  {
    if (!m_isRecordered)
    {
      QMessageBox::message("Error", "No message was recordered !");
    }
    else
    {
      if (!*m_subjLine->text())
      {
        if (!QMessageBox::query("Warning", "Set subject to \"No subject\" ?"))
          return;

        m_subjLine->setText("No subject");
      }

      // Send recordered message 
      accept();
    }
  }
  else
  {
    QMessageBox::message("Error", "You must specify recipient name !");
  }
}



void RecDlg::OnAudioRead(int fd)
{
#ifdef USE_GSM
  gsm_signal sample[SIZE];
  gsm_frame  buf;
#else
  char       sample[SIZE];
  char       buf[SIZE];
#endif

  // Read data from audio device
  if (m_audio->read((char *)sample, SIZE) != SIZE)
  {
    cleanup();
    QMessageBox::message("Error", "Can't read from audio device !");
    reject();
    return;
  }

#ifdef USE_GSM
  // Compress with GSM
  gsm_encode(m_gsmHandle, sample, buf);
#else
  memcpy(buf, sample, SIZE);
#endif

  // And write it into temporary file
  if (write(m_tempFile, (char *)buf, sizeof(buf)) != sizeof(buf))
  {
    cleanup();
    QMessageBox::message("Error", "Can't to write to temporary file.");
    reject();
    return;
  }
  
  // Number of seconds elapsed
  QString secStr;
  secStr.setNum(m_time.elapsed()/1000);

  // Number of Kbytes read
  QString byteStr;
  m_bytesRecordered += sizeof(buf);
  byteStr.setNum(m_bytesRecordered/1024);

  // Change label
  m_stateLbl->setText(secStr + " sec / " + byteStr + " Kb");
}

void RecDlg::OnTempFileRead()
{
#ifdef USE_GSM
  gsm_signal sample[SIZE];
  gsm_frame  buf;
#else
  char       sample[SIZE];
  char       buf[SIZE];
#endif

  // Read data from file
  int ret;
  if ((ret=read(m_tempFile, (char *)buf, sizeof(buf))) != sizeof(buf))
  {
    // EOF ?
    if (ret == 0)
    {
      cleanup();

      // Enable buttons
      m_playBtn->setText("Listen");
      m_recordBtn->setEnabled(TRUE);
      m_sendBtn->setEnabled(TRUE);
      m_cancelBtn->setEnabled(TRUE);

      m_isPlaying = FALSE;

      return;
    }

    QMessageBox::message("Error", "Can't read from temporary file.");
    cleanup();
    reject();
    return;
  }

#ifdef USE_GSM
  // Decompress with GSM
  if (gsm_decode(m_gsmHandle, buf, sample) < 0)
  {
    cleanup();
    QMessageBox::message("Error", "Error decompressing audio !");
    reject();
    return;
  }
#else
  memcpy(sample, buf, SIZE);
#endif

  // And write it to audio device
  if (m_audio->write((char *)sample, SIZE) != SIZE)
  {
    cleanup();
    QMessageBox::message("Error", "Can't to write to audio device.");
    reject();
    return;
  }
  
  // Number of seconds elapsed
  QString secStr;
  secStr.setNum(m_time.elapsed()/1000);

  // Change label
  m_stateLbl->setText(secStr + " seconds");
}
