#include <stdlib.h>
#include <signal.h>
#include <sys/wait.h>

#include "defs.h"
#include "server.h"
#include "util.h"

int OnNewClient(int sock, int numClients);

int listenSocket;
extern int g_sock; // Client socket

////////////////////////////////////////////////////////////////////////////
// Signals that cause server stop
void StopSignal(int sig)
{
  fprintf(stderr, "\nCaught signal (%d) - exiting...\n", sig);

  close(listenSocket);
  exit(1);
}

///////////////////////////////////////////////////////////////////////////
// Main entrance
int main(int argc, char *argv[])
{
  int clientSocket, numClients=0;
  sockaddr_in addr;
  int addrLen;
  fd_set readBitmask;

  // Catch signals
  signal(SIGINT,  StopSignal);
  signal(SIGTERM, StopSignal);
  signal(SIGSTOP, StopSignal);

  // Create listen socket
  if ((listenSocket = socket(PF_INET, SOCK_STREAM, 0)) < 0)
  {
    perror("socket");
    return -1;
  }

  // Bind it
  memset(&addr, 0, sizeof(addr));
  addr.sin_family      = AF_INET;
  addr.sin_addr.s_addr = INADDR_ANY;
  addr.sin_port        = (argc < 2)? htons(LISTEN_PORT) : htons(atoi(argv[1]));

  if (bind(listenSocket, (sockaddr *)&addr, sizeof(sockaddr_in)) < 0)
  {
    perror("bind");
    fprintf(stderr, "Another instance may be running.\n");
    return -1;
  }

  // Enable listening
  if (listen(listenSocket,5) < 0)
  {
    perror("listen");
    return -1;
  }

  while (1)
  {
    // Wait for connection
    FD_ZERO(&readBitmask);
    FD_SET(listenSocket, &readBitmask);
    if (select(listenSocket+1, &readBitmask, NULL, NULL, NULL) < 0)
    {
      perror("select(listen)");
      return -1;
    }

    // Accept new connection
    clientSocket = accept(listenSocket, (sockaddr *)&addr, &addrLen);
    if (clientSocket < 0)
    {
      perror("accept");
      return -1;
    }
    else // Connection accepted
    {
      int pid;
      
      numClients++;
      DEBUG_CODE(printf("Client %d connected.\n", numClients);)

      // Fork child process
      if ((pid=fork()) == 0)
      {
        // Fork one more time - the next child will start talking
        // with client and this one (parent) will exit.
        // This way second child will be reparented by the init
        // process (pid = 1).
        if ((pid=fork()) == 0)
        {
          int ret = OnNewClient(clientSocket, numClients);
          DEBUG_CODE(printf("Client %d disconnected.\n", numClients);)
          exit(ret);
        }
        else if (pid < 0)
        {
          perror("fork");
          return -1;
        }
        else // Parent
        {
          exit(0);
        }
      }
      else if (pid < 0)
      {
        perror("fork");
        return -1;
      }
      close(clientSocket);

      // Wait for first child (it will exit soon)
      int status;
      waitpid(pid, &status, 0);
    }
  } // while(1)
}
