#include <time.h>
#include <signal.h>
#include <dirent.h>

#include "defs.h"
#include "server.h"
#include "messages.h"
#include "util.h"

// User mails
static Mail s_mail[MAX_MAILS];
static int s_mailNum;

// User name
static char s_userName[MAX_USERNAME+1];

// Recipient name (specified by user)
static char s_rcptName[MAX_USERNAME+1];
static char s_subjName[MAX_STRLEN+1];
static bool s_waitForMailData;

// Our socket
static int s_sock;

// List of known actions
static Action AL[] =
{
  { "help",     OnHelp     },
  { "user",     OnUserName },
  { "list",     OnList     },
  { "read",     OnReadMail },
  { "send",     OnSendMail },
  { "del",      OnDelMail  },
  { "quit",     OnQuit     }
};
#define NUM_ACTIONS (sizeof(AL)/sizeof(Action))

// Return codes for actions
#define ACT_FATAL  -1
#define ACT_ERROR  FALSE
#define ACT_OK     TRUE

#define SEND(msg) if (SendMsg(s_sock, msg, strlen(msg)+1) < 0) return ACT_FATAL; DEBUG_CODE(printf("S: %s\n", msg))



//
//                         Helper functions
//


/////////////////////////////////////////////////////////////////////////////
// Add file to list of user mails that will be sent to client
#define READLINE(s) if (fgets(s, MAX_STRLEN, f) < 0) \
                    { \
                        perror("fgets"); \
                        fclose(f); \
                        return FALSE; \
                    } \
                    else \
                    { \
                      s[strlen(s)-1] = 0; \
                    }
int AddToMailList(char *fname)
{
  // Open file
  FILE *f;
  if ((f=fopen(fname, "r")) == NULL)
  {
    perror("fopen");

    // Notify user
    char s[MAX_STRLEN];
    sprintf(s, ERR_MAIL_OPEN_MSG, fname);
    SEND(s);

    return FALSE;
  }

  // Read mail info from file

  // We need filename in case we'll have to send this mail to user
  getcwd(s_mail[s_mailNum].fname, PATH_MAX);
  strcat(s_mail[s_mailNum].fname, "/");
  strcat(s_mail[s_mailNum].fname, fname);

  // Position of mail data
  long dataPos;
  if (fread(&dataPos, sizeof(long), 1, f) < 0)
  {
    perror("fread");
    fclose(f);
    return FALSE;
  }
  s_mail[s_mailNum].dataPos = ntohl(dataPos);

  // Sender address, subject and entered date
  READLINE(s_mail[s_mailNum].from);
  READLINE(s_mail[s_mailNum].subject);
  READLINE(s_mail[s_mailNum].date);

  s_mailNum++;

  fclose(f);
  return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Create mail list
int CreateMailList()
{
  int fd;
  char dir[PATH_MAX];

  // Get mailbox location
  if (!GetMailBoxDir(s_userName, dir))
  {
    char s[MAX_STRLEN+PATH_MAX];
    sprintf(s, NO_MAILBOX_MSG, dir);
    SEND(s);
    return FALSE;
  }
    
  // Open mailbox
  DIR *dirStream;
  if ((dirStream=opendir(dir)) == NULL)
  {
    char s[MAX_STRLEN+PATH_MAX];
    sprintf(s, NO_MB_ACCESS_MSG, dir);
    SEND(s);
    return FALSE;
  }
    
  // Remember current dir and change to mailbox dir
  char prevDir[PATH_MAX+1];
  getcwd(prevDir, PATH_MAX);
  chdir(dir);

  // Look for entries in this mailbox (files in directory)
  int ret = TRUE;
  struct dirent *entry;
  while ((entry=readdir(dirStream)) != NULL)
  {
    // Skip all dot files
    if (*(entry->d_name) == '.')
      continue;

    if (!AddToMailList(entry->d_name))
    {
      ret = FALSE;
      break;
    }
  }

  closedir(dirStream);
  chdir(prevDir);
  return ret;
}


//
//                               Actions
//


/////////////////////////////////////////////////////////////////////////////
// Show help
int OnHelp()
{
  SEND(HELP_MSG);
  return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Remember user name
int OnUserName()
{
  char *user = strtok(NULL, DELIMITERS);
  if (!user)
  {
    SEND(NO_USERNAME_MSG);
    return FALSE;
  }
  else
  {
    char dir[PATH_MAX];

    // Get mailbox location
    if (!GetMailBoxDir(user, dir))
    {
      char s[MAX_STRLEN+PATH_MAX];
      sprintf(s, NO_MAILBOX_MSG, dir);
      SEND(s);
      return FALSE;
    }
    else
    {
      strcpy(s_userName, user);
      SEND(SMALLHELP_MSG);
      return TRUE;
    }
  }
}

/////////////////////////////////////////////////////////////////////////////
// Show user awaiting mail messages
int OnList()
{
  if (!*s_userName)
  {
    SEND(NO_USERNAME_MSG);
    return FALSE;
  }
  else
  {
    char s[MAX_STRLEN];

    // Create mail list if doesn't exist yet
    if (s_mailNum == 0)
    {
      int ret = CreateMailList();
      if (ret == FALSE || ret < 0)
        return ret;
    }

    // No mail
    if (s_mailNum == 0)
    {
      SEND(NO_MAIL_MSG);
      return TRUE;
    }

    // Show it to user
    char *msg = new char[MAX_MSGSIZE];
    sprintf(msg, "+You have %d messages:\n", s_mailNum);
    for (int i=0; i<s_mailNum; i++)
    {
      // Pad each item with spaces
      sprintf(s, "%-35s%-15s %s\n",
              s_mail[i].from, 
              s_mail[i].date, 
              s_mail[i].subject);
      strcat(msg, s);
    }
    SEND(msg);
    delete msg;

    return TRUE;
  }
}

/////////////////////////////////////////////////////////////////////////////
// User wants to read mail
int OnReadMail()
{
  if (!*s_userName)
  {
    SEND(NO_USERNAME_MSG);
    return FALSE;
  }
  else
  {
    char *p = strtok(NULL, DELIMITERS);
    if (!p)
    {
      SEND(NO_INDX_MSG);
      return FALSE;
    }
    else
    {
      // Remember index
      int index = atoi(p);
      
      // Create mail list if doesn't exist yet
      if (s_mailNum == 0)
      {
        int ret = CreateMailList();
        if (ret == FALSE || ret < 0)
          return ret;
      }

      // No mail
      if (s_mailNum == 0)
      {
        SEND(NO_MAIL_MSG);
        return TRUE;
      }

      // Check validity of index
      if (index < 0 || index >= s_mailNum)
      {
        SEND(BAD_INDX_MSG);
        return FALSE;
      }

      // Open audio mail file
      FILE *f;
      if ((f=fopen(s_mail[index].fname, "r")) == NULL)
      {
        perror("fopen");

        // Notify user
        char s[MAX_STRLEN];
        sprintf(s, ERR_MAIL_OPEN_MSG, s_mail[index].fname);
        SEND(s);

        return FALSE;
      }

      // Calc. length of data (from the end of file)
      fseek(f, 0, SEEK_END);
      long dataLen = ftell(f) - s_mail[index].dataPos;

      // Go to start of mail data
      fseek(f, s_mail[index].dataPos, SEEK_SET);

      // Read till the end of file
      char *buf = (char *)malloc(dataLen);
      int r;
      if ((r=fread(buf, dataLen, 1, f)) < 0)
      {
        perror("fread");

        // Notify user
        char s[MAX_STRLEN];
        sprintf(s, ERR_MAIL_READ_MSG, s_mail[index].fname);
        SEND(s);

        return FALSE;
      }

      // Send it to client
      if (SendMsg(s_sock, buf, dataLen) < 0)
        return ACT_FATAL;

      free(buf);
      return TRUE;
    }
  }
}

/////////////////////////////////////////////////////////////////////////////
// User wants to delete mail
int OnDelMail()
{
  if (!*s_userName)
  {
    SEND(NO_USERNAME_MSG);
    return FALSE;
  }
  else
  {
    char *p = strtok(NULL, DELIMITERS);
    if (!p)
    {
      SEND(NO_INDX_MSG);
      return FALSE;
    }
    else
    {
      // Remember index
      int index = atoi(p);
      
      // Create mail list if doesn't exist yet
      if (s_mailNum == 0)
      {
        int ret = CreateMailList();
        if (ret == FALSE || ret < 0)
          return ret;
      }

      // No mail
      if (s_mailNum == 0)
      {
        SEND(NO_MAIL_MSG);
        return TRUE;
      }

      // Check validity of index
      if (index < 0 || index >= s_mailNum)
      {
        SEND(BAD_INDX_MSG);
        return FALSE;
      }

      // Delete audio mail file
      if (unlink(s_mail[index].fname) < 0)
      {
        perror("unlink");

        // Notify user
        char s[MAX_STRLEN];
        sprintf(s, ERR_MAIL_DEL_MSG, s_mail[index].fname);
        SEND(s);

        return FALSE;
      }

      SEND(DEL_OK_MSG);

      // Rebuild list
      s_mailNum = 0;
      int ret = CreateMailList();
      if (ret == FALSE || ret < 0)
        return ret;

      return TRUE;
    }
  }
}


/////////////////////////////////////////////////////////////////////////////
// User wants to send mail
int OnSendMail()
{
  if (!*s_userName)
  {
    SEND(NO_USERNAME_MSG);
    return FALSE;
  }
  else
  {
    char *p = strtok(NULL, DELIMITERS);
    if (!p)
    {
      SEND(NO_RCPT_MSG);
      return FALSE;
    }
    else
    {
      // Remember recipient name
      strcpy(s_rcptName, p);

      // Check if recipient has mailbox
      char dir[PATH_MAX];

      // Get mailbox location
      if (!GetMailBoxDir(s_rcptName, dir))
      {
        char s[MAX_STRLEN+PATH_MAX];
        sprintf(s, NO_MAILBOX_MSG, dir);
        SEND(s);
        *s_rcptName = 0;
        return FALSE;
      }
    
      // Get subject
      p += strlen(p) + 1;
      if (p)
        strcpy(s_subjName, p);
      else
        strcpy(s_subjName, "No subject");

      // Ok, user may proceed sending mail
      SEND(RCPT_OK_MSG);
      s_waitForMailData = TRUE;

      return TRUE;
    }
  }
}

/////////////////////////////////////////////////////////////////////////////
// Create file in user mail directory
#define WRITELINE(s) if (fputs(s, f) < 0) \
                    { \
                        perror("fputs"); \
                        fclose(f); \
                        return FALSE; \
                    } \
                    else \
                    { \
                      fputs("\n", f); \
                    }
int OnMailData(char *buf, int size)
{
  // Get mailbox location
  char dir[PATH_MAX];
  if (!GetMailBoxDir(s_rcptName, dir))
  {
    char s[MAX_STRLEN+PATH_MAX];
    sprintf(s, NO_MAILBOX_MSG, dir);
    SEND(s);
    return FALSE;
  }
    
  // Construct unique filename
  char fname[PATH_MAX];
  strcpy(fname, dir);
  strcat(fname, "/auXXXXXX");
  mktemp(fname);

  // Create file
  FILE *f;
  if ((f=fopen(fname, "w")) == NULL)
  {
    perror("fopen");

    // Notify user
    char s[MAX_STRLEN];
    sprintf(s, ERR_MAIL_OPEN_MSG, fname);
    SEND(s);

    return FALSE;
  }

  // Write all available info into file
  char str[MAX_STRLEN];
  
  // Skip 4 bytes for data location
  fseek(f, 4, SEEK_SET);

  // User address
  struct sockaddr_in addr;
  int addrLen = sizeof(addr);
  if (getpeername(s_sock, (struct sockaddr *)&addr, &addrLen) < 0)
  {
    perror("getpeername");
    fclose(f);
    return FALSE;
  }
  char *ipAddr = inet_ntoa(addr.sin_addr);
  struct hostent *he;
  if ((he=gethostbyaddr(ipAddr, strlen(ipAddr), AF_INET)) != NULL)
    sprintf(str, "%s@%s", s_userName, he->h_name);
  else
    sprintf(str, "%s@%s", s_userName, ipAddr);
  WRITELINE(str);

  // Subject
  WRITELINE(s_subjName);

  // Current date
  time_t t;
  time(&t);
  strcpy(str, ctime(&t));
  *(str+strlen(str)-1)='\0';
  WRITELINE(str);

  // Write mail data location into first 4 bytes of file
  long pos = ftell(f), npos = htonl(pos);
  fseek(f, 0, SEEK_SET);
  if (fwrite(&npos, sizeof(long), 1, f) < 0)
  {
    perror("fwrite");
    fclose(f);
    return FALSE;
  }

  // Go back to data location and write there our buf
  fseek(f, pos, SEEK_SET);
  if (fwrite(buf, size, 1, f) < 0)
  {
    perror("fwrite");
    fclose(f);
    return FALSE;
  }

  // Done with file
  fclose(f);

  // Send ACK to client
  SEND(SEND_OK_MSG);
  
  return TRUE;
}


/////////////////////////////////////////////////////////////////////////////
// Bye-bye....
int OnQuit()
{
  SEND(BYE_MSG);
  return ACT_FATAL; // Will cause exit
}




////////////////////////////////////////////////////////////////////////////
// Signals that will cause this process to exit
void Client_StopSignal(int sig)
{
  fprintf(stderr, "Client %d exitting on signal %d.\n", getpid(), sig);
  close(s_sock);
  exit(1);
}

/////////////////////////////////////////////////////////////////////////////
// Handle new connection
int OnNewClient(int sock, int numClients)
{
  fd_set readBitmask;
  s_sock = sock;

  // Catch signals
  signal(SIGINT,  Client_StopSignal);
  signal(SIGTERM, Client_StopSignal);
  signal(SIGSTOP, Client_StopSignal);

  // Welcome client
  char s[256];
  time_t val;
  time(&val);
  char *timeStr = ctime(&val);
  *(timeStr+(strlen(timeStr)-1)) = 0;
  sprintf(s, WELCOME_MSG, timeStr);
  SEND(s);

  // Work
  while (1)
  {
    FD_ZERO(&readBitmask);
    FD_SET(sock, &readBitmask);
    if (select(sock+1, &readBitmask, NULL, NULL, NULL) < 0)
    {
      perror("select(read)");
      return -1;
    }

    // Read client message
    int size;
    char *buf;
    if ((buf=RecvMsg(sock, &size)) == NULL)
    {
      // If size == 0 client simply disconnected and we should
      // leave silently.
      if (size != 0)
        fprintf(stderr, "Client %d: low-level error. Exitting...\n", getpid());
      close(s_sock);
      return -1;
    }

    // Data that client sends after the 'send' command must
    // be handled differently.
    if (s_waitForMailData)
    {
      s_waitForMailData = FALSE;
      OnMailData(buf, size);
    }
    else // Any other protocol command
    {
      DEBUG_CODE(printf("C: %s\n", buf);)

      // Act accordingly
      char *bufCmd = strtok(buf, DELIMITERS);
      strlwr(bufCmd);
      int breakCode = FALSE;
      for (int i=0; i<NUM_ACTIONS; i++)
      {
        if (strcmp(AL[i].cmd, bufCmd) == 0)
        {
          if (AL[i].action() == ACT_FATAL)
          {
            fprintf(stderr, "Client %d: Fatal error. Exitting...\n", getpid());
            close(s_sock);
            return -1;
          }
          else
          {
            breakCode = TRUE;
            break;
          }
        }
      }
    
      // Command wasn't found
      if (!breakCode)
      {
        SEND(BAD_CMD_MSG);
      }
    }

    free(buf);
  } // while 1

  return 0;
}
