#include <malloc.h>
#include <ctype.h>
#include <sys/stat.h>

#include "defs.h"
#include "server.h"

//
// Server helper functions
//

/////////////////////////////////////////////////////////////////////////////
// Find user's mailbox directory
int GetMailBoxDir(char *user, char *dir)
{
  char prefix[PATH_MAX+1], *p;

  // Create path name
  p = getenv("AUSERVER_DIR");
  if (p)
    strcpy(prefix, p);
  else
    getcwd(prefix, PATH_MAX);

  strcat(prefix, "/");
  strcat(prefix, user);
  strcpy(dir, prefix);

  // Check it
  struct stat sbuf;
  if (stat(prefix, &sbuf) < 0 || !S_ISDIR(sbuf.st_mode))
    return FALSE;
  
  return TRUE;
}

//
// Common helper functions
//

/////////////////////////////////////////////////////////////////////////////
// Fill sockaddr_in structure given host name and port
int FillAddr(char *host, int port, sockaddr_in *psin)
{
  hostent *he;
  char localhost[] = "localhost";

  // Clear structure
  memset(psin, 0, sizeof(sockaddr_in));
   
  // Fill it
  psin->sin_family = AF_INET;
  psin->sin_port   = htons(port);

  // Default is - localhost
  if (!host)
    host = localhost;

  // Resolve
  if ((he=gethostbyaddr(host, strlen(host), AF_INET)) != NULL ||
      (he=gethostbyname(host)) != NULL)
  {
    memcpy(&(psin->sin_addr.s_addr), he->h_addr_list[0], he->h_length);
    return TRUE;
  }
  else
  {
#ifdef LINUX
    herror("resolve");
#else
    perror("resolve");
#endif
    return FALSE;
  }
}

/////////////////////////////////////////////////////////////////////////////
// Send given message with it's size prepended
int SendMsg(int sock, char *msg, int msgSize)
{
  int ret;
  long bufSize = sizeof(long)+msgSize;
  long net_msgSize = htonl(msgSize);

  // Allocate buffer for outgoing message
  char *buf = (char *)malloc(bufSize);
  if (!buf)
  {
    perror("malloc(send)");
    return -1;
  }
  
  // Add message size in front of message
  memcpy(buf, &net_msgSize, sizeof(long));
  memcpy(buf+sizeof(long), msg, msgSize);

  // printf("send: size %d\n", msgSize);

  // Send
  if ((ret=send(sock, buf, bufSize, 0)) < 0)
  {
    perror("send");
  }

  // Paranoia check
  if (ret != bufSize)
    fprintf(stderr, "SendMsg: Sent %d bytes instead of %ld\n", ret, bufSize);

  free(buf);
  return ret;
}

/////////////////////////////////////////////////////////////////////////////
// Receive message (first long should specify size of message after it)
#define READ_EXACT_SIZE(_buf, _size) \
 receivedSize = 0; \
 do \
 { \
   if ((ret=recv(sock, _buf+receivedSize, _size-receivedSize, 0)) < 0) \
    { \
      perror("recv"); \
      return NULL; \
    } \
    receivedSize += ret; \
  } while(receivedSize != _size && ret != 0); \
  \
  if (ret == 0) \
  { \
    *msgSize = 0; \
    return NULL; \
  }

char *RecvMsg(int sock, int *msgSize)
{
  // Read client message size
  long size, receivedSize;
  int ret;

  // Read message length
  READ_EXACT_SIZE(((char *)&size), sizeof(long));

  // Check message size
  size = ntohl(size);
  if (size > MAX_MSGSIZE)
  {
    fprintf(stderr, "RecvMsg: %ld > MAX_MSGSIZE(%d)\n", size, MAX_MSGSIZE);
    return NULL;
  }

  //printf("recv: size %ld\n", size);

  // Allocate buffer for message
  char *buf = (char *)malloc(size);
  if (!buf)
  {
    perror("malloc(recv)");
    return NULL;
  }
    
  // Read message
  READ_EXACT_SIZE(buf, size);

  *msgSize = size;

  return buf;
}


////////////////////////////////////////////////////////////////////////////
// Uppercase string
char *strupr(char *s)
{
  if (!s) return NULL;

  char *p = s;
  while (*p) { *p = toupper(*p); ++p; }
  return s;
}

////////////////////////////////////////////////////////////////////////////
// Lowercase string
char *strlwr(char *s)
{
  if (!s) return NULL;

  char *p = s;
  while (*p) { *p = tolower(*p); ++p; }
  return s;
}
