/*
 * This file implements the classes to handle  with any (?) mixer-device
 * currently only tested with /dev/mixer
 * the main program that use this should do a mixer::open_mixer at startup
 * and a mixer::close_mixer when closing the program.
 *
 * $Log: mixer.cpp,v $
 * Revision 1.5  1997/10/24 13:35:39  ral
 * Changes between OSS and USS implemented, if the
 * mixer is compiled with OSS it can not display mixerinfos
 * with USS. Why?
 *
 * Revision 1.4  1997/10/22 11:45:43  ral
 * on startup it reads the mixer info
 * the mute state never used 'cause macros are obsolete
 *
 * Revision 1.3  1997/10/20 14:55:25  ral
 * read_value changed
 * (re-reading the recsrc-value)
 * Todo: read and set the mute-state
 *
 * Revision 1.2  1997/10/20 11:14:27  ral
 * First version stable, some error checks schould  be implemented
 *
 * Revision 1.1  1997/10/20 11:06:20  ral
 * Initial revision
 *
 */
#include "mixer.hh"
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <math.h>
#include <string.h>

#ifndef rcsid
static char rcsid[]="$Id: mixer.cpp,v 1.5 1997/10/24 13:35:39 ral Exp $";
#endif

mixer::mixer()
{
}

// init of the static members
// of course - they are only better global variables.
// -> you can open onyl ones with every session of program
int mixer::mixer_fd=0;
int mixer::devmask=0, mixer::recmask=0, mixer::recsrc=0, mixer::stereodev = 0;
char*mixer::names[SOUND_MIXER_NRDEVICES]=SOUND_DEVICE_LABELS;
int mixer::exclusiv_record=0;
char mixer::DevName[51]="";
mixer::~mixer()
{
}

// open the mixer and writes the current state to the variables
// next version will include a re-read function to scan again
int mixer::open_mixer(const char * devname)
{
    if (mixer_fd)
	return 1;
    if (!devname)
	mixer_fd = open("/dev/mixer",O_RDWR);
    else
	mixer_fd = open(devname,O_RDWR);
    if (mixer_fd < 1)
	return mixer_fd;
    // reading the devices
    int i = ioctl(mixer_fd,  SOUND_MIXER_READ_DEVMASK,&devmask);
    if (i == -1)
	return i;
    //  which are record sources?
    i = ioctl(mixer_fd, SOUND_MIXER_READ_RECMASK,&recmask);
    if (i == -1)
	return i;
    // which recoords are enabled?
    i = ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC,&recsrc);
    if (i == -1)
	return i;
    // which are stereo-devices?
    i = ioctl(mixer_fd, SOUND_MIXER_READ_STEREODEVS,&stereodev);
    if (i == -1)
	return i;
    int Cap;
    // the other capabilities
    i = ioctl(mixer_fd, SOUND_MIXER_READ_CAPS,&Cap);
    if (i == -1)
	return i;
    exclusiv_record = ((1<<SOUND_CAP_EXCL_INPUT) & Cap) ? 1:0;
    struct minfo info;
#ifdef UNIX_SOUND_SYSTEM
    // when compiled with USS
    // it always run under current OSS 3.8
    i   = ioctl(mixer_fd,SOUND_MIXER_INFO,&info);
    if (i == -1)
	return i;
    memset(DevName,'\0',51);
    strcpy(DevName,info.id);
    strcat(DevName," ");
    strcat(DevName,info.name);
#else
    //We get trouble when the program is compiled with OSS
    // and runs with USS ->  we check the version again
    int version;
    i = ioctl(mixer_fd,OSS_GETVERSION,&version);
    // in OSS it returns 0 and the versionvalue
    if (i > -1) {
	i   = ioctl(mixer_fd,SOUND_MIXER_INFO,&info);
	if (i == -1)
	    return i;
	memset(DevName,'\0',51);
	strcpy(DevName,info.id);
	strcat(DevName," ");
	strcat(DevName,info.name);
    }
    else {
	//in USS it returns -1 -> we can not  retrieve the soundinfo
	strcpy(DevName,"Unknown");
    }
#endif
    return 1;
}

// check if dev is a valid device on your soundcard
int mixer::is_valid_dev(int dev)
{
    if (!mixer_fd)
	return 0;
    return (((1 << dev) & devmask) ? 1 : 0);
}

void mixer::set_recmask()
{
    ioctl(mixer_fd,SOUND_MIXER_READ_RECMASK,&recmask);
}


// this is a single mixer-device class.
mixer_dev::mixer_dev(int nDev)
{
    devnumber = nDev;
    devnumber = (devnumber > SOUND_MIXER_NRDEVICES-1) ? SOUND_MIXER_NRDEVICES-1 : devnumber;
    devnumber = (devnumber < 0) ? 0 : devnumber;
    balance = value = 0;
    Name = 0;
    read_values();
    Name = names[devnumber];
}

// set the value of the device
void mixer_dev::set_value(int nValue)
{
    if (!get_mixer_fd())
	return;
    int oldv = value;
    value = nValue;
    int ltemp, rtemp;
    ltemp = value;
    rtemp = value;
    // want a better calculate of values
    double wert;
    if (is_stereo()) {
	if (balance < 50) {
	    wert = (double)balance / 50.0 * (double)value;
	    (double)rtemp = rint(wert);
	}
	else {
	    wert = (100.0-(double)balance)/50.0 * (double)value;
	    (double)ltemp = rint(wert);
	}
    }
    int temp = (rtemp << 8) | ltemp;
    ioctl(get_mixer_fd(),MIXER_WRITE(devnumber),&temp);
}

//scans for the values of the device
void mixer_dev::read_values()
{
    if (!get_mixer_fd())
	return;
    ioctl(get_mixer_fd(), MIXER_READ(devnumber),&value);
    int ltemp, rtemp;
    ltemp = value & 0x7f;
    rtemp = (value >> 8)& 0x7f;
    double b;
    int max = (ltemp > rtemp) ? ltemp : rtemp;
    if (max) {
	b = (ltemp > rtemp) ? 50.0 * (double)rtemp / (double)max : 100.0 - (50.0 * (double)ltemp / (double)max);
	balance = (int)rint(b);
    }
    else
	balance = 50;
    if ((1 << devnumber) & stereodev) {
	value = max;
    }
    else
	value = value & 0x7f;
    ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC,&recsrc);
}

//simple
void mixer_dev::set_balance(int nBalance)
{
    if (!get_mixer_fd())
	return;
    balance = nBalance;
    set_value(value);
}

// set if this device should use as a record source
void mixer_dev::set_record(int On)
{
    int o = On;
    o = (o < 0 ) ? 0 : o;
    o = (o > 1) ? 1 :  o;
    if (is_recordsrc()) {
	if (o) {
	    //may be, there is a device which allows only one recordsource
	    //to select at ones
	    if (is_exclusiv())
		recsrc=0;
	    recsrc |= ( 1 << devnumber);
	}
	else {
	    recsrc &= ~(1 << devnumber);
	}
	if (ioctl(mixer_fd,SOUND_MIXER_WRITE_RECSRC,&recsrc) == -1) {
	    perror("WRITE_RECSRC");
	}
    }
}
