/*------------------------------------------------------------------------*/
/*                                                                        */
/*  File:       profile.cpp                                               */
/*                                                                        */
/*  Contains:   Functions getProfileString(), getProfileInt(),            */
/*              writeProfileString()                                      */
/*  Purpose:    MS-Windows Style .ini File Interface for C++              */
/*  Author:     XuYiFeng, China                                           */
/*  Date:       25.04.1995                                                */
/*------------------------------------------------------------------------*/
/*
 * Some options were changed to work with linux.
 * Id set all fstreams to there
 * 'real' - classe (ofstream or ifstream) for better reading the file :)
 * Rajko Albrecht (ral) 1997
 * First inserted: writeProfileInt
 *
 * $Log: profile.cpp,v $
 * Revision 1.5  1997/09/15 11:46:09  ral
 * made some optimisations for -O3 flag
 *
 * Revision 1.4  1997/07/18 21:33:41  ral
 * design-bug in getProfileSection fixed
 *
 * Revision 1.3  1997/07/18 16:30:19  ral
 * function removeLastSpace inserted
 * (getProfileSection() require that)
 *
 * Revision 1.2  1997/07/18 14:34:02  ral
 * Function getProfileSection implemented
 *
 * Revision 1.1  1997/07/18 12:25:38  ral
 * Initial revision
 *
 */

#ifndef rcsid
static char rcsid[]="$Id: profile.cpp,v 1.5 1997/09/15 11:46:09 ral Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <fstream.h>

#ifdef __linux__
#include <strstream.h>
#include <unistd.h>
#endif

#include "helpers.h"
#include "profile.h"

#ifndef False
#define False 0
#endif

#ifndef True
#define True  1
#endif

//--------------------------------------------------------------------------
// removeLastSpace: shorts a string if there are specialchars at end
//--------------------------------------------------------------------------
void removeLastSpace(char *buf)
{
    if (!buf)
	return;
    int i = strlen(buf);
    int j = buf[i-1];
    while (isspace(j)) {
	buf[i-1] = '\0';
	i = strlen(buf);
	j = buf[i];
    }
}

//--------------------------------------------------------------------------
// titlePos: get a section title position & length in a string.
//--------------------------------------------------------------------------

static char *titlePos( char *buf, int *len ) 
{
    char *p = buf, *q;

    while( *p && isspace(*p) ) p++;
    if( *p != '[' )
        return 0;

    q = p+1;
    while( *q && *q != ']' ) q++;
    if( *q != ']' )
        return 0;
    if( len )
        *len = int(q - p - 1);
    return p+1;
}

//-------------------------------
// check if the line is a comment
//-------------------------------
static int isComment(char*bufPtr)
{
    if (!bufPtr)
	return 0;
    return (bufPtr[0] == ';');
}

//--------------------------------------------------------------------------
// isTitleLine: check if a string is a section title line
//--------------------------------------------------------------------------

static int isTitleLine( char *bufPtr ) 
{
    if (isComment(bufPtr))        // of course a comment is not a titleline
	return 0;
    return titlePos( bufPtr, 0 ) != 0;
}

//--------------------------------------------------------------------------
// containTitle: check if a string contains a section a title
//--------------------------------------------------------------------------

static int containTitle( char *buf, const char *section ) {

    char *p;
    int len;

    p = titlePos( buf, &len );
    if( p ) {
#ifdef WIN32
       if( strlen( section ) == (unsigned)len && strnicmp( section, p, len ) == 0 )
#else
       if( strlen( section ) == (unsigned)len && strncasecmp( section, p, len ) == 0 )	 
#endif	   
           return True;
    }
    return False;
}

//--------------------------------------------------------------------------
// gotoSection: move file position to start line of a section
//--------------------------------------------------------------------------

static int gotoSection( ifstream &is, const char *section ) 
{
    char line[257];
    memset(line,'\0',257);
    while( is.getline( line, 256 ) )
       if( containTitle( line, section ) )
           return True;
    return False;
}

//--------------------------------------------------------------------------
// textPos: get content's position of an entry
//--------------------------------------------------------------------------

static char *textPos( char *buf, const char *entry ) {

    if( buf[0] == ';' ) // it is comment line
        return 0;

    char *p = strchr( buf, '=' );
    if( !p )
        return 0;

    int len = int(p - buf);
#ifdef WIN32   
    if( strlen(entry) == (unsigned)len && strnicmp( buf, entry, len ) == 0 )
#else     
    if( strlen(entry) == (unsigned)len && strncasecmp( buf, entry, len ) == 0 )
#endif       
        return p+1;

    return 0;
}

//--------------------------------------------------------------------------
// stripQuotationChar: strip a pair of quotation chars in a string
//--------------------------------------------------------------------------

static void stripQuotationChar( char *buf ) {

    char *p;
    char *q;

    p = buf;
    while( *p && isspace(*p) ) p++;
    if( !(*p == '\"' || *p == '\'') )
	return;
    q = p+strlen(p);
    while( *q != *p && q > p ) q--;
    if( q == p )
        return;
    int len = int(q - p - 1);
    memmove( buf, p+1, len );
    buf[len] = 0;
}

//--------------------------------------------------------------------------
// readEntry: read contents of entry
//--------------------------------------------------------------------------

static int readEntry( ifstream &is, const char *entry, char *buf,
                      int bufSize, int strip ) {

    char lineBuf[256];
    char *p, *cur;
    int  len;

    cur  = buf;
    *cur = '\0';
    len  = -1;
    while( is.getline( lineBuf, 256 ) ) {
        if( isTitleLine( lineBuf ) )       // section is ended
            break;

        p = textPos( lineBuf, entry );     // not equal this entry
        if( p == 0 )
            continue;

        if( strip )
            stripQuotationChar( p );

        len = strlen(p);
        if( bufSize-1 < len )
            len = bufSize-1;

        strncpy( cur, p, len );
        cur[len] = 0;
        break;
    }

    return len;
}

//--------------------------------------------------------------------------
// getProfileString:
//--------------------------------------------------------------------------

int getProfileString( const char *section,
                      const char *entry,
                      const char *defaultString,
                      char *buffer,
                      int   bufLen,
                      const char *fileName ) {

    
    ifstream input( fileName, ios::in|ios::nocreate );
    int len = -1;
    if( input ) {
		if( gotoSection( input, section ) )
        len = readEntry(input, entry, buffer, bufLen, True);
	}

    if( len < 0 ) {             //can not read entry, use default string
        strncpy( buffer, defaultString, bufLen-1 );
	buffer[bufLen-1] = 0;
        len = strlen(buffer);
    }
    return len;
}

//--------------------------------------------------------------------------
// getProfileInt:
//--------------------------------------------------------------------------

int getProfileInt( const char *section,
                   const char *entry,
                   int defaultInt,
                   const char *fileName ) {
   
    char buf[256];
    char iBuf[34];   //"34" is max space "itoa" requires under 32 bit C++
#ifndef WIN32
    ostrstream outputbuffer(iBuf,33);
#endif
#ifdef WIN32
    itoa( defaultInt, iBuf, 10 );
#else
    outputbuffer << defaultInt << ends;
#endif
    getProfileString( section, entry, iBuf, buf, 256, fileName );
    return atoi( buf );
}

static void writeEntry(ofstream &os, const char *entry, const char *string) {

    os << entry << '=' << string << endl;
}

//--------------------------------------------------------------------------
// writeProfileString:
//--------------------------------------------------------------------------

int writeProfileString( const char *section,
                        const char *entry,
                        const char *string,
                        const char *fileName ) 
{
    char buf  [256];
    char*path;
    int  titleFound, entryFound;
    path = tmpnam(0);
    ifstream is( fileName, ios::in  );
    ofstream os( path,     ios::out );
    if( !os || entry == 0 )     //maybe cannot create file or invalid entry
        return 0;
    titleFound = False;
    entryFound = False;
    while( is.getline(buf, 256) ) {
        os << buf << endl;
        if( containTitle(buf, section) ) {
            titleFound = True;
            break;
        }
    }
    if( !titleFound ) {         // add section
        os << '[' << section << ']' << endl;
        writeEntry( os, entry, string );
    }
    else {
	while( is.getline(buf, 256) ) {
            if( isTitleLine( buf ) )     // section ended, but still not
                break;                   // found the entry
            if( textPos( buf, entry ) ) {// entry found, so rewrite it
                entryFound = True;
                break;
            }
	    os << buf << endl;
        }
        writeEntry( os, entry, string );
        if( is.gcount() > 0 && !entryFound )
            os << buf << endl;
        while( is.getline(buf, 256) )  // copy left lines
            os << buf << endl;
    }
    is.close();
    os.close();
    unlink( fileName );
#ifdef WIN32
    rename( path, fileName );
#else
    if (move_file(path,fileName) == -1)
      return 0;
#endif
    return strlen(string);
}

int writeProfileInt( const char *section,
		     const char *entry,
		     int Int,
		     const char *fileName ) {
   
    char buf[256];
    char iBuf[34];   //"34" is max space "itoa" requires under 32 bit C++
#ifndef WIN32
    ostrstream outputbuffer(iBuf,33);
#endif
#ifdef WIN32
    itoa( Int, iBuf, 10 );
#else
    outputbuffer << Int << ends;
#endif
    if ( (unsigned)writeProfileString( section, entry, iBuf,  fileName) != strlen(iBuf))
      return -1;
    return 0;
}

/*
 * This funktion write all items in a section into Buffer.
 * Buffer must have size of Size.
 * It will  write the items as an array of strings, last string is 
 * marked with two '\0'.
 * it return the size of copied buffer, if Size is to small, it returns the
 * required size. If Buffer == NULL or / and Size == NULL, it returns the 
 * required size too. (required = length of all strings+ all terminators + one
 * array terminator)
 */
int getProfileSection(  const char *section,
			char *Buffer,
			int Size,
			const char *fileName )
{
  int counter = 0;
  int iSize = 0;
  int oldsize = 0;
  char iBuf[256];
  memset(iBuf,'\0',256);
  char *BP = NULL;
  char delem='=';
  if (!fileName || !section)
      return -1;
  ifstream i(fileName,ios::nocreate | ios::in);
  if (!i)
      return -1;
  if (!gotoSection(i,section))
      return 0;
  if (Size && !Buffer) 
      Size = 0;
  if (Buffer && Size)
      memset(Buffer,'\0',Size);
  while ( !i.eof()) {
      i.getline(iBuf,255);
      if (!i)
	  break;
      if (isTitleLine(iBuf))
	  break;
      removeLastSpace(iBuf);
      counter = strlen(iBuf);
      if (counter && !isComment(iBuf)) {
	oldsize = iSize;
	iSize += counter;
	iSize += 1;
	  if (Size) {
	      if (oldsize < Size-1)
		  BP = &Buffer[oldsize];
	      if (iSize < Size)
		  strcat(BP,iBuf);
	  }
	  memset(iBuf,'\0',256);
      }
  }
  if (iSize)
      iSize += 1;
  return iSize;
}
