/* A soundmixer for linux with a nice text-interface for those non-X-ers.
 * (C)1998 Bram Avontuur (brama@stack.nl)
 */
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>

#ifdef HAVE_NCURSES_CURSES_H
#include <ncurses/curses.h>
#elif HAVE_NCURSES_H
#include <ncurses.h>
#elif HAVE_CURSES_H
#include <curses.h>
#else
#warning "Can't find any ncurses include file!"
#endif

#ifdef HAVE_SYS_SOUNDCARD_H
#include <sys/soundcard.h>
#elif HAVE_SOUNDCARD_H
#include <soundcard.h>
#else
#error "Couldn't find any soundcard.h"
#endif

#define MIXER_DEVICE "/dev/mixer"
#define MIN(x, y) ((x) < (y) ? (x) : (y))

#define BOTH_CHANNELS 0x11
#define RIGHT_CHANNEL 0x10
#define LEFT_CHANNEL  0x01

#define MYVERSION "<<MixIt 2.0>>"

/* globals */
int
	*supported,
	mixer;
char
	*sources[] = SOUND_DEVICE_LABELS;
WINDOW
	**sbars;
short
	currentbar,
	minbar, /* index-nr. of topmost on-screen bar. */
	maxspos, /* index-nr. of bottommost on-screen bar. */
	currentspos; /* index-nr. of screen-position of current bar */

/* Prototypes */
void draw_scrollbar(int, int);
void change_bar(short, short, short, short);
void redraw_all_bars();

int
main(int argc, char *argv[])
{
	int
		i,
		key,
		volume,
		nrbars = 0;

	unsigned int setting;
	unsigned int j;

	supported = NULL;
	sbars = NULL;
	
	if ( (mixer = open(MIXER_DEVICE, O_RDWR)) < 0)
	{
		fprintf(stderr, "%s: can't open /dev/mixer: ", argv[0]);
		perror("");
		exit(1);
	}

	ioctl(mixer, MIXER_READ(SOUND_MIXER_DEVMASK), &setting);
	printf("devmask = %u\n", setting);

	/* Determine supported devices */
	for (j = 0; j < SOUND_MIXER_NRDEVICES; j++)
	{
		if (setting & (1 << j)) /*Device sources[j] is supported.*/
		{
			supported = (int *)realloc(supported, (++nrbars) * sizeof(int));
			supported[nrbars - 1] = j;
			printf("Device %s is supported.\n", sources[j]); fflush(stdout);
		}
	}

	sbars = (WINDOW**)malloc(nrbars * sizeof(WINDOW*));

	/* initialize curses screen management */
	initscr();
	start_color();
	cbreak();
	noecho();
	nonl();
	intrflush(stdscr, FALSE);
	keypad(stdscr, TRUE);

	init_pair(0, COLOR_WHITE, COLOR_BLACK);
	init_pair(1, COLOR_GREEN, COLOR_BLACK);
	init_pair(2, COLOR_YELLOW, COLOR_BLACK);
	init_pair(3, COLOR_RED, COLOR_BLACK);

	border(0, 0, 0, 0, 0, 0, 0, 0);
	if (COLS < 80 || LINES < 8)
	{
		mvprintw(1,1, "You'll need at least an 80x8 terminal, sorry.");
		getch();
		endwin();
		exit(1);
	}

	/* On my system, the mozart souncard I have (with OTI-601 chipset) 
	 * produces a lot of noise and fuss after initialisation. Setting 
	 * SOUND_MIXER_LINE2 to zero stops this (what actual setting does this
	 * change on the mozart card???)
	 */
#ifdef MOZART
	volume = 0;
	ioctl(mixer, MIXER_WRITE(SOUND_MIXER_LINE2), &volume);
#endif

	mvprintw(0, (COLS - strlen(MYVERSION)) / 2, MYVERSION);

	maxspos = ((LINES - 5) / 3) - 1; /* 5 nonbar-lines, 3 lines/bar */
	if ( (maxspos + 1) > nrbars)
		maxspos = nrbars - 1;
	currentspos = 0;
	minbar = 0;
	currentbar = 0;

	for ( i = 0; i < MIN(nrbars, (maxspos + 1)); i++)
	{
		mvprintw(3 + 3 * i, 65, "L");
		mvprintw(4 + 3 * i, 65, "R");
	}
	mvprintw(1, 10, "0        1         2         3         4         5");
	/* mvprintw(2, 10, "12345678901234567890123456789012345678901234567890");*/
	mvprintw(LINES - 2, 2, "(C)1998 Bram Avontuur (brama@stack.nl)");
	refresh();

	/* Draw all Scrollbars */
	redraw_all_bars();

	/* get input */
	while ((key = getch()) != KEY_F(12) && key != 'q')
	{
		short oldbar = currentbar;

		switch(key)
		{
			case KEY_DOWN:
				if (++currentbar >= nrbars) /* flip da dip, B! */
				{ 
					currentbar = 0;
					minbar = 0;
					currentspos = 0;
					redraw_all_bars();
				}
				else
				{
					if (currentspos == maxspos) /* scroll bars up by one */
					{
						minbar++;
						redraw_all_bars();
					}
					else
					{
						currentspos++;	
						draw_scrollbar(oldbar, currentspos - 1);
						draw_scrollbar(currentbar, currentspos);
					}
				}
				break;
			case KEY_UP:
				if (--currentbar < 0)
				{
					currentbar = nrbars - 1;
					currentspos = maxspos;
					minbar = nrbars - maxspos - 1;
					redraw_all_bars();
				}
				else
				{
					if (!currentspos)
					{
						minbar--;
						redraw_all_bars();
					}
					else
					{
						currentspos--;
						draw_scrollbar(oldbar, currentspos + 1);
						draw_scrollbar(currentbar, currentspos);
					}
				}
				break;
			case KEY_RIGHT:
				change_bar(currentbar,  2, 0, BOTH_CHANNELS);
				break;
			case KEY_LEFT:
				change_bar(currentbar, -2, 0, BOTH_CHANNELS);
				break;
			case KEY_PPAGE:
				change_bar(currentbar, 2, 0, LEFT_CHANNEL);
				break;
			case KEY_HOME:
				change_bar(currentbar, -2, 0, LEFT_CHANNEL);
				break;
			case KEY_NPAGE:
				change_bar(currentbar, 2, 0, RIGHT_CHANNEL);
				break;
			case '0':
				change_bar(currentbar, 0, 1, BOTH_CHANNELS);
				break;
			case '1':
				change_bar(currentbar, 20, 1, BOTH_CHANNELS);
				break;
			case '2':
				change_bar(currentbar, 40, 1, BOTH_CHANNELS);
				break;
			case '3':
				change_bar(currentbar, 60, 1, BOTH_CHANNELS);
				break;
			case '4':
				change_bar(currentbar, 80, 1, BOTH_CHANNELS);
				break;
			case '5':
				change_bar(currentbar, 100, 1, BOTH_CHANNELS);
				break;
			case KEY_END:
				change_bar(currentbar, -2, 0, RIGHT_CHANNEL);
				break;
		}
	}
	endwin();
	return 0;
}

void
draw_scrollbar(int i, int spos)
{
	int
		j;
	unsigned int
		setting, volumes[2];

	sbars[i] = newwin(2, 58, 3 + 3 * spos, 2);
	mvwprintw(sbars[i], 0, 0, sources[supported[i]]);

	if (i == currentbar)
	{
		mvwchgat(sbars[i], 0, 0, strlen(sources[supported[i]]), A_REVERSE, 
			0, NULL);
	}

	/* get sound-settings */
	ioctl(mixer, MIXER_READ(supported[i]), &setting);
	/* setting = setting / 2; */
	
	volumes[0] = setting & 0x000000FF;
	volumes[1] = (setting & 0x0000FF00)>>8;
	volumes[0] = MIN(100, volumes[0]);
	volumes[1] = MIN(100, volumes[1]);
	volumes[0] /= 2;
	volumes[1] /= 2;

	wmove(sbars[i], 0, 8);
	for (j = 0; j < volumes[0]; j++)
		waddch(sbars[i], '#');
	for (j = volumes[0]; j < 50; j++)
		waddch(sbars[i], '.');

	wmove(sbars[i], 1, 8);
	for (j = 0; j < volumes[1]; j++)
		waddch(sbars[i], '#');
	for (j = volumes[1]; j < 50; j++)
		waddch(sbars[i], '.');

	/* fix up some nice colors */
	mvwchgat(sbars[i], 0, 8,  17, A_BOLD, 1, NULL);
	mvwchgat(sbars[i], 0, 25, 16, A_BOLD, 2, NULL);
	mvwchgat(sbars[i], 0, 41, 17, A_BOLD, 3, NULL);
	mvwchgat(sbars[i], 1, 8,  17, A_BOLD, 1, NULL);
	mvwchgat(sbars[i], 1, 25, 16, A_BOLD, 2, NULL);
	mvwchgat(sbars[i], 1, 41, 17, A_BOLD, 3, NULL);

	wrefresh(sbars[i]);
	delwin(sbars[i]);
	move(0, 0);
}

/* bar     : Which bar to modify (prolly currentsbar)
 * amount  : Amount by which to change OR (if absolute != 0) the absolute
 *         : value (0<=value<=100) to change to.
 * absolute: !0 if amount is an absolute value (and not relative to current
 *         : setting)
 * channels: One of LEFT_CHANNEL/RIGHT_CHANNEL/BOTH_CHANNELS
 */
void
change_bar(short bar, short amount, short absolute, short channels)
{
	int
		volumes[2],
		setting;

	ioctl(mixer, MIXER_READ(supported[bar]), &setting);
	volumes[0] = setting & 0x000000FF;
	volumes[1] = (setting & 0x0000FF00)>>8;

	if (absolute)
	{
		if (amount < 0)
			amount = 0;
		else if (amount > 100)
			amount = 100;
	}

	if (channels == BOTH_CHANNELS || channels == LEFT_CHANNEL)
	{
		if (!absolute)
			volumes[0] += amount;
		else
			volumes[0] = amount;

		if (volumes[0] < 0)
			volumes[0] = 0;
		if (volumes[0] > 100)
			volumes[0] = 100;
	}

	if (channels == BOTH_CHANNELS || channels == RIGHT_CHANNEL)
	{	
		if (!absolute)
			volumes[1] += amount;
		else
			volumes[1] = amount;

		if (volumes[1] < 0)
			volumes[1] = 0;
		if (volumes[1] > 100)
			volumes[1] = 100;
	}
	setting = (volumes[1]<<8) + volumes[0];
	ioctl(mixer, MIXER_WRITE(supported[bar]), &setting);
	draw_scrollbar(bar, currentspos);
}

void
redraw_all_bars()
{
	int i;

	for (i = 0; i < (maxspos + 1); i++)
		draw_scrollbar(minbar + i, i);
}
	
