/*
    TkMix v1.2: An X11 audio mixer for OSS.
    Copyright (C) 1996 Mike Welch (mike@tara.demon.co.uk)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <tcl.h>
#include <tk.h>
#include <linux/soundcard.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>

#include "tkmix.h"

extern char menu_tcl[];

void usage()
{
	fprintf (stderr,"Usage: tkmix [options]\n");
	fprintf (stderr,"Options:\n");
	fprintf (stderr,"  -h          Disply usage information\n");
	fprintf (stderr,"  -p          Don't poll mixer device\n");
	fprintf (stderr,"  -v          Volume only\n");
#ifdef PAS
	fprintf (stderr,"  -f          Toggle Stereo Field controls (PAS only)\n");
	fprintf (stderr,"  -l          Toggle Loudness controls (PAS only)\n");
	fprintf (stderr,"  -m          Toggle Mute controls (PAS only)\n");
#endif
}

int parse_args(int argc,char *argv[], struct MixerData *mix_dat)
{
	int i;

	mix_dat->flags=TKM_DEFAULT_FLAGS;

	for (i=1;i<argc;i++) {
		if (!strcmp(argv[i], "-h")) {
			/* Usage info */
			usage();
			exit(1);
		}

		if (!strcmp(argv[i], "-p")) {
			/* No poll */
			mix_dat->flags^=TKM_POLL;
			continue;
		}

		if (!strcmp(argv[i], "-v")) {
			/* Volume only */
			mix_dat->flags^=TKM_VOLUME_ONLY;
			continue;
		}

		if (!strcmp(argv[i], "-f")) {
			/* Toggle field controls */
			mix_dat->flags^=TKM_ENHANCE;
			continue;
		}

		if (!strcmp(argv[i],"-m")) {
			/* Toggle mute control */
			mix_dat->flags^=TKM_MUTE;
			continue;
		}

		if (!strcmp(argv[i],"-l")) {
			/* Toggle loudness control */
			mix_dat->flags^=TKM_LOUDNESS;
			continue;
		}

		fprintf(stderr,"Unknown option: %s\n",argv[i]);
		usage();
		exit(1);
	}
}

int GetMixerInfo(struct MixerData *mix_dat) 
{
	/* A hack for the time being */
	/* HTF do you find out what mixer /dev/mixer points to? */
#ifdef PAS
	mix_dat->card=SNDCARD_PAS;
#endif

    /* Open the Mixer */
    if ((mix_dat->mixer_fd = open(TKM_MIXER_DEVICE, O_RDWR)) <0) {
		fprintf(stderr, "Error opening %s\n",TKM_MIXER_DEVICE);
		return(0);
    }

    /* Read bit mask of available devices */
    if (ioctl(mix_dat->mixer_fd, SOUND_MIXER_READ_DEVMASK, &mix_dat->devmask) == -1) {
		fprintf(stderr,"Error obtaining DEVMASK\n");
		return(0);
    }

#ifdef DEBUG
	printf("DevMask:%x\n",mix_dat->devmask);
#endif

    /* Read bit mask of recording devices */
    if (ioctl(mix_dat->mixer_fd, SOUND_MIXER_READ_RECMASK, &mix_dat->recmask) == -1) {
		fprintf(stderr,"Error obtaining RECMASK\n");
		return(0);
    }

#ifdef DEBUG
	printf("RecMask:%x\n",mix_dat->recmask);
#endif

    /* Read bit mask of stereo devices */
    if (ioctl(mix_dat->mixer_fd, SOUND_MIXER_READ_STEREODEVS, &mix_dat->stereodevs) == -1) {
		fprintf(stderr,"Error obtaining STEREODEVS\n");
		return(0);
    }

#ifdef DEBUG
	printf("StereoMask:%x\n",mix_dat->stereodevs);
#endif

	/* Enable Extra features on a per card basis */
	switch(mix_dat->card) {
	case SNDCARD_PAS:
		break;
	default:
		mix_dat->flags&= ~(TKM_ENHANCE|TKM_MUTE|TKM_LOUDNESS);
	}

	return(1);
}

int main(int argc, char *argv[])
{
    
    struct MixerData mix_dat;
    struct Tcl_DString *dstring;
#ifdef DEBUG
    FILE *tclfile;
#endif

	/* Parse command line args */
	if (!parse_args(argc,argv,&mix_dat)) {
		usage();
		exit(1);
	}

	if (!GetMixerInfo(&mix_dat)) {
		exit(1);
	}

    /* Make our Tcl interpreter */
    mix_dat.interp = Tcl_CreateInterp();

    /* Call our AppInit procedures */
	if (Tcl_MixAppInit (&mix_dat) != TCL_OK) {
		fprintf(stderr,"Tcl_AppInit failed\n");
		exit(1);
    }

#ifdef DEBUG
    tclfile=fopen("gui.tcl","w");
    fprintf(tclfile,"%s\n",menu_tcl);
#endif

    /* Build and evaluate the Tcl script to layout the GUI */
    if (Tcl_Eval(mix_dat.interp,menu_tcl) == TCL_ERROR) {
		fprintf(stderr,"Can't layout menu bar\n");
		exit(1);
    }
    if (!(dstring=(struct Tcl_DString *)malloc(sizeof(struct Tcl_DString)))) {
		fprintf(stderr,"Error mallocing Tcl_DString\n");
		exit(1);
    }
	Tcl_DStringInit(dstring);
	if (BuildTclScript(&mix_dat,dstring) == TCL_ERROR) {
		fprintf(stderr,"Error building Tcl script\n");
		exit(1);
    }

#ifdef DEBUG
    fprintf(tclfile,"%s\n",Tcl_DStringValue(dstring));
    fclose(tclfile);
#endif

    if (Tcl_Eval(mix_dat.interp, Tcl_DStringValue(dstring)) == TCL_ERROR) {
		fprintf(stderr,"%s\n",Tcl_DStringValue(dstring));
		fprintf(stderr,"Malformed GUI definition (mail the code above to the author)\n");
		exit(1);
    }
    Tcl_DStringFree(dstring);
    free(dstring);

	init_controls(&mix_dat);

    /* Let Tcl sort the rest out */
    Tk_MainLoop();

    /* Time to leave now, call exit in case we've re-defined it to
       do something cool */
    Tcl_Eval(mix_dat.interp,"exit");
    exit(1);
}
