/*
    TkMix v1.2: An X11 audio mixer for OSS.
    Copyright (C) 1996 Mike Welch (mike@tara.demon.co.uk)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/ 

#include <linux/soundcard.h>
#include <tcl.h>
#include <tk.h>
#include "tkmix.h"

/*
 * Produce some Tcl code in dstring to update all the widgets to 
 * what the mixer devices says they should be
 */
int update_all(struct MixerData *mix_dat,struct Tcl_DString *dstring)
{
    int i,temp;
    char buffer[200];
    char *devnames[SOUND_MIXER_NRDEVICES] = SOUND_DEVICE_LABELS;

    /* Read bit mask of recording sources */
    if (ioctl(mix_dat->mixer_fd, SOUND_MIXER_READ_RECSRC, &mix_dat->recsrc) == -1) {
		fprintf(stderr,"Error obtaining RECSRC\n");
		exit(1);
    }

	/* Set the value of the mixer devices */
    for(i=0;i<=SOUND_MIXER_NRDEVICES;i++) {
		if((1 << i) & mix_dat->devmask) {

			/* Set the green indicator if the device is selected */
			/* for recording */
			if ((1 << i) & mix_dat->recsrc) {
   				sprintf(buffer, ".top.levframe%d.recbut%d select\n",i,i);
   				Tcl_DStringAppend(dstring, buffer,-1);
	   	 	} else {
   				sprintf(buffer, ".top.levframe%d.recbut%d deselect\n",i,i);
	   	 		Tcl_DStringAppend(dstring, buffer,-1);
	   	 	}

			/* Read the value of the mixer */
			if(ioctl(mix_dat->mixer_fd,MIXER_READ(i),&temp) == -1) {
				sprintf(buffer,"Unable to read status of device %s\n",i);
				Tcl_SetResult(mix_dat->interp,buffer,TCL_VOLATILE);
				return(TCL_ERROR);
	    	}

			/* Set the volume scale */
	    	if ((1 << i) & mix_dat->stereodevs) {
				/* It's a stereo device so do both channels */
	    		sprintf(buffer, ".top.levframe%d.topframe.levell set %d\n",i,(temp & 0x7F));
	    		Tcl_DStringAppend(dstring, buffer,-1);
	    		sprintf(buffer, ".top.levframe%d.topframe.levelr set %d\n",i,((temp >> 8)  & 0x7F));
	    		Tcl_DStringAppend(dstring, buffer,-1);
	    	} else {
				/* It's a mono device */
	    		sprintf(buffer, ".top.levframe%d.levelm set %d\n",i,(temp & 0x7F));
	    		Tcl_DStringAppend(dstring, buffer,-1);
	    	}
		}
    }

    /* Update widgets for Loudness, Mute and Enhance*/

	/* Do Mute first */
    if((mix_dat->flags & TKM_MUTE)) {
    	temp=-1;
    	if(ioctl(mix_dat->mixer_fd,SOUND_MIXER_PRIVATE3,&temp) == -1) {
			Tcl_SetResult(mix_dat->interp,"Unable to get MUTE status",TCL_VOLATILE);
	    	return(TCL_ERROR);
		}

		if (temp) {
    	    sprintf(buffer, ".bottom.mute.mbutton select\n");
		} else {
    	    sprintf(buffer, ".bottom.mute.mbutton deselect\n");
		}
   	    Tcl_DStringAppend(dstring, buffer,-1);
   	}

	/* Do Loudness */
    if(mix_dat->flags & TKM_LOUDNESS) {
    	temp=-1;
		if(ioctl(mix_dat->mixer_fd,SOUND_MIXER_PRIVATE1,&temp) == -1) {
			Tcl_SetResult(mix_dat->interp,"Unable to get LOUDNESS status",TCL_VOLATILE);
		    return(TCL_ERROR);
		}
        if (temp) {
            sprintf(buffer, ".bottom.loud.lbutton select\n");
        } else {
            sprintf(buffer, ".bottom.loud.lbutton deselect\n");
        } 
    	Tcl_DStringAppend(dstring, buffer,-1);
    }

	/* Do Enhance */
    if(mix_dat->flags &TKM_ENHANCE) {
    	temp=-1;
		if(ioctl(mix_dat->mixer_fd,SOUND_MIXER_PRIVATE2,&temp) == -1) {
			Tcl_SetResult(mix_dat->interp,"Unable to get ENHANCE status",TCL_VOLATILE);
		    return(TCL_ERROR);
		}
		if (temp==0) {
			sprintf(buffer, "set enhance None");
		} else {
    		sprintf(buffer, "set enhance %d%%\n",temp);
		}
    	Tcl_DStringAppend(dstring, buffer,-1);
    }

	return (TCL_OK);
}

/*
 * Called once every TKM_POLL_INTERVAL by Tcl 
 */
void PollMixer(struct MixerData *mix_dat)
{
	struct Tcl_DString dstring;

	Tcl_DStringInit(&dstring);

	if(update_all(mix_dat,&dstring)==TCL_ERROR) {
		Tk_BackgroundError(mix_dat->interp);
	} else {
		Tcl_Eval(mix_dat->interp, Tcl_DStringValue(&dstring));
		Tcl_DStringFree(&dstring);
		Tk_CreateTimerHandler(TKM_POLL_INTERVAL, (Tk_TimerProc *)PollMixer, mix_dat);
	}
}

/*
 * Initialise the controls and call the polling routine if needed
 */
void init_controls(struct MixerData *mix_dat)
{
	struct Tcl_DString *dstring;
	/* FXIME */
	/* This is where we'll load the settings from a file */

	if (!(dstring=(struct Tcl_DString *)malloc(sizeof(struct Tcl_DString)))) {
		fprintf(stderr,"Error mallocing Tcl_DString\n");
		exit(1);
	}

	Tcl_DStringInit(dstring);
	if (update_all(mix_dat,dstring)==TCL_ERROR) {
		Tk_BackgroundError(mix_dat->interp);
	} else {
		Tcl_Eval(mix_dat->interp, Tcl_DStringValue(dstring));
	}

	Tcl_DStringFree(dstring);
	free(dstring);

	if (mix_dat->flags & TKM_POLL) {
		Tk_CreateTimerHandler(TKM_POLL_INTERVAL, (Tk_TimerProc *)PollMixer, mix_dat);
	}
}
