
/*
    xfmix - audio mixer
    Copyright (C) 1995  Radek Doulik

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <forms.h>
#include "light.h"
#include "xfmix.xpm"

#include <xpm.h>

#include <signal.h>
#include <unistd.h>
#include <sys/time.h>
#include <stdio.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>

FL_FORM  *mix_form, *about_form, *help_form;
FL_OBJECT *master_lock, *active_box;

#define LEFT  (1<<24)
#define RIGHT (1<<25)
#define PAD   100

struct mixer {
  int handle;

  int devs;

  int devmask;
  int stereomask;
  int recmask;
  int recsrc;
} mix;

struct mix_dev {
  FL_OBJECT *l, *r;
  FL_OBJECT *fl_rec;
  FL_OBJECT *fl_mute;
  FL_OBJECT *fl_lock;

  int left, right;                                       /* values on L&R channels */
  int muted_l, muted_r;
  int stereo;
  int dev_num;
  int dev_bit;
} *mdevs;

int sig_alarm=0;
static struct sigaction sig_alarm_action;
static struct itimerval sig_alarm_delay={
  {0, 33333},
  {0, 33333},
};

char *err_msg[]={ "Cannot open mixer",
		  "Error reading from mixer",
		  "No mixer devices found",
		  "Error writing to mixer",
		  "\nTry xfmix -h for help\n",
		  "\nxfmix (c) Radek Doulik 1995\n\n"
		  "usage :\n\n"
		  "\t-l ... lock all channels\n"
		  "\t-a ... starts in nonactive mode\n"
		  "\t-h ... displays this message\n"
		};

int start_locked=0,
    start_nonactive=0;

void end_error(int err) {
  fprintf(stderr, "%s\n", err_msg[err-1]);
  exit(err);
}

char *dev_name="/dev/mixer";
char *dev_labels[]=SOUND_DEVICE_LABELS;

void get_mix_info() {                                     /* open mixer and look for capabilities */

  if ((mix.handle=open(dev_name, O_RDWR))<0) end_error(1);

  if (ioctl(mix.handle, SOUND_MIXER_READ_DEVMASK, &mix.devmask)==-1) end_error(2);
  if (ioctl(mix.handle, SOUND_MIXER_READ_STEREODEVS, &mix.stereomask)==-1) end_error(2);
  if (ioctl(mix.handle, SOUND_MIXER_READ_RECMASK, &mix.recmask)==-1) end_error(2);

  if (!mix.devmask) end_error(3);
}

void set_volume(int dev_num, int left, int right) {
  unsigned char temp[4];

  temp[0]=(unsigned char)left;
  temp[1]=(unsigned char)right;
  temp[2]=temp[3]=0;
                                                          /* send it to mixer */
  ioctl(mix.handle, MIXER_WRITE(dev_num), temp);
}

void update_mixer_rec() {
  int i,j, k;

  if (ioctl(mix.handle, SOUND_MIXER_READ_RECSRC, &mix.recsrc)==-1) end_error(2);

  for (i=0; i<mix.devs; i++) {
    j=mdevs[i].dev_bit;
    if ((mix.devmask & j) && (mix.recmask & j)) {
      k=fl_get_lightbox(mdevs[i].fl_rec);
      if ((mix.recsrc & j)!=(k<<(mdevs[i].dev_num)))
	fl_set_lightbox(mdevs[i].fl_rec, mix.recsrc & j);
    }
  }
}

void update_mixer_values() {
  int i, t1, t2;
  unsigned char temp[4];

  for (i=0; i<mix.devs; i++) {
    ioctl(mix.handle, MIXER_READ(mdevs[i].dev_num), &temp);
    t1=PAD*temp[0]/PAD;
    if (mdevs[i].left!=t1) {
      mdevs[i].left=t1;
      fl_set_slider_value(mdevs[i].l, (double)temp[0]/PAD);
    }

    if (mdevs[i].stereo) {
      t2=PAD*temp[1]/PAD;
      if (mdevs[i].right!=t2) {
	mdevs[i].right=t2;
	fl_set_slider_value(mdevs[i].r, (double)temp[1]/PAD);
      }
    }
  }
}

void sig_alarm_handler(int n) {                                 /* app timer */
  sig_alarm=1;
};

int mix_idle_cb(XEvent *xe, void *v) {                  /* in idle time update meter */
  if (!sig_alarm) return 1;

  update_mixer_values();
  update_mixer_rec();

  sig_alarm=0;

  return 1;
}

void lock_dev(int i, int ns) {
  if (ns) {
    int m=fl_get_lightbox(mdevs[i].fl_mute);

    mdevs[i].left=mdevs[i].right=(mdevs[i].left+mdevs[i].right)/2;
    set_volume(mdevs[i].dev_num, mdevs[i].left, mdevs[i].right);
    fl_set_slider_value(mdevs[i].l, (m) ? 0 : (double)mdevs[i].left/PAD);
    if (mdevs[i].stereo) fl_set_slider_value(mdevs[i].r, (m) ? 0 : (double)mdevs[i].right/PAD);
  }
  fl_set_lightbox(mdevs[i].fl_lock, ns);
  fl_set_object_color(mdevs[i].l, FL_COL1, (ns) ? FL_INACTIVE : FL_RED);
  if (mdevs[i].stereo) fl_set_object_color(mdevs[i].r, FL_COL1, (ns) ? FL_INACTIVE : FL_BLUE);
}

void mix_master_lock_cb(FL_OBJECT *ob, long l) {          /* callback for master lock */
  int ns=fl_get_lightbox(ob), i;

  for (i=0;i<mix.devs;i++) {
    if (mdevs[i].stereo)
      if (fl_get_lightbox(mdevs[i].fl_lock)!=ns)
        lock_dev(i, ns);
  }
}

void mix_master_mute_cb(FL_OBJECT *ob, long l) {          /* callback for master mute */
  int st=1-fl_get_lightbox(ob);
  fl_set_lightbox(ob, st);  
  set_volume(SOUND_MIXER_MUTE, st, 0);
}

void mix_mute_cb(FL_OBJECT *ob, long data) {              /* callback for channel mute */
  int st=fl_get_lightbox(ob);
  int l, r;
  
  if (st) {
    mdevs[data].muted_l=mdevs[data].left;
    mdevs[data].muted_r=mdevs[data].right;
    set_volume(mdevs[data].dev_num, l=0, r=0); 
  } else {
    mdevs[data].left=mdevs[data].muted_l;
    mdevs[data].right=mdevs[data].muted_r;
    set_volume(mdevs[data].dev_num, l=mdevs[data].left, r=mdevs[data].right);
  }

  fl_set_slider_value(mdevs[data].l, (double)l/PAD);
  if (mdevs[data].stereo) fl_set_slider_value(mdevs[data].r, (double)r/PAD);
}

void mix_slider_cb(FL_OBJECT *ob, long data) {            /* callback for mixer sliders */
  int vol, idx=data & 255;
  double v;
  XEvent *xe=(XEvent*)fl_last_event();
  int b1=(xe->type==MotionNotify) ? ((((XMotionEvent*)xe)->state) & Button1Mask)
                                  : ((((XButtonEvent*)xe)->button)==Button1);

  v=fl_get_slider_value(ob);                              /* get current volume */
  vol=(int)(PAD*v);
  
  if (mdevs[idx].stereo) b1|=fl_get_lightbox(mdevs[idx].fl_lock);
  else b1=1;

  if (b1) {
    if (data & LEFT) mdevs[idx].left=vol;
    else mdevs[idx].right=vol;
  } else {                                                /* other than 1st button */
    int dv=0;                                             /* change both sliders */

    if (data & LEFT) {
      dv=vol-mdevs[idx].left;
      mdevs[idx].left=vol;
      mdevs[idx].right+=dv;
      if (mdevs[idx].right > PAD) mdevs[idx].right=PAD;
      else if (mdevs[idx].right<0) mdevs[idx].right=0;
      fl_set_slider_value(mdevs[idx].r, (double)mdevs[idx].right/PAD);
    } else {
      dv=vol-mdevs[idx].right;
      mdevs[idx].right=vol;
      mdevs[idx].left+=dv;
      if (mdevs[idx].left > PAD) mdevs[idx].left=PAD;
      else if (mdevs[idx].left<0) mdevs[idx].left=0;
      fl_set_slider_value(mdevs[idx].l, (double)mdevs[idx].left/PAD);
    }
  }

  if (mdevs[idx].stereo)
    if (fl_get_lightbox(mdevs[idx].fl_lock)) {              /* locking feature */
      if (data & LEFT) {
        mdevs[idx].right=vol;
        fl_set_slider_value(mdevs[idx].r, v);
      } else {
        mdevs[idx].left=vol;
        fl_set_slider_value(mdevs[idx].l, v);
      }
  }

  set_volume(mdevs[idx].dev_num, mdevs[idx].left, mdevs[idx].right);
  if ((mdevs[idx].left>0 || mdevs[idx].right>0) && fl_get_lightbox(mdevs[idx].fl_mute))
    fl_set_lightbox(mdevs[idx].fl_mute, 0);
}

void mix_lock_cb(FL_OBJECT *ob, long data) {             /* callback for locking */
  int ns=fl_get_lightbox(ob);

  lock_dev(data, ns);
  if (!ns && fl_get_lightbox(master_lock))
    fl_set_lightbox(master_lock, 0);
}

void mix_rec_cb(FL_OBJECT *ob, long data) {              /* callback for recording */

  if (fl_get_lightbox(ob)) {
    mix.recsrc|=mdevs[data].dev_bit;
  } else mix.recsrc&=(0xffffffff ^ mdevs[data].dev_bit);

  if (ioctl(mix.handle, SOUND_MIXER_WRITE_RECSRC, &mix.recsrc)==-1) end_error(4);

  update_mixer_rec();
}

void mix_end_cb(FL_OBJECT *o, long l) {
  exit(0);
}

void about_end_cb(FL_OBJECT *o, long l) {
  FL_FORM *f=(FL_FORM*)l;

  fl_hide_form(f);
  fl_free_form(f);
  fl_activate_all_forms();
}

void mix_help_cb(FL_OBJECT *o, long l) {
  FL_OBJECT *ob;

  fl_deactivate_all_forms();

  help_form = fl_bgn_form(FL_UP_BOX, 260, 150);

  fl_add_text(FL_NORMAL_TEXT,20,25,80,90,
	      "use 'l' to lock / unlock all channels\n"
	      "      'm' to mute/unmute master (if supported)\n"
	      "use left mouse button to change single channel\n"
	      "      and right to change both"
	      );
  fl_add_frame(FL_DOWN_FRAME,10,40,240,95,"");
  ob=fl_add_text(FL_NORMAL_TEXT,10,95,65,40,"help");
    fl_set_object_lsize(ob,FL_HUGE_SIZE);
    fl_set_object_lstyle(ob,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
  ob=fl_add_text(FL_NORMAL_TEXT,185,95,55,40,"???");
    fl_set_object_lsize(ob,FL_HUGE_SIZE);
    fl_set_object_lstyle(ob,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
  ob=fl_add_button(FL_NORMAL_BUTTON,185,10,65,20,"Close");
    fl_set_object_callback(ob, about_end_cb, (long)help_form);
  fl_end_form();

  fl_show_form(help_form, FL_PLACE_MOUSE, FL_TRANSIENT, "xfmix help");
}

void mix_about_cb(FL_OBJECT *o, long l) {
  FL_OBJECT *ob;

  fl_deactivate_all_forms();

  about_form = fl_bgn_form(FL_UP_BOX, 220, 140);

  fl_add_box(FL_UP_BOX,0,0,220,140,"");
  fl_add_frame(FL_DOWN_FRAME,10,40,200,90,"");
  fl_add_text(FL_NORMAL_TEXT,10,50,200,35,
	      "xforms based mixer      for Linux/X11R6\n\n"
	      "(c) 1995 R.Doulik     rodo@earn.cvut.cz");
  ob=fl_add_text(FL_NORMAL_TEXT,10,95,90,30,"xfmix");
    fl_set_object_lsize(ob, FL_HUGE_SIZE);
    fl_set_object_lstyle(ob, FL_NORMAL_STYLE+FL_SHADOW_STYLE);
  fl_add_text(FL_NORMAL_TEXT,105,95,100,20,"version 0.2");
  ob=fl_add_button(FL_NORMAL_BUTTON,145,10,65,20,"Close");
    fl_set_object_callback(ob, about_end_cb, (long)about_form);

  fl_end_form();

  fl_show_form(about_form, FL_PLACE_MOUSE, FL_TRANSIENT, "xfmix about");
}

void mix_active_cb(FL_OBJECT *ob, long l) {
  int st=fl_get_lightbox(ob);

  if (st) {
    fl_set_idle_callback(mix_idle_cb, NULL);
  } else {
    fl_set_idle_callback(NULL, NULL);
  }
}

void create_form_mix() {
  FL_OBJECT* ob;
  unsigned int x,i,j,k;

  get_mix_info();

  for (x=i=mix.devs=0, j=1; i<20; i++, j<<=1)             /* count form width */
    if (mix.devmask & j) {
      x+=45;
      mix.devs++;
    }

  mdevs=(struct mix_dev*)malloc(sizeof(struct mix_dev)*mix.devs);

  fl_set_border_width(-2);
  mix_form=fl_bgn_form(FL_FLAT_BOX, x+75, 260);

  x+=15;
  ob=fl_add_button(FL_NORMAL_BUTTON, x, 53, 50, 20, "Quit");
    fl_set_object_callback(ob, mix_end_cb, 0);
    fl_set_object_lstyle(ob,FL_BOLD_STYLE);
  ob=fl_add_button(FL_NORMAL_BUTTON, x, 232, 50, 20, "Help");
    fl_set_object_callback(ob, mix_help_cb, 0);
    fl_set_object_lstyle(ob,FL_BOLD_STYLE);
  ob=fl_add_button(FL_NORMAL_BUTTON, x, 207, 50, 20, "About");
    fl_set_object_callback(ob, mix_about_cb, 0);
    fl_set_object_lstyle(ob,FL_BOLD_STYLE);
  active_box=ob=fl_add_lightbox(FL_NORMAL_LIGHTBOX, x+2, 170, 50, 8, "active");
    fl_set_object_callback(ob, mix_active_cb, 0);
    fl_set_object_lstyle(ob,FL_BOLD_STYLE);
    fl_set_lightbox(ob, 1);

  if (mix.devmask & (1<<SOUND_MIXER_MUTE)) {
    ob=fl_add_lightbox(FL_NORMAL_LIGHTBOX, x+5, 25, 45, 8, "mute");
      fl_set_object_callback(ob, mix_master_mute_cb, 0);
      fl_set_object_shortcut(ob, "mM", 0);
      fl_set_object_lstyle(ob,FL_BOLD_STYLE);
  }

  master_lock=ob=fl_add_lightbox(FL_NORMAL_LIGHTBOX, x+5, 10, 45, 8, "lock");
    fl_set_object_color(ob, FL_COL1, FL_WHITE); 
    fl_set_object_callback(ob, mix_master_lock_cb, 0);
    fl_set_object_shortcut(ob, "lL", 0);
    fl_set_object_lstyle(ob,FL_BOLD_STYLE);

  for (i=k=0, j=1, x=10; i<20; i++, j<<=1) {
    if (mix.devmask & j) {
      mdevs[k].stereo=((mix.stereomask & j)!=0);
      mdevs[k].dev_num=i;
      mdevs[k].dev_bit=j;
      mdevs[k].left=mdevs[k].right=-1;

      ob=fl_add_text(FL_NORMAL_TEXT, x, 52, 40, 20, dev_labels[i]);
        fl_set_object_lstyle(ob,FL_BOLD_STYLE);
      fl_set_border_width(-1);
      fl_add_frame(FL_DOWN_FRAME,x,55, 40, 195,"");
      fl_set_border_width(-2);

      if (mix.recmask & j) {
	mdevs[k].fl_rec=ob=fl_add_lightbox(FL_NORMAL_LIGHTBOX, x, 40, 45, 8, "rec");
	  fl_set_object_color(ob, FL_COL1, FL_RED); 
          fl_set_object_callback(ob, mix_rec_cb, k);
	  fl_set_object_lstyle(ob,FL_BOLD_STYLE);
      }
      mdevs[k].fl_mute=ob=fl_add_lightbox(FL_NORMAL_LIGHTBOX, x, 25, 45, 8, "mute");
        fl_set_object_callback(ob, mix_mute_cb, k);
        fl_set_object_lstyle(ob,FL_BOLD_STYLE);

      if (mdevs[k].stereo) {
        mdevs[k].fl_lock=ob=fl_add_lightbox(FL_NORMAL_LIGHTBOX, x, 10, 45, 8, "lock");
          fl_set_object_callback(ob, mix_lock_cb, k);
          fl_set_object_color(ob, FL_COL1, FL_WHITE); 
	  fl_set_object_lstyle(ob,FL_BOLD_STYLE);

	mdevs[k].r=ob=fl_add_valslider(FL_VERT_NICE_SLIDER,x+20,70,20,180,"");
	  fl_set_object_boxtype(ob, FL_FLAT_BOX);
	  fl_set_object_color(ob, FL_COL1,
			      (fl_get_lightbox(mdevs[k].fl_lock)) ? FL_INACTIVE : FL_BLUE);
          fl_set_object_callback(ob, mix_slider_cb, k | RIGHT);
          fl_set_slider_bounds(ob, 1, 0);
	  fl_set_object_lstyle(ob,FL_BOLD_STYLE);
      }

      mdevs[k].l=ob=fl_add_valslider(FL_VERT_NICE_SLIDER,x+((mdevs[k].stereo) ? 0 : 5),70, (mdevs[k].stereo) ? 20 : 30, 180, "");
        fl_set_object_boxtype(ob, FL_FLAT_BOX);
        fl_set_object_color(ob, FL_COL1,
			     (mdevs[k].stereo) ? ((fl_get_lightbox(mdevs[k].fl_lock)) ? FL_INACTIVE : FL_RED) : FL_INACTIVE);
        fl_set_object_callback(ob, mix_slider_cb, k | LEFT);
        fl_set_slider_bounds(ob, 1, 0);
        fl_set_object_lstyle(ob,FL_BOLD_STYLE);

      x+=45;
      k++;
    }
  }

  fl_end_form();

  update_mixer_values();
  update_mixer_rec();
}

void resolve_options(int n, char *argv[]) {
  int o;

  while ((o=getopt(n, argv, "lah"))>=0) {
    switch (o) {
    case 'l':
      start_locked=1;
      break;
    case 'a':
      start_nonactive=1;
      break;
    case 'h':
      end_error(6);
      break;
    case '?':
      end_error(5);
      break;
    }
  }
}

void check_options() {
  if (start_locked) {
    fl_set_lightbox(master_lock, 1);
    mix_master_lock_cb(master_lock, 0);
  }
  if (start_nonactive) {
    fl_set_lightbox(active_box, 0);
    mix_active_cb(active_box, 0);
  }
}

int main(int argc, char *argv[]) {
  Pixmap icon, icon_shape;
  XWMHints *wm_hints;

  resolve_options(argc, argv);

  /* initialize forms library */
  fl_initialize(argv[0], "xfmix", 0, 0, &argc, argv);
  
  /* create main window form */   
  create_form_mix();
  
  /* now we can begin */
  fl_show_form(mix_form, FL_PLACE_FREE, FL_FULLBORDER, "Mixer");

  /* set my icon */
  XpmCreatePixmapFromData(fl_get_display(), mix_form->window, xfmix_icon, &icon, &icon_shape,
			  NULL);

  wm_hints=XAllocWMHints();
  wm_hints->flags=IconPixmapHint | IconMaskHint;
  wm_hints->icon_pixmap=icon;
  wm_hints->icon_mask=icon_shape;
  XSetWMHints(fl_get_display(), mix_form->window, wm_hints);

  sig_alarm_action.sa_handler=sig_alarm_handler;
  sig_alarm_action.sa_flags=SA_NOMASK;
  sig_alarm_action.sa_restorer=NULL;
    
  sigaction(SIGALRM, &sig_alarm_action, 0);
  setitimer(ITIMER_REAL, &sig_alarm_delay, 0);

  fl_set_idle_callback(mix_idle_cb, NULL);

  check_options();

  while(1) fl_do_forms();         /* do it forever :) */

  XFree(wm_hints);
}
