/*
  cddaplay.c  --  play back a cdda stream
  Copyright (C) 1998-2000 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 *	$Id: cddaplay.c,v 1.3 2000/12/20 20:58:18 solyga Exp $
 */

#include	"cddaplay.h"


int
display_help( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s): ", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( HELP_CHANNEL, "play back cdda file.\n" );
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] [file]\n", pn );
  fprintf( HELP_CHANNEL, "switches:\n" );
  fprintf( HELP_CHANNEL, "  -h\t write this info to %s and exit sucessfully\n", HELP_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -v\t raise verbosity level on %s by one\n",VERBOSE_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -V\t display version and compilation info on %s\n",VERSION_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -x\t swap byte order\n" );
  fprintf( HELP_CHANNEL, "controllers:\n");
  fprintf( HELP_CHANNEL, "  -D STRING  \t dsp device name (default is %s)\n", DEFAULT_DSP_DEVICE );
  fprintf( HELP_CHANNEL, "  -F STRING  \t frames to play, frames or msf (default is 0 to EOF)\n" );
  fprintf( HELP_CHANNEL, "  -S NUMBER  \t sample speed in Hz out of [%d,%d] (default is %d)\n", MIN_DSP_SPEED, MAX_DSP_SPEED, DEFAULT_DSP_SPEED );
  return( 0 );
}


int
display_version( char* pn ) {
  fprintf( VERSION_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( VERSION_CHANNEL, "Compilation settings:\n" );
  fprintf( VERSION_CHANNEL, "  DEFAULT_DSP_SPEED      : %d\n", DEFAULT_DSP_SPEED );
  fprintf( VERSION_CHANNEL, "  DEFAULT_DSP_SAMPLESIZE : %d\n", DEFAULT_DSP_SAMPLESIZE );
  fprintf( VERSION_CHANNEL, "  DEFAULT_DSP_DEVICE     : %s\n", DEFAULT_DSP_DEVICE );
  fprintf( VERSION_CHANNEL, "  DEFAULT_DSP_STEREO     : %d\n", DEFAULT_DSP_STEREO );
  fprintf( VERSION_CHANNEL, "  DEFAULT_SWAP           : %d\n", DEFAULT_SWAP );
  return( 0 );
}


long getlongoptarg( char opt, char* arg,long vmn, long vmx, char* fn ) {
  char *p;
  long val;
  val=strtol(arg,&p,0);
  if(p==arg || *p!='\0')
  { /* nothing read */
    fprintf(ERROR_CHANNEL,"%s: Argument of option -%c (`%s') must be a number.\n",
            fn,opt,arg);
    exit(1);
  }
  if(val<vmn||val>vmx)
  { /* out of interval */
    fprintf(ERROR_CHANNEL,"%s: Argument value of option -%c (%ld) must be in [%ld,%ld].\n",
            fn,opt,val,vmn,vmx);
    exit(1);
  }
  return(val);
}


ssize_t
my_read( int fd, void* buf, size_t count ) {
/*
 * (attempts to) read exactly count bytes from fd into buf
 * returns number of bytes read or -1 on error
 * retval < count indicates EOF (or EAGAIN when non-blocking)
 * started 1998-01-01
 */
  unsigned char* p= buf;
  ssize_t nbr;
  ssize_t tnbr= 0;
  size_t rem= count;
  do {
    if( (nbr=read(fd,p+tnbr,rem)) == -1 ) {
      if( errno == EAGAIN ) return( tnbr );
      else                  return( -1 );
    }
    tnbr+= nbr;
    rem-= nbr;
  } while( nbr>0 && rem>0 );
  return( tnbr );
}


char*
str_dsc( char* str1, char* str2) {
/*
returns pointer to first char of str1 not contained in str2
started 1997-02-09
*/
  char *p1, *p2;
  for ( p1=str1; ; p1++ ) {
    if ( *p1 == '\0' ) return( p1 );
    for ( p2=str2; ; p2++ ) {
      if ( *p2 == '\0' ) return( p1 );
      if ( *p2 == *p1 ) break;
    }
  }
}


/*
 * 	$Id: cddaplay.c,v 1.3 2000/12/20 20:58:18 solyga Exp $
 */
int
str2fra( char* s, char** es ) {
/*
 * convert string to signed frames
 * return INT_MAX if value too large
 * *es == s indicates nothing read
 * no hex values allowed
 * started 2000-12-17
 */
  char* p= s;
  char* pold;
  int min= 0;
  int sec= 0;
  int fra= 0;
  int sign= 1;
  if( s == NULL ) {
    if( es != NULL ) *es= s;
    return( 0 );
  }
  /* get sign */
  p= str_dsc( p, " \t" );
  if( *p == '-' ) {
    sign= -1;
    p++;
  }
  /* assuming to read frames */
  p= str_dsc( p, " \t" );
  if( *p != ':' ) {
    pold= p;
    fra= strtol( p, &p, 10 );
    if( p == pold ) { /* nothing read ==> error */
      if( es != NULL ) *es= s;
      return( 0 );
    }
    if( *p != ':' ) goto STR2FRA_RETURN;
  }
  p++;          /* skip the ':' */
  /* were seconds (or minutes), assuming again to read frames */
  sec= fra; fra= 0;
  if( *p == '-'  ||  *p == '+' ) goto STR2FRA_RETURN;
  if( *p != ':' ) {
    pold= p;
    fra= strtol( p, &p, 10 );
    if( p == pold ) { /* nothing read ==> shortcut for 0 ==> done */
      fra= 0;
      goto STR2FRA_RETURN;
    }
    if( *p != ':' ) goto STR2FRA_RETURN;
  }
  p++;          /* skip the ':' */
  /* were mm:ss:, assuming again to read frames */
  min= sec; sec= fra; fra= 0;
  if( *p == '-'  ||  *p == '+' ) goto STR2FRA_RETURN;
  pold= p;
  fra= strtol( p, &p, 10 );
  if( p == pold ) { /* nothing read ==> shortcut for 0 ==> done */
    fra= 0;
    goto STR2FRA_RETURN;
  }
STR2FRA_RETURN:
  if( fra < 0  ||  sec < 0  ||  min < 0 ) goto STR2FRA_ERROR;
  /* max is INT_MAX/CD_FRAMESIZE_RAW = 913045 = 202:53:70 */
  if( min > 200 ) goto STR2FRA_ERROR;
  min+= sec/CD_SECS; sec%= CD_SECS;
  sec+= fra/CD_FRAMES; fra%= CD_FRAMES;
  min+= sec/CD_SECS; sec%= CD_SECS;
  if( min > 200 ) goto STR2FRA_ERROR;
  if( es != NULL ) *es= p;
  return( sign*((min*CD_SECS+sec)*CD_FRAMES+fra) );
STR2FRA_ERROR:
  if( es != NULL ) *es= s;
  return( INT_MAX );
}


char
fra2sig( int f ) {
/* convert frames to sign character */
  return( f < 0 ? '-' : ' ' );
}


int
fra2min( int f ) {
/* convert frames to minutes of msf */
  f= f < 0 ? -f : f;
  return( f/CD_FRAMES/CD_SECS );
}


int
fra2sec( int f ) {
/* convert frames to seconds of msf */
  f= f < 0 ? -f : f;
  return( (f - fra2min(f)*CD_SECS*CD_FRAMES)/CD_FRAMES );
}


int
fra2fra( int f ) {
/* convert frames to frames of msf */
  f= f < 0 ? -f : f;
  return( f - (fra2min(f)*CD_SECS+fra2sec(f))*CD_FRAMES );
}


char *
basename( const char* filename ) {
  /* from glibc-2.1 */
  char *p = strrchr (filename, '/');
  return p ? p + 1 : (char *) filename;
}


int
get_frame( char* pn, char* s, char** ep ) {
  char* p= s;
  int fra= str2fra( p, &p );
  if( fra >= INT_MAX ) {
    fprintf( ERROR_CHANNEL, "%s: Invalid frame argument `%s'. %s.\n",
             pn, s, "Value too large" );
    if( ep != NULL ) *ep = s;
    return( -1 );
  }
  if( fra < 0 ) {
    fprintf( ERROR_CHANNEL, "%s: Invalid frame argument `%s'. %s.\n",
             pn, s, "Number must not be negative" );
    if( ep != NULL ) *ep = s;
    return( -1 );
  }
  if( p == s ) {
    fprintf( ERROR_CHANNEL, "%s: Invalid frame argument `%s'.\n",
             pn, s );
    if( ep != NULL ) *ep = s;
    return( -1 );
  }
    if( ep != NULL ) *ep = p;
  return( fra );
}


int
main( int argc, char** argv) {
/*
SOLYGA --------------------
SOLYGA main(argc, argv) cddaplay

SOLYGA started      : Sun Jan 18 22:05:04 MET 1998 @beast
Hint: big-endian-format is     highest byte first
      little-endian-format is  lowest byte first   (80x86 machines)
*/
  char* fpn= *argv;
  char* pn= basename( fpn );
  int retval= RETVAL_OK;
  /***** sample stuff *****/
  int dsp_speed= DEFAULT_DSP_SPEED;
  int dsp_samplesize= DEFAULT_DSP_SAMPLESIZE;
  int dsp_stereo= DEFAULT_DSP_STEREO;
  long dsp_buffer_size;
  char *dsp_buffer;
  long real_buffer_size;
  char default_dsp_fn[]= DEFAULT_DSP_DEVICE;
  char *dsp_fn= default_dsp_fn;
  int dsp_fd;
  /***** cdr file stuff *****/
  char default_in_fn[]= "stdin";
  int in_fd= STDIN_FILENO;
  char *in_fn= default_in_fn;
  long nbr,nbw;			/* no. of bytes read/written , one buffer */
  long tnbr= 0L;		/* total no. of bytes read */
  long tnbw= 0L;		/* total no. of bytes written */
  int fra0= 0;
  int fra1= INT_MAX/CD_FRAMESIZE_RAW;	/* indicates until EOF */
  off_t off, off0, off1;	/* offsets for one file */
  off_t in_fs= 0;		/* input file size */
  /***** options stuff *****/
  int c;
  int verbose= 0;
  int swap= DEFAULT_SWAP;
  /***** misc. stuff *****/
  int tmp;
  int i;
  char* p;

  /***** process options *****/
  while((c=getopt(argc,argv,"hvVxD:F:S:"))!=EOF)
    switch(c) {
      case 'h': /* display help to HELP_CHANNEL and exit sucessfully */
        display_help( pn );
        retval= RETVAL_OK; goto DIE_NOW;
      case 'V': /* display version to VERSION_CHANNEL and exit sucessfully */
        display_version( pn );
        retval= RETVAL_OK; goto DIE_NOW;
      case 'v': /* be verbose */
        verbose++;
        break;
      case 'x': /* swap byte order (play little-endian file) */
        swap= FLIP( swap );
        break;
      case 'D': /* set dsp device */
        dsp_fn= optarg;
        break;
      case 'S': /* set sample speed (frequency) */
        dsp_speed=(int)getlongoptarg(c,optarg,MIN_DSP_SPEED,MAX_DSP_SPEED,pn);
        break;
      case 'F': /* set frames to play */
        p= optarg;
        if( (fra0=get_frame(pn,p,&p)) == -1 ) {
          retval= RETVAL_ERROR; goto DIE_NOW;
        }
        p= str_dsc( p, " \t" );
        switch( *p ) {
          case '\0':
            break;
          case '-':
            p++;
            if( *p != '\0' ) {
              p= str_dsc( p, " \t" );
              if( (fra1=get_frame(pn,p,&p)) == -1 ) {
                retval= RETVAL_ERROR; goto DIE_NOW;
              }
            }
            break;
          case '+':
            p++;
            if( *p != '\0' ) {
              p= str_dsc( p, " \t" );
              if( (fra1=get_frame(pn,p,&p)) == -1 ) {
                retval= RETVAL_ERROR; goto DIE_NOW;
              }
              fra1+= fra0;
            }
            break;
          default:
            fprintf( ERROR_CHANNEL, "%s: Invalid frame argument `%s'. %s.\n",
                     pn, optarg, "Use `f0-f1' or `f0+df'" );
            retval= RETVAL_ERROR; goto DIE_NOW;
            break;
        }
        if( fra1 < fra0 ) {
          fprintf( ERROR_CHANNEL, "%s: Invalid frame argument `%s'. %s.\n",
                   pn, optarg, "Cannot play backwards :-)" );
          retval= RETVAL_ERROR; goto DIE_NOW;
        }
        break;      
      case '?': /* refer to -h and exit unsucessfully */
        fprintf( ERROR_CHANNEL, "%s: Try `%s -h' for more information.\n",
                 pn, fpn );
        retval= RETVAL_ERROR; goto DIE_NOW;
      default : /* program error */
        fprintf( ERROR_CHANNEL, "%s: Bug detected! Send an email to %s.\n",
                pn, MY_EMAIL_ADDRESS );
        retval= RETVAL_BUG; goto DIE_NOW;
    }

/* open audio device and set parameters */
/* the sequence is important:
 * open dsp device
 * set samplesize
 * set stereo
 * set sample speed
 * get audio buffer size
 * write to soundcard
 * close dsp device
 */
  if( (dsp_fd=open(dsp_fn,O_WRONLY,0)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Can't open `%s' for writing. %s.\n",
            pn, dsp_fn, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  tmp= dsp_samplesize;
  ioctl( dsp_fd, SNDCTL_DSP_SAMPLESIZE, &dsp_samplesize );
  if( tmp != dsp_samplesize ) {
    fprintf( ERROR_CHANNEL, "%s: Can't set sample size to %d (set to %d).",
            pn, tmp, dsp_samplesize );
    if( dsp_samplesize == -1 ) {
      fprintf( ERROR_CHANNEL, " %s.\n", sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    fprintf( ERROR_CHANNEL, "\n" );
  }
  if( ioctl(dsp_fd,SNDCTL_DSP_STEREO,&dsp_stereo) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Can't set mono/stereo mode. %s.\n ",
            pn, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  tmp= dsp_speed;
  ioctl( dsp_fd, SNDCTL_DSP_SPEED, &dsp_speed );
  if( dsp_speed != tmp ) {
    fprintf( ERROR_CHANNEL,
             "%s: Can't set sample frequency to %d Hz (set to %d Hz).",
             pn, tmp, dsp_speed );
    if( dsp_speed == -1 ) {
      fprintf( ERROR_CHANNEL, " %s.\n", sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    else fprintf( ERROR_CHANNEL, "\n" );
  }
  ioctl( dsp_fd, SNDCTL_DSP_GETBLKSIZE, &dsp_buffer_size );
  if( dsp_buffer_size < 128 || dsp_buffer_size > (2*65536) ) {
    if( dsp_buffer_size == -1 )
      fprintf( ERROR_CHANNEL, "%s: Can't get audio buffer size. %s.\n",
               pn, sys_errlist[errno] );
    else
      fprintf( ERROR_CHANNEL, "%s: Invalid audio buffer size %ld.\n",
              pn, dsp_buffer_size );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  real_buffer_size= dsp_buffer_size > CD_FRAMESIZE_RAW ? dsp_buffer_size : CD_FRAMESIZE_RAW;
  if( (dsp_buffer=malloc(real_buffer_size)) == NULL ) {
    fprintf( ERROR_CHANNEL, "%s: Can't allocate %ld bytes for audio buffer.\n",
             pn, real_buffer_size );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }


/* give some info */
  if( verbose > 2 ) {
    fprintf( VERBOSE_CHANNEL, "%s: Sampling frequency %d Hz, ",
             pn, dsp_speed );
    if( dsp_stereo ) fprintf( VERBOSE_CHANNEL, "stereo, " );
    else fprintf( VERBOSE_CHANNEL, "mono, " );
    fprintf( VERBOSE_CHANNEL, "sample size %d bit.\n", dsp_samplesize );
    fprintf( VERBOSE_CHANNEL,"%s: Byte order ", pn );
    if(swap) fprintf( VERBOSE_CHANNEL,"swapped.\n" );
    else fprintf( VERBOSE_CHANNEL,"unswapped.\n" );
  }


/* open input */
  if( argc > optind ) {
    if( strcmp(argv[optind],"-") != 0 ) {
      in_fn= argv[optind];
      if( (in_fd=open(in_fn,O_RDONLY)) == -1 ) {
        fprintf( ERROR_CHANNEL, "%s: Cannot open `%s' readonly. %s.\n",
                 pn, in_fn, sys_errlist[errno] );
        retval= RETVAL_ERROR; goto DIE_NOW;
      }
    }
  }


/* lseek input */
  off0= fra0 * CD_FRAMESIZE_RAW;
  off1= fra1 * CD_FRAMESIZE_RAW;
  if( in_fd != STDIN_FILENO ) {
    if( (in_fs=lseek(in_fd,(off_t)0,SEEK_END)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot lseek `%s'. %s.\n",
               pn, in_fn, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    if( (off=lseek(in_fd,off0<in_fs?off0:in_fs,SEEK_SET)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot lseek `%s'. %s.\n",
               pn, in_fn, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    if( in_fs%CD_FRAMESIZE_RAW ) {
      fprintf( ERROR_CHANNEL, "%s: Input has non-integer number of frames!\n",
               pn );
    }
    if( fra1 == INT_MAX/CD_FRAMESIZE_RAW ) {
      fra1= in_fs/CD_FRAMESIZE_RAW;
      fra1+= in_fs%CD_FRAMESIZE_RAW ? 1 : 0;
    }
  }
  else {
    long frames= fra0;
    off= 0;
    while( frames-- > 0 ) {
      if( (nbr=my_read(in_fd,dsp_buffer,CD_FRAMESIZE_RAW)) == -1 ) {
        fprintf( ERROR_CHANNEL, "%s: Cannot read from stdin. %s.\n",
                 pn, sys_errlist[errno] );
        retval= RETVAL_ERROR; goto DIE_NOW;
      }
      off+= nbr;
      if( nbr < CD_FRAMESIZE_RAW ) break;
    } 
  }
  if( off < off0 ) {
    fprintf( ERROR_CHANNEL, "%s: Frame offset exeeds filesize of:\n", pn );
    fprintf( ERROR_CHANNEL, "%s: %ld bytes, %ld frames, %02d:%02d:%02d msf.\n",
             pn, off, off/CD_FRAMESIZE_RAW,
             fra2min(off/CD_FRAMESIZE_RAW),
             fra2sec(off/CD_FRAMESIZE_RAW),
             fra2fra(off/CD_FRAMESIZE_RAW) );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }

/* info on frames to play */
  if( verbose > 1 ) {
    fprintf( VERBOSE_CHANNEL,"%s: Playing ", pn );
    fprintf( VERBOSE_CHANNEL,"%02d:%02d:%02d - ", 
             fra2min(fra0), fra2sec(fra0), fra2fra(fra0) );
    if( fra1 == INT_MAX/CD_FRAMESIZE_RAW )
      fprintf( VERBOSE_CHANNEL,"EOF msf.\n" );
    else {
      fprintf( VERBOSE_CHANNEL,"%02d:%02d:%02d, ", 
               fra2min(fra1), fra2sec(fra1), fra2fra(fra1) );
      fprintf( VERBOSE_CHANNEL,"%02d:%02d:%02d msf.\n", 
               fra2min(fra1-fra0), fra2sec(fra1-fra0), fra2fra(fra1-fra0) );
    }
    fprintf( VERBOSE_CHANNEL,"%s: Playing %d - ", pn, fra0 );
    if( fra1 == INT_MAX/CD_FRAMESIZE_RAW )
      fprintf( VERBOSE_CHANNEL,"EOF frames.\n" );
    else
      fprintf( VERBOSE_CHANNEL,"%d, %d frames.\n", fra1, fra1-fra0 );
  }


/* playback input */
  while( 1 ) {
    if( (nbr=my_read(in_fd,dsp_buffer,dsp_buffer_size)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot read input. %s.\n",
               pn, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    if( fra1<INT_MAX/CD_FRAMESIZE_RAW  &&  tnbr+nbr>off1-off0 ) {
      nbr= off1-off0-tnbr;
      for( i=nbr; i<dsp_buffer_size; i++ ) { dsp_buffer[i]= 0x00; }
    }
    tnbr+= nbr;
    if( swap )
      for( i=0; i<nbr; i+=2 ) {
        tmp= dsp_buffer[i];
        dsp_buffer[i]= dsp_buffer[i+1];
        dsp_buffer[i+1]= tmp;
      }
    if( (nbw=write(dsp_fd,dsp_buffer,nbr)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot write to dsp device. %s.\n",
               pn, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    tnbw+= nbw;
    if( nbw < nbr ) {
      fprintf( ERROR_CHANNEL,
               "%s: Cannot flush dsp buffer, %ld bytes skipped.\n",
               pn, nbr-nbw );
    }
    if( nbr < dsp_buffer_size ) break;
  }


/* close stuff */
  if( in_fd != STDIN_FILENO  &&  close(in_fd) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot close `%s'. %s.\n",
             pn, in_fn, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  if( close(dsp_fd) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot close `%s'. %s.\n",
    pn, dsp_fn, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }


/* final infos */
  if( tnbr%CD_FRAMESIZE_RAW ) {
    fprintf( ERROR_CHANNEL, "%s: Input had non-integer number of frames!\n",
             pn );
  }
  if( verbose ) {
    int fras= tnbr/CD_FRAMESIZE_RAW;
    fprintf( VERBOSE_CHANNEL,"%s: Played ", pn );
    fprintf( VERBOSE_CHANNEL,"%02d:%02d:%02d - ", 
             fra2min(fra0), fra2sec(fra0), fra2fra(fra0) );
    fprintf( VERBOSE_CHANNEL,"%02d:%02d:%02d, ", 
             fra2min(fra0+fras), fra2sec(fra0+fras), fra2fra(fra0+fras) );
    fprintf( VERBOSE_CHANNEL,"%02d:%02d:%02d msf.\n", 
             fra2min(fras), fra2sec(fras), fra2fra(fras) );
    fprintf( VERBOSE_CHANNEL,"%s: Played %d - %d, %d frames.\n",
             pn, fra0, fra0+fras, fras );
  }


DIE_NOW:
  if( retval != RETVAL_OK ) {
    fprintf( VERBOSE_CHANNEL, "%s: %ld/%ld bytes read/written.\n",
             pn, tnbr, tnbw );
  }
  close( in_fd );
  close( dsp_fd );
  exit( retval );
}
